<?php
header('Content-Type: application/json');
// ── Includes ──────────────────────────────────────────────────────────
require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';
// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
// Validate input
if (!isset($input['mac']) || !isset($input['endpoints'])) {
    http_response_code(400);
    echo json_encode(['error' => 'MAC address and endpoints are required']);
    exit;
}
$mac = trim($input['mac']);
$endpoints = $input['endpoints'];
// Validate MAC address format
if (!preg_match('/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/', $mac)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid MAC address format']);
    exit;
}
// Validate endpoints is an array and not empty
if (!is_array($endpoints) || empty($endpoints)) {
    http_response_code(400);
    echo json_encode(['error' => 'At least one endpoint must be selected']);
    exit;
}
// Join endpoints with commas
$endpointString = implode(',', $endpoints);
try {
    // Database connection
    $db = new PDO(
        "mysql:host={$host_gscroll};dbname={$database_gscroll};charset=utf8mb4",
        $username_gscroll,
        $password_gscroll
    );
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Check if MAC address exists
    $checkStmt = $db->prepare("SELECT COUNT(*) FROM glance_classic WHERE MAC = :mac");
    $checkStmt->execute(['mac' => $mac]);
    $exists = $checkStmt->fetchColumn() > 0;
    
    if (!$exists) {
        // Insert new record if MAC doesn't exist
        $stmt = $db->prepare("INSERT INTO glance_classic (MAC, EndPoint, needsUpdate) VALUES (:mac, :endpoint, 1)");
        $stmt->execute([
            'mac' => $mac,
            'endpoint' => $endpointString
        ]);
    } else {
        // Update existing record
        $stmt = $db->prepare("UPDATE glance_classic SET EndPoint = :endpoint, needsUpdate = 1 WHERE MAC = :mac");
        $stmt->execute([
            'endpoint' => $endpointString,
            'mac' => $mac
        ]);
    }
    
    // Also update glance_classic_profile table to set needsUpdate to 'T'
    $profileStmt = $db->prepare("UPDATE glance_classic_profile SET needsUpdate = 'T' WHERE MAC = :mac");
    $profileStmt->execute(['mac' => $mac]);
    
    // Check if the operation was successful
    if ($stmt->rowCount() > 0 || $exists) {
        echo json_encode([
            'success' => true,
            'message' => $exists ? 'Endpoint updated successfully' : 'New device added successfully',
            'selected_count' => count($endpoints),
            'profile_updated' => $profileStmt->rowCount() > 0
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'No changes were made']);
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    exit;
}
?>