<?php
/**
 * Save Selections API Endpoint
 */

require_once '../../lib/database.php';
require_once '../../lib/functions.php';

header('Content-Type: application/json');

// Check authentication
if (!isAuthenticated('jackmorton')) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Handle POST requests only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['mac']) || !isset($input['selections'])) {
    echo json_encode(['success' => false, 'error' => 'MAC address and selections required']);
    exit;
}

$mac = strtoupper(sanitizeInput($input['mac']));
$selections = $input['selections'];

// Validate MAC format
if (!preg_match('/^([0-9A-F]{2}[:-]){5}([0-9A-F]{2})$/i', $mac)) {
    echo json_encode(['success' => false, 'error' => 'Invalid MAC address format']);
    exit;
}

// Validate selections is array
if (!is_array($selections)) {
    echo json_encode(['success' => false, 'error' => 'Selections must be an array']);
    exit;
}

// Limit selections to 6 items
if (count($selections) > 6) {
    echo json_encode(['success' => false, 'error' => 'Maximum 6 selections allowed']);
    exit;
}

// Ensure all selections are integers
$selections = array_map('intval', $selections);
$selections = array_unique($selections);
$selections = array_values($selections); // Re-index

// Save selections
if (saveDeviceSelections($mac, $selections)) {
    echo json_encode([
        'success' => true,
        'message' => 'Selections saved successfully',
        'count' => count($selections)
    ]);
} else {
    echo json_encode(['success' => false, 'error' => 'Failed to save selections']);
}