<?php
/**
 * GLANCE Mobile - Jack Morton Entry Point
 */

// Include shared libraries
require_once __DIR__ . '/../lib/database.php';
require_once __DIR__ . '/../lib/functions.php';
require_once __DIR__ . '/../lib/image_processing.php';

// Load company configuration
$companyName = 'jackmorton';
$config = getCompanyConfig($companyName);

if (!$config) {
    die('Company configuration not found. Please check config.json file.');
}

// Initialize session company
if (!isset($_SESSION['company'])) {
    $_SESSION['company'] = $companyName;
}

// Get current page
$page = $_GET['page'] ?? 'login';

// Check authentication for protected pages
$protectedPages = ['home', 'content-library', 'upload', 'select-device'];
if (in_array($page, $protectedPages) && !isAuthenticated($companyName)) {
    header('Location: ?page=login');
    exit;
}

// Handle logout
if ($page === 'logout') {
    session_destroy();
    setcookie('mac_address', '', time() - 3600, '/');
    setcookie('mac_mode', '', time() - 3600, '/');
    header('Location: ?page=login');
    exit;
}

// After login, if authenticated but no page specified, go to home
if (!$page && isAuthenticated($companyName)) {
    header('Location: ?page=home');
    exit;
}

// If authenticated and trying to access login, redirect to home
if ($page === 'login' && isAuthenticated($companyName)) {
    header('Location: ?page=home');
    exit;
}

// Get current MAC if authenticated
$currentMAC = isAuthenticated($companyName) ? getCurrentMAC() : null;

// On home page, handle MAC setup
if ($page === 'home' && isAuthenticated($companyName)) {
    // For pre-controlled only mode, auto-assign first MAC if none set
    if (!$currentMAC && $config['features']['mac_modes']['precontrolled'] && !$config['features']['mac_modes']['entered']) {
        $availableMACs = getCompanyMACs($config['comp_group']);
        if (!empty($availableMACs)) {
            setCurrentMAC($availableMACs[0]['MAC'], 'precontrolled');
            $currentMAC = $availableMACs[0]['MAC'];
        }
    }
}

// Remove select-device from valid pages
$validPages = ['login', 'home', 'content-library', 'logout'];
if (!in_array($page, $validPages)) {
    $page = 'login';
}

// Get current MAC if authenticated
$currentMAC = isAuthenticated($companyName) ? getCurrentMAC() : null;

// Redirect to device selection if no MAC is set
if ($page === 'home' && !$currentMAC) {
    header('Location: ?page=select-device');
    exit;
}

// Page routing
$pageFile = __DIR__ . '/pages/' . $page . '.php';
if (!file_exists($pageFile)) {
    $page = 'login';
    $pageFile = __DIR__ . '/pages/login.php';
}

// Start output
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black">
    <title><?php echo htmlspecialchars($config['company_name']); ?> - GLANCE Mobile</title>
    
    <!-- Base styles -->
    <link rel="stylesheet" href="../lib/css/base.css">
    
    <!-- Company theme -->
    <style>
        :root {
            <?php foreach ($config['theme'] as $key => $value): ?>
                <?php if (strpos($key, 'color') !== false): ?>
                    --<?php echo str_replace('_', '-', $key); ?>: <?php echo $value; ?>;
                <?php endif; ?>
            <?php endforeach; ?>
        }
    </style>
    
    <!-- Company config for JavaScript -->
    <script id="company-config" type="application/json">
        <?php echo json_encode($config); ?>
    </script>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <header class="header">
            <?php if ($page === 'home' && $currentMAC): ?>
                <div class="hamburger" id="hamburger">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            <?php else: ?>
                <div style="width: 24px;"></div>
            <?php endif; ?>
            
            <div class="header-logo">
                <?php if (file_exists(__DIR__ . '/../lib/glance_assets/glance-logo.png')): ?>
                    <img src="../lib/glance_assets/glance-logo.png" alt="GLANCE" style="height: 30px; width: auto;">
                <?php else: ?>
                    <span class="header-title">GLANCE</span>
                <?php endif; ?>
            </div>
            
            <div style="width: 24px;"></div>
        </header>
        
        <!-- Side Menu -->
        <?php if ($page === 'home' && $currentMAC): ?>
        <nav class="side-menu" id="side-menu">
            <h2 style="margin-bottom: 20px;">Settings</h2>
            <a href="#" class="menu-item" id="clear-company-selections">Clear Company Selections</a>
            <a href="#" class="menu-item" id="clear-all-selections">Clear All Selections</a>
            <a href="?page=select-device" class="menu-item">Change Device</a>
            <a href="?page=logout" class="menu-item">Logout</a>
        </nav>
        <div class="menu-overlay" id="menu-overlay"></div>
        <?php endif; ?>
        
        <!-- Main Content -->
        <main class="content">
            <?php include $pageFile; ?>
        </main>
    </div>
    
    <!-- Main JavaScript -->
    <script src="../lib/js/app.js"></script>
</body>
</html>