<?php
/**
 * GLANCE Mobile - Shared Functions
 */

require_once __DIR__ . '/database.php';

/**
 * Check if user is authenticated with company key
 * @param string $companyName Company identifier
 * @return bool
 */
function isAuthenticated($companyName) {
    if (!isset($_SESSION['authenticated']) || !isset($_SESSION['company'])) {
        return false;
    }
    return $_SESSION['authenticated'] === true && $_SESSION['company'] === $companyName;
}

/**
 * Authenticate user with company key
 * @param string $companyKey User-provided key
 * @param array $config Company configuration
 * @return bool
 */
function authenticateUser($companyKey, $config) {
    return isset($config['company_key']) && $config['company_key'] === $companyKey;
}

/**
 * Get current MAC address from session/cookie
 * @return string|null
 */
function getCurrentMAC() {
    // Check session first
    if (isset($_SESSION['mac_address'])) {
        return $_SESSION['mac_address'];
    }
    
    // Check cookie
    if (isset($_COOKIE['mac_address'])) {
        $_SESSION['mac_address'] = $_COOKIE['mac_address'];
        return $_COOKIE['mac_address'];
    }
    
    return null;
}

/**
 * Set current MAC address
 * @param string $mac MAC address
 * @param string $mode 'entered' or 'precontrolled'
 */
function setCurrentMAC($mac, $mode) {
    $_SESSION['mac_address'] = $mac;
    $_SESSION['mac_mode'] = $mode;
    $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
    setcookie('mac_address', $mac, time() + (86400 * 30), '/', '', $secure, true); // 30 days
    setcookie('mac_mode', $mode, time() + (86400 * 30), '/', '', $secure, true);
}

/**
 * Get current MAC mode
 * @return string 'entered' or 'precontrolled'
 */
function getMACMode() {
    if (isset($_SESSION['mac_mode'])) {
        return $_SESSION['mac_mode'];
    }
    if (isset($_COOKIE['mac_mode'])) {
        return $_COOKIE['mac_mode'];
    }
    return 'entered';
}

/**
 * Get device selections from database
 * @param string $mac MAC address
 * @return array Selections data
 */
function getDeviceSelections($mac) {
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM glance_scroll WHERE MAC = ? LIMIT 1");
        $stmt->execute([$mac]);
        $result = $stmt->fetch();
        
        if ($result) {
            $result['Selections'] = json_decode($result['Selections'], true) ?: [];
            return $result;
        }
        
        // Return default structure if not found
        return [
            'MAC' => $mac,
            'Selections' => [],
            'NeedsUpdate' => true
        ];
    } catch (PDOException $e) {
        error_log("Failed to get device selections: " . $e->getMessage());
        return [
            'MAC' => $mac,
            'Selections' => [],
            'NeedsUpdate' => true
        ];
    }
}

/**
 * Save device selections to database
 * @param string $mac MAC address
 * @param array $selections Array of selection IDs
 * @return bool Success status
 */
function saveDeviceSelections($mac, $selections) {
    try {
        $db = getDB();
        $stmt = $db->prepare("
            INSERT INTO glance_scroll (MAC, Selections, NeedsUpdate) 
            VALUES (?, ?, TRUE)
            ON DUPLICATE KEY UPDATE 
                Selections = VALUES(Selections),
                NeedsUpdate = TRUE,
                updated_at = CURRENT_TIMESTAMP
        ");
        return $stmt->execute([$mac, json_encode($selections)]);
    } catch (PDOException $e) {
        error_log("Failed to save device selections: " . $e->getMessage());
        return false;
    }
}

/**
 * Get content library for company
 * @param string $compGroup Company group
 * @param string|null $contentType Filter by type (image, text, api)
 * @return array Content items
 */
function getContentLibrary($compGroup, $contentType = null) {
    try {
        $db = getDB();
        $query = "SELECT * FROM content_library WHERE comp_group = ? AND is_active = 1";
        $params = [$compGroup];
        
        if ($contentType) {
            $query .= " AND content_type = ?";
            $params[] = $contentType;
        }
        
        $query .= " ORDER BY created_at DESC";
        
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        $results = $stmt->fetchAll();
        
        // Decode JSON data for each item
        foreach ($results as &$item) {
            $item['content_data'] = json_decode($item['content_data'], true) ?: [];
        }
        
        return $results;
    } catch (PDOException $e) {
        error_log("Failed to get content library: " . $e->getMessage());
        return [];
    }
}

/**
 * Add content to library
 * @param string $compGroup Company group
 * @param string $type Content type
 * @param string $name Content name
 * @param array $data Content data
 * @param string|null $path File path for images
 * @return int|false Insert ID or false on failure
 */
function addContentToLibrary($compGroup, $type, $name, $data, $path = null) {
    try {
        $db = getDB();
        $stmt = $db->prepare("
            INSERT INTO content_library (comp_group, content_type, content_name, content_path, content_data) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$compGroup, $type, $name, $path, json_encode($data)]);
        return $db->lastInsertId();
    } catch (PDOException $e) {
        error_log("Failed to add content to library: " . $e->getMessage());
        return false;
    }
}

/**
 * Check daily upload limit
 * @param string $compGroup Company group
 * @return array ['allowed' => bool, 'count' => int, 'limit' => int]
 */
function checkUploadLimit($compGroup) {
    try {
        $db = getDB();
        $today = date('Y-m-d');
        $limit = 100; // TODO: Make this configurable per company
        
        $stmt = $db->prepare("
            SELECT upload_count 
            FROM upload_tracking 
            WHERE comp_group = ? AND upload_date = ?
        ");
        $stmt->execute([$compGroup, $today]);
        $result = $stmt->fetch();
        
        $count = $result ? (int)$result['upload_count'] : 0;
        return ['allowed' => $count < $limit, 'count' => $count, 'limit' => $limit];
    } catch (PDOException $e) {
        error_log("Failed to check upload limit: " . $e->getMessage());
        return ['allowed' => true, 'count' => 0, 'limit' => 100];
    }
}

/**
 * Increment upload count
 * @param string $compGroup Company group
 * @return bool Success status
 */
function incrementUploadCount($compGroup) {
    try {
        $db = getDB();
        $today = date('Y-m-d');
        
        $stmt = $db->prepare("
            INSERT INTO upload_tracking (comp_group, upload_date, upload_count) 
            VALUES (?, ?, 1)
            ON DUPLICATE KEY UPDATE upload_count = upload_count + 1
        ");
        return $stmt->execute([$compGroup, $today]);
    } catch (PDOException $e) {
        error_log("Failed to increment upload count: " . $e->getMessage());
        return false;
    }
}

/**
 * Clear selections for company or all
 * @param string $mac MAC address
 * @param string $compGroup Company group
 * @param bool $allSelections Clear all selections or just company ones
 * @return bool Success status
 */
function clearSelections($mac, $compGroup, $allSelections = false) {
    try {
        if ($allSelections) {
            // Clear all selections for this MAC
            return saveDeviceSelections($mac, []);
        } else {
            // Clear only company-specific selections
            $device = getDeviceSelections($mac);
            $selections = $device['Selections'];
            
            // Get company content IDs
            $companyContent = getContentLibrary($compGroup);
            $companyContentIds = array_column($companyContent, 'id');
            
            // Filter out company selections
            $newSelections = array_filter($selections, function($id) use ($companyContentIds) {
                return !in_array($id, $companyContentIds);
            });
            
            return saveDeviceSelections($mac, array_values($newSelections));
        }
    } catch (Exception $e) {
        error_log("Failed to clear selections: " . $e->getMessage());
        return false;
    }
}

/**
 * Generate unique filename
 * @param string $baseName Base filename without extension
 * @param string $extension File extension
 * @param string $uploadDir Upload directory
 * @return string Unique filename with extension
 */
function generateUniqueFilename($baseName, $extension, $uploadDir) {
    $counter = 0;
    $filename = $baseName;
    
    while (file_exists($uploadDir . '/' . $filename . '.' . $extension)) {
        $counter++;
        $filename = $baseName . '_' . $counter;
    }
    
    return $filename . '.' . $extension;
}