<?php
// debug_dashboard_flow.php - Debug the exact dashboard flow
require_once '../../config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<pre>";
echo "=== DASHBOARD FLOW DEBUG ===\n\n";

// Connect
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Helper functions (exact copy from dashboard_api.php)
function binaryToIp($binaryIp) {
    if (strlen($binaryIp) === 4) {
        $u = unpack('N', $binaryIp);
        return long2ip($u[1]);
    }
    return null;
}

function batchGetLocations($ips) {
    $valid = array_filter($ips, function($ip){
        return filter_var($ip, FILTER_VALIDATE_IP)
            && filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE);
    });
    if (!$valid) return [];

    $endpoint = 'http://ip-api.com/batch?fields=status,query,lat,lon,city,regionName';
    $opts = [
        'http'=>[
            'method'=>'POST',
            'header'=>'Content-Type: application/json',
            'content'=>json_encode(array_values($valid)),
            'timeout'=>5
        ]
    ];
    $ctx = stream_context_create($opts);
    $resp = @file_get_contents($endpoint, false, $ctx);
    if (!$resp) return [];

    $out = [];
    foreach (json_decode($resp, true) as $r) {
        if (!empty($r['status']) && $r['status'] === 'success') {
            $out[$r['query']] = [
                'state'=> $r['regionName'] ?? 'Unknown',
                'city' => $r['city']      ?? '',
                'lat'  => $r['lat']       ?? null,
                'lon'  => $r['lon']       ?? null,
            ];
        }
    }
    return $out;
}

// Run the EXACT same query as dashboard
echo "1. Running dashboard query...\n";
$sql = "
  SELECT
    t1.mac_address,
    t1.ip_address,
    t1.session_count,
    t1.last_ping_time,
    sm.SN
  FROM glance_scroll_tracking t1
  JOIN (
    SELECT mac_address, MAX(session_count) AS max_session
    FROM glance_scroll_tracking
    GROUP BY mac_address
  ) t2 ON t1.mac_address = t2.mac_address
       AND t1.session_count = t2.max_session
  LEFT JOIN serial_mapping sm
    ON t1.mac_address = sm.MAC
  GROUP BY t1.mac_address
  ORDER BY t1.last_ping_time DESC
  LIMIT 5
";

$result = mysqli_query($con, $sql);
if (!$result) {
    die("Query failed: " . mysqli_error($con));
}

echo "✓ Query executed successfully\n\n";

$devices = [];
$ipList = [];

echo "2. Processing devices (following exact dashboard flow)...\n\n";

// Step 1: Collect IPs + raw rows (exact copy)
$buffer = [];
$rowNum = 0;
while ($r = mysqli_fetch_assoc($result)) {
    $rowNum++;
    echo "Device #$rowNum:\n";
    echo "  MAC: {$r['mac_address']}\n";
    echo "  Binary IP length: " . strlen($r['ip_address']) . "\n";
    echo "  Binary IP (hex): " . bin2hex($r['ip_address']) . "\n";
    
    $ip = binaryToIp($r['ip_address']);
    echo "  Converted IP: " . ($ip ?: "NULL") . "\n";
    
    if ($ip) {
        $ipList[] = $ip;
        echo "  ✓ Added to ipList\n";
    } else {
        echo "  ❌ IP conversion failed - not added to ipList\n";
    }
    
    $buffer[] = ['row' => $r, 'ip' => $ip];
    echo "\n";
}

echo "3. IP List Summary:\n";
echo "  Total IPs collected: " . count($ipList) . "\n";
echo "  Unique IPs: " . count(array_unique($ipList)) . "\n";
echo "  IPs: " . implode(", ", array_unique($ipList)) . "\n\n";

// Step 2: Lookup geolocation
echo "4. Calling batchGetLocations...\n";
$locs = batchGetLocations($ipList);
echo "  Locations returned: " . count($locs) . "\n";
if (!empty($locs)) {
    echo "  Location data:\n";
    foreach ($locs as $ip => $loc) {
        echo "    $ip => {$loc['city']}, {$loc['state']}\n";
    }
} else {
    echo "  ⚠️ NO LOCATIONS RETURNED!\n";
}
echo "\n";

// Step 3: Build devices array (exact copy)
echo "5. Building devices array...\n\n";
foreach ($buffer as $idx => $entry) {
    $r  = $entry['row'];
    $ip = $entry['ip'];
    
    echo "Processing buffer entry #" . ($idx + 1) . ":\n";
    echo "  MAC: {$r['mac_address']}\n";
    echo "  IP: " . ($ip ?: "NULL") . "\n";
    
    // Geo info logic (exact copy)
    $state = 'Unknown';
    $city  = '';
    $lat   = null;
    $lon   = null;
    
    echo "  Checking location for IP '$ip'...\n";
    echo "  isset(\$locs['$ip']): " . (isset($locs[$ip]) ? "YES" : "NO") . "\n";
    
    if (isset($locs[$ip])) {
        $state = $locs[$ip]['state'];
        $city  = $locs[$ip]['city'];
        $lat   = $locs[$ip]['lat'];
        $lon   = $locs[$ip]['lon'];
        echo "  ✓ Location found: $city, $state\n";
    } elseif ($ip && !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE)) {
        $state = 'Local Network';
        echo "  → IP is private/local network\n";
    } else {
        echo "  ❌ No location found - will show as 'Unknown'\n";
    }
    
    echo "  Final state: $state\n\n";
}

// Test with specific problem case
echo "6. Testing edge cases...\n";

// Check for NULL or empty IPs
$nullCheckSql = "
    SELECT COUNT(*) as null_count 
    FROM glance_scroll_tracking 
    WHERE ip_address IS NULL OR LENGTH(ip_address) = 0
";
$nullResult = mysqli_query($con, $nullCheckSql);
$nullRow = mysqli_fetch_assoc($nullResult);
echo "  Records with NULL/empty IP: " . $nullRow['null_count'] . "\n";

// Check for IPs with wrong binary length
$lengthCheckSql = "
    SELECT LENGTH(ip_address) as len, COUNT(*) as count 
    FROM glance_scroll_tracking 
    WHERE ip_address IS NOT NULL
    GROUP BY LENGTH(ip_address)
";
$lengthResult = mysqli_query($con, $lengthCheckSql);
echo "  IP binary lengths in database:\n";
while ($row = mysqli_fetch_assoc($lengthResult)) {
    echo "    Length {$row['len']}: {$row['count']} records\n";
}

// Check for duplicate processing
echo "\n7. Checking for duplicate IP processing...\n";
$dupIps = array_count_values($ipList);
foreach ($dupIps as $ip => $count) {
    if ($count > 1) {
        echo "  IP $ip appears $count times\n";
    }
}

mysqli_close($con);
echo "\n=== END OF DEBUG ===\n";
echo "</pre>";
?>