<?php
// update_and_debug.php

// Enable error reporting (development only)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include configuration file (defines $host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll)
require_once '../config.php';

/**
 * Helper function to extract an event ID from a $ref URL.
 */
function parseEventId($refUrl) {
    $path = parse_url($refUrl, PHP_URL_PATH);
    $parts = explode('/', trim($path, '/'));
    return end($parts);
}

//------------------------------------------------
// 1. Connect to the database and update API_DB timestamp
//------------------------------------------------
$con = new mysqli($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if ($con->connect_error) {
    die("Connection failed: " . $con->connect_error);
}
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$update_stmt = $con->prepare("UPDATE API_DB SET TIME_UPDATED = ? WHERE API = 'NASCAR'");
if (!$update_stmt) {
    die("Prepare failed (update API_DB): " . $con->error);
}
$update_stmt->bind_param("s", $time_updated);
if (!$update_stmt->execute()) {
    die("Execute failed (update API_DB): " . $update_stmt->error);
}
$update_stmt->close();
echo "<p>API_DB updated with time: {$time_updated}</p>";

//------------------------------------------------
// 2. Ensure extra columns and keys exist
//------------------------------------------------
// (a) Ensure column "racer_number" exists
$colCheck = $con->query("SHOW COLUMNS FROM race_list LIKE 'racer_number'");
if ($colCheck->num_rows == 0) {
    $alter_sql = "ALTER TABLE race_list ADD COLUMN racer_number INT";
    if ($con->query($alter_sql) === TRUE) {
        echo "<p>Column 'racer_number' added successfully.</p>";
    } else {
        echo "<p>Error adding 'racer_number': " . $con->error . "</p>";
    }
}
// (b) Ensure column "unique_key" exists
$colCheck2 = $con->query("SHOW COLUMNS FROM race_list LIKE 'unique_key'");
if ($colCheck2->num_rows == 0) {
    $alter_sql2 = "ALTER TABLE race_list ADD COLUMN unique_key INT";
    if ($con->query($alter_sql2) === TRUE) {
        echo "<p>Column 'unique_key' added successfully.</p>";
    } else {
        echo "<p>Error adding 'unique_key': " . $con->error . "</p>";
    }
}
// (c) Ensure unique key on (event_in, athlete_id) exists
$keyCheck = $con->query("SHOW INDEX FROM race_list WHERE Key_name = 'unique_event_athlete'");
if ($keyCheck->num_rows == 0) {
    $unique_key_sql = "ALTER TABLE race_list ADD UNIQUE KEY unique_event_athlete (event_in, athlete_id)";
    if (!$con->query($unique_key_sql)) {
        echo "<p>Error adding unique key: " . $con->error . "</p>";
    } else {
        echo "<p>Unique key 'unique_event_athlete' added successfully.</p>";
    }
} else {
    echo "<p>Unique key 'unique_event_athlete' already exists.</p>";
}

//------------------------------------------------
// 3. Fetch the NASCAR scoreboard API JSON using cURL
//------------------------------------------------
$scoreboard_url = "https://site.api.espn.com/apis/site/v2/sports/racing/nascar-premier/scoreboard";
$ch = curl_init($scoreboard_url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT        => 30,
]);
$scoreboard_response = curl_exec($ch);
if (curl_errno($ch)) {
    die("cURL Error (scoreboard): " . curl_error($ch));
}
curl_close($ch);
$scoreboard_data = json_decode($scoreboard_response, true);
if (!$scoreboard_data) {
    die("Error decoding JSON from scoreboard API.");
}
echo "<p>Scoreboard data retrieved.</p>";

//------------------------------------------------
// 4. Build a normalized events list from the calendar array
//------------------------------------------------
if (isset($scoreboard_data['leagues'][0]['calendar']) && is_array($scoreboard_data['leagues'][0]['calendar'])) {
    $calendarEvents = $scoreboard_data['leagues'][0]['calendar'];
} else {
    die("No calendar events found in API data.");
}
$events = [];
foreach ($calendarEvents as $e) {
    if (!isset($e['startDate']) || !isset($e['label'])) continue;
    $eventTime = new DateTime($e['startDate'], new DateTimeZone("UTC"));
    $label = $e['label'];
    $eventIdExtracted = "";
    if (isset($e['event']['$ref'])) {
        $eventIdExtracted = parseEventId($e['event']['$ref']);
    }
    $events[] = [
        'label' => $label,
        'date'  => $e['startDate'],
        'id'    => $eventIdExtracted
    ];
}
echo "<p>" . count($events) . " events processed from the calendar.</p>";

//------------------------------------------------
// 5. Debug: Prepare debug information for scheduled events
//------------------------------------------------
$now = new DateTime("now", new DateTimeZone("UTC"));
$debugEvents = [];
foreach ($events as $e) {
    $eventTime = new DateTime($e['date'], new DateTimeZone("UTC"));
    $diffSeconds = $eventTime->getTimestamp() - $now->getTimestamp();
    $diffHuman = gmdate("H:i:s", abs($diffSeconds));
    $debugEvents[] = [
        'label' => $e['label'],
        'start_time' => $eventTime->format('Y-m-d\TH:i:s\Z'),
        'diff_seconds' => $diffSeconds,
        'diff_human' => ($diffSeconds >= 0 ? "+" : "-") . $diffHuman
    ];
}

//------------------------------------------------
// 6. Find most recent event with competitor data first
//------------------------------------------------
$eventWithCompetitors = null;
$selectedEventFromScoreboard = null;

// First check events in scoreboard for competitor data
if (isset($scoreboard_data['events']) && is_array($scoreboard_data['events'])) {
    // Sort events by date (most recent first)
    usort($scoreboard_data['events'], function($a, $b) {
        $dateA = isset($a['date']) ? strtotime($a['date']) : 0;
        $dateB = isset($b['date']) ? strtotime($b['date']) : 0;
        return $dateB - $dateA; // Most recent first
    });
    
    // Find most recent event with competitor data
    foreach ($scoreboard_data['events'] as $event) {
        if (isset($event['competitions'][0]['competitors']) && 
            count($event['competitions'][0]['competitors']) > 0) {
            $eventWithCompetitors = $event;
            $selectedEventFromScoreboard = $event;
            break;
        }
    }
}

// If we found an event with competitors, use that
if ($eventWithCompetitors) {
    $eventId = $eventWithCompetitors['id'];
    $latestRaceLabel = $eventWithCompetitors['name'] ?? $eventWithCompetitors['shortName'] ?? 'Unknown Race';
    $latestRaceTime = $eventWithCompetitors['date'] ?? 'Unknown Time';
    $competitorSource = $eventWithCompetitors['competitions'][0]['competitors'] ?? [];
    $selectedEventTime = new DateTime($latestRaceTime, new DateTimeZone("UTC"));
    echo "<p style='color:green; font-weight:bold;'>Using most recent event with competitor data: " . htmlspecialchars($latestRaceLabel) . " at " . htmlspecialchars($latestRaceTime) . "</p>";
} else {
    // No event with competitors found, fallback to next upcoming event
    $upcoming = array_filter($events, function($e) use ($now) {
        $eventTime = new DateTime($e['date'], new DateTimeZone("UTC"));
        return $eventTime > $now;
    });
    
    if (!empty($upcoming)) {
        usort($upcoming, function($a, $b) {
             return strtotime($a['date']) - strtotime($b['date']);
        });
        $selected_event = $upcoming[0];
        echo "<p style='color:orange; font-weight:bold;'>No event with competitor data found. Using next upcoming event.</p>";
    } else {
        // If no upcoming events, choose the most recent past event.
        usort($events, function($a, $b) {
             return strtotime($b['date']) - strtotime($a['date']);
        });
        $selected_event = $events[0];
        echo "<p style='color:orange; font-weight:bold;'>No event with competitor data found. Using most recent past event.</p>";
    }
    
    $eventId = $selected_event['id'] ?? '';
    $latestRaceLabel = $selected_event['label'] ?? 'Unknown Race';
    $latestRaceTime = $selected_event['date'] ?? 'Unknown Time';
    $selectedEventTime = new DateTime($latestRaceTime, new DateTimeZone("UTC"));
    
    // Try to get competitor data for the selected event
    $competitorSource = [];
    if (isset($scoreboard_data['events']) && is_array($scoreboard_data['events'])) {
        foreach ($scoreboard_data['events'] as $ev) {
            if (isset($ev['id']) && $ev['id'] === $eventId) {
                if (isset($ev['competitions'][0]['competitors'])) {
                    $competitorSource = $ev['competitions'][0]['competitors'];
                }
                $selectedEventFromScoreboard = $ev;
                break;
            }
        }
    }
}

//------------------------------------------------
// 7. Debug: Calculate countdown until selected race
//------------------------------------------------
$countdown = "";
if ($selectedEventTime > $now) {
    $diffSeconds = $selectedEventTime->getTimestamp() - $now->getTimestamp();
    $days = floor($diffSeconds / 86400);
    $hours = floor(($diffSeconds % 86400) / 3600);
    $minutes = floor(($diffSeconds % 3600) / 60);
    $seconds = $diffSeconds % 60;
    $countdown = "{$days} days, {$hours} hours, {$minutes} minutes, {$seconds} seconds";
} else {
    $countdown = "Selected event is in the past.";
}
echo "<p>Countdown: " . $countdown . "</p>";
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>NASCAR Race Data Debug</title>
    <style>
        body { font-family: Arial, sans-serif; }
        table { border-collapse: collapse; width: 90%; margin: 20px auto; }
        th, td { border: 1px solid #333; padding: 8px; text-align: center; }
        th { background-color: #f0f0f0; }
        .debug-table { margin-bottom: 40px; }
    </style>
</head>
<body>
    <h1 style="text-align:center;">Debug Information</h1>

    <h2 style="text-align:center;">Current UTC Time</h2>
    <p style="text-align:center; font-weight:bold;"><?php echo $now->format('Y-m-d\TH:i:s\Z'); ?></p>

    <h2 style="text-align:center;">Scheduled Events</h2>
    <table class="debug-table">
        <thead>
            <tr>
                <th>Label</th>
                <th>Start Time (UTC)</th>
                <th>Time Diff (sec)</th>
                <th>Time Diff (H:M:S)</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($debugEvents as $debug): ?>
            <tr>
                <td><?= htmlspecialchars($debug['label']) ?></td>
                <td><?= htmlspecialchars($debug['start_time']) ?></td>
                <td><?= htmlspecialchars($debug['diff_seconds']) ?></td>
                <td><?= htmlspecialchars($debug['diff_human']) ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <h2 style="text-align:center;">Selected Race</h2>
    <table class="debug-table">
        <thead>
            <tr>
                <th>Label</th>
                <th>Start Time (UTC)</th>
                <th>Event ID</th>
                <th>Countdown</th>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td><?= htmlspecialchars($latestRaceLabel) ?></td>
                <td><?= htmlspecialchars($latestRaceTime) ?></td>
                <td><?= htmlspecialchars($eventId) ?></td>
                <td><?= htmlspecialchars($countdown) ?></td>
            </tr>
        </tbody>
    </table>

    <?php
    //------------------------------------------------
    // 8. Filter competitors to only include those with an 'order' field
    //------------------------------------------------
    $activeCompetitors = [];
    foreach ($competitorSource as $competitor) {
        // Only include competitors with an 'order' field (they're actually racing)
        if (isset($competitor['order'])) {
            $activeCompetitors[] = $competitor;
        }
    }
    
    // Sort active competitors by their order value
    usort($activeCompetitors, function($a, $b) {
        return ($a['order'] ?? 999) - ($b['order'] ?? 999);
    });
    
    echo "<p style='text-align:center; font-weight:bold;'>Active competitor count: " . count($activeCompetitors) . "</p>";

    //------------------------------------------------
    // 9. Prepare final competitor list (exactly 16 entries)
    //------------------------------------------------
    $finalCompetitors = [];
    
    // First, add all active competitors
    foreach ($activeCompetitors as $competitor) {
        $finalCompetitors[] = $competitor;
    }
    
    // If we have less than 16, pad with default entries
    $currentCount = count($finalCompetitors);
    if ($currentCount < 16) {
        echo "<p style='text-align:center; color:orange;'>Only {$currentCount} active competitors. Padding with " . (16 - $currentCount) . " default entries.</p>";
        for ($i = $currentCount; $i < 16; $i++) {
            $finalCompetitors[] = ['default' => true];
        }
    } elseif ($currentCount > 16) {
        // If we have more than 16, take only the first 16
        $finalCompetitors = array_slice($finalCompetitors, 0, 16);
        echo "<p style='text-align:center; color:orange;'>More than 16 competitors found. Using only the first 16.</p>";
    }

    //------------------------------------------------
    // Debug: Show final competitors array
    //------------------------------------------------
    echo "<h3>Final Competitors (16 entries):</h3>";
    echo "<pre>";
    foreach ($finalCompetitors as $idx => $comp) {
        if (isset($comp['default'])) {
            echo ($idx + 1) . ". [DEFAULT ENTRY]\n";
        } else {
            $name = $comp['athlete']['shortName'] ?? $comp['athlete']['displayName'] ?? 'Unknown';
            $order = $comp['order'] ?? 'N/A';
            echo ($idx + 1) . ". Order: {$order}, Name: {$name}, ID: {$comp['id']}\n";
        }
    }
    echo "</pre>";

    //------------------------------------------------
    // 10. Delete all existing rows from race_list
    //------------------------------------------------
    $deleteQuery = "DELETE FROM race_list";
    if (!$con->query($deleteQuery)) {
         echo "<p style='color:red;'>Error deleting rows: " . $con->error . "</p>";
    } else {
         echo "<p>Existing rows deleted from race_list.</p>";
    }

    //------------------------------------------------
    // 11. Insert competitor data into race_list table
    //------------------------------------------------
    $con->begin_transaction();
    $insert_sql = "INSERT INTO race_list (
                            unique_key, race_time, race_title, racer_name, race_id, rankValue, League, status, athlete_id, event_in, car_type, racer_number
                       ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $con->prepare($insert_sql);
    if (!$stmt) {
         die("Prepare failed (insertion): " . $con->error);
    }
    
    // Set race title and time (always inserted)
    $raceTitleDB = strtoupper($latestRaceLabel);
    $raceTimeDB = $latestRaceTime;
    $statusDB = '';
    
    // Get status from the selected event if available
    if ($selectedEventFromScoreboard && isset($selectedEventFromScoreboard['status']['type']['name'])) {
        $statusDB = strtoupper($selectedEventFromScoreboard['status']['type']['name']);
    }
    
    echo "<pre>Insertion debug: Race Title: {$raceTitleDB}, Race Time: {$raceTimeDB}, Status: {$statusDB}</pre>";
    
    // Insert exactly 16 rows
    for ($i = 0; $i < 16; $i++) {
        $competitor = $finalCompetitors[$i];
        $unique_key = $i + 1;
        
        if (isset($competitor['default']) && $competitor['default'] === true) {
            // Default row: minimal data
            $athlete_id = -($i + 1); // Negative to avoid duplicates
            $racerName = "";
            $race_id = 0;
            $rankValue = $i + 1;
            $LeagueDB = "";
        } else {
            // Active competitor
            $athlete_id = $competitor['id'] ?? 0;
            $racerName = "";
            
            // Extract racer name
            if (isset($competitor['athlete'])) {
                if (isset($competitor['athlete']['shortName'])) {
                    $racerName = $competitor['athlete']['shortName'];
                } elseif (isset($competitor['athlete']['displayName'])) {
                    $racerName = $competitor['athlete']['displayName'];
                } elseif (isset($competitor['athlete']['fullName'])) {
                    $racerName = $competitor['athlete']['fullName'];
                }
            }
            
            $race_id = $i + 1;
            $rankValue = $competitor['order'] ?? ($i + 1);
            $LeagueDB = "NASCAR";
        }
        
        $racer_number = 0;
        $car_type = "";
        
        // Debug output
        echo "<pre>Inserting row " . ($i + 1) . ": unique_key={$unique_key}, race_time={$raceTimeDB}, race_title={$raceTitleDB}, racer_name={$racerName}, race_id={$race_id}, rankValue={$rankValue}, League={$LeagueDB}, status={$statusDB}, athlete_id={$athlete_id}, event_in={$eventId}, car_type={$car_type}, racer_number={$racer_number}</pre>";
        
        $stmt->bind_param("isssiisssssi", $unique_key, $raceTimeDB, $raceTitleDB, $racerName, $race_id, $rankValue, $LeagueDB, $statusDB, $athlete_id, $eventId, $car_type, $racer_number);
        if (!$stmt->execute()) {
            echo "<pre>Insert error on row " . ($i + 1) . ": " . $stmt->error . "</pre>";
        }
    }
    
    if ($con->commit()) {
         echo "<p style='color:green;'>Transaction committed successfully. 16 rows inserted into race_list.</p>";
    } else {
         echo "<p style='color:red;'>Transaction commit failed: " . $con->error . "</p>";
    }
    $stmt->close();

    //------------------------------------------------
    // 12. Fetch detailed event data (including vehicle info) using the details API
    //------------------------------------------------
    $details_url = "https://sports.core.api.espn.com/v2/sports/racing/leagues/nascar-premier/events/{$eventId}?lang=en&region=us";
    $ch2 = curl_init($details_url);
    curl_setopt_array($ch2, [
         CURLOPT_RETURNTRANSFER => true,
         CURLOPT_FOLLOWLOCATION => true,
         CURLOPT_TIMEOUT        => 30,
    ]);
    $details_response = curl_exec($ch2);
    if (curl_errno($ch2)) {
         echo "<p style='color:red;'>cURL Error (details): " . curl_error($ch2) . "</p>";
         $details_data = [];
    } else {
        curl_close($ch2);
        $details_data = json_decode($details_response, true);
        if (!$details_data) {
             echo "<p style='color:red;'>Error decoding JSON from details API.</p>";
             $details_data = [];
        }
    }

    //------------------------------------------------
    // 13. Build mappings: athlete_id => vehicle info (manufacturer and number)
    //------------------------------------------------
    $vehicleMap = [];      // For car_type (manufacturer)
    $vehicleNumberMap = []; // For racer_number (vehicle number)
    if (isset($details_data['competitions']) && is_array($details_data['competitions'])) {
         foreach ($details_data['competitions'] as $competition) {
              if (isset($competition['competitors']) && is_array($competition['competitors'])) {
                   foreach ($competition['competitors'] as $competitor) {
                        $athlete_id = $competitor['id'] ?? 0;
                        if ($athlete_id && isset($competitor['vehicle'])) {
                             $vehicle = $competitor['vehicle'];
                             $manufacturer = $vehicle['manufacturer'] ?? '';
                             $number = $vehicle['number'] ?? '';
                             $vehicleMap[$athlete_id] = $manufacturer;
                             $vehicleNumberMap[$athlete_id] = (int)$number;
                        }
                   }
              }
         }
    }
    echo "<p style='text-align:center; font-weight:bold;'>Vehicle info retrieved for " . count($vehicleMap) . " competitors.</p>";

    //------------------------------------------------
    // 14. Update race_list rows with vehicle info using athlete_id as identifier
    //------------------------------------------------
    if (count($vehicleMap) > 0) {
        $updateStmt = $con->prepare("UPDATE race_list SET car_type = ?, racer_number = ? WHERE athlete_id = ? AND event_in = ?");
        if (!$updateStmt) {
             echo "<p style='color:red;'>Prepare update failed: " . $con->error . "</p>";
        } else {
            $query = "SELECT unique_key, athlete_id FROM race_list WHERE event_in = ? ORDER BY unique_key";
            $stmt = $con->prepare($query);
            $stmt->bind_param("s", $eventId);
            $stmt->execute();
            $result = $stmt->get_result();
            $rowsData = [];
            while ($row = $result->fetch_assoc()) {
                 $rowsData[$row['unique_key']] = $row['athlete_id'];
            }
            $stmt->close();
            
            foreach ($rowsData as $unique_key => $athleteId) {
                 $carType = isset($vehicleMap[$athleteId]) ? $vehicleMap[$athleteId] : "";
                 $vehicleNumber = isset($vehicleNumberMap[$athleteId]) ? $vehicleNumberMap[$athleteId] : 0;
                 if ($carType !== "" || $vehicleNumber !== 0) {
                     $updateStmt->bind_param("siss", $carType, $vehicleNumber, $athleteId, $eventId);
                     if (!$updateStmt->execute()) {
                          echo "<p style='color:red;'>Update error for athlete {$athleteId}: " . $updateStmt->error . "</p>";
                     } else {
                          echo "<p>Updated row for athlete {$athleteId} with car_type '{$carType}' and racer_number '{$vehicleNumber}'.</p>";
                     }
                 }
            }
            $updateStmt->close();
        }
        echo "<p style='text-align:center; font-weight:bold;'>Vehicle info updated in database.</p>";
    }

    //------------------------------------------------
    // 15. Query the updated race_list data for display
    //------------------------------------------------
    $query = "SELECT * FROM race_list ORDER BY unique_key";
    $stmt = $con->prepare($query);
    $stmt->execute();
    $result = $stmt->get_result();
    $finalRows = [];
    while ($row = $result->fetch_assoc()) {
         $finalRows[] = $row;
    }
    $stmt->close();
    $con->close();
    ?>
    <h2 style="text-align:center;">Final Race Data (<?= htmlspecialchars($latestRaceLabel ?? 'Unknown Race') ?>)</h2>
    <table>
        <thead>
            <tr>
                <th>Unique Key</th>
                <th>Race ID</th>
                <th>Race Time</th>
                <th>Race Title</th>
                <th>Racer Name</th>
                <th>Rank Value</th>
                <th>League</th>
                <th>Status</th>
                <th>Athlete ID</th>
                <th>Event In</th>
                <th>Car Type</th>
                <th>Racer Number</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($finalRows as $row): ?>
            <tr>
                <td><?= htmlspecialchars($row['unique_key'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['race_id'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['race_time'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['race_title'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['racer_name'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['rankValue'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['League'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['status'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['athlete_id'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['event_in'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['car_type'] ?? '') ?></td>
                <td><?= htmlspecialchars($row['racer_number'] ?? '') ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    
    <!-- Final step: cURL the external link -->
    <?php
    $wrapper_url = SERVER_URL . "Scrolling/App_Output/API/NASCAR/wrapperNASCAR.php";
    $ch = curl_init($wrapper_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $wrapper_response = curl_exec($ch);
    if (curl_errno($ch)) {
         echo "<p style='text-align:center; color:red;'>cURL Error calling wrapperNASCAR.php: " . curl_error($ch) . "</p>";
    } else {
         echo "<p style='text-align:center; font-weight:bold;'>Wrapper API response:</p>";
         echo "<div style='width:90%; margin:0 auto; border:1px solid #333; padding:10px;'>" . htmlspecialchars($wrapper_response) . "</div>";
    }
    curl_close($ch);
    ?>
    
</body>
</html>