<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// PLAYOFF FLAG - Set this to true during playoffs to use playoff records
$PLAY_OFF = true; // Set to true for playoffs, false for regular season

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp = date('Y-m-d H:i:s');

// Check if columns exist and add if needed
$columns_to_check = ['scoreValue', 'lastUpdatedTime', 'lastPlay', 'lastPlayerID', 'lastPlayerName', 
                     'lastPlayerPos', 'lastPlayerTeamID', 'jersey', 'broadcast'];

foreach ($columns_to_check as $column) {
    $check_column = "SHOW COLUMNS FROM `sports_list` LIKE '$column'";
    $column_result = mysqli_query($con, $check_column);
    if (mysqli_num_rows($column_result) == 0) {
        if ($column == 'lastUpdatedTime') {
            $type = 'DATETIME';
        } elseif ($column == 'broadcast') {
            $type = 'VARCHAR(50)';
        } else {
            $type = 'VARCHAR(255)';
        }
        $add_column = "ALTER TABLE `sports_list` ADD COLUMN `$column` $type DEFAULT NULL";
        mysqli_query($con, $add_column);
    }
}

// Helper function to convert hex to RGB
function hexToRgb($hex) {
    $hex = str_replace(['#', '"'], '', $hex);
    if (strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    
    if (strlen($hex) != 6) {
        return array('r' => '000', 'g' => '000', 'b' => '000', 'rgb' => '000,000,000');
    }
    
    $r = str_pad(hexdec(substr($hex, 0, 2)), 3, '0', STR_PAD_LEFT);
    $g = str_pad(hexdec(substr($hex, 2, 2)), 3, '0', STR_PAD_LEFT);
    $b = str_pad(hexdec(substr($hex, 4, 2)), 3, '0', STR_PAD_LEFT);
    
    return array('r' => $r, 'g' => $g, 'b' => $b, 'rgb' => "$r,$g,$b");
}

// Function to check if update should be paused due to recent score
function shouldPauseUpdate($game_num, $con) {
    $check_query = "SELECT scoreValue, lastUpdatedTime FROM sports_list 
                    WHERE Game = $game_num AND Conference = 'MLB'";
    $result = mysqli_query($con, $check_query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $currentScoreValue = $row['scoreValue'] ?? '';
        $lastUpdatedTime = $row['lastUpdatedTime'] ?? null;
        
        if (!empty($currentScoreValue) && 
            $currentScoreValue != 'NO-SCORE' && 
            $currentScoreValue != ' ' &&
            !empty($lastUpdatedTime)) {
            
            $lastUpdateTimestamp = strtotime($lastUpdatedTime);
            $currentTimestamp = time();
            $timeDifference = $currentTimestamp - $lastUpdateTimestamp;
            
            if ($timeDifference < 150) {
                return array(
                    'pause' => true,
                    'time_remaining' => 150 - $timeDifference
                );
            }
        }
    }
    
    return array('pause' => false, 'time_remaining' => 0);
}

// API CALL
$api_call = "http://site.api.espn.com/apis/site/v2/sports/baseball/mlb/scoreboard";
// $api_call = "http://gscroll.us/updateAPIs/sampleData/MLB_SCORE_HR.txt";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error #:" . $err);
}

// PARSE JSON PROPERLY
$data = json_decode($result, true);

if (!$data || !isset($data['events'])) {
    die("Error parsing API response or no events found");
}

$events = $data['events'];
$loop_length = count($events);

// Store all data for table display
$games_data = [];

// Process each game
for ($i = 1; $i <= $loop_length; $i++) {
    $event = $events[$i - 1];
    
    if (!$event || !isset($event['competitions'][0])) {
        continue;
    }
    
    $competition = $event['competitions'][0];
    
    // Extract basic game info
    $status = $competition['status'] ?? [];
    $period = $status['period'] ?? 0;
    $statusType = $status['type'] ?? [];
    
    // Game date/time
    $game_time = $event['date'] ?? '';
    $game_time_2 = '';
    if (!empty($game_time)) {
        $game_time_2 = date("g:i a", strtotime($game_time) - 14400); // Eastern time
    }
    
    // Extract broadcast information
    $broadcast = '';
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $broadcast = $broadcast_data['names'][0] ?? '';
        }
    }
    
    // Determine game status
    $game_being_played = 'NOT_STARTED';
    if (isset($statusType['completed']) && $statusType['completed']) {
        $game_being_played = 'GAME_DONE';
    } elseif (isset($statusType['state'])) {
        switch($statusType['state']) {
            case 'pre':
                $game_being_played = 'NOT_STARTED';
                break;
            case 'in':
                $game_being_played = 'IN_SESSION';
                break;
            case 'post':
                $game_being_played = 'GAME_DONE';
                break;
        }
    }
    
    // Check for playoff game and extract playoff records
    $playoff_records = [];
    $is_playoff_game = false;
    if ($PLAY_OFF && isset($competition['series']) && $competition['series']['type'] === 'playoff') {
        $is_playoff_game = true;
        $series_competitors = $competition['series']['competitors'] ?? [];
        
        // Collect all teams' wins
        $team_wins = [];
        foreach ($series_competitors as $series_comp) {
            $team_id = $series_comp['id'] ?? '';
            $wins = $series_comp['wins'] ?? 0;
            $team_wins[$team_id] = $wins;
        }
        
        // Calculate playoff records (each team's losses = opponent's wins)
        foreach ($series_competitors as $series_comp) {
            $team_id = $series_comp['id'] ?? '';
            $wins = $series_comp['wins'] ?? 0;
            
            // Find opponent's wins (which are this team's losses)
            $losses = 0;
            foreach ($team_wins as $tid => $w) {
                if ($tid != $team_id) {
                    $losses = $w;
                    break;
                }
            }
            
            $playoff_records[$team_id] = $wins . '-' . $losses;
        }
    }
    
    // Get competitors (teams)
    $competitors = $competition['competitors'] ?? [];
    $home_team = null;
    $away_team = null;
    
    foreach ($competitors as $competitor) {
        if (($competitor['homeAway'] ?? '') == 'home') {
            $home_team = $competitor;
        } else {
            $away_team = $competitor;
        }
    }
    
    // Initialize default values
    $team_name1 = '';
    $location_team1 = '';
    $team_abbrev_1 = '';
    $team1_score = '0';
    $team_id_1 = '';
    $team1_record = '0-0';
    $team1_color_rgb = '"000,000,000"';
    $team1_color_r = '000';
    $team1_color_g = '000';
    $team1_color_b = '000';
    $team1HE = '';
    
    $team_name2 = '';
    $location_team2 = '';
    $team_abbrev_2 = '';
    $team2_score = '0';
    $team_id_2 = '';
    $team2_record = '0-0';
    $team2_color_rgb = '"000,000,000"';
    $team2_color_r = '000';
    $team2_color_g = '000';
    $team2_color_b = '000';
    $team2HE = '';
    
    // Team 1 (Home) data
    if ($home_team) {
        $team1_info = $home_team['team'] ?? [];
        $team_name1 = $team1_info['name'] ?? '';
        $location_team1 = $team1_info['location'] ?? '';
        $team_abbrev_1 = $team1_info['abbreviation'] ?? '';
        $team1_score = $home_team['score'] ?? '0';
        $team_id_1 = $team1_info['id'] ?? '';
        
        // Team 1 color
        $team1_color_hex = $team1_info['color'] ?? '000000';
        if ($team1_color_hex == '000000' || empty($team1_color_hex)) {
            $team1_color_hex = $team1_info['alternateColor'] ?? '000000';
        }
        $team1_colors = hexToRgb($team1_color_hex);
        $team1_color_r = $team1_colors['r'];
        $team1_color_g = $team1_colors['g'];
        $team1_color_b = $team1_colors['b'];
        $team1_color_rgb = '"' . $team1_colors['rgb'] . '"';
        
        // Team 1 record - Check for playoff flag
        if ($PLAY_OFF && $is_playoff_game && isset($playoff_records[$team_id_1])) {
            $team1_record = $playoff_records[$team_id_1];
        } else {
            // Use regular season record
            $team1_records = $home_team['records'] ?? [];
            foreach ($team1_records as $record) {
                if (($record['type'] ?? '') == 'total') {
                    $team1_record = $record['summary'] ?? '0-0';
                    break;
                }
            }
        }
        
        // Team 1 hits and errors
        if ($game_being_played == 'IN_SESSION') {
            $hits1 = '0';
            $errors1 = '0';
            $stats = $home_team['statistics'] ?? [];
            foreach ($stats as $stat) {
                if ($stat['name'] == 'hits') {
                    $hits1 = $stat['displayValue'] ?? '0';
                }
                if ($stat['name'] == 'errors') {
                    $errors1 = $stat['displayValue'] ?? '0';
                }
            }
            $team1HE = $hits1 . ',' . $errors1;
        }
    }
    
    // Team 2 (Away) data
    if ($away_team) {
        $team2_info = $away_team['team'] ?? [];
        $team_name2 = $team2_info['name'] ?? '';
        $location_team2 = $team2_info['location'] ?? '';
        $team_abbrev_2 = $team2_info['abbreviation'] ?? '';
        $team2_score = $away_team['score'] ?? '0';
        $team_id_2 = $team2_info['id'] ?? '';
        
        // Team 2 color
        $team2_color_hex = $team2_info['color'] ?? '000000';
        if ($team2_color_hex == '000000' || empty($team2_color_hex)) {
            $team2_color_hex = $team2_info['alternateColor'] ?? '000000';
        }
        $team2_colors = hexToRgb($team2_color_hex);
        $team2_color_r = $team2_colors['r'];
        $team2_color_g = $team2_colors['g'];
        $team2_color_b = $team2_colors['b'];
        $team2_color_rgb = '"' . $team2_colors['rgb'] . '"';
        
        // Team 2 record - Check for playoff flag
        if ($PLAY_OFF && $is_playoff_game && isset($playoff_records[$team_id_2])) {
            $team2_record = $playoff_records[$team_id_2];
        } else {
            // Use regular season record
            $team2_records = $away_team['records'] ?? [];
            foreach ($team2_records as $record) {
                if (($record['type'] ?? '') == 'total') {
                    $team2_record = $record['summary'] ?? '0-0';
                    break;
                }
            }
        }
        
        // Team 2 hits and errors
        if ($game_being_played == 'IN_SESSION') {
            $hits2 = '0';
            $errors2 = '0';
            $stats = $away_team['statistics'] ?? [];
            foreach ($stats as $stat) {
                if ($stat['name'] == 'hits') {
                    $hits2 = $stat['displayValue'] ?? '0';
                }
                if ($stat['name'] == 'errors') {
                    $errors2 = $stat['displayValue'] ?? '0';
                }
            }
            $team2HE = $hits2 . ',' . $errors2;
        }
    }
    
    // In-game situation data
    $who_has_ball = '';
    $score_value = '';
    $last_play_text = '';
    $lastPlayerID = '';
    $lastPlayerName = '';
    $lastPlayerPos = '';
    $lastPlayerTeamID = '';
    $lastPlayerJersey = '';
    $driveDescription = '';
    $onBases = '';
    $outsB = '';
    
    // EXTRACT IN-GAME DATA PROPERLY
    if ($game_being_played == 'IN_SESSION') {
        $situation = $competition['situation'] ?? [];
        
        // Get the shortDetail for inning info from status
        $shortDetail = '';
        if (isset($status['type']['shortDetail'])) {
            $shortDetail = $status['type']['shortDetail'];
        } elseif (isset($status['type']['detail'])) {
            $shortDetail = $status['type']['detail'];
        }
        
        // Extract inning number from shortDetail (like "Bot 5th" or "Top 3rd")
        if (preg_match('/(\d+)/', $shortDetail, $matches)) {
            $period = $matches[1];  // Update period with the actual inning number
        }
        
        // Determine who has ball based on Top/Bot in shortDetail - FIXED LOGIC
        $shortDetailLower = strtolower($shortDetail);
        if (strpos($shortDetailLower, 'top') !== false) {
            $who_has_ball = $team_abbrev_2; // Away team bats in top
        } elseif (strpos($shortDetailLower, 'bot') !== false) {
            $who_has_ball = $team_abbrev_1; // Home team bats in bottom  
        } elseif (strpos($shortDetailLower, 'mid') !== false || strpos($shortDetailLower, 'middle') !== false) {
            $who_has_ball = 'mid';
        } elseif (strpos($shortDetailLower, 'end') !== false) {
            $who_has_ball = 'end';
        }
        
        // EXTRACT LAST PLAY DATA
        if (isset($situation['lastPlay'])) {
            $lastPlay = $situation['lastPlay'];
            
            // Get the play text
            if (isset($lastPlay['text'])) {
                $last_play_text = strip_tags(trim($lastPlay['text']));
                if (strlen($last_play_text) > 255) {
                    $last_play_text = substr($last_play_text, 0, 252) . '...';
                }
            }
            
            // Override who_has_ball if play text indicates middle/end of inning
            $playTextLower = strtolower($last_play_text);
            if (strpos($playTextLower, 'middle of') !== false || strpos($playTextLower, 'mid ') !== false) {
                $who_has_ball = 'mid';
            }
            if (strpos($playTextLower, 'end of') !== false) {
                $who_has_ball = 'end';
            }
            
            // Get score value
            $scoreVal = intval($lastPlay['scoreValue'] ?? 0);
            
            // Determine score type
            if ($scoreVal == 0) {
                $score_value = "NO-SCORE";
            } elseif ($scoreVal == 1) {
                $score_value = 'R-1';
            } elseif ($scoreVal == 2) {
                $score_value = 'R-2';
            } elseif ($scoreVal == 3) {
                $score_value = 'R-3';
            } elseif ($scoreVal == 4) {
                $score_value = 'R-4';
            } else {
                $score_value = 'R-' . $scoreVal;
            }
            
            // Check for home run in play text or play type
            $playType = $lastPlay['type']['text'] ?? '';
            if (stripos($last_play_text, 'home run') !== false || 
                stripos($last_play_text, 'homer') !== false ||
                stripos($last_play_text, 'grand slam') !== false ||
                stripos($playType, 'home run') !== false) {
                if ($scoreVal > 0) {
                    $score_value = 'H-' . $scoreVal;
                }
            }
            
            // EXTRACT PLAYER INFORMATION
            if (isset($lastPlay['athletesInvolved']) && is_array($lastPlay['athletesInvolved']) && count($lastPlay['athletesInvolved']) > 0) {
                $firstPlayer = $lastPlay['athletesInvolved'][0];
                $lastPlayerID = $firstPlayer['id'] ?? '';
                $lastPlayerName = $firstPlayer['fullName'] ?? $firstPlayer['displayName'] ?? '';
                $lastPlayerPos = $firstPlayer['position'] ?? '';
                $lastPlayerJersey = $firstPlayer['jersey'] ?? '';
                
                // Get team ID from player data
                if (isset($firstPlayer['team']['id'])) {
                    $lastPlayerTeamID = $firstPlayer['team']['id'];
                }
            }
        }
        
        // Get current batter info if no last play player
        if (empty($lastPlayerName) && isset($situation['batter']['athlete'])) {
            $batter = $situation['batter']['athlete'];
            $lastPlayerID = $batter['id'] ?? '';
            $lastPlayerName = $batter['fullName'] ?? $batter['displayName'] ?? '';
            $lastPlayerPos = $batter['position'] ?? '';
            $lastPlayerJersey = $batter['jersey'] ?? '';
            if (isset($batter['team']['id'])) {
                $lastPlayerTeamID = $batter['team']['id'];
            }
        }
        
        // Get base runners - properly format as 0,0,0 or 1,1,0 etc
        $onFirst = '0';
        $onSecond = '0';
        $onThird = '0';
        
        if (isset($situation['onFirst']) && $situation['onFirst'] === true) {
            $onFirst = '1';
        }
        if (isset($situation['onSecond']) && $situation['onSecond'] === true) {
            $onSecond = '1';
        }
        if (isset($situation['onThird']) && $situation['onThird'] === true) {
            $onThird = '1';
        }
        
        $onBases = $onFirst . ',' . $onSecond . ',' . $onThird;
        
        // Get outs - format as single digit
        $outsB = isset($situation['outs']) ? strval($situation['outs']) : '0';
        
    } else {
        // Not in session - clear all in-game fields with single space
        $who_has_ball = ' ';
        $score_value = ' ';
        $last_play_text = ' ';
        $lastPlayerID = ' ';
        $lastPlayerName = ' ';
        $lastPlayerPos = ' ';
        $lastPlayerTeamID = ' ';
        $lastPlayerJersey = ' ';
        $driveDescription = ' ';
        $onBases = ' ';
        $outsB = ' ';
    }
    
    // Betting odds extraction
    $favorite_team = '';
    $favorite_points = '';
    $overUnder = '';
    
    $odds = $competition['odds'] ?? [];
    if (!empty($odds) && isset($odds[0])) {
        $oddData = $odds[0];
        if (isset($oddData['details'])) {
            $details = $oddData['details'] ?? '';
            if (strpos($details, $team_abbrev_1) !== false) {
                $favorite_team = $team_abbrev_1;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            } elseif (strpos($details, $team_abbrev_2) !== false) {
                $favorite_team = $team_abbrev_2;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            }
        }
        $overUnder = $oddData['overUnder'] ?? '';
    }
    
    // CHECK FOR PAUSE DUE TO RECENT SCORE
    $pauseInfo = shouldPauseUpdate($i, $con);
    $isPaused = $pauseInfo['pause'];
    $timeRemaining = $pauseInfo['time_remaining'];
    
    // Check if this is a new score (not yet in database)
    $isNewScore = false;
    if (isset($scoreVal) && $scoreVal > 0 && ($game_being_played == 'IN_SESSION')) {
        $check_score = "SELECT scoreValue FROM sports_list WHERE Game = $i AND Conference = 'MLB'";
        $score_result = mysqli_query($con, $check_score);
        if ($score_result && mysqli_num_rows($score_result) > 0) {
            $current_db_score = mysqli_fetch_assoc($score_result)['scoreValue'] ?? '';
            if ($current_db_score != $score_value) {
                $isNewScore = true;
            }
        }
    }
    
    // Store data for display
    $games_data[] = [
        'Game' => $i,
        'Period' => $period,
        'Game_Played' => $game_being_played,
        'Game_Time' => $game_time_2,
        'Game_Date' => $game_time,
        'broadcast' => $broadcast,
        'scoreValue' => $score_value,
        'lastPlay' => $last_play_text,
        'lastPlayerID' => $lastPlayerID,
        'lastPlayerName' => $lastPlayerName,
        'lastPlayerPos' => $lastPlayerPos,
        'lastPlayerTeamID' => $lastPlayerTeamID,
        'jersey' => $lastPlayerJersey,
        'Team_1_Name' => $team_name1,
        'Team_1_Record' => $team1_record,
        'Team_1_Score' => $team1_score,
        'Team_1_Location' => $location_team1,
        'Team_1_Abbrev' => $team_abbrev_1,
        'Team_1_Color' => $team1_color_rgb,
        'Team_1_Color_R' => $team1_color_r,
        'Team_1_Color_G' => $team1_color_g,
        'Team_1_Color_B' => $team1_color_b,
        'Team_2_Name' => $team_name2,
        'Team_2_Record' => $team2_record,
        'Team_2_Score' => $team2_score,
        'Team_2_Location' => $location_team2,
        'Team_2_Abbrev' => $team_abbrev_2,
        'Team_2_Color' => $team2_color_rgb,
        'Team_2_Color_R' => $team2_color_r,
        'Team_2_Color_G' => $team2_color_g,
        'Team_2_Color_B' => $team2_color_b,
        'Who_Has_Ball' => $who_has_ball,
        'Favorite_Team' => $favorite_team,
        'Favorite_Points' => $favorite_points,
        'OverUnder' => $overUnder,
        'outsB' => $outsB,
        'onBases' => $onBases,
        'team1HE' => $team1HE,
        'team2HE' => $team2HE,
        'is_paused' => $isPaused,
        'time_remaining' => $timeRemaining,
        'is_new_score' => $isNewScore,
        'is_playoff' => $is_playoff_game
    ];
    
    // UPDATE DATABASE - SELECTIVE UPDATES DURING PAUSE
    if (!$isPaused) {
        // FULL UPDATE - Game is not paused
        $query = "
            UPDATE sports_list 
            SET 
                Clock = '0:00',
                Period = '".addslashes($period)."',
                Game_Played = '".addslashes($game_being_played)."',
                Game_Time = '".addslashes($game_time_2)."',
                Game_Date = '".addslashes($game_time)."',
                broadcast = '".addslashes($broadcast)."',
                scoreValue = '".addslashes($score_value)."',
                lastPlay = '".addslashes($last_play_text)."',
                lastPlayerID = '".addslashes($lastPlayerID)."',
                lastPlayerName = '".addslashes($lastPlayerName)."',
                lastPlayerPos = '".addslashes($lastPlayerPos)."',
                lastPlayerTeamID = '".addslashes($lastPlayerTeamID)."',
                jersey = '".addslashes($lastPlayerJersey)."',
                lastUpdatedTime = '".addslashes($current_timestamp)."',
                
                Team_1_Name = '".addslashes($team_name1)."',
                Team_1_Record = '".addslashes($team1_record)."',
                Team_1_Score = '".addslashes($team1_score)."',
                Team_1_Location = '".addslashes($location_team1)."',
                Team_1_Abbrev = '".addslashes($team_abbrev_1)."',
                Team_1_Color = '".addslashes($team1_color_rgb)."',
                Team_1_Color_R = '".addslashes($team1_color_r)."',
                Team_1_Color_G = '".addslashes($team1_color_g)."',
                Team_1_Color_B = '".addslashes($team1_color_b)."',
                
                Team_2_Name = '".addslashes($team_name2)."',
                Team_2_Record = '".addslashes($team2_record)."',
                Team_2_Score = '".addslashes($team2_score)."',
                Team_2_Location = '".addslashes($location_team2)."',
                Team_2_Abbrev = '".addslashes($team_abbrev_2)."',
                Team_2_Color = '".addslashes($team2_color_rgb)."',
                Team_2_Color_R = '".addslashes($team2_color_r)."',
                Team_2_Color_G = '".addslashes($team2_color_g)."',
                Team_2_Color_B = '".addslashes($team2_color_b)."',
                
                Who_Has_Ball = '".addslashes($who_has_ball)."',
                Favorite_Team = '".addslashes($favorite_team)."',
                Favorite_Points = '".addslashes($favorite_points)."',
                OverUnder = '".addslashes($overUnder)."',
                outsB = '".addslashes($outsB)."',
                onBases = '".addslashes($onBases)."',
                team1HE = '".addslashes($team1HE)."',
                team2HE = '".addslashes($team2HE)."'
            WHERE 
                Game = $i AND 
                Conference = 'MLB'
        ";
        
        $query_run = mysqli_query($con, $query);
        
        if (!$query_run) {
            $games_data[count($games_data) - 1]['error'] = mysqli_error($con);
        } else {
            $games_data[count($games_data) - 1]['error'] = '';
        }
    } else {
        // PARTIAL UPDATE - Game is paused, only update scores
        $query = "
            UPDATE sports_list 
            SET 
                Team_1_Score = '".addslashes($team1_score)."',
                Team_2_Score = '".addslashes($team2_score)."'
            WHERE 
                Game = $i AND 
                Conference = 'MLB'
        ";
        
        $query_run = mysqli_query($con, $query);
        
        if (!$query_run) {
            $games_data[count($games_data) - 1]['error'] = 'PAUSED - Score update failed: ' . mysqli_error($con);
        } else {
            $games_data[count($games_data) - 1]['error'] = 'PAUSED - Scores updated';
        }
    }
    
    // Debug output for first game
    if ($i == 1 && $game_being_played == 'IN_SESSION') {
        echo "<!-- DEBUG Game 1: lastPlay='$last_play_text', Player='$lastPlayerName', ID='$lastPlayerID', WhoBall='$who_has_ball', Broadcast='$broadcast' -->\n";
    }
}

// HTML Output
?>
<!DOCTYPE html>
<html>
<head>
    <title>MLB Games Database Update - <?php echo $time_updated; ?></title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f5f5f5;
        }
        h1 {
            color: #333;
            text-align: center;
            background: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .summary {
            background: #fff;
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .playoff-mode {
            background: #ff9800;
            color: white;
            padding: 10px;
            margin: 20px 0;
            border-radius: 5px;
            text-align: center;
            font-weight: bold;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            background: #fff;
            margin: 20px 0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            font-size: 11px;
        }
        th {
            background: #2c3e50;
            color: white;
            padding: 8px 4px;
            text-align: left;
            font-size: 10px;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        td {
            padding: 6px 4px;
            border-bottom: 1px solid #ddd;
            font-size: 10px;
        }
        tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        tr:hover {
            background-color: #f0f0f0;
        }
        .paused-row {
            background-color: #fff59d !important;
        }
        .paused-row:hover {
            background-color: #fff176 !important;
        }
        .playoff-game {
            background-color: #e8f5e9 !important;
        }
        .playoff-game:hover {
            background-color: #c8e6c9 !important;
        }
        .countdown-timer {
            background: #ff9800;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 11px;
            display: inline-block;
        }
        .status-updated {
            background: #4caf50;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 11px;
            display: inline-block;
        }
        .playoff-badge {
            background: #9c27b0;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 11px;
            display: inline-block;
        }
        .broadcast-cell {
            background: #e3f2fd;
            font-weight: bold;
        }
        .status-in-session { background: #d4edda; font-weight: bold; }
        .status-game-done { background: #f8d7da; }
        .status-not-started { background: #fff3cd; }
        .truncate {
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        .error {
            color: red;
            font-weight: bold;
        }
        .wrapper {
            overflow-x: auto;
        }
        .pause-info {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 10px;
            margin: 10px 0;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <h1>MLB Games Database Update</h1>
    
    <?php if ($PLAY_OFF): ?>
    <div class="playoff-mode">
        🏆 PLAYOFF MODE ACTIVE 🏆<br>
        Displaying playoff series records instead of regular season records
    </div>
    <?php endif; ?>
    
    <div class="summary">
        <strong>Update Time:</strong> <?php echo $time_updated; ?><br>
        <strong>Total Games Processed:</strong> <?php echo $loop_length; ?><br>
        <strong>Database:</strong> <?php echo $database_gscroll; ?><br>
        <strong>Mode:</strong> <?php echo $PLAY_OFF ? 'PLAYOFF' : 'Regular Season'; ?>
    </div>
    
    <?php 
    // Collect all paused and newly scored games
    $score_status = [];
    foreach ($games_data as $game) {
        if ((isset($game['is_paused']) && $game['is_paused']) || (isset($game['is_new_score']) && $game['is_new_score'])) {
            $next_update_time = time() + ($game['is_new_score'] ? 150 : $game['time_remaining']);
            $score_status[] = [
                'game' => $game['Game'],
                'teams' => $game['Team_2_Abbrev'] . ' @ ' . $game['Team_1_Abbrev'],
                'score' => $game['Team_2_Score'] . '-' . $game['Team_1_Score'],
                'scoreValue' => $game['scoreValue'],
                'next_update' => date('g:i:s a', $next_update_time),
                'seconds_left' => $game['is_new_score'] ? 150 : $game['time_remaining']
            ];
        }
    }
    
    if (count($score_status) > 0): ?>
    <div style="background: #fff3cd; border: 2px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 5px;">
        <h3 style="margin-top: 0; color: #856404;">⏰ Score Update Status - Next Update Times</h3>
        <table style="width: 100%; font-size: 13px; border-collapse: collapse;">
            <thead>
                <tr style="background: #ffc107;">
                    <th style="padding: 8px; text-align: left;">Game #</th>
                    <th style="padding: 8px; text-align: left;">Matchup</th>
                    <th style="padding: 8px; text-align: left;">Score</th>
                    <th style="padding: 8px; text-align: left;">Score Type</th>
                    <th style="padding: 8px; text-align: left;">Next Update Time</th>
                    <th style="padding: 8px; text-align: left;">Countdown</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($score_status as $status): ?>
                <tr style="background: #fff9e6;">
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><?php echo $status['game']; ?></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><strong><?php echo $status['teams']; ?></strong></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><?php echo $status['score']; ?></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;">
                        <span style="background: #ff9800; color: white; padding: 2px 6px; border-radius: 3px;">
                            <?php echo $status['scoreValue']; ?>
                        </span>
                    </td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><strong><?php echo $status['next_update']; ?></strong></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;">
                        <span style="background: #dc3545; color: white; padding: 2px 6px; border-radius: 3px;">
                            <?php echo $status['seconds_left']; ?>s remaining
                        </span>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>
    
    <div class="wrapper">
        <table>
            <thead>
                <tr>
                    <th>Game</th>
                    <th>Type</th>
                    <th>Pause Status</th>
                    <th>Status</th>
                    <th>Inning</th>
                    <th>Game Time</th>
                    <th>Broadcast</th>
                    <th>Score Value</th>
                    <th>Who Has Ball</th>
                    <th>Outs</th>
                    <th>On Base</th>
                    <th>Last Play</th>
                    <th>Player Name</th>
                    <th>Player ID</th>
                    <th>Jersey</th>
                    <th>Pos</th>
                    <th>Team ID</th>
                    <th colspan="8">Team 1 (Home)</th>
                    <th colspan="8">Team 2 (Away)</th>
                    <th>Favorite</th>
                    <th>Spread</th>
                    <th>O/U</th>
                </tr>
            </thead>
            <tbody>
                <?php 
                $paused_count = 0;
                $updated_count = 0;
                $new_score_count = 0;
                $playoff_count = 0;
                
                foreach ($games_data as $game): 
                    $row_class = '';
                    if (isset($game['is_playoff']) && $game['is_playoff']) {
                        $row_class = 'playoff-game';
                        $playoff_count++;
                    }
                    if (isset($game['is_new_score']) && $game['is_new_score']) {
                        $row_class = 'paused-row';
                        $new_score_count++;
                    } elseif (isset($game['is_paused']) && $game['is_paused']) {
                        $row_class = 'paused-row';
                        $paused_count++;
                    } else {
                        $updated_count++;
                    }
                ?>
                <tr class="<?php echo $row_class; ?>">
                    <td><?php echo $game['Game']; ?></td>
                    <td>
                        <?php if (isset($game['is_playoff']) && $game['is_playoff']): ?>
                            <span class="playoff-badge">PLAYOFF</span>
                        <?php else: ?>
                            Regular
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if (isset($game['is_new_score']) && $game['is_new_score']): ?>
                            <span class="countdown-timer">NEW SCORE!</span>
                        <?php elseif (isset($game['is_paused']) && $game['is_paused']): ?>
                            <span class="countdown-timer">
                                <?php echo $game['time_remaining']; ?>s left
                            </span>
                        <?php else: ?>
                            <span class="status-updated">UPDATED</span>
                        <?php endif; ?>
                    </td>
                    <td class="status-<?php echo strtolower(str_replace('_', '-', $game['Game_Played'])); ?>">
                        <?php echo $game['Game_Played']; ?>
                    </td>
                    <td><?php echo $game['Period']; ?></td>
                    <td><?php echo $game['Game_Time']; ?></td>
                    <td class="broadcast-cell"><?php echo $game['broadcast'] ?: '-'; ?></td>
                    <td><?php echo $game['scoreValue']; ?></td>
                    <td><?php echo $game['Who_Has_Ball']; ?></td>
                    <td><?php echo $game['outsB']; ?></td>
                    <td><?php echo $game['onBases']; ?></td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['lastPlay']); ?>">
                        <strong><?php echo htmlspecialchars($game['lastPlay']); ?></strong>
                    </td>
                    <td><?php echo $game['lastPlayerName']; ?></td>
                    <td><?php echo $game['lastPlayerID']; ?></td>
                    <td><?php echo $game['jersey']; ?></td>
                    <td><?php echo $game['lastPlayerPos']; ?></td>
                    <td><?php echo $game['lastPlayerTeamID']; ?></td>
                    <td><?php echo $game['Team_1_Name']; ?></td>
                    <td><strong><?php echo $game['Team_1_Abbrev']; ?></strong></td>
                    <td><?php echo $game['Team_1_Record']; ?></td>
                    <td><strong><?php echo $game['Team_1_Score']; ?></strong></td>
                    <td><?php echo $game['team1HE']; ?></td>
                    <td><?php echo $game['Team_1_Color']; ?></td>
                    <td><?php echo $game['Team_1_Color_R']; ?></td>
                    <td><?php echo $game['Team_1_Color_G']; ?></td>
                    <td><?php echo $game['Team_2_Name']; ?></td>
                    <td><strong><?php echo $game['Team_2_Abbrev']; ?></strong></td>
                    <td><?php echo $game['Team_2_Record']; ?></td>
                    <td><strong><?php echo $game['Team_2_Score']; ?></strong></td>
                    <td><?php echo $game['team2HE']; ?></td>
                    <td><?php echo $game['Team_2_Color']; ?></td>
                    <td><?php echo $game['Team_2_Color_R']; ?></td>
                    <td><?php echo $game['Team_2_Color_G']; ?></td>
                    <td><?php echo $game['Favorite_Team']; ?></td>
                    <td><?php echo $game['Favorite_Points']; ?></td>
                    <td><?php echo $game['OverUnder']; ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <?php if ($paused_count > 0 || $new_score_count > 0): ?>
    <div class="pause-info">
        <strong>⏸️ Pause Status:</strong><br>
        <?php if ($new_score_count > 0): ?>
            <?php echo $new_score_count; ?> game(s) just scored (will be paused for 150 seconds)<br>
        <?php endif; ?>
        <?php if ($paused_count > 0): ?>
            <?php echo $paused_count; ?> game(s) currently paused from previous scores<br>
        <?php endif; ?>
        <?php echo $updated_count; ?> game(s) updated normally<br>
        <small>Games are paused for 150 seconds after a run scores to allow users to see the scoring play. Only team scores update during pause.</small>
    </div>
    <?php endif; ?>
    
    <div class="summary">
        <strong>Update completed successfully!</strong><br>
        Total games processed: <?php echo $loop_length; ?><br>
        Games updated: <?php echo $updated_count; ?><br>
        Games with new scores: <?php echo $new_score_count; ?><br>
        Games paused: <?php echo $paused_count; ?><br>
        <?php if ($playoff_count > 0): ?>
        <strong>Playoff games: <?php echo $playoff_count; ?></strong><br>
        <?php endif; ?>
        <small>Broadcast information included for all games</small>
    </div>
</body>
</html>

<?php

// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/MLB_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
} else {
    echo $response;
}
curl_close($ch);

// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/Scoring_ALERT.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
} else {
    echo $response; // ADD THIS LINE to output the response
}
curl_close($ch);

// Close database connection
mysqli_close($con);

?>