<?php

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Set timezone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp = date('Y-m-d H:i:s');

// Ensure database columns exist
$check_column = "SHOW COLUMNS FROM `sports_list` LIKE 'lastUpdatedTime'";
$column_result = mysqli_query($con, $check_column);
if (mysqli_num_rows($column_result) == 0) {
    $add_column = "ALTER TABLE `sports_list` ADD COLUMN `lastUpdatedTime` DATETIME DEFAULT NULL";
    mysqli_query($con, $add_column);
}

// Check for broadcast column and add if needed
$check_broadcast = "SHOW COLUMNS FROM `sports_list` LIKE 'broadcast'";
$broadcast_result = mysqli_query($con, $check_broadcast);
if (mysqli_num_rows($broadcast_result) == 0) {
    $add_broadcast = "ALTER TABLE `sports_list` ADD COLUMN `broadcast` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_broadcast);
}

// Update API timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NHL'";
mysqli_query($con, $sql);

/**
 * Check if update should be skipped due to timing restriction (2-minute rule for goals)
 */
function shouldSkipUpdate($game_num, $scoreValue, $con) {
    // Only skip if this is a GOAL event
    if ($scoreValue != 'GOAL') {
        return false;
    }
    
    // Query the last update time for this game
    $check_query = "SELECT lastUpdatedTime, scoreValue FROM sports_list 
                   WHERE Game = $game_num AND Conference = 'NHL'";
    $result = mysqli_query($con, $check_query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $lastUpdate = $row['lastUpdatedTime'];
        $lastScoreValue = $row['scoreValue'];
        
        // If there's a previous update time and it was also a GOAL
        if (!empty($lastUpdate) && $lastScoreValue == 'GOAL') {
            $timeDifference = time() - strtotime($lastUpdate);
            
            // Skip if less than 2 minutes since last goal
            if ($timeDifference < 120) {
                return true;
            }
        }
    }
    
    return false;
}

/**
 * Convert hex color to RGB
 */
function hexToRgb($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) != 6) return "255,255,255";
    
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    // Force to white if too dark
    if ($r <= 16 && $g <= 16 && $b <= 16) {
        $r = $g = $b = 255;
    }
    
    return "$r,$g,$b";
}

/**
 * Extract game data from competition object
 */
function extractGameData($event, $game_num) {
    $gameData = [
        'game_num' => $game_num,
        'clock' => '0:00',
        'period' => 0,
        'game_status' => 'NOT_STARTED',
        'game_time' => '',
        'game_date' => '',
        'broadcast' => '',
        'score_value' => 'NO-SCORE',
        'team1' => [],
        'team2' => [],
        'last_play' => '',
        'last_player' => [],
        'who_has_possession' => '',
        'drive_desc' => '',
        'favorite' => [],
        'over_under' => ''
    ];
    
    $competition = $event['competitions'][0] ?? [];
    
    // Get game timing and status
    $status = $competition['status'] ?? [];
    $gameData['clock'] = $status['displayClock'] ?? '0:00';
    $gameData['period'] = $status['period'] ?? 0;
    
    // Determine game status
    $statusType = $status['type']['name'] ?? '';
    switch ($statusType) {
        case 'STATUS_SCHEDULED':
            $gameData['game_status'] = 'NOT_STARTED';
            break;
        case 'STATUS_IN_PROGRESS':
            $gameData['game_status'] = 'IN_SESSION';
            break;
        case 'STATUS_END_PERIOD':
            $gameData['game_status'] = 'HALF_TIME';
            break;
        case 'STATUS_FINAL':
            $gameData['game_status'] = 'GAME_DONE';
            break;
    }
    
    // Get game time
    $gameData['game_date'] = $event['date'] ?? '';
    $gameData['game_time'] = date("g:i a", strtotime($gameData['game_date']));
    
    // Extract broadcast information
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $gameData['broadcast'] = $broadcast_data['names'][0] ?? '';
        }
    }
    
    // Get teams (home is index 0, away is index 1 in the API)
    $competitors = $competition['competitors'] ?? [];
    
    foreach ($competitors as $competitor) {
        $teamData = [
            'id' => $competitor['team']['id'] ?? '',
            'name' => $competitor['team']['name'] ?? '',
            'location' => $competitor['team']['location'] ?? '',
            'abbreviation' => $competitor['team']['abbreviation'] ?? '',
            'score' => $competitor['score'] ?? '0',
            'record' => '',
            'color' => '',
            'color_rgb' => '255,255,255'
        ];
        
        // Get team colors
        $color = $competitor['team']['color'] ?? '000000';
        $altColor = $competitor['team']['alternateColor'] ?? '';
        
        if ($color == '000000' && !empty($altColor)) {
            $color = $altColor;
        }
        
        $teamData['color'] = $color;
        $teamData['color_rgb'] = hexToRgb($color);
        
        // Get team record
        $records = $competitor['records'] ?? [];
        foreach ($records as $record) {
            if ($record['type'] == 'ytd' || $record['name'] == 'YTD') {
                $teamData['record'] = $record['summary'] ?? '';
                break;
            }
        }
        
        // Assign to team1 (home) or team2 (away)
        if ($competitor['homeAway'] == 'home') {
            $gameData['team1'] = $teamData;
        } else {
            $gameData['team2'] = $teamData;
        }
    }
    
    // Get in-game data if the game is active
    if (in_array($gameData['game_status'], ['IN_SESSION', 'HALF_TIME'])) {
        $situation = $competition['situation'] ?? [];
        $lastPlay = $situation['lastPlay'] ?? [];
        
        if (!empty($lastPlay)) {
            $gameData['last_play'] = $lastPlay['text'] ?? '';
            $gameData['score_value'] = ($lastPlay['scoreValue'] ?? 0) == 1 ? 'GOAL' : 'NO-SCORE';
            
            // Get player involved
            $athletes = $lastPlay['athletesInvolved'] ?? [];
            if (!empty($athletes[0])) {
                $athlete = $athletes[0];
                $gameData['last_player'] = [
                    'id' => $athlete['id'] ?? '',
                    'name' => $athlete['displayName'] ?? $athlete['fullName'] ?? '',
                    'position' => $athlete['position'] ?? '',
                    'jersey' => $athlete['jersey'] ?? '',
                    'team_id' => $athlete['team']['id'] ?? ''
                ];
            }
            
            // Determine possession from last play team
            $lastPlayTeam = $lastPlay['team']['id'] ?? '';
            if ($lastPlayTeam == $gameData['team1']['id']) {
                $gameData['who_has_possession'] = $gameData['team1']['abbreviation'];
            } elseif ($lastPlayTeam == $gameData['team2']['id']) {
                $gameData['who_has_possession'] = $gameData['team2']['abbreviation'];
            }
            
            // Check for goal in text
            if (stripos($gameData['last_play'], 'goal') !== false && stripos($gameData['last_play'], 'goalie') === false) {
                $gameData['score_value'] = 'GOAL';
            }
        }
        
        // Set period description
        if ($gameData['period'] <= 3) {
            $gameData['drive_desc'] = "Period " . $gameData['period'];
        } elseif ($gameData['period'] == 4) {
            $gameData['drive_desc'] = "Overtime";
        } elseif ($gameData['period'] == 5) {
            $gameData['drive_desc'] = "Shootout";
        }
    }
    
    // Get betting data
    $odds = $competition['odds'][0] ?? [];
    if (!empty($odds)) {
        $gameData['over_under'] = $odds['overUnder'] ?? '';
        
        // Determine favorite
        $homeOdds = $odds['homeTeamOdds'] ?? [];
        $awayOdds = $odds['awayTeamOdds'] ?? [];
        
        if (!empty($homeOdds['favorite']) && $homeOdds['favorite']) {
            $gameData['favorite'] = [
                'team' => $gameData['team1']['abbreviation'],
                'spread' => $odds['spread'] ?? ''
            ];
        } elseif (!empty($awayOdds['favorite']) && $awayOdds['favorite']) {
            $gameData['favorite'] = [
                'team' => $gameData['team2']['abbreviation'],
                'spread' => $odds['spread'] ?? ''
            ];
        }
    }
    
    return $gameData;
}

// Make API call
$api_url = "http://site.api.espn.com/apis/site/v2/sports/hockey/nhl/scoreboard";
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$response = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error: " . $err);
}

// Parse JSON response
$data = json_decode($response, true);
if (empty($data)) {
    die("Failed to parse API response");
}

// Process each game
$events = $data['events'] ?? [];
$games_processed = [];
$skipped_updates = [];

foreach ($events as $index => $event) {
    $game_num = $index + 1;
    $gameData = extractGameData($event, $game_num);
    
    // First, check what's CURRENTLY in the database for this game
    $current_db_query = "SELECT scoreValue, lastUpdatedTime FROM sports_list 
                        WHERE Game = $game_num AND Conference = 'NHL'";
    $current_db_result = mysqli_query($con, $current_db_query);
    $current_db_data = mysqli_fetch_assoc($current_db_result);
    
    $skip_update = false;
    
    // Debug output
    echo "<!-- Game $game_num Debug:\n";
    echo "  Current DB scoreValue: " . ($current_db_data['scoreValue'] ?? 'NULL') . "\n";
    echo "  Current DB lastUpdated: " . ($current_db_data['lastUpdatedTime'] ?? 'NULL') . "\n";
    echo "  New API scoreValue: " . $gameData['score_value'] . "\n";
    echo "  Broadcast: " . $gameData['broadcast'] . "\n";
    
    // If the database currently shows a GOAL was recorded
    if ($current_db_data && $current_db_data['scoreValue'] == 'GOAL') {
        $lastUpdate = $current_db_data['lastUpdatedTime'];
        
        if (!empty($lastUpdate)) {
            $timeDifference = time() - strtotime($lastUpdate);
            echo "  Time since last GOAL: $timeDifference seconds\n";
            
            // If less than 2 minutes since the goal was recorded, skip ALL updates for this game
            if ($timeDifference < 120) {
                $skip_update = true;
                $skipped_updates[] = $game_num;
                echo "  SKIPPING UPDATE - Within 2 minute window\n";
            }
        }
    }
    echo "-->\n\n";
    
    if ($skip_update) {
        $gameData['skipped'] = true;
    } else {
        $gameData['skipped'] = false;
        
        // Prepare data for database update
        $update_data = [
            'Clock' => mysqli_real_escape_string($con, $gameData['clock']),
            'Period' => intval($gameData['period']),
            'Game_Played' => mysqli_real_escape_string($con, $gameData['game_status']),
            'Game_Time' => mysqli_real_escape_string($con, $gameData['game_time']),
            'Game_Date' => mysqli_real_escape_string($con, $gameData['game_date']),
            'broadcast' => mysqli_real_escape_string($con, $gameData['broadcast']),
            'scoreValue' => mysqli_real_escape_string($con, $gameData['score_value']),
            'Team_1_Name' => mysqli_real_escape_string($con, $gameData['team1']['name']),
            'Team_2_Name' => mysqli_real_escape_string($con, $gameData['team2']['name']),
            'Team_1_Record' => mysqli_real_escape_string($con, $gameData['team1']['record']),
            'Team_2_Record' => mysqli_real_escape_string($con, $gameData['team2']['record']),
            'Team_1_Score' => intval($gameData['team1']['score']),
            'Team_2_Score' => intval($gameData['team2']['score']),
            'Team_1_Location' => mysqli_real_escape_string($con, $gameData['team1']['location']),
            'Team_2_Location' => mysqli_real_escape_string($con, $gameData['team2']['location']),
            'Team_1_Abbrev' => mysqli_real_escape_string($con, $gameData['team1']['abbreviation']),
            'Team_2_Abbrev' => mysqli_real_escape_string($con, $gameData['team2']['abbreviation']),
            'Team_1_Color' => '"' . mysqli_real_escape_string($con, $gameData['team1']['color_rgb']) . '"',
            'Team_2_Color' => '"' . mysqli_real_escape_string($con, $gameData['team2']['color_rgb']) . '"',
            'Who_Has_Ball' => mysqli_real_escape_string($con, $gameData['who_has_possession']),
            'lastPlay' => mysqli_real_escape_string($con, $gameData['last_play']),
            'OverUnder' => mysqli_real_escape_string($con, $gameData['over_under']),
            'lastPlayerID' => mysqli_real_escape_string($con, $gameData['last_player']['id'] ?? ''),
            'lastPlayerName' => mysqli_real_escape_string($con, $gameData['last_player']['name'] ?? ''),
            'lastPlayerPos' => mysqli_real_escape_string($con, $gameData['last_player']['position'] ?? ''),
            'lastPlayerTeamID' => mysqli_real_escape_string($con, $gameData['last_player']['team_id'] ?? ''),
            'jersey' => mysqli_real_escape_string($con, $gameData['last_player']['jersey'] ?? ''),
            'driveDesc' => mysqli_real_escape_string($con, $gameData['drive_desc']),
            'Favorite_Team' => mysqli_real_escape_string($con, $gameData['favorite']['team'] ?? ''),
            'Favorite_Points' => mysqli_real_escape_string($con, $gameData['favorite']['spread'] ?? ''),
            'lastUpdatedTime' => $current_timestamp
        ];
        
        // Extract RGB values for separate columns
        $rgb1 = explode(',', $gameData['team1']['color_rgb']);
        $update_data['Team_1_Color_R'] = $rgb1[0] ?? '255';
        $update_data['Team_1_Color_G'] = $rgb1[1] ?? '255';
        $update_data['Team_1_Color_B'] = $rgb1[2] ?? '255';
        
        $rgb2 = explode(',', $gameData['team2']['color_rgb']);
        $update_data['Team_2_Color_R'] = $rgb2[0] ?? '255';
        $update_data['Team_2_Color_G'] = $rgb2[1] ?? '255';
        $update_data['Team_2_Color_B'] = $rgb2[2] ?? '255';
        
        // Build UPDATE query
        $set_clauses = [];
        foreach ($update_data as $field => $value) {
            $set_clauses[] = "$field='$value'";
        }
        
        $query = "UPDATE sports_list SET " . implode(', ', $set_clauses) . 
                 " WHERE Game = $game_num AND Conference = 'NHL'";
        
        mysqli_query($con, $query);
    }
    
    $games_processed[] = $gameData;
}

// HTML Output
?>
<!DOCTYPE html>
<html>
<head>
    <title>NHL Games Database Update</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f5f5f5;
        }
        .header {
            text-align: center;
            color: #2c3e50;
            margin-bottom: 30px;
        }
        .update-time {
            text-align: center;
            color: #7f8c8d;
            font-size: 14px;
            margin-bottom: 20px;
        }
        
        /* Database Table Styles */
        .database-table {
            width: 100%;
            background: white;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow-x: auto;
        }
        .database-table table {
            width: 100%;
            border-collapse: collapse;
            font-size: 11px;
        }
        .database-table th {
            background: #2c3e50;
            color: white;
            padding: 8px 4px;
            text-align: left;
            font-size: 10px;
            position: sticky;
            top: 0;
        }
        .database-table td {
            padding: 6px 4px;
            border-bottom: 1px solid #ddd;
            font-size: 10px;
        }
        .database-table tr:hover {
            background-color: #f5f5f5;
        }
        .status-not-started { color: #95a5a6; }
        .status-in-session { color: #27ae60; font-weight: bold; }
        .status-half-time { color: #f39c12; }
        .status-game-done { color: #34495e; }
        .skipped-row {
            background-color: #fff9c4 !important;
        }
        .goal-indicator {
            background: #ff5252;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
        }
        .update-indicator {
            display: inline-block;
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 10px;
            font-weight: bold;
        }
        .update-indicator.updated {
            background: #c8e6c9;
            color: #2e7d32;
        }
        .update-indicator.skipped {
            background: #fff176;
            color: #f57c00;
        }
        .player-info {
            background: #e3f2fd;
            padding: 2px;
            border-radius: 3px;
        }
        .broadcast-cell {
            background: #e3f2fd;
            font-weight: bold;
        }
        
        /* Card Grid Styles */
        .games-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .game-card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 15px;
            position: relative;
        }
        .game-card.skipped {
            background: #fff9c4;
        }
        .game-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }
        .game-status {
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
            font-weight: bold;
        }
        .teams-container {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 15px 0;
        }
        .team {
            flex: 1;
            text-align: center;
        }
        .team-name {
            font-weight: bold;
            margin-bottom: 5px;
        }
        .team-score {
            font-size: 32px;
            font-weight: bold;
            margin: 10px 0;
        }
        .team-record {
            font-size: 12px;
            color: #666;
        }
        .vs {
            padding: 0 20px;
            color: #999;
        }
        .game-info {
            background: #f8f9fa;
            padding: 10px;
            border-radius: 4px;
            margin-top: 15px;
            font-size: 13px;
        }
        .game-info-row {
            display: flex;
            justify-content: space-between;
            margin: 5px 0;
        }
        .game-info-label {
            color: #666;
        }
        .skip-indicator {
            position: absolute;
            top: 10px;
            right: 10px;
            background: #ffc107;
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: bold;
        }
        .last-play {
            background: #e1f5fe;
            padding: 10px;
            border-radius: 4px;
            margin-top: 10px;
            font-size: 13px;
        }
        .summary-section {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-top: 30px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }
        .stat-box {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 4px;
            text-align: center;
        }
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #2c3e50;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
        .skipped-warning {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 10px;
            margin: 10px auto;
            width: 80%;
            border-radius: 5px;
            text-align: center;
        }
    </style>
</head>
<body>
    <h1 class="header">NHL Games Database Update</h1>
    <div class="update-time">Last Updated: <?= $time_updated ?> Eastern</div>
    
    <?php if (count($skipped_updates) > 0): ?>
    <div class="skipped-warning">
        <strong>⚠️ Timing Restriction Active:</strong> Games <?= implode(', ', $skipped_updates) ?> 
        were not updated due to recent goal (2-minute hold period)
    </div>
    <?php endif; ?>
    
    <!-- Detailed Database Table -->
    <h2 style="margin-top: 30px;">Database Update Details</h2>
    <div class="database-table">
        <table>
            <thead>
                <tr>
                    <th>Game</th>
                    <th>Update</th>
                    <th>Status</th>
                    <th>Time</th>
                    <th>Date</th>
                    <th>Broadcast</th>
                    <th>Away Team</th>
                    <th>Away Score</th>
                    <th>Away Record</th>
                    <th>Home Team</th>
                    <th>Home Score</th>
                    <th>Home Record</th>
                    <th>Period</th>
                    <th>Clock</th>
                    <th>Score Event</th>
                    <th>Last Player</th>
                    <th>Player ID</th>
                    <th>Position</th>
                    <th>Jersey</th>
                    <th>Team ID</th>
                    <th>Last Play</th>
                    <th>Possession</th>
                    <th>Drive Desc</th>
                    <th>Favorite</th>
                    <th>Spread</th>
                    <th>O/U</th>
                    <th>Updated</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($games_processed as $game): ?>
                <tr class="<?= $game['skipped'] ? 'skipped-row' : '' ?>">
                    <td><?= $game['game_num'] ?></td>
                    <td>
                        <?php if ($game['skipped']): ?>
                            <span class="update-indicator skipped">SKIPPED</span>
                        <?php else: ?>
                            <span class="update-indicator updated">UPDATED</span>
                        <?php endif; ?>
                    </td>
                    <td class="status-<?= strtolower(str_replace('_', '-', $game['game_status'])) ?>">
                        <?= $game['game_status'] ?>
                    </td>
                    <td><?= $game['game_time'] ?></td>
                    <td><?= date('Y-m-d', strtotime($game['game_date'])) ?></td>
                    <td class="broadcast-cell"><?= $game['broadcast'] ?: '-' ?></td>
                    <td><strong><?= $game['team2']['abbreviation'] ?></strong></td>
                    <td><strong><?= $game['team2']['score'] ?></strong></td>
                    <td><?= $game['team2']['record'] ?></td>
                    <td><strong><?= $game['team1']['abbreviation'] ?></strong></td>
                    <td><strong><?= $game['team1']['score'] ?></strong></td>
                    <td><?= $game['team1']['record'] ?></td>
                    <td><?= $game['period'] ?></td>
                    <td><?= $game['clock'] ?></td>
                    <td>
                        <?php if ($game['score_value'] == 'GOAL'): ?>
                            <span class="goal-indicator">GOAL</span>
                        <?php else: ?>
                            <?= $game['score_value'] ?>
                        <?php endif; ?>
                    </td>
                    <td class="player-info">
                        <?= $game['last_player']['name'] ?? '-' ?>
                    </td>
                    <td><?= $game['last_player']['id'] ?? '-' ?></td>
                    <td><?= $game['last_player']['position'] ?? '-' ?></td>
                    <td><?= $game['last_player']['jersey'] ?? '-' ?></td>
                    <td><?= $game['last_player']['team_id'] ?? '-' ?></td>
                    <td style="max-width: 200px; overflow: hidden; text-overflow: ellipsis;" 
                        title="<?= htmlspecialchars($game['last_play']) ?>">
                        <?= !empty($game['last_play']) ? 
                            htmlspecialchars(substr($game['last_play'], 0, 50)) . 
                            (strlen($game['last_play']) > 50 ? '...' : '') : '-' ?>
                    </td>
                    <td><?= $game['who_has_possession'] ?: '-' ?></td>
                    <td><?= $game['drive_desc'] ?: '-' ?></td>
                    <td><?= $game['favorite']['team'] ?? '-' ?></td>
                    <td><?= $game['favorite']['spread'] ?? '-' ?></td>
                    <td><?= $game['over_under'] ?: '-' ?></td>
                    <td style="font-size: 9px;">
                        <?php 
                        $time_ago = time() - strtotime($current_timestamp);
                        if ($time_ago < 60) {
                            echo $time_ago . "s ago";
                        } elseif ($time_ago < 3600) {
                            echo round($time_ago / 60) . "m ago";
                        } else {
                            echo date('g:i a', strtotime($current_timestamp));
                        }
                        ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Game Cards Grid -->
    <h2 style="margin-top: 30px;">Game Overview</h2>
    <div class="games-grid">
        <?php foreach ($games_processed as $game): ?>
        <div class="game-card <?= $game['skipped'] ? 'skipped' : '' ?>">
            <?php if ($game['skipped']): ?>
                <div class="skip-indicator">SKIPPED (2-min rule)</div>
            <?php endif; ?>
            
            <div class="game-header">
                <span>Game #<?= $game['game_num'] ?></span>
                <span class="game-status status-<?= strtolower(str_replace('_', '-', $game['game_status'])) ?>">
                    <?= $game['game_status'] ?>
                </span>
            </div>
            
            <div class="teams-container">
                <div class="team">
                    <div class="team-name"><?= $game['team2']['abbreviation'] ?></div>
                    <div class="team-score"><?= $game['team2']['score'] ?></div>
                    <div class="team-record"><?= $game['team2']['record'] ?></div>
                </div>
                <div class="vs">@</div>
                <div class="team">
                    <div class="team-name"><?= $game['team1']['abbreviation'] ?></div>
                    <div class="team-score"><?= $game['team1']['score'] ?></div>
                    <div class="team-record"><?= $game['team1']['record'] ?></div>
                </div>
            </div>
            
            <?php if (in_array($game['game_status'], ['IN_SESSION', 'HALF_TIME'])): ?>
            <div class="game-info">
                <div class="game-info-row">
                    <span class="game-info-label">Period:</span>
                    <span><?= $game['drive_desc'] ?></span>
                </div>
                <div class="game-info-row">
                    <span class="game-info-label">Clock:</span>
                    <span><?= $game['clock'] ?></span>
                </div>
                <?php if ($game['broadcast']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Broadcast:</span>
                    <span style="font-weight: bold; color: #1976d2;"><?= $game['broadcast'] ?></span>
                </div>
                <?php endif; ?>
                <?php if ($game['who_has_possession']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Last Play By:</span>
                    <span><?= $game['who_has_possession'] ?></span>
                </div>
                <?php endif; ?>
                <?php if ($game['score_value'] == 'GOAL'): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Event:</span>
                    <span class="goal-indicator">GOAL!</span>
                </div>
                <?php endif; ?>
            </div>
            
            <?php if ($game['last_play']): ?>
            <div class="last-play">
                <strong>Last Play:</strong> <?= htmlspecialchars($game['last_play']) ?>
                <?php if (!empty($game['last_player']['name'])): ?>
                    <div class="player-info">
                        Player: #<?= $game['last_player']['jersey'] ?> <?= $game['last_player']['name'] ?> 
                        (<?= $game['last_player']['position'] ?>)
                    </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php else: ?>
            <div class="game-info">
                <div class="game-info-row">
                    <span class="game-info-label">Game Time:</span>
                    <span><?= $game['game_time'] ?></span>
                </div>
                <?php if ($game['broadcast']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Broadcast:</span>
                    <span style="font-weight: bold; color: #1976d2;"><?= $game['broadcast'] ?></span>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>
    </div>
    
    <div class="summary-section">
        <h2>Update Summary</h2>
        <div class="stats-grid">
            <div class="stat-box">
                <div class="stat-value"><?= count($games_processed) ?></div>
                <div class="stat-label">Total Games</div>
            </div>
            <div class="stat-box">
                <div class="stat-value"><?= count($games_processed) - count($skipped_updates) ?></div>
                <div class="stat-label">Games Updated</div>
            </div>
            <div class="stat-box">
                <div class="stat-value"><?= count($skipped_updates) ?></div>
                <div class="stat-label">Games Skipped</div>
            </div>
            <div class="stat-box">
                <div class="stat-value">
                    <?= count(array_filter($games_processed, function($g) { return $g['game_status'] == 'IN_SESSION'; })) ?>
                </div>
                <div class="stat-label">Live Games</div>
            </div>
        </div>
        
        <div style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 4px;">
            <h3 style="margin: 0 0 10px 0;">Database Information</h3>
            <ul style="margin: 0; padding-left: 20px;">
                <li>Database: <?= $database_gscroll ?></li>
                <li>Table: sports_list</li>
                <li>Conference: NHL</li>
                <li>API Source: ESPN NHL Scoreboard</li>
                <li>Broadcast Information: Included for all games</li>
                <li>Timing Restriction: 2 minutes after goal</li>
                <li>Update Time: <?= $time_updated ?></li>
            </ul>
        </div>
    </div>
</body>
</html>

<?php
// Close database connection
mysqli_close($con);

// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/NHL_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
curl_close($ch);


// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/Scoring_ALERT.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
} else {
    echo $response; // ADD THIS LINE to output the response
}
curl_close($ch);

?>