<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp = date('Y-m-d H:i:s');

// Check if columns exist and add if needed
$check_column = "SHOW COLUMNS FROM `sports_list` LIKE 'scoreValue'";
$column_result = mysqli_query($con, $check_column);
if (mysqli_num_rows($column_result) == 0) {
    $add_column = "ALTER TABLE `sports_list` ADD COLUMN `scoreValue` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_column);
}

// Check for broadcast column and add if needed
$check_broadcast = "SHOW COLUMNS FROM `sports_list` LIKE 'broadcast'";
$broadcast_result = mysqli_query($con, $check_broadcast);
if (mysqli_num_rows($broadcast_result) == 0) {
    $add_broadcast = "ALTER TABLE `sports_list` ADD COLUMN `broadcast` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_broadcast);
}

// NEW FUNCTION: Get correct field goal kicker information
function getFieldGoalKicker($eventId) {
    if (!$eventId) return null;
    
    $url = "https://site.api.espn.com/apis/site/v2/sports/football/nfl/summary?event={$eventId}";
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_USERAGENT => 'Mozilla/5.0',
    ]);
    
    $raw = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($err || $code !== 200 || !$raw) {
        return null; // Fail silently, use original data
    }
    
    $data = json_decode($raw, true);
    if (!$data) return null;
    
    // Get scoring plays
    $scoringPlays = null;
    if (isset($data['scoringPlays']) && is_array($data['scoringPlays']) && $data['scoringPlays']) {
        $scoringPlays = $data['scoringPlays'];
    } elseif (isset($data['gamecast']['scoringPlays']) && is_array($data['gamecast']['scoringPlays']) && $data['gamecast']['scoringPlays']) {
        $scoringPlays = $data['gamecast']['scoringPlays'];
    }
    
    if (!$scoringPlays) return null;
    
    // Find the last field goal
    $lastFieldGoal = null;
    foreach (array_reverse($scoringPlays) as $play) {
        $typeText = $play['type']['text'] ?? '';
        if (stripos($typeText, 'field goal') !== false) {
            $lastFieldGoal = $play;
            break;
        }
    }
    
    if (!$lastFieldGoal) return null;
    
    // Extract player info
    $playerId = $lastFieldGoal['athletes'][0]['id'] ?? null;
    $jersey = $lastFieldGoal['athletes'][0]['jersey'] ?? null;
    $fullname = $lastFieldGoal['athletes'][0]['fullName'] ?? null;
    $teamId = $lastFieldGoal['team']['id'] ?? null;
    
    // If we don't have player info, try to parse from text and match in boxscore
    if ((!$playerId || !$jersey) && isset($lastFieldGoal['text'])) {
        $text = $lastFieldGoal['text'];
        $kickerName = null;
        
        // Parse kicker name from text patterns
        $patterns = [
            '/^([A-Za-z\.\'\- ]+?)\s+\d+\s*(?:Yd|yd|YDS)?\s*Field Goal/i',
            '/^([A-Za-z\.\'\- ]+?)\s+Made\s+Field Goal/i',
            '/^([A-Za-z\.\'\- ]+?)\s+Field Goal/i',
        ];
        
        foreach ($patterns as $p) {
            if (preg_match($p, $text, $m)) {
                $kickerName = trim($m[1]);
                break;
            }
        }
        
        // Try to find kicker in boxscore
        if ($kickerName && isset($data['boxscore']['players']) && is_array($data['boxscore']['players'])) {
            foreach ($data['boxscore']['players'] as $teamBlock) {
                // Filter by team if we have it
                if ($teamId && isset($teamBlock['team']['id']) && 
                    (string)$teamBlock['team']['id'] !== (string)$teamId) {
                    continue;
                }
                
                if (!isset($teamBlock['statistics']) || !is_array($teamBlock['statistics'])) continue;
                
                foreach ($teamBlock['statistics'] as $statGroup) {
                    if (($statGroup['name'] ?? '') !== 'kicking') continue;
                    if (!isset($statGroup['athletes']) || !is_array($statGroup['athletes'])) continue;
                    
                    foreach ($statGroup['athletes'] as $athRow) {
                        $ath = $athRow['athlete'] ?? null;
                        if (!$ath) continue;
                        
                        $name = $ath['displayName'] ?? '';
                        
                        // Match on name (case-insensitive)
                        if (mb_strtolower($name) === mb_strtolower($kickerName) ||
                            mb_strtolower($name) === mb_strtolower(trim(preg_replace('/\s+Jr\.?|\s+Sr\.?/i', '', $kickerName)))) {
                            $playerId = $playerId ?: ($ath['id'] ?? null);
                            $jersey = $jersey ?: ($ath['jersey'] ?? null);
                            $fullname = $fullname ?: $name;
                            break 3; // Break out of all loops
                        }
                    }
                }
            }
        }
    }
    
    return [
        'playerId' => $playerId,
        'playerName' => $fullname,
        'jersey' => $jersey,
        'teamId' => $teamId
    ];
}

// Helper function to convert hex to RGB
function hexToRgb($hex) {
    $hex = str_replace(['#', '"'], '', $hex);
    if (strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    
    if (strlen($hex) != 6) {
        return array('r' => '000', 'g' => '000', 'b' => '000', 'rgb' => '000,000,000');
    }
    
    $r = str_pad(hexdec(substr($hex, 0, 2)), 3, '0', STR_PAD_LEFT);
    $g = str_pad(hexdec(substr($hex, 2, 2)), 3, '0', STR_PAD_LEFT);
    $b = str_pad(hexdec(substr($hex, 4, 2)), 3, '0', STR_PAD_LEFT);
    
    return array('r' => $r, 'g' => $g, 'b' => $b, 'rgb' => "$r,$g,$b");
}

// Function to check if update should be paused due to recent score
function shouldPauseUpdate($game_num, $con) {
    // Query current database values
    $check_query = "SELECT scoreValue, lastUpdatedTime FROM sports_list 
                    WHERE Game = $game_num AND League = 'NFL'";
    $result = mysqli_query($con, $check_query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $currentScoreValue = $row['scoreValue'] ?? '';
        $lastUpdatedTime = $row['lastUpdatedTime'] ?? null;
        
        // Check if there's a score in the database (not NO-SCORE or empty)
        if (!empty($currentScoreValue) && 
            $currentScoreValue != 'NO-SCORE' && 
            $currentScoreValue != ' ' &&
            !empty($lastUpdatedTime)) {
            
            // Calculate time difference in seconds
            $lastUpdateTimestamp = strtotime($lastUpdatedTime);
            $currentTimestamp = time();
            $timeDifference = $currentTimestamp - $lastUpdateTimestamp;
            
            // If less than 150 seconds, pause the update
            if ($timeDifference < 150) {
                return array(
                    'pause' => true,
                    'time_remaining' => 150 - $timeDifference
                );
            }
        }
    }
    
    return array('pause' => false, 'time_remaining' => 0);
}

// Function to determine game status
function getGameStatus($status) {
    if (!isset($status['type'])) {
        return 'NOT_STARTED';
    }
    
    $state = $status['type']['state'] ?? '';
    $period = $status['period'] ?? 0;
    $clock = $status['displayClock'] ?? '0:00';
    
    if ($status['type']['completed'] ?? false) {
        return 'GAME_DONE';
    }
    
    switch($state) {
        case 'pre':
            return 'NOT_STARTED';
        case 'in':
            if ($period == 2 && $clock == '0:00') {
                return 'HALF_TIME';
            }
            return 'IN_SESSION';
        case 'post':
            return 'GAME_DONE';
        default:
            return 'NOT_STARTED';
    }
}

// Function to extract player stats
function extractPlayerStats($leaders, $type) {
    $statsMap = [
        'QB' => 'passingYards',
        'rusher' => 'rushingYards',
        'Receiver' => 'receivingYards'
    ];
    
    $statType = $statsMap[$type] ?? '';
    if (!$statType || !is_array($leaders)) return ['_', '!'];
    
    foreach ($leaders as $leader) {
        if (($leader['name'] ?? '') == $statType && !empty($leader['leaders'])) {
            $leaderData = $leader['leaders'][0] ?? null;
            if ($leaderData) {
                $athlete = $leaderData['athlete'] ?? null;
                $team = $leaderData['team'] ?? null;
                
                if ($athlete && $team) {
                    $playerId = $athlete['id'] ?? '';
                    $fullName = $athlete['fullName'] ?? '';
                    $displayValue = $leaderData['displayValue'] ?? '';
                    $jersey = $athlete['jersey'] ?? '';
                    $position = $athlete['position']['abbreviation'] ?? '';
                    $teamId = $team['id'] ?? '';
                    
                    $formatted = "DN: $fullName | STATS: $displayValue | ID: $playerId | JERSEY: $jersey | POS: $position | TEAM: $teamId";
                    return ['_' . $formatted, '!' . $formatted];
                }
            }
        }
    }
    
    return ['_', '!'];
}

// API CALL
$api_call = "http://site.api.espn.com/apis/site/v2/sports/football/nfl/scoreboard";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error #:" . $err);
}

// Parse JSON response
$data = json_decode($result, true);

if (!$data || !isset($data['events'])) {
    die("Error parsing API response or no events found");
}

$events = $data['events'];
$loop_length = count($events);

// Store all data for table display
$games_data = [];

// Process each game
for ($i = 1; $i <= $loop_length; $i++) {
    $event = $events[$i - 1];
    
    if (!$event || !isset($event['competitions'][0])) {
        continue;
    }
    
    $competition = $event['competitions'][0];
    
    // Extract the event ID for field goal lookups
    $eventId = $event['id'] ?? null;
    
    // Extract basic game info
    $status = $competition['status'] ?? [];
    $clock = $status['displayClock'] ?? '0:00';
    $quarter = $status['period'] ?? 0;
    
    // Game date/time
    $game_time = $event['date'] ?? '';
    $game_time_2 = '';
    if (!empty($game_time)) {
        $game_time_2 = date("g:i a", strtotime($game_time) - 14400); // Eastern time
    }
    
    // Extract broadcast information
    $broadcast = '';
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $broadcast = $broadcast_data['names'][0] ?? '';
        }
    }
    
    // Determine game status
    $game_being_played = getGameStatus($status);
    
    // Get competitors (teams)
    $competitors = $competition['competitors'] ?? [];
    $home_team = null;
    $away_team = null;
    
    foreach ($competitors as $competitor) {
        if (($competitor['homeAway'] ?? '') == 'home') {
            $home_team = $competitor;
        } else {
            $away_team = $competitor;
        }
    }
    
    // Initialize default values
    $team_name1 = '';
    $location_team1 = '';
    $team_abbrev_1 = '';
    $team1_score = '0';
    $team_id_1 = '';
    $team1_record = '0-0';
    $team1_color_rgb = '"000,000,000"';
    $team1_color_r = '000';
    $team1_color_g = '000';
    $team1_color_b = '000';
    
    $team_name2 = '';
    $location_team2 = '';
    $team_abbrev_2 = '';
    $team2_score = '0';
    $team_id_2 = '';
    $team2_record = '0-0';
    $team2_color_rgb = '"000,000,000"';
    $team2_color_r = '000';
    $team2_color_g = '000';
    $team2_color_b = '000';
    
    // Team 1 (Home) data
    if ($home_team) {
        $team1_info = $home_team['team'] ?? [];
        $team_name1 = $team1_info['name'] ?? '';
        $location_team1 = $team1_info['location'] ?? '';
        $team_abbrev_1 = $team1_info['abbreviation'] ?? '';
        $team1_score = $home_team['score'] ?? '0';
        $team_id_1 = $team1_info['id'] ?? '';
        
        // Team 1 color
        $team1_color_hex = $team1_info['color'] ?? '000000';
        if ($team1_color_hex == '000000' || empty($team1_color_hex)) {
            $team1_color_hex = $team1_info['alternateColor'] ?? '000000';
        }
        $team1_colors = hexToRgb($team1_color_hex);
        $team1_color_r = $team1_colors['r'];
        $team1_color_g = $team1_colors['g'];
        $team1_color_b = $team1_colors['b'];
        $team1_color_rgb = '"' . $team1_colors['rgb'] . '"';
        
        // Team 1 record
        $team1_records = $home_team['records'] ?? [];
        foreach ($team1_records as $record) {
            if (($record['type'] ?? '') == 'total') {
                $team1_record = $record['summary'] ?? '0-0';
                break;
            }
        }
    }
    
    // Team 2 (Away) data
    if ($away_team) {
        $team2_info = $away_team['team'] ?? [];
        $team_name2 = $team2_info['name'] ?? '';
        $location_team2 = $team2_info['location'] ?? '';
        $team_abbrev_2 = $team2_info['abbreviation'] ?? '';
        $team2_score = $away_team['score'] ?? '0';
        $team_id_2 = $team2_info['id'] ?? '';
        
        // Team 2 color
        $team2_color_hex = $team2_info['color'] ?? '000000';
        if ($team2_color_hex == '000000' || empty($team2_color_hex)) {
            $team2_color_hex = $team2_info['alternateColor'] ?? '000000';
        }
        $team2_colors = hexToRgb($team2_color_hex);
        $team2_color_r = $team2_colors['r'];
        $team2_color_g = $team2_colors['g'];
        $team2_color_b = $team2_colors['b'];
        $team2_color_rgb = '"' . $team2_colors['rgb'] . '"';
        
        // Team 2 record
        $team2_records = $away_team['records'] ?? [];
        foreach ($team2_records as $record) {
            if (($record['type'] ?? '') == 'total') {
                $team2_record = $record['summary'] ?? '0-0';
                break;
            }
        }
    }
    
    // In-game situation data
    $who_has_ball = '';
    $short_distance_text = '';
    $pos_text = '';
    $is_red_zone = '';
    $score_value = '';
    $last_play_text = '';
    $lastPlayerID = '';
    $lastPlayerName = '';
    $lastPlayerPos = '';
    $lastPlayerTeamID = '';
    $lastPlayerJersey = '';
    $driveDescription = '';
    
    if ($game_being_played == 'IN_SESSION' || $game_being_played == 'HALF_TIME') {
        $situation = $competition['situation'] ?? [];
        
        // Extract last play text
        if (isset($situation['lastPlay'])) {
            $lastPlay = $situation['lastPlay'];
            
            if (isset($lastPlay['text'])) {
                $last_play_text = strip_tags(trim($lastPlay['text']));
                if (strlen($last_play_text) > 255) {
                    $last_play_text = substr($last_play_text, 0, 252) . '...';
                }
            }
            
            // Get score value
            $scoreVal = intval($lastPlay['scoreValue'] ?? 0);
            
             // FAILSAFE: If scoreValue is 6 or 1 (touchdown-related) and lastPlay contains "kick", override to "TOUCHDOWN"
    if (($scoreVal == 6 || $scoreVal == 1) && stripos($last_play_text, 'kick') !== false) {
        $last_play_text = 'TOUCHDOWN!';
    }
            
            // Set score_value based on scoreVal (1 and 6 both = TOUCHDOWN for display continuity)
            switch($scoreVal) {
                case 0: $score_value = "NO-SCORE"; break;
                case 1: $score_value = "TOUCHDOWN"; break;  // Extra point shows as TD
                case 2: $score_value = "TWO-PT"; break;
                case 3: $score_value = "FIELD-GOAL-GOOD"; break;
                case 6: $score_value = "TOUCHDOWN"; break;
                default: $score_value = "SCORE-" . $scoreVal; break;
            }
            
            // Track which team scored for who_has_ball
            if ($scoreVal > 0 && isset($lastPlay['team']['id'])) {
                $scoring_team_id = $lastPlay['team']['id'];
                
                // Immediately set who_has_ball based on scoring team
                if ($scoring_team_id == $team_id_1) {
                    $who_has_ball = $team_abbrev_1;
                } elseif ($scoring_team_id == $team_id_2) {
                    $who_has_ball = $team_abbrev_2;
                }
            }
            
            // Extract drive description
            if (isset($lastPlay['drive']['description'])) {
                $driveDescription = strip_tags(trim($lastPlay['drive']['description']));
                if (strlen($driveDescription) > 100) {
                    $driveDescription = substr($driveDescription, 0, 97) . '...';
                }
            }
            
            // Extract player information (THIS GETS THE PUNTER FOR FIELD GOALS)
            if (isset($lastPlay['athletesInvolved'][0])) {
                $firstPlayer = $lastPlay['athletesInvolved'][0];
                $lastPlayerID = $firstPlayer['id'] ?? '';
                $lastPlayerName = $firstPlayer['fullName'] ?? $firstPlayer['displayName'] ?? '';
                $lastPlayerPos = $firstPlayer['position'] ?? '';
                $lastPlayerJersey = $firstPlayer['jersey'] ?? '';
                $lastPlayerTeamID = $firstPlayer['team']['id'] ?? '';
            }
            
            // NEW CODE: Override with correct kicker data if this is a field goal
            if ($scoreVal == 3 && $eventId) {
                $kickerData = getFieldGoalKicker($eventId);
                if ($kickerData && $kickerData['playerId']) {
                    // Override the player data with the correct kicker information
                    $lastPlayerID = $kickerData['playerId'];
                    $lastPlayerName = $kickerData['playerName'] ?? '';
                    $lastPlayerJersey = $kickerData['jersey'] ?? '';
                    $lastPlayerTeamID = $kickerData['teamId'] ?? $lastPlayerTeamID;
                    $lastPlayerPos = 'K'; // Kicker position
                }
            }
        }
        
        // Possession info
        $possession_id = $situation['possession'] ?? '';
        if (!empty($possession_id)) {
            if ($possession_id == $team_id_1) {
                $who_has_ball = $team_abbrev_1;
            } elseif ($possession_id == $team_id_2) {
                $who_has_ball = $team_abbrev_2;
            }
        }
        
        // Down and distance
        $short_distance_text = $situation['shortDownDistanceText'] ?? '';
        $pos_text = $situation['possessionText'] ?? '';
        
        // Clean possession text
        if (!empty($pos_text) && strpos($pos_text, 'at') !== false) {
            $pos_text = substr($pos_text, strpos($pos_text, "at") + 2);
            $pos_text = str_replace(['"', ' '], ['', '-'], $pos_text);
            $pos_text = ltrim($pos_text, '-');
        }
        
        // Red zone
        $is_red_zone = '';
        if (isset($situation['isRedZone']) && $situation['isRedZone'] === true) {
            $is_red_zone = 'true';
        }
        
        // Clear fields if scoring play (but keep who_has_ball which was set above)
        if ($scoreVal > 0) {
            $short_distance_text = ' ';
            $pos_text = ' ';
            $is_red_zone = ' ';
        }
    } else {
        // Not in session - clear all in-game fields
        $is_red_zone = ' ';
        $who_has_ball = ' ';
        $short_distance_text = ' ';
        $pos_text = ' ';
        $score_value = ' ';
        $last_play_text = ' ';
        $lastPlayerID = ' ';
        $lastPlayerName = ' ';
        $lastPlayerPos = ' ';
        $lastPlayerTeamID = ' ';
        $lastPlayerJersey = ' ';
        $driveDescription = ' ';
    }
    
    // Betting odds extraction
    $favorite_team = '';
    $favorite_points = '';
    $overUnder = '';
    
    $odds = $competition['odds'] ?? [];
    if (!empty($odds) && isset($odds[0])) {
        $oddData = $odds[0];
        if (isset($oddData['details'])) {
            $details = $oddData['details'] ?? '';
            if (strpos($details, $team_abbrev_1) !== false) {
                $favorite_team = $team_abbrev_1;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            } elseif (strpos($details, $team_abbrev_2) !== false) {
                $favorite_team = $team_abbrev_2;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            }
        }
        $overUnder = $oddData['overUnder'] ?? '';
    }
    
    // Player stats extraction
    $leaders = $competition['leaders'] ?? [];
    
    if ($game_being_played == "NOT_STARTED") {
        $rusher1 = '_';
        $rusher2 = '!';
        $QB1 = '_';
        $QB2 = '!';
        $Rec1 = '_';
        $Rec2 = '!';
    } else {
        list($rusher1, $rusher2) = extractPlayerStats($leaders, 'rusher');
        list($QB1, $QB2) = extractPlayerStats($leaders, 'QB');
        list($Rec1, $Rec2) = extractPlayerStats($leaders, 'Receiver');
    }
    
    // CHECK FOR PAUSE DUE TO RECENT SCORE
    $pauseInfo = shouldPauseUpdate($i, $con);
    $isPaused = $pauseInfo['pause'];
    $timeRemaining = $pauseInfo['time_remaining'];
    
    // Check if this is a new score (not yet in database)
    $isNewScore = false;
    if ($scoreVal > 0 && ($game_being_played == 'IN_SESSION' || $game_being_played == 'HALF_TIME')) {
        // Query to check if this score is already in the database
        $check_score = "SELECT scoreValue FROM sports_list WHERE Game = $i AND League = 'NFL'";
        $score_result = mysqli_query($con, $check_score);
        if ($score_result && mysqli_num_rows($score_result) > 0) {
            $current_db_score = mysqli_fetch_assoc($score_result)['scoreValue'] ?? '';
            // If database doesn't have this score yet, it's new
            if ($current_db_score != $score_value) {
                $isNewScore = true;
            }
        }
    }
    
    
    // Clean possession text  
if (!empty($pos_text)) {
    // If it contains "at", extract everything after it
    if (strpos($pos_text, 'at') !== false) {
        $pos_text = substr($pos_text, strpos($pos_text, "at") + 2);
    }
    // Clean up the text
    $pos_text = trim($pos_text); // Remove leading/trailing spaces
    $pos_text = str_replace('"', '', $pos_text); // Remove quotes
    $pos_text = str_replace(' ', '-', $pos_text); // Replace ALL spaces with dashes
}
    
    // Store data for display
    $games_data[] = [
        'Game' => $i,
        'Clock' => $clock,
        'Period' => $quarter,
        'Game_Played' => $game_being_played,
        'Game_Time' => $game_time_2,
        'Game_Date' => $game_time,
        'broadcast' => $broadcast,
        'scoreValue' => $score_value,
        'lastPlay' => $last_play_text,
        'lastPlayerID' => $lastPlayerID,
        'lastPlayerName' => $lastPlayerName,
        'lastPlayerPos' => $lastPlayerPos,
        'lastPlayerTeamID' => $lastPlayerTeamID,
        'jersey' => $lastPlayerJersey,
        'driveDesc' => $driveDescription,
        'Team_1_Name' => $team_name1,
        'Team_1_Record' => $team1_record,
        'Team_1_Score' => $team1_score,
        'Team_1_Location' => $location_team1,
        'Team_1_Abbrev' => $team_abbrev_1,
        'Team_1_Color' => $team1_color_rgb,
        'Team_1_Color_R' => $team1_color_r,
        'Team_1_Color_G' => $team1_color_g,
        'Team_1_Color_B' => $team1_color_b,
        'Team_2_Name' => $team_name2,
        'Team_2_Record' => $team2_record,
        'Team_2_Score' => $team2_score,
        'Team_2_Location' => $location_team2,
        'Team_2_Abbrev' => $team_abbrev_2,
        'Team_2_Color' => $team2_color_rgb,
        'Team_2_Color_R' => $team2_color_r,
        'Team_2_Color_G' => $team2_color_g,
        'Team_2_Color_B' => $team2_color_b,
        'Who_Has_Ball' => $who_has_ball,
        'Distance' => $short_distance_text,
        'Possesion' => $pos_text,
        'Favorite_Team' => $favorite_team,
        'Favorite_Points' => $favorite_points,
        'OverUnder' => $overUnder,
        'Red_Zone' => $is_red_zone,
        'Ru1' => $rusher1,
        'Ru2' => $rusher2,
        'QB1' => $QB1,
        'QB2' => $QB2,
        'Rec1' => $Rec1,
        'Rec2' => $Rec2,
        'is_paused' => $isPaused,
        'time_remaining' => $timeRemaining,
        'is_new_score' => $isNewScore
    ];
    
    // UPDATE DATABASE - SELECTIVE UPDATES DURING PAUSE
    if (!$isPaused) {
        // FULL UPDATE - Game is not paused
        $query = "
            UPDATE sports_list 
            SET 
                Clock = '".addslashes($clock)."',
                Period = '".addslashes($quarter)."',
                Game_Played = '".addslashes($game_being_played)."',
                Game_Time = '".addslashes($game_time_2)."',
                Game_Date = '".addslashes($game_time)."',
                broadcast = '".addslashes($broadcast)."',
                scoreValue = '".addslashes($score_value)."',
                lastPlay = '".addslashes($last_play_text)."',
                lastPlayerID = '".addslashes($lastPlayerID)."',
                lastPlayerName = '".addslashes($lastPlayerName)."',
                lastPlayerPos = '".addslashes($lastPlayerPos)."',
                lastPlayerTeamID = '".addslashes($lastPlayerTeamID)."',
                jersey = '".addslashes($lastPlayerJersey)."',
                driveDesc = '".addslashes($driveDescription)."',
                lastUpdatedTime = '".addslashes($current_timestamp)."',
                
                Team_1_Name = '".addslashes($team_name1)."',
                Team_1_Record = '".addslashes($team1_record)."',
                Team_1_Score = '".addslashes($team1_score)."',
                Team_1_Location = '".addslashes($location_team1)."',
                Team_1_Abbrev = '".addslashes($team_abbrev_1)."',
                Team_1_Color = '".addslashes($team1_color_rgb)."',
                Team_1_Color_R = '".addslashes($team1_color_r)."',
                Team_1_Color_G = '".addslashes($team1_color_g)."',
                Team_1_Color_B = '".addslashes($team1_color_b)."',
                
                Team_2_Name = '".addslashes($team_name2)."',
                Team_2_Record = '".addslashes($team2_record)."',
                Team_2_Score = '".addslashes($team2_score)."',
                Team_2_Location = '".addslashes($location_team2)."',
                Team_2_Abbrev = '".addslashes($team_abbrev_2)."',
                Team_2_Color = '".addslashes($team2_color_rgb)."',
                Team_2_Color_R = '".addslashes($team2_color_r)."',
                Team_2_Color_G = '".addslashes($team2_color_g)."',
                Team_2_Color_B = '".addslashes($team2_color_b)."',
                
                Who_Has_Ball = '".addslashes($who_has_ball)."',
                Distance = '".addslashes($short_distance_text)."',
                Possesion = '".addslashes($pos_text)."',
                Favorite_Team = '".addslashes($favorite_team)."',
                Favorite_Points = '".addslashes($favorite_points)."',
                OverUnder = '".addslashes($overUnder)."',
                Red_Zone = '".addslashes($is_red_zone)."',
                Ru1 = '".addslashes($rusher1)."',
                Ru2 = '".addslashes($rusher2)."',
                QB1 = '".addslashes($QB1)."',
                QB2 = '".addslashes($QB2)."',
                Rec1 = '".addslashes($Rec1)."',
                Rec2 = '".addslashes($Rec2)."'
            WHERE 
                Game = $i AND 
                League = 'NFL'
        ";
        
        $query_run = mysqli_query($con, $query);
        
        if (!$query_run) {
            $games_data[count($games_data) - 1]['error'] = mysqli_error($con);
        } else {
            $games_data[count($games_data) - 1]['error'] = '';
        }
    } else {
        // PARTIAL UPDATE - Game is paused, only update scores
        // This allows extra points to update the score while preserving the touchdown message
        $query = "
            UPDATE sports_list 
            SET 
                Team_1_Score = '".addslashes($team1_score)."',
                Team_2_Score = '".addslashes($team2_score)."'
            WHERE 
                Game = $i AND 
                League = 'NFL'
        ";
        
        $query_run = mysqli_query($con, $query);
        
        if (!$query_run) {
            $games_data[count($games_data) - 1]['error'] = 'PAUSED - Score update failed: ' . mysqli_error($con);
        } else {
            $games_data[count($games_data) - 1]['error'] = 'PAUSED - Scores updated';
        }
    }
}

// HTML Output
?>
<!DOCTYPE html>
<html>
<head>
    <title>NFL Games Database Update - <?php echo $time_updated; ?></title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f5f5f5;
        }
        h1 {
            color: #333;
            text-align: center;
            background: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .summary {
            background: #fff;
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        table {
            width: 100%;
            border-collapse: collapse;
            background: #fff;
            margin: 20px 0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            font-size: 11px;
        }
        th {
            background: #2c3e50;
            color: white;
            padding: 8px 4px;
            text-align: left;
            font-size: 10px;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        td {
            padding: 6px 4px;
            border-bottom: 1px solid #ddd;
            font-size: 10px;
        }
        tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        tr:hover {
            background-color: #f0f0f0;
        }
        .paused-row {
            background-color: #fff59d !important;
        }
        .paused-row:hover {
            background-color: #fff176 !important;
        }
        .countdown-timer {
            background: #ff9800;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 11px;
            display: inline-block;
        }
        .status-updated {
            background: #4caf50;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 11px;
            display: inline-block;
        }
        .status-in-session { background: #d4edda; font-weight: bold; }
        .status-game-done { background: #f8d7da; }
        .status-not-started { background: #fff3cd; }
        .status-half-time { background: #d1ecf1; }
        .truncate {
            max-width: 150px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        .error {
            color: red;
            font-weight: bold;
        }
        .wrapper {
            overflow-x: auto;
        }
        .pause-info {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 10px;
            margin: 10px 0;
            border-radius: 5px;
        }
        .broadcast-cell {
            background: #e3f2fd;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <h1>NFL Games Database Update</h1>
    
    <div class="summary">
        <strong>Update Time:</strong> <?php echo $time_updated; ?><br>
        <strong>Total Games Processed:</strong> <?php echo $loop_length; ?><br>
        <strong>Database:</strong> <?php echo $database_gscroll; ?>
    </div>
    
    <?php 
    // Collect all paused and newly scored games
    $score_status = [];
    foreach ($games_data as $game) {
        if ((isset($game['is_paused']) && $game['is_paused']) || (isset($game['is_new_score']) && $game['is_new_score'])) {
            $next_update_time = time() + ($game['is_new_score'] ? 150 : $game['time_remaining']);
            $score_status[] = [
                'game' => $game['Game'],
                'teams' => $game['Team_2_Abbrev'] . ' @ ' . $game['Team_1_Abbrev'],
                'score' => $game['Team_2_Score'] . '-' . $game['Team_1_Score'],
                'scoreValue' => $game['scoreValue'],
                'next_update' => date('g:i:s a', $next_update_time),
                'seconds_left' => $game['is_new_score'] ? 150 : $game['time_remaining']
            ];
        }
    }
    
    if (count($score_status) > 0): ?>
    <div style="background: #fff3cd; border: 2px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 5px;">
        <h3 style="margin-top: 0; color: #856404;">⏰ Score Update Status - Next Update Times</h3>
        <table style="width: 100%; font-size: 13px; border-collapse: collapse;">
            <thead>
                <tr style="background: #ffc107;">
                    <th style="padding: 8px; text-align: left;">Game #</th>
                    <th style="padding: 8px; text-align: left;">Matchup</th>
                    <th style="padding: 8px; text-align: left;">Score</th>
                    <th style="padding: 8px; text-align: left;">Score Type</th>
                    <th style="padding: 8px; text-align: left;">Next Update Time</th>
                    <th style="padding: 8px; text-align: left;">Countdown</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($score_status as $status): ?>
                <tr style="background: #fff9e6;">
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><?php echo $status['game']; ?></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><strong><?php echo $status['teams']; ?></strong></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><?php echo $status['score']; ?></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;">
                        <span style="background: #ff9800; color: white; padding: 2px 6px; border-radius: 3px;">
                            <?php echo $status['scoreValue']; ?>
                        </span>
                    </td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;"><strong><?php echo $status['next_update']; ?></strong></td>
                    <td style="padding: 8px; border-bottom: 1px solid #ffc107;">
                        <span style="background: #dc3545; color: white; padding: 2px 6px; border-radius: 3px;">
                            <?php echo $status['seconds_left']; ?>s remaining
                        </span>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <p style="margin: 10px 0 0 0; font-size: 12px; color: #856404;">
            <strong>Note:</strong> Rows highlighted in yellow will remain frozen (except scores) until the countdown expires.
        </p>
    </div>
    <?php endif; ?>
    
    <div class="wrapper">
        <table>
            <thead>
                <tr>
                    <th>Game</th>
                    <th>Pause Status</th>
                    <th>Status</th>
                    <th>Clock</th>
                    <th>Period</th>
                    <th>Game Time</th>
                    <th>Game Date</th>
                    <th>Broadcast</th>
                    <th>Score Value</th>
                    <th>Who Has Ball</th>
                    <th>Distance</th>
                    <th>Position</th>
                    <th>Red Zone</th>
                    <th>Last Play</th>
                    <th>Last Player ID</th>
                    <th>Last Player Name</th>
                    <th>Last Player Pos</th>
                    <th>Last Player Team</th>
                    <th>Jersey</th>
                    <th>Drive Desc</th>
                    <th colspan="9">Team 1 (Home)</th>
                    <th colspan="9">Team 2 (Away)</th>
                    <th>Favorite</th>
                    <th>Spread</th>
                    <th>O/U</th>
                    <th colspan="6">Leaders</th>
                    <th>Update Status</th>
                </tr>
                <tr>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th>Name</th>
                    <th>Abbrev</th>
                    <th>Location</th>
                    <th>Record</th>
                    <th>Score</th>
                    <th>Color</th>
                    <th>R</th>
                    <th>G</th>
                    <th>B</th>
                    <th>Name</th>
                    <th>Abbrev</th>
                    <th>Location</th>
                    <th>Record</th>
                    <th>Score</th>
                    <th>Color</th>
                    <th>R</th>
                    <th>G</th>
                    <th>B</th>
                    <th></th>
                    <th></th>
                    <th></th>
                    <th>QB1</th>
                    <th>QB2</th>
                    <th>Ru1</th>
                    <th>Ru2</th>
                    <th>Rec1</th>
                    <th>Rec2</th>
                    <th></th>
                </tr>
            </thead>
            <tbody>
                <?php 
                $paused_count = 0;
                $updated_count = 0;
                $new_score_count = 0;
                
                foreach ($games_data as $game): 
                    $row_class = '';
                    if (isset($game['is_new_score']) && $game['is_new_score']) {
                        $row_class = 'paused-row';
                        $new_score_count++;
                    } elseif (isset($game['is_paused']) && $game['is_paused']) {
                        $row_class = 'paused-row';
                        $paused_count++;
                    } else {
                        $updated_count++;
                    }
                ?>
                <tr class="<?php echo $row_class; ?>">
                    <td><?php echo $game['Game']; ?></td>
                    <td>
                        <?php if (isset($game['is_new_score']) && $game['is_new_score']): ?>
                            <span class="countdown-timer">NEW SCORE!</span>
                        <?php elseif (isset($game['is_paused']) && $game['is_paused']): ?>
                            <span class="countdown-timer">
                                <?php echo $game['time_remaining']; ?>s left
                            </span>
                        <?php else: ?>
                            <span class="status-updated">UPDATED</span>
                        <?php endif; ?>
                    </td>
                    <td class="status-<?php echo strtolower(str_replace('_', '-', $game['Game_Played'])); ?>">
                        <?php echo $game['Game_Played']; ?>
                    </td>
                    <td><?php echo $game['Clock']; ?></td>
                    <td><?php echo $game['Period']; ?></td>
                    <td><?php echo $game['Game_Time']; ?></td>
                    <td><?php echo substr($game['Game_Date'], 0, 16); ?></td>
                    <td class="broadcast-cell"><?php echo $game['broadcast']; ?></td>
                    <td><?php echo $game['scoreValue']; ?></td>
                    <td><?php echo $game['Who_Has_Ball']; ?></td>
                    <td><?php echo $game['Distance']; ?></td>
                    <td><?php echo $game['Possesion']; ?></td>
                    <td><?php echo $game['Red_Zone']; ?></td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['lastPlay']); ?>">
                        <?php echo htmlspecialchars($game['lastPlay']); ?>
                    </td>
                    <td><?php echo $game['lastPlayerID']; ?></td>
                    <td><?php echo $game['lastPlayerName']; ?></td>
                    <td><?php echo $game['lastPlayerPos']; ?></td>
                    <td><?php echo $game['lastPlayerTeamID']; ?></td>
                    <td><?php echo $game['jersey']; ?></td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['driveDesc']); ?>">
                        <?php echo htmlspecialchars($game['driveDesc']); ?>
                    </td>
                    <td><?php echo $game['Team_1_Name']; ?></td>
                    <td><strong><?php echo $game['Team_1_Abbrev']; ?></strong></td>
                    <td><?php echo $game['Team_1_Location']; ?></td>
                    <td><?php echo $game['Team_1_Record']; ?></td>
                    <td><strong><?php echo $game['Team_1_Score']; ?></strong></td>
                    <td><?php echo $game['Team_1_Color']; ?></td>
                    <td><?php echo $game['Team_1_Color_R']; ?></td>
                    <td><?php echo $game['Team_1_Color_G']; ?></td>
                    <td><?php echo $game['Team_1_Color_B']; ?></td>
                    <td><?php echo $game['Team_2_Name']; ?></td>
                    <td><strong><?php echo $game['Team_2_Abbrev']; ?></strong></td>
                    <td><?php echo $game['Team_2_Location']; ?></td>
                    <td><?php echo $game['Team_2_Record']; ?></td>
                    <td><strong><?php echo $game['Team_2_Score']; ?></strong></td>
                    <td><?php echo $game['Team_2_Color']; ?></td>
                    <td><?php echo $game['Team_2_Color_R']; ?></td>
                    <td><?php echo $game['Team_2_Color_G']; ?></td>
                    <td><?php echo $game['Team_2_Color_B']; ?></td>
                    <td><?php echo $game['Favorite_Team']; ?></td>
                    <td><?php echo $game['Favorite_Points']; ?></td>
                    <td><?php echo $game['OverUnder']; ?></td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['QB1']); ?>">
                        <?php echo htmlspecialchars(substr($game['QB1'], 0, 20)); ?>
                    </td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['QB2']); ?>">
                        <?php echo htmlspecialchars(substr($game['QB2'], 0, 20)); ?>
                    </td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['Ru1']); ?>">
                        <?php echo htmlspecialchars(substr($game['Ru1'], 0, 20)); ?>
                    </td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['Ru2']); ?>">
                        <?php echo htmlspecialchars(substr($game['Ru2'], 0, 20)); ?>
                    </td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['Rec1']); ?>">
                        <?php echo htmlspecialchars(substr($game['Rec1'], 0, 20)); ?>
                    </td>
                    <td class="truncate" title="<?php echo htmlspecialchars($game['Rec2']); ?>">
                        <?php echo htmlspecialchars(substr($game['Rec2'], 0, 20)); ?>
                    </td>
                    <td>
                        <?php 
                        if ($game['error'] == 'PAUSED') {
                            echo '<span style="color: orange; font-weight: bold;">PAUSED</span>';
                        } elseif (!empty($game['error'])) {
                            echo '<span class="error">' . $game['error'] . '</span>';
                        } else {
                            echo '<span style="color: green;">✓</span>';
                        }
                        ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    
    <?php if ($paused_count > 0 || $new_score_count > 0): ?>
    <div class="pause-info">
        <strong>⏸️ Pause Status:</strong><br>
        <?php if ($new_score_count > 0): ?>
            <?php echo $new_score_count; ?> game(s) just scored (will be paused for 150 seconds)<br>
        <?php endif; ?>
        <?php if ($paused_count > 0): ?>
            <?php echo $paused_count; ?> game(s) currently paused from previous scores<br>
        <?php endif; ?>
        <?php echo $updated_count; ?> game(s) updated normally<br>
        <small>Games are paused for 150 seconds after a score to allow users to see the scoring play. Only team scores update during pause.</small>
    </div>
    <?php endif; ?>
    
    <div class="summary">
        <strong>Update completed successfully!</strong><br>
        Total games processed: <?php echo $loop_length; ?><br>
        Games updated: <?php echo $updated_count; ?><br>
        Games with new scores: <?php echo $new_score_count; ?><br>
        Games paused: <?php echo $paused_count; ?><br>
        <small>Yellow highlighted rows indicate games with recent scoring (150-second hold)</small><br>
        <small>Blue highlighted cells show broadcast/channel information</small>
    </div>
</body>
</html>


<?php

// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/NFL_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
} else {
    echo $response; // ADD THIS LINE to output the response
}
curl_close($ch);

// Call external API endpoint
$url = SERVER_URL . "/GLED-V2/App_Output/API/Games/Scoring_ALERT.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
} else {
    echo $response; // ADD THIS LINE to output the response
}
curl_close($ch);

// Close database connection
mysqli_close($con);

?>