<?php
/**
 * College Basketball Upset Detection API
 * Smart timing: Shows last week's data Mon-Thu 3PM, refreshes Thu 3PM-Sun
 */

require_once '../config.php';

// Start session and error reporting
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

if (!$con) {
    die("<div style='background-color: #f8d7da; color: #721c24; padding: 20px; margin: 20px; border-radius: 5px; border: 1px solid #f5c6cb;'>
         <h2>⚠️ Database Connection Failed</h2>
         <p><strong>Error:</strong> " . htmlspecialchars(mysqli_connect_error()) . "</p>
         <p>Please check your database configuration in config.php</p>
         </div>");
}

mysqli_set_charset($con, "utf8mb4");

// Conference ID to name mapping
$CONFERENCE_NAMES = [
    100 => 'Top 25',
    8 => 'Big 12',
    4 => 'Big East',
    7 => 'Big Ten',
    23 => 'SEC',
    2 => 'ACC',
    21 => 'Pac-12',
    44 => 'Mountain West',
    46 => 'WCC',
    62 => 'American',
    3 => 'Atlantic 10',
    11 => 'Conference USA'
];

/**
 * Check if we should fetch new data based on day/time
 */
function shouldFetchNewData() {
    $currentTime = new DateTime('now', new DateTimeZone('America/New_York')); // Using EST/EDT
    $dayOfWeek = (int)$currentTime->format('N'); // 1=Monday, 7=Sunday
    $hour = (int)$currentTime->format('G'); // 0-23 hour format
    
    // Monday (1), Tuesday (2), Wednesday (3): Never fetch
    if ($dayOfWeek >= 1 && $dayOfWeek <= 3) {
        return false;
    }
    
    // Thursday (4): Only fetch if after 3 PM (15:00)
    if ($dayOfWeek == 4) {
        return $hour >= 15;
    }
    
    // Friday (5), Saturday (6), Sunday (7): Always fetch
    if ($dayOfWeek >= 5 && $dayOfWeek <= 7) {
        return true;
    }
    
    return false;
}

/**
 * Get current status message for display
 */
function getStatusMessage() {
    $currentTime = new DateTime('now', new DateTimeZone('America/New_York'));
    $dayName = $currentTime->format('l'); // Full day name
    $time = $currentTime->format('g:i A T');
    
    if (!shouldFetchNewData()) {
        if ($currentTime->format('N') == 4 && $currentTime->format('G') < 15) {
            return "Showing last week's upsets (new data available after 3 PM ET today)";
        }
        return "Showing last week's upsets (new data available Thursday 3 PM ET)";
    }
    
    return "Live data - Updated: $time";
}

/**
 * Setup database tables
 */
function setupDatabaseTables($con) {
    try {
        // Create upset_summary table
        $createSummaryTable = "
        CREATE TABLE IF NOT EXISTS upset_summary (
            id INT AUTO_INCREMENT PRIMARY KEY,
            league VARCHAR(10) DEFAULT NULL,
            fetch_date DATE NOT NULL,
            timestamp DATETIME NOT NULL,
            total_upsets INT NOT NULL DEFAULT 0,
            ranked_teams_upset INT NOT NULL DEFAULT 0,
            biggest_upset_game_id VARCHAR(50) DEFAULT NULL,
            biggest_upset_spread DECIMAL(5,2) DEFAULT NULL,
            biggest_upset_winner_abbrev VARCHAR(10) DEFAULT NULL,
            biggest_upset_loser_abbrev VARCHAR(10) DEFAULT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_fetch_date (fetch_date),
            INDEX idx_league (league)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        mysqli_query($con, $createSummaryTable);
        
        // Create upsets table
        $createUpsetsTable = "
        CREATE TABLE IF NOT EXISTS upsets (
            id INT AUTO_INCREMENT PRIMARY KEY,
            league VARCHAR(10) DEFAULT NULL,
            fetch_date DATE NOT NULL,
            game_id VARCHAR(50) NOT NULL,
            game_date DATETIME NOT NULL,
            conference VARCHAR(50) NOT NULL,
            upset_type ENUM('ranking', 'betting') NOT NULL,
            spread DECIMAL(5,2) DEFAULT NULL,
            winner_id VARCHAR(20) NOT NULL,
            winner_name VARCHAR(100) NOT NULL,
            winner_abbreviation VARCHAR(10) NOT NULL,
            winner_score INT NOT NULL,
            winner_rank INT DEFAULT NULL,
            winner_record VARCHAR(20) DEFAULT NULL,
            winner_conference VARCHAR(50) DEFAULT NULL,
            loser_id VARCHAR(20) NOT NULL,
            loser_name VARCHAR(100) NOT NULL,
            loser_abbreviation VARCHAR(10) NOT NULL,
            loser_score INT NOT NULL,
            loser_rank INT DEFAULT NULL,
            loser_record VARCHAR(20) DEFAULT NULL,
            loser_conference VARCHAR(50) DEFAULT NULL,
            venue VARCHAR(200) DEFAULT NULL,
            attendance INT DEFAULT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_game_date (fetch_date, game_id),
            INDEX idx_fetch_date (fetch_date),
            INDEX idx_game_date (game_date),
            INDEX idx_conference (conference),
            INDEX idx_upset_type (upset_type),
            INDEX idx_league (league)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        mysqli_query($con, $createUpsetsTable);
        
        return ['success' => true, 'message' => 'Tables verified/created successfully'];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Fetch game data from ESPN API
 */
function fetchGamesForDate($date) {
    $url = "http://site.api.espn.com/apis/site/v2/sports/basketball/mens-college-basketball/scoreboard?dates={$date}&limit=200";
    
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            error_log("Error fetching date {$date}: HTTP {$httpCode}");
            return [];
        }
        
        $data = json_decode($response, true);
        return $data['events'] ?? [];
        
    } catch (Exception $e) {
        error_log("Error fetching date {$date}: " . $e->getMessage());
        return [];
    }
}

/**
 * Determine if a game is an upset
 */
function isUpset($game) {
    $competition = $game['competitions'][0];
    $competitors = $competition['competitors'];
    
    $home = null;
    $away = null;
    foreach ($competitors as $competitor) {
        if ($competitor['homeAway'] === 'home') {
            $home = $competitor;
        } else {
            $away = $competitor;
        }
    }
    
    if (!$home || !$away) {
        return null;
    }
    
    if ($competition['status']['type']['state'] !== 'post') {
        return null;
    }
    
    $homeRank = $home['curatedRank']['current'] ?? 99;
    $awayRank = $away['curatedRank']['current'] ?? 99;
    $winner = $home['winner'] ? $home : $away;
    $loser = $home['winner'] ? $away : $home;
    
    // Check for ranking upset: unranked beats ranked
    if ($homeRank <= 25 && $awayRank > 25 && $away['winner']) {
        return createUpsetObject('ranking', $winner, $loser, $game, null);
    }
    if ($awayRank <= 25 && $homeRank > 25 && $home['winner']) {
        return createUpsetObject('ranking', $winner, $loser, $game, null);
    }
    
    // Check for ranking upset: lower ranked beats higher ranked
    if ($homeRank <= 25 && $awayRank <= 25) {
        if ($homeRank < $awayRank && $away['winner']) {
            return createUpsetObject('ranking', $winner, $loser, $game, null);
        }
        if ($awayRank < $homeRank && $home['winner']) {
            return createUpsetObject('ranking', $winner, $loser, $game, null);
        }
    }
    
    // Check for betting upset
    $odds = $competition['odds'][0] ?? null;
    if ($odds) {
        $homeWasFavored = $odds['homeTeamOdds']['favorite'] ?? false;
        $awayWasFavored = $odds['awayTeamOdds']['favorite'] ?? false;
        $spread = abs($odds['spread'] ?? 0);
        
        if ($homeWasFavored && $away['winner'] && $spread >= 7) {
            return createUpsetObject('betting', $winner, $loser, $game, $spread);
        }
        if ($awayWasFavored && $home['winner'] && $spread >= 7) {
            return createUpsetObject('betting', $winner, $loser, $game, $spread);
        }
    }
    
    return null;
}

/**
 * Create upset object with all relevant data
 */
function createUpsetObject($type, $winner, $loser, $game, $spread) {
    $winnerRank = isset($winner['curatedRank']['current']) && $winner['curatedRank']['current'] <= 25 
        ? $winner['curatedRank']['current'] 
        : null;
    
    $loserRank = isset($loser['curatedRank']['current']) && $loser['curatedRank']['current'] <= 25 
        ? $loser['curatedRank']['current'] 
        : null;
    
    $winnerAbbrev = !empty($winner['team']['abbreviation']) ? $winner['team']['abbreviation'] : strtoupper(substr($winner['team']['displayName'], 0, 3));
    $loserAbbrev = !empty($loser['team']['abbreviation']) ? $loser['team']['abbreviation'] : strtoupper(substr($loser['team']['displayName'], 0, 3));
    
    return [
        'gameId' => $game['id'],
        'date' => $game['date'],
        'type' => $type,
        'spread' => $spread,
        'winner' => [
            'id' => $winner['id'],
            'name' => $winner['team']['displayName'],
            'abbreviation' => $winnerAbbrev,
            'score' => $winner['score'],
            'rank' => $winnerRank,
            'record' => $winner['records'][0]['summary'] ?? 'N/A'
        ],
        'loser' => [
            'id' => $loser['id'],
            'name' => $loser['team']['displayName'],
            'abbreviation' => $loserAbbrev,
            'score' => $loser['score'],
            'rank' => $loserRank,
            'record' => $loser['records'][0]['summary'] ?? 'N/A'
        ],
        'venue' => $game['competitions'][0]['venue']['fullName'] ?? 'Unknown',
        'attendance' => $game['competitions'][0]['attendance'] ?? 0
    ];
}

/**
 * Get all college basketball upsets from API
 */
function getCollegeBasketballUpsets() {
    $results = [
        'timestamp' => date('c'),
        'mysqlTimestamp' => date('Y-m-d H:i:s'),
        'totalUpsets' => 0,
        'rankedTeamsUpset' => 0,
        'biggestUpsetBySpread' => null,
        'conferences' => [],
        'allUpsets' => []
    ];
    
    // Get yesterday's and today's dates in YYYYMMDD format (Pacific Time)
    date_default_timezone_set('America/Los_Angeles');
    $today = date('Ymd');
    $yesterday = date('Ymd', strtotime('-1 day'));
    
    // Fetch games from yesterday and today
    $allGames = [];
    $allGames = array_merge($allGames, fetchGamesForDate($yesterday));
    $allGames = array_merge($allGames, fetchGamesForDate($today));
    
    // DEBUG: Let's see what we're getting
    echo "<!-- DEBUG INFO:\n";
    echo "Yesterday date: $yesterday\n";
    echo "Today date: $today\n";
    echo "Total games fetched: " . count($allGames) . "\n";

    $completedGames = 0;
    foreach ($allGames as $game) {
        $status = $game['competitions'][0]['status']['type']['state'] ?? 'unknown';
        if ($status === 'post') {
            $completedGames++;
            $home = null;
            $away = null;
            foreach ($game['competitions'][0]['competitors'] as $competitor) {
                if ($competitor['homeAway'] === 'home') $home = $competitor;
                else $away = $competitor;
            }
            if ($home && $away) {
                echo "\nCompleted game: ";
                echo $away['team']['displayName'] . " (#" . ($away['curatedRank']['current'] ?? 'UR') . ") ";
                echo $away['score'] . " @ ";
                echo $home['team']['displayName'] . " (#" . ($home['curatedRank']['current'] ?? 'UR') . ") ";
                echo $home['score'] . "\n";
            }
        }
    }
    echo "Completed games: $completedGames\n";
    
    echo "Completed games: $completedGames\n";

    // DEBUG: Show conference structure for first game
    if (count($allGames) > 0) {
        echo "\n=== CONFERENCE DATA STRUCTURE ===\n";
        $firstGame = $allGames[0];
        if (isset($firstGame['competitions'][0]['competitors'][0]['team'])) {
            $team = $firstGame['competitions'][0]['competitors'][0]['team'];
            echo "Team name: " . ($team['displayName'] ?? 'N/A') . "\n";
            echo "Conference ID: " . ($team['conferenceId'] ?? 'N/A') . "\n";
            if (isset($team['conference'])) {
                echo "Conference object exists:\n";
                print_r($team['conference']);
            } else {
                echo "No conference object found\n";
            }
            if (isset($team['conferenceAbbreviation'])) {
                echo "Conference abbreviation: " . $team['conferenceAbbreviation'] . "\n";
            }
        }
        echo "=================================\n";
    }


    echo "-->\n";
    
    $processedGameIds = [];
    
    foreach ($allGames as $game) {
        if (isset($processedGameIds[$game['id']])) {
            continue;
        }
        
        $upset = isUpset($game);
        if ($upset) {
            $processedGameIds[$game['id']] = true;
            
            // Get conference name from game data
            $confId = $game['competitions'][0]['competitors'][0]['team']['conferenceId'] ?? null;
            $confName = 'UNKNOWN';
            if ($confId !== null) {
                global $CONFERENCE_NAMES;
                $confName = $CONFERENCE_NAMES[$confId] ?? "Conference $confId";
            }
            $upset['conference'] = $confName;
            
            $results['allUpsets'][] = $upset;
            $results['totalUpsets']++;
            
            if ($upset['loser']['rank']) {
                $results['rankedTeamsUpset']++;
            }
            
            if ($upset['spread'] && 
                (!$results['biggestUpsetBySpread'] || 
                 $upset['spread'] > $results['biggestUpsetBySpread']['spread'])) {
                $results['biggestUpsetBySpread'] = $upset;
            }
            
            // Group by conference
            if (!isset($results['conferences'][$confName])) {
                $results['conferences'][$confName] = [
                    'upsetCount' => 0,
                    'upsets' => []
                ];
            }
            $results['conferences'][$confName]['upsetCount']++;
            $results['conferences'][$confName]['upsets'][] = $upset;
        }
    }
    
    return $results;
}

/**
 * Update database with upset data
 */
function updateDatabase($upsets, $con, $clearAll = false) {
    $fetchDate = date('Y-m-d');
    
    mysqli_begin_transaction($con);
    
    try {
        if ($clearAll) {
            // Clear only NCAAB data when refreshing
            mysqli_query($con, "DELETE FROM upsets WHERE league = 'NCAAB' OR league IS NULL");
            mysqli_query($con, "DELETE FROM upset_summary WHERE league = 'NCAAB' OR league IS NULL");
        }
        
        // Insert summary data
        $totalUpsets = (int)$upsets['totalUpsets'];
        $rankedUpsets = (int)$upsets['rankedTeamsUpset'];
        $timestamp = $upsets['mysqlTimestamp'];
        
        $biggestUpsetId = 'NULL';
        $biggestUpsetSpread = 'NULL';
        $biggestUpsetWinner = 'NULL';
        $biggestUpsetLoser = 'NULL';
        
        if ($upsets['biggestUpsetBySpread']) {
            $biggestUpsetId = "'" . mysqli_real_escape_string($con, $upsets['biggestUpsetBySpread']['gameId']) . "'";
            $biggestUpsetSpread = $upsets['biggestUpsetBySpread']['spread'];
            $biggestUpsetWinner = "'" . mysqli_real_escape_string($con, $upsets['biggestUpsetBySpread']['winner']['abbreviation']) . "'";
            $biggestUpsetLoser = "'" . mysqli_real_escape_string($con, $upsets['biggestUpsetBySpread']['loser']['abbreviation']) . "'";
        }
        
        $summaryQuery = "INSERT INTO upset_summary 
            (league, fetch_date, timestamp, total_upsets, ranked_teams_upset, 
             biggest_upset_game_id, biggest_upset_spread, 
             biggest_upset_winner_abbrev, biggest_upset_loser_abbrev) 
            VALUES ('NCAAB', '$fetchDate', '$timestamp', $totalUpsets, $rankedUpsets, 
                    $biggestUpsetId, $biggestUpsetSpread, 
                    $biggestUpsetWinner, $biggestUpsetLoser)";
        
        if (!mysqli_query($con, $summaryQuery)) {
            throw new Exception("Failed to insert summary: " . mysqli_error($con));
        }
        
        // Insert individual upsets
        foreach ($upsets['allUpsets'] as $upset) {
            $gameId = mysqli_real_escape_string($con, $upset['gameId']);
            $gameDate = date('Y-m-d H:i:s', strtotime($upset['date']));
            $conference = mysqli_real_escape_string($con, $upset['conference']);
            $upsetType = $upset['type'];
            $spread = $upset['spread'] ? $upset['spread'] : 'NULL';
            
            $winnerId = mysqli_real_escape_string($con, $upset['winner']['id']);
            $winnerName = mysqli_real_escape_string($con, $upset['winner']['name']);
            $winnerAbbrev = mysqli_real_escape_string($con, $upset['winner']['abbreviation']);
            $winnerScore = (int)$upset['winner']['score'];
            $winnerRank = $upset['winner']['rank'] ? (int)$upset['winner']['rank'] : 'NULL';
            $winnerRecord = mysqli_real_escape_string($con, $upset['winner']['record']);
            
            $loserId = mysqli_real_escape_string($con, $upset['loser']['id']);
            $loserName = mysqli_real_escape_string($con, $upset['loser']['name']);
            $loserAbbrev = mysqli_real_escape_string($con, $upset['loser']['abbreviation']);
            $loserScore = (int)$upset['loser']['score'];
            $loserRank = $upset['loser']['rank'] ? (int)$upset['loser']['rank'] : 'NULL';
            $loserRecord = mysqli_real_escape_string($con, $upset['loser']['record']);
            
            $venue = mysqli_real_escape_string($con, $upset['venue']);
            $attendance = (int)$upset['attendance'];
            
            $upsetQuery = "INSERT INTO upsets 
                (league, fetch_date, game_id, game_date, conference, upset_type, spread,
                 winner_id, winner_name, winner_abbreviation, winner_score, winner_rank, winner_record, winner_conference,
                 loser_id, loser_name, loser_abbreviation, loser_score, loser_rank, loser_record, loser_conference,
                 venue, attendance)
                VALUES 
                ('NCAAB', '$fetchDate', '$gameId', '$gameDate', '$conference', '$upsetType', $spread,
                 '$winnerId', '$winnerName', '$winnerAbbrev', $winnerScore, $winnerRank, '$winnerRecord', '$conference',
                 '$loserId', '$loserName', '$loserAbbrev', $loserScore, $loserRank, '$loserRecord', '$conference',
                 '$venue', $attendance)";
            
            if (!mysqli_query($con, $upsetQuery)) {
                throw new Exception("Failed to insert upset: " . mysqli_error($con));
            }
        }
        
        mysqli_commit($con);
        
        $message = "Successfully inserted " . count($upsets['allUpsets']) . " upset(s)";
        if ($upsets['rankedTeamsUpset'] > 0) {
            $message .= " (including {$upsets['rankedTeamsUpset']} ranked team" . ($upsets['rankedTeamsUpset'] != 1 ? 's' : '') . ")";
        }
        
        return ['success' => true, 'message' => $message];
        
    } catch (Exception $e) {
        mysqli_rollback($con);
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Get upsets from database
 */
function getUpsetsFromDatabase($con) {
    $results = [
        'timestamp' => date('c'),
        'mysqlTimestamp' => date('Y-m-d H:i:s'),
        'totalUpsets' => 0,
        'rankedTeamsUpset' => 0,
        'biggestUpsetBySpread' => null,
        'conferences' => [],
        'allUpsets' => [],
        'fromDatabase' => true
    ];
    
    // Get summary - filter by NCAAB league
    $summaryQuery = "SELECT * FROM upset_summary WHERE league = 'NCAAB' ORDER BY fetch_date DESC LIMIT 1";
    $summaryResult = mysqli_query($con, $summaryQuery);
    
    if ($summaryResult && mysqli_num_rows($summaryResult) > 0) {
        $summary = mysqli_fetch_assoc($summaryResult);
        $results['totalUpsets'] = $summary['total_upsets'];
        $results['rankedTeamsUpset'] = $summary['ranked_teams_upset'];
        $results['mysqlTimestamp'] = $summary['timestamp'];
    }
    
    // Get all upsets - filter by NCAAB league
    $upsetsQuery = "SELECT * FROM upsets WHERE league = 'NCAAB' ORDER BY game_date DESC";
    $upsetsResult = mysqli_query($con, $upsetsQuery);
    
    if ($upsetsResult) {
        while ($row = mysqli_fetch_assoc($upsetsResult)) {
            $upset = [
                'gameId' => $row['game_id'],
                'date' => $row['game_date'],
                'type' => $row['upset_type'],
                'spread' => $row['spread'],
                'conference' => $row['conference'],
                'winner' => [
                    'id' => $row['winner_id'],
                    'name' => $row['winner_name'],
                    'abbreviation' => $row['winner_abbreviation'],
                    'score' => $row['winner_score'],
                    'rank' => $row['winner_rank'],
                    'record' => $row['winner_record']
                ],
                'loser' => [
                    'id' => $row['loser_id'],
                    'name' => $row['loser_name'],
                    'abbreviation' => $row['loser_abbreviation'],
                    'score' => $row['loser_score'],
                    'rank' => $row['loser_rank'],
                    'record' => $row['loser_record']
                ],
                'venue' => $row['venue'],
                'attendance' => $row['attendance']
            ];
            
            $results['allUpsets'][] = $upset;
            
            // Track biggest upset
            if ($upset['spread'] && 
                (!$results['biggestUpsetBySpread'] || 
                 $upset['spread'] > $results['biggestUpsetBySpread']['spread'])) {
                $results['biggestUpsetBySpread'] = $upset;
            }
            
            // Group by conference
            if (!isset($results['conferences'][$row['conference']])) {
                $results['conferences'][$row['conference']] = [
                    'upsetCount' => 0,
                    'upsets' => []
                ];
            }
            $results['conferences'][$row['conference']]['upsetCount']++;
            $results['conferences'][$row['conference']]['upsets'][] = $upset;
        }
    }
    
    return $results;
}

/**
 * Display results as HTML table
 */
function displayHTMLTable($upsets, $dbUpdateResult = null, $statusMessage = '', $curlResult = null) {
    $dataSource = isset($upsets['fromDatabase']) ? 'Database' : 'Live API';
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>College Basketball Upsets</title>
        <style>
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                margin: 20px;
                background-color: #f5f5f5;
            }
            .summary {
                background: white;
                padding: 20px;
                border-radius: 8px;
                margin-bottom: 20px;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            .summary h1 {
                color: #333;
                margin-top: 0;
            }
            .status-bar {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 10px 15px;
                border-radius: 5px;
                margin-bottom: 20px;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            .status-message {
                font-weight: 500;
            }
            .data-source {
                font-size: 12px;
                opacity: 0.9;
            }
            .stats {
                display: flex;
                gap: 30px;
                margin-bottom: 20px;
                flex-wrap: wrap;
            }
            .stat {
                flex: 1;
                min-width: 150px;
            }
            .stat-label {
                font-size: 12px;
                color: #666;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            .stat-value {
                font-size: 24px;
                font-weight: bold;
                color: #2c3e50;
            }
            .alert {
                padding: 15px;
                border-radius: 5px;
                margin-bottom: 20px;
            }
            .alert-success {
                background-color: #d4edda;
                color: #155724;
                border: 1px solid #c3e6cb;
            }
            .alert-error {
                background-color: #f8d7da;
                color: #721c24;
                border: 1px solid #f5c6cb;
            }
            .alert-info {
                background-color: #d1ecf1;
                color: #0c5460;
                border: 1px solid #bee5eb;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                background: white;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            th {
                background-color: #2c3e50;
                color: white;
                padding: 12px;
                text-align: left;
                position: sticky;
                top: 0;
                z-index: 10;
            }
            td {
                padding: 10px 12px;
                border-bottom: 1px solid #ddd;
            }
            tr:hover {
                background-color: #f8f9fa;
            }
            .winner {
                color: #27ae60;
                font-weight: bold;
            }
            .loser {
                color: #e74c3c;
            }
            .ranked {
                background-color: #fff3cd;
                padding: 2px 6px;
                border-radius: 3px;
                font-size: 11px;
                margin-left: 4px;
            }
            .upset-type {
                padding: 3px 8px;
                border-radius: 3px;
                font-size: 11px;
                text-transform: uppercase;
                font-weight: bold;
            }
            .upset-ranking {
                background-color: #d4edda;
                color: #155724;
            }
            .upset-betting {
                background-color: #cce5ff;
                color: #004085;
            }
            .no-upsets {
                text-align: center;
                color: #666;
                padding: 20px;
                font-style: italic;
            }
            .team-abbrev {
                font-weight: bold;
                margin-right: 4px;
            }
            @media (max-width: 768px) {
                table {
                    font-size: 12px;
                }
                th, td {
                    padding: 8px 6px;
                }
                .stats {
                    flex-direction: column;
                }
            }
        </style>
    </head>
    <body>
        <div class="summary">
            <h1>🏀 College Basketball Upsets</h1>
            
            <div class="status-bar">
                <span class="status-message">📊 <?php echo $statusMessage; ?></span>
                <span class="data-source">Source: <?php echo $dataSource; ?></span>
            </div>
            
            <?php if (isset($dbUpdateResult)): ?>
                <?php if ($dbUpdateResult['success']): ?>
                    <div class="alert alert-success">
                        ✅ <strong>Database Updated:</strong> <?php echo htmlspecialchars($dbUpdateResult['message']); ?>
                    </div>
                <?php else: ?>
                    <div class="alert alert-error">
                        ⚠️ <strong>Database Error:</strong> <?php echo htmlspecialchars($dbUpdateResult['message']); ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            
            <?php if (isset($curlResult)): ?>
                <div class="alert <?php echo $curlResult['success'] ? 'alert-success' : 'alert-error'; ?>">
                    <?php echo $curlResult['success'] ? '✅' : '⚠️'; ?> <strong>Image Generation:</strong> <?php echo htmlspecialchars($curlResult['message']); ?>
                </div>
            <?php endif; ?>
            
            <div class="stats">
                <div class="stat">
                    <div class="stat-label">Total Upsets</div>
                    <div class="stat-value"><?php echo $upsets['totalUpsets']; ?></div>
                </div>
                <div class="stat">
                    <div class="stat-label">Ranked Teams Upset</div>
                    <div class="stat-value"><?php echo $upsets['rankedTeamsUpset']; ?></div>
                </div>
                <?php if ($upsets['biggestUpsetBySpread']): ?>
                <div class="stat">
                    <div class="stat-label">Biggest Spread Upset</div>
                    <div class="stat-value">
                        <?php echo $upsets['biggestUpsetBySpread']['winner']['abbreviation']; ?> 
                        (+<?php echo number_format($upsets['biggestUpsetBySpread']['spread'], 1); ?>)
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <div style="font-size: 12px; color: #666;">
                <strong>Data From:</strong> <?php echo date('F j, Y g:i A', strtotime($upsets['mysqlTimestamp'])); ?>
            </div>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Conference</th>
                    <th>Type</th>
                    <th>Winner</th>
                    <th>Score</th>
                    <th>Loser</th>
                    <th>Score</th>
                    <th>Spread</th>
                    <th>Venue</th>
                    <th>Attendance</th>
                </tr>
            </thead>
            <tbody>
                <?php
                if (count($upsets['allUpsets']) > 0) {
                    foreach ($upsets['allUpsets'] as $upset) {
                        ?>
                        <tr>
                            <td><?php echo str_replace('_', ' ', $upset['conference']); ?></td>
                            <td>
                                <span class="upset-type upset-<?php echo $upset['type']; ?>">
                                    <?php echo $upset['type']; ?>
                                </span>
                            </td>
                            <td class="winner">
                                <span class="team-abbrev"><?php echo $upset['winner']['abbreviation']; ?></span>
                                <?php echo $upset['winner']['name']; ?>
                                <?php if ($upset['winner']['rank']): ?>
                                    <span class="ranked">#<?php echo $upset['winner']['rank']; ?></span>
                                <?php endif; ?>
                                <br><small style="color: #666;"><?php echo $upset['winner']['record']; ?></small>
                            </td>
                            <td class="winner"><?php echo $upset['winner']['score']; ?></td>
                            <td class="loser">
                                <span class="team-abbrev"><?php echo $upset['loser']['abbreviation']; ?></span>
                                <?php echo $upset['loser']['name']; ?>
                                <?php if ($upset['loser']['rank']): ?>
                                    <span class="ranked">#<?php echo $upset['loser']['rank']; ?></span>
                                <?php endif; ?>
                                <br><small style="color: #666;"><?php echo $upset['loser']['record']; ?></small>
                            </td>
                            <td class="loser"><?php echo $upset['loser']['score']; ?></td>
                            <td>
                                <?php echo $upset['spread'] ? '+' . number_format($upset['spread'], 1) : '-'; ?>
                            </td>
                            <td><?php echo $upset['venue']; ?></td>
                            <td><?php echo number_format($upset['attendance']); ?></td>
                        </tr>
                        <?php
                    }
                } else {
                    echo '<tr><td colspan="9" class="no-upsets">No upsets found</td></tr>';
                }
                ?>
            </tbody>
        </table>
    </body>
    </html>
    <?php
}

/**
 * Trigger createNCAABUpsets.php to generate images for all upsets
 */
function triggerCreateUpsets() {
    $url = SERVER_URL . "Scrolling/createNCAABUpsets.php";
    
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120); // Longer timeout for processing all upsets
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'message' => "CURL Error: " . $error];
        }
        
        if ($httpCode !== 200) {
            return ['success' => false, 'message' => "HTTP Error: Code " . $httpCode];
        }
        
        // Parse the response to see how many images were generated
        if (strpos($response, 'Completed processing all upsets') !== false) {
            // Extract the summary from the response
            preg_match('/Total: (\d+)/', $response, $totalMatch);
            preg_match('/Successful: (\d+)/', $response, $successMatch);
            preg_match('/Errors: (\d+)/', $response, $errorMatch);
            
            $total = isset($totalMatch[1]) ? $totalMatch[1] : '?';
            $successful = isset($successMatch[1]) ? $successMatch[1] : '?';
            $errors = isset($errorMatch[1]) ? $errorMatch[1] : '?';
            
            return [
                'success' => true, 
                'message' => "Generated images for $successful/$total upsets" . ($errors > 0 ? " ($errors errors)" : "")
            ];
        } else {
            return ['success' => true, 'message' => "Image generation triggered successfully"];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => "Exception: " . $e->getMessage()];
    }
}

// Main execution
try {
    // Setup database tables
    $tableSetup = setupDatabaseTables($con);
    if (!$tableSetup['success']) {
        throw new Exception("Table setup failed: " . $tableSetup['message']);
    }
    
    // Get status message
    $statusMessage = getStatusMessage();
    
    $curlResult = null;
    
    // Check if we should fetch new data
    if (shouldFetchNewData() || isset($_GET['force'])) {
        // Fetch fresh data from ESPN API
        $upsets = getCollegeBasketballUpsets();
        
        // Update database (clear all NCAAB and reinsert)
        $dbUpdateResult = updateDatabase($upsets, $con, true);
        
        // If database update was successful and we have upsets, trigger image generation
        if ($dbUpdateResult['success'] && $upsets['totalUpsets'] > 0) {
            $curlResult = triggerCreateUpsets();
        }
        
        // Display HTML table with fresh data
        displayHTMLTable($upsets, $dbUpdateResult, $statusMessage, $curlResult);
    } else {
        // Just display existing data from database
        $upsets = getUpsetsFromDatabase($con);
        
        // Display HTML table with database data
        displayHTMLTable($upsets, null, $statusMessage);
    }
    
} catch (Exception $e) {
    ?>
    <!DOCTYPE html>
    <html>
    <head><title>Error</title></head>
    <body>
        <div style="background-color: #f8d7da; color: #721c24; padding: 20px; margin: 20px; border-radius: 5px;">
            <h1>⚠️ Error</h1>
            <p><strong><?php echo htmlspecialchars($e->getMessage()); ?></strong></p>
            <details>
                <summary>Stack Trace</summary>
                <pre><?php echo htmlspecialchars($e->getTraceAsString()); ?></pre>
            </details>
        </div>
    </body>
    </html>
    <?php
    error_log("Error in upset detection: " . $e->getMessage());
}

// Close database connection
mysqli_close($con);

// CLI execution for testing
if (php_sapi_name() === 'cli') {
    echo "College Basketball Upset Detector - CLI Mode\n";
    echo "==========================================\n";
    
    $currentTime = new DateTime('now', new DateTimeZone('America/New_York'));
    echo "Current Time (ET): " . $currentTime->format('l, F j, Y g:i A T') . "\n";
    echo "Should fetch new data: " . (shouldFetchNewData() ? "YES" : "NO") . "\n";
    echo "Status: " . getStatusMessage() . "\n\n";
    
    if (shouldFetchNewData()) {
        echo "Fetching fresh data from ESPN API...\n";
        $upsets = getCollegeBasketballUpsets();
        
        if ($upsets['totalUpsets'] > 0) {
            echo "Triggering image generation...\n";
            $curlResult = triggerCreateUpsets();
            echo "Image Generation: " . $curlResult['message'] . "\n";
        }
    } else {
        echo "Loading data from database...\n";
        $upsets = getUpsetsFromDatabase($con);
    }
    
    echo "Total Upsets: " . $upsets['totalUpsets'] . "\n";
    echo "Ranked Teams Upset: " . $upsets['rankedTeamsUpset'] . "\n";
    
    if ($upsets['biggestUpsetBySpread']) {
        echo "Biggest Upset: " . $upsets['biggestUpsetBySpread']['winner']['name'] . 
             " beat " . $upsets['biggestUpsetBySpread']['loser']['name'] . 
             " (+" . $upsets['biggestUpsetBySpread']['spread'] . " spread)\n";
    }
    
    echo "\nUpsets by Conference:\n";
    foreach ($upsets['conferences'] as $conf => $data) {
        if ($data['upsetCount'] > 0) {
            echo "  " . str_replace('_', ' ', $conf) . ": " . $data['upsetCount'] . "\n";
        }
    }
}
?>