<?php
// nfl_draft_view_and_update.php

//–– Enable error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

//–– 0) Load DB credentials & connect
require_once '../../config.php'; // Adjust path if needed
$con = new mysqli($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if ($con->connect_error) {
    die("DB Connection failed: " . $con->connect_error);
}

//–– 1) Fetch the Draft JSON
$url = 'https://site.web.api.espn.com/apis/site/v2/sports/football/nfl/draft?season=2025&region=us&lang=en';
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT        => 30,
    CURLOPT_HTTPHEADER     => [
        'Accept: application/json',
        'User-Agent: PHP Draft Script/1.0'
    ]
]);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    error_log('CURL error: ' . curl_error($ch));
    die('Failed to fetch data from ESPN API');
}
curl_close($ch);

//–– 2) Decode response
$data = json_decode($response, true);
if (empty($data['picks']) || !is_array($data['picks'])) {
    error_log('Invalid or empty JSON response: ' . json_encode($data));
    die('Invalid or empty JSON response');
}

//–– 3) Mappings
$positionMap = [
    '8'=>'QB','9'=>'RB','1'=>'WR','7'=>'TE','10'=>'FB',
    '46'=>'OT','47'=>'OG','91'=>'C','32'=>'DT','30'=>'LB',
    '264'=>'EDGE','29'=>'CB','36'=>'S','96'=>'LS','80'=>'PK','94'=>'P'
];
$teamMap = [
    22=>'ARI',1=>'ATL',33=>'BAL',2=>'BUF',29=>'CAR',3=>'CHI',4=>'CIN',5=>'CLE',
    6=>'DAL',7=>'DEN',8=>'DET',9=>'GB',34=>'HOU',11=>'IND',30=>'JAX',12=>'KC',
    13=>'LV',24=>'LAC',14=>'LAR',15=>'MIA',16=>'MIN',17=>'NE',18=>'NO',19=>'NYG',
    20=>'NYJ',21=>'PHI',23=>'PIT',25=>'SF',26=>'SEA',27=>'TB',10=>'TEN',28=>'WSH'
];
$colorMap = [
    22=>['primary'=>'#a40227','secondary'=>'#ffffff'],1=>['primary'=>'#a71930','secondary'=>'#000000'],
    33=>['primary'=>'#29126f','secondary'=>'#000000'],2=>['primary'=>'#00338d','secondary'=>'#d50a0a'],
    29=>['primary'=>'#0085ca','secondary'=>'#000000'],3=>['primary'=>'#0b1c3a','secondary'=>'#e64100'],
    4=>['primary'=>'#fb4f14','secondary'=>'#000000'],5=>['primary'=>'#472a08','secondary'=>'#ff3c00'],
    6=>['primary'=>'#002a5c','secondary'=>'#b0b7bc'],7=>['primary'=>'#0a2343','secondary'=>'#fc4c02'],
    8=>['primary'=>'#0076b6','secondary'=>'#bbbbbb'],9=>['primary'=>'#204e32','secondary'=>'#ffb612'],
    34=>['primary'=>'#00143f','secondary'=>'#c41230'],11=>['primary'=>'#003b75','secondary'=>'#ffffff'],
    30=>['primary'=>'#007487','secondary'=>'#d7a22a'],12=>['primary'=>'#e31837','secondary'=>'#ffb612'],
    13=>['primary'=>'#000000','secondary'=>'#a5acaf'],24=>['primary'=>'#0080c6','secondary'=>'#ffc20e'],
    14=>['primary'=>'#003594','secondary'=>'#ffd100'],15=>['primary'=>'#008e97','secondary'=>'#fc4c02'],
    16=>['primary'=>'#4f2683','secondary'=>'#ffc62f'],17=>['primary'=>'#002a5c','secondary'=>'#c60c30'],
    18=>['primary'=>'#d3bc8d','secondary'=>'#000000'],19=>['primary'=>'#003c7f','secondary'=>'#c9243f'],
    20=>['primary'=>'#115740','secondary'=>'#ffffff'],21=>['primary'=>'#06424d','secondary'=>'#000000'],
    23=>['primary'=>'#000000','secondary'=>'#ffb612'],25=>['primary'=>'#aa0000','secondary'=>'#b3995d'],
    26=>['primary'=>'#002a5c','secondary'=>'#69be28'],27=>['primary'=>'#bd1c36','secondary'=>'#3e3a35'],
    10=>['primary'=>'#4b92db','secondary'=>'#002a5c'],28=>['primary'=>'#5a1414','secondary'=>'#ffb612']
];

//–– 4) Group all picks into days, with new ON_THE_CLOCK logic
$days = [
    'Day 1 (Round 1)'     => [],
    'Day 2 (Rounds 2–3)'  => [],
    'Day 3 (Rounds 4–7)'  => []
];

$foundOnClock = false;

foreach ($data['picks'] as $p) {
    $roundRaw = (int)($p['round'] ?? 0);
    if ($roundRaw < 1) continue;

    if ($roundRaw === 1)         $dayKey = 'Day 1 (Round 1)';
    elseif ($roundRaw <= 3)      $dayKey = 'Day 2 (Rounds 2–3)';
    else                         $dayKey = 'Day 3 (Rounds 4–7)';

    // determine status: keep SELECTION_MADE, first ON_THE_CLOCK, blank other ON_THE_CLOCK
    $rawStatus = $p['status'] ?? '';
    if ($rawStatus === 'SELECTION_MADE') {
        $status = 'SELECTION_MADE';
    } elseif ($rawStatus === 'ON_THE_CLOCK') {
        if (! $foundOnClock) {
            $status = 'ON_THE_CLOCK';
            $foundOnClock = true;
        } else {
            $status = '';
        }
    } else {
        $status = '';
    }

    // team & colors
    $teamId    = (int)($p['teamId'] ?? 0);
    $teamAbbr  = $teamMap[$teamId] ?? 'Unknown';
    list($r,$g,$b)    = sscanf($colorMap[$teamId]['primary']   ?? '#000000', "#%02x%02x%02x");
    list($r2,$g2,$b2) = sscanf($colorMap[$teamId]['secondary'] ?? '#000000', "#%02x%02x%02x");

    // default athlete fields
    $name         = '';
    $posAbbrev    = '';
    $height       = '';
    $weight       = '';
    $collegeAbbr  = '';
    $rank         = 0;
    $overallRank  = 0;
    $grade        = 0;

    if ($status === 'SELECTION_MADE') {
        $ath = $p['athlete'] ?? [];
        $name       = $ath['displayName'] ?? '';
        $posAbbrev  = $positionMap[$ath['position']['id'] ?? ''] ?? '';
        $height     = $ath['displayHeight'] ?? '';
        $weight     = $ath['displayWeight'] ?? '';
        $collegeAbbr= $ath['team']['abbreviation'] ?? '';
        foreach ($ath['attributes'] ?? [] as $a) {
            if (!empty($a['name']) && isset($a['displayValue'])) {
                if ($a['name'] === 'rank')    $rank        = (int)$a['displayValue'];
                if ($a['name'] === 'overall') $overallRank = (int)$a['displayValue'];
                if ($a['name'] === 'grade')   $grade       = (int)$a['displayValue'];
            }
        }
    }

    $days[$dayKey][] = [
        'Pick'         => (int)$p['pick'],
        'Overall'      => (int)$p['overall'],
        'Round'        => $roundRaw,
        'Traded'       => !empty($p['traded']) ? 1 : 0,
        'TradeNote'    => $p['tradeNote'] ?? '',
        'Name'         => $name,
        'Position'     => $posAbbrev,
        'Height'       => $height,
        'Weight'       => $weight,
        'College'      => $collegeAbbr,
        'Team'         => $teamAbbr,
        'R'            => $r,  'G'   => $g,  'B'   => $b,
        'R_2'          => $r2, 'G_2' => $g2, 'B_2' => $b2,
        'Rank'         => $rank,
        'OverallRank'  => $overallRank,
        'Grade'        => $grade,
        'Status'       => $status
    ];
}

//–– 5) Prepare UPDATE statement
$sql = <<<SQL
UPDATE nfl_draft
SET
    `round`      = ?,
    traded       = ?,
    trade_note   = ?,
    name         = ?,
    position     = ?,
    height       = ?,
    weight       = ?,
    college      = ?,
    team         = ?,
    R            = ?, G      = ?, B      = ?,
    R_2          = ?, G_2    = ?, B_2    = ?,
    `rank`       = ?,
    overall_rank = ?,
    grade        = ?,
    status       = ?,
    day1         = ?,
    day2         = ?,
    day3         = ?
WHERE overall = ?
SQL;

$stmt = $con->prepare($sql);
if (!$stmt) {
    die("Prepare failed: " . $con->error);
}

// types: ii + 7s + 9i + s + 4i = 23
$types =
    'ii'            // round, traded
  . str_repeat('s', 7)  // trade_note, name, position, height, weight, college, team
  . str_repeat('i', 9)  // R, G, B, R_2, G_2, B_2, rank, overall_rank, grade
  . 's'                // status
  . 'iiii';            // day1, day2, day3, overall

//–– 6) Execute updates
$con->begin_transaction();
try {
    foreach ($days as $dayLabel => $rows) {
        $d1 = $dayLabel === 'Day 1 (Round 1)'    ? 1 : 0;
        $d2 = $dayLabel === 'Day 2 (Rounds 2–3)' ? 1 : 0;
        $d3 = $dayLabel === 'Day 3 (Rounds 4–7)' ? 1 : 0;

        foreach ($rows as $r) {
            $stmt->bind_param(
                $types,
                $r['Round'],
                $r['Traded'],
                $r['TradeNote'],
                $r['Name'],
                $r['Position'],
                $r['Height'],
                $r['Weight'],
                $r['College'],
                $r['Team'],
                $r['R'],   $r['G'],   $r['B'],
                $r['R_2'], $r['G_2'], $r['B_2'],
                $r['Rank'],
                $r['OverallRank'],
                $r['Grade'],
                $r['Status'],
                $d1, $d2, $d3,
                $r['Overall']
            );
            if (!$stmt->execute()) {
                error_log("UPDATE failed for overall {$r['Overall']}: " . $stmt->error);
            }
        }
    }
    $con->commit();
} catch (\Exception $e) {
    $con->rollback();
    die("Transaction failed: " . $e->getMessage());
}
$stmt->close();
$con->close();

//–– 7) Render HTML
// ——————————————————————————————————————————
// Trigger the Drafts.php endpoint once everything else has run
// ——————————————————————————————————————————
$ch = curl_init('https://gscroll.us/Scrolling/App_Output/API/Drafts.php');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);          // give it up to 10 seconds
// if you need to send headers, uncomment and edit the next line:
// curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer YOUR_TOKEN']);

$response = curl_exec($ch);
if (curl_errno($ch)) {
    error_log('Drafts API cURL error: ' . curl_error($ch));
} else {
    // optionally inspect or log the response:
    // error_log('Drafts API response: ' . $response);
}
curl_close($ch);


?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>2025 NFL Draft Results</title>
    <style>
        body  { font-family: sans-serif; padding: 1em; }
        table { border-collapse: collapse; width: 100%; margin-bottom: 2em; }
        th, td { border: 1px solid #ccc; padding: 0.5em; text-align: left; }
        th     { background: #eee; }
        h2     { margin-top: 2em; }
    </style>
</head>
<body>
<h1>2025 NFL Draft Results by Day</h1>
<?php 
$totalPicks = 0;
foreach ($days as $dayTitle => $picks):
    $totalPicks += count($picks);
?>
    <h2><?= htmlspecialchars($dayTitle) ?></h2>
    <?php if (empty($picks)): ?>
        <p>No picks.</p>
    <?php else: ?>
        <p>Total picks: <?= count($picks) ?></p>
        <table>
            <thead>
                <tr>
                    <th>Pick</th><th>Overall</th><th>Round</th><th>Name</th>
                    <th>Position</th><th>Height</th><th>Weight</th>
                    <th>College</th><th>Team</th><th>Traded</th>
                    <th>Trade Note</th><th>Status</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($picks as $row): ?>
                    <tr>
                        <td><?= $row['Pick'] ?></td>
                        <td><?= $row['Overall'] ?></td>
                        <td><?= $row['Round'] ?></td>
                        <td><?= htmlspecialchars($row['Name']) ?></td>
                        <td><?= htmlspecialchars($row['Position']) ?></td>
                        <td><?= htmlspecialchars($row['Height']) ?></td>
                        <td><?= htmlspecialchars($row['Weight']) ?></td>
                        <td><?= htmlspecialchars($row['College']) ?></td>
                        <td style="
                            background-color: rgb(<?= $row['R'] ?>,<?= $row['G'] ?>,<?= $row['B'] ?>);
                            color: rgb(<?= $row['R_2'] ?>,<?= $row['G_2'] ?>,<?= $row['B_2'] ?>);
                        "><?= htmlspecialchars($row['Team']) ?></td>
                        <td><?= $row['Traded'] ? 'Yes' : 'No' ?></td>
                        <td><?= htmlspecialchars($row['TradeNote']) ?></td>
                        <td><?= htmlspecialchars($row['Status']) ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
<?php endforeach; ?>
<p><strong>Total picks updated in database: <?= $totalPicks ?></strong></p>
</body>
</html>
