<?php
// update_nfl_draft.php - Single script to fetch and populate 2025 NFL Draft data

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

//–– 1) Load DB credentials & connect
require_once '../../config.php';
$con = new mysqli($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if ($con->connect_error) {
    die("DB Connection failed: " . $con->connect_error);
}

//–– 2) Fetch the Draft JSON from ESPN
echo "Fetching 2025 NFL Draft data from ESPN...\n";
$url = 'https://site.web.api.espn.com/apis/site/v2/sports/football/nfl/draft?season=2025&region=us&lang=en';
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT        => 30,
    CURLOPT_HTTPHEADER     => [
        'Accept: application/json',
        'User-Agent: PHP Draft Script/1.0'
    ]
]);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    error_log('CURL error: ' . curl_error($ch));
    die('Failed to fetch data from ESPN API');
}
curl_close($ch);

//–– 3) Decode response
$data = json_decode($response, true);
if (empty($data['picks']) || !is_array($data['picks'])) {
    error_log('Invalid or empty JSON response: ' . json_encode($data));
    die('Invalid or empty JSON response');
}

//–– 4) Mappings
$positionMap = [
    '8'=>'QB','9'=>'RB','1'=>'WR','7'=>'TE','10'=>'FB',
    '46'=>'OT','47'=>'OG','91'=>'C','32'=>'DT','30'=>'LB',
    '264'=>'EDGE','29'=>'CB','36'=>'S','96'=>'LS','80'=>'PK','94'=>'P'
];
$teamMap = [
    22=>'ARI',1=>'ATL',33=>'BAL',2=>'BUF',29=>'CAR',3=>'CHI',4=>'CIN',5=>'CLE',
    6=>'DAL',7=>'DEN',8=>'DET',9=>'GB',34=>'HOU',11=>'IND',30=>'JAX',12=>'KC',
    13=>'LV',24=>'LAC',14=>'LAR',15=>'MIA',16=>'MIN',17=>'NE',18=>'NO',19=>'NYG',
    20=>'NYJ',21=>'PHI',23=>'PIT',25=>'SF',26=>'SEA',27=>'TB',10=>'TEN',28=>'WSH'
];
$colorMap = [
    22=>['primary'=>'#a40227','secondary'=>'#ffffff'],1=>['primary'=>'#a71930','secondary'=>'#000000'],
    33=>['primary'=>'#29126f','secondary'=>'#000000'],2=>['primary'=>'#00338d','secondary'=>'#d50a0a'],
    29=>['primary'=>'#0085ca','secondary'=>'#000000'],3=>['primary'=>'#0b1c3a','secondary'=>'#e64100'],
    4=>['primary'=>'#fb4f14','secondary'=>'#000000'],5=>['primary'=>'#472a08','secondary'=>'#ff3c00'],
    6=>['primary'=>'#002a5c','secondary'=>'#b0b7bc'],7=>['primary'=>'#0a2343','secondary'=>'#fc4c02'],
    8=>['primary'=>'#0076b6','secondary'=>'#bbbbbb'],9=>['primary'=>'#204e32','secondary'=>'#ffb612'],
    34=>['primary'=>'#00143f','secondary'=>'#c41230'],11=>['primary'=>'#003b75','secondary'=>'#ffffff'],
    30=>['primary'=>'#007487','secondary'=>'#d7a22a'],12=>['primary'=>'#e31837','secondary'=>'#ffb612'],
    13=>['primary'=>'#000000','secondary'=>'#a5acaf'],24=>['primary'=>'#0080c6','secondary'=>'#ffc20e'],
    14=>['primary'=>'#003594','secondary'=>'#ffd100'],15=>['primary'=>'#008e97','secondary'=>'#fc4c02'],
    16=>['primary'=>'#4f2683','secondary'=>'#ffc62f'],17=>['primary'=>'#002a5c','secondary'=>'#c60c30'],
    18=>['primary'=>'#d3bc8d','secondary'=>'#000000'],19=>['primary'=>'#003c7f','secondary'=>'#c9243f'],
    20=>['primary'=>'#115740','secondary'=>'#ffffff'],21=>['primary'=>'#06424d','secondary'=>'#000000'],
    23=>['primary'=>'#000000','secondary'=>'#ffb612'],25=>['primary'=>'#aa0000','secondary'=>'#b3995d'],
    26=>['primary'=>'#002a5c','secondary'=>'#69be28'],27=>['primary'=>'#bd1c36','secondary'=>'#3e3a35'],
    10=>['primary'=>'#4b92db','secondary'=>'#002a5c'],28=>['primary'=>'#5a1414','secondary'=>'#ffb612']
];

//–– 5) Process picks and track ON_THE_CLOCK status
$picks = [];
$foundOnClock = false;

foreach ($data['picks'] as $p) {
    $roundRaw = (int)($p['round'] ?? 0);
    if ($roundRaw < 1) continue;

    // Determine status: keep SELECTION_MADE, first ON_THE_CLOCK, blank others
    $rawStatus = $p['status'] ?? '';
    if ($rawStatus === 'SELECTION_MADE') {
        $status = 'SELECTION_MADE';
    } elseif ($rawStatus === 'ON_THE_CLOCK') {
        if (!$foundOnClock) {
            $status = 'ON_THE_CLOCK';
            $foundOnClock = true;
        } else {
            $status = '';
        }
    } else {
        $status = '';
    }

    // Team & colors
    $teamId   = (int)($p['teamId'] ?? 0);
    $teamAbbr = $teamMap[$teamId] ?? 'Unknown';
    list($r,$g,$b)    = sscanf($colorMap[$teamId]['primary']   ?? '#000000', "#%02x%02x%02x");
    list($r2,$g2,$b2) = sscanf($colorMap[$teamId]['secondary'] ?? '#000000', "#%02x%02x%02x");

    // Default athlete fields
    $name        = '';
    $posAbbrev   = '';
    $height      = '';
    $weight      = '';
    $collegeAbbr = '';
    $rank        = 0;
    $overallRank = 0;
    $grade       = 0;

    if ($status === 'SELECTION_MADE') {
        $ath = $p['athlete'] ?? [];
        $name        = $ath['displayName'] ?? '';
        $posAbbrev   = $positionMap[$ath['position']['id'] ?? ''] ?? '';
        $height      = $ath['displayHeight'] ?? '';
        $weight      = $ath['displayWeight'] ?? '';
        $collegeAbbr = $ath['team']['abbreviation'] ?? '';
        foreach ($ath['attributes'] ?? [] as $a) {
            if (!empty($a['name']) && isset($a['displayValue'])) {
                if ($a['name'] === 'rank')    $rank        = (int)$a['displayValue'];
                if ($a['name'] === 'overall') $overallRank = (int)$a['displayValue'];
                if ($a['name'] === 'grade')   $grade       = (int)$a['displayValue'];
            }
        }
    }

    // Determine day flags
    $day1 = ($roundRaw === 1) ? 1 : 0;
    $day2 = ($roundRaw >= 2 && $roundRaw <= 3) ? 1 : 0;
    $day3 = ($roundRaw >= 4) ? 1 : 0;

    $picks[] = [
        'Pick'         => (int)$p['pick'],      // Pick within round (1-32)
        'Overall'      => (int)$p['overall'],   // Overall pick (1-257) - UNIQUE
        'Round'        => $roundRaw,
        'Traded'       => !empty($p['traded']) ? 1 : 0,
        'TradeNote'    => $p['tradeNote'] ?? '',
        'Name'         => $name,
        'Position'     => $posAbbrev,
        'Height'       => $height,
        'Weight'       => $weight,
        'College'      => $collegeAbbr,
        'Team'         => $teamAbbr,
        'R'            => $r,  'G'   => $g,  'B'   => $b,
        'R_2'          => $r2, 'G_2' => $g2, 'B_2' => $b2,
        'Rank'         => $rank,
        'OverallRank'  => $overallRank,
        'Grade'        => $grade,
        'Status'       => $status,
        'Day1'         => $day1,
        'Day2'         => $day2,
        'Day3'         => $day3
    ];
}

echo "Processed " . count($picks) . " picks from ESPN API\n";

//–– 6) Clear the table FIRST (outside transaction)
if (!$con->query("DELETE FROM `nfl_draft`")) {
    die("Delete failed: " . $con->error);
}
echo "Cleared existing data from table\n";

//–– 7) Insert fresh data in transaction
$con->begin_transaction();
try {
    // Prepare INSERT statement with all 24 columns
    $sql = <<<SQL
INSERT INTO nfl_draft (
    `pick`, `overall`, `round`, `traded`, `trade_note`,
    `name`, `position`, `Height`, `Weight`, `college`, `team`,
    `R`, `G`, `B`, `R_2`, `G_2`, `B_2`,
    `rank`, `overall_rank`, `grade`,
    `STATUS`, `day1`, `day2`, `day3`
) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
SQL;

    $stmt = $con->prepare($sql);
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $con->error);
    }

    // Bind types: 24 parameters
    $types = 
        'iii'              // pick, overall, round
      . 'is'               // traded, trade_note
      . 'ssssss'           // name, position, Height, Weight, college, team
      . str_repeat('i', 6) // R, G, B, R_2, G_2, B_2
      . str_repeat('i', 3) // rank, overall_rank, grade
      . 's'                // STATUS
      . str_repeat('i', 3); // day1, day2, day3

    // Insert each pick
    $insertCount = 0;
    foreach ($picks as $p) {
        $stmt->bind_param(
            $types,
            $p['Pick'], $p['Overall'], $p['Round'], $p['Traded'], $p['TradeNote'],
            $p['Name'], $p['Position'], $p['Height'], $p['Weight'], $p['College'], $p['Team'],
            $p['R'], $p['G'], $p['B'], $p['R_2'], $p['G_2'], $p['B_2'],
            $p['Rank'], $p['OverallRank'], $p['Grade'],
            $p['Status'], $p['Day1'], $p['Day2'], $p['Day3']
        );

        if (!$stmt->execute()) {
            throw new Exception("Insert failed for pick {$p['Overall']}: " . $stmt->error);
        }
        $insertCount++;
    }

    $con->commit();
    $stmt->close();

    echo "Successfully inserted {$insertCount} picks\n";

} catch (Exception $e) {
    $con->rollback();
    die("Transaction failed: " . $e->getMessage());
}

//–– 8) Trigger the Drafts.php endpoint
$ch = curl_init('https://gscroll.us/Scrolling/App_Output/API/Drafts.php');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    error_log('Drafts API cURL error: ' . curl_error($ch));
} else {
    echo "Drafts.php API triggered successfully\n";
}
curl_close($ch);

$con->close();

echo "\n2025 NFL Draft data updated successfully!\n";
?>