<?php
/**
 * ESPN API Debug Tool - Shows raw data from all endpoints
 * Usage: espn_debug.php?event=401752836
 */

header('Content-Type: text/html; charset=utf-8');

$eventId = $_GET['event'] ?? '401752836';

function http_get_json(string $url, int $timeoutSec = 8): ?array {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_CONNECTTIMEOUT => $timeoutSec,
        CURLOPT_TIMEOUT => $timeoutSec,
        CURLOPT_HTTPHEADER => [
            'Accept: application/json',
            'User-Agent: GLANCE-LED/1.0 (+https://glance-led.com)'
        ],
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);

    if ($err || $code >= 400 || !$body) return null;
    $json = json_decode($body, true);
    return is_array($json) ? $json : null;
}

// Fetch data from both endpoints
$summaryUrl = "https://site.api.espn.com/apis/site/v2/sports/football/college-football/summary?event={$eventId}";
$pbpUrl = "https://site.api.espn.com/apis/site/v2/sports/football/college-football/playbyplay?event={$eventId}";

$summary = http_get_json($summaryUrl);
$pbp = http_get_json($pbpUrl);

?>
<!DOCTYPE html>
<html>
<head>
    <title>ESPN API Debug - Event <?= htmlspecialchars($eventId) ?></title>
    <style>
        body { font-family: monospace; margin: 20px; background: #f5f5f5; }
        .section { 
            background: white; 
            border: 1px solid #ddd; 
            padding: 15px; 
            margin-bottom: 20px;
            border-radius: 5px;
        }
        h2 { color: #333; border-bottom: 2px solid #007acc; padding-bottom: 5px; }
        h3 { color: #007acc; margin-top: 15px; }
        pre { 
            background: #f8f8f8; 
            padding: 10px; 
            overflow-x: auto; 
            border-left: 3px solid #007acc;
            max-height: 500px;
            overflow-y: auto;
        }
        .highlight { background: yellow; font-weight: bold; }
        .error { color: red; font-weight: bold; }
        .success { color: green; font-weight: bold; }
        .json-key { color: #d14; font-weight: bold; }
        table { border-collapse: collapse; width: 100%; margin-top: 10px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background: #007acc; color: white; }
        tr:nth-child(even) { background: #f9f9f9; }
        .null-value { color: #999; font-style: italic; }
    </style>
</head>
<body>
    <h1>ESPN API Debug Tool</h1>
    <p>Event ID: <strong><?= htmlspecialchars($eventId) ?></strong></p>
    
    <?php if ($summary): ?>
        
        <!-- Summary Overview -->
        <div class="section">
            <h2>1. Summary Endpoint Analysis</h2>
            
            <h3>A. Available Top-Level Fields</h3>
            <ul>
                <?php foreach (array_keys($summary) as $key): ?>
                    <li><?= htmlspecialchars($key) ?> 
                        <?php if ($key === 'situation'): ?>
                            <span class="success">✓ FOUND</span>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ul>
            
            <?php if (isset($summary['situation'])): ?>
                <h3>B. Situation Field Structure</h3>
                <pre><?= htmlspecialchars(json_encode($summary['situation'], JSON_PRETTY_PRINT)) ?></pre>
                
                <?php if (isset($summary['situation']['lastPlay'])): ?>
                    <h3>C. Last Play Details</h3>
                    <table>
                        <tr>
                            <th>Field</th>
                            <th>Value</th>
                        </tr>
                        <tr>
                            <td>Play ID</td>
                            <td><?= htmlspecialchars($summary['situation']['lastPlay']['id'] ?? 'null') ?></td>
                        </tr>
                        <tr>
                            <td>Text</td>
                            <td><?= htmlspecialchars($summary['situation']['lastPlay']['text'] ?? 'null') ?></td>
                        </tr>
                        <tr>
                            <td>Score Value</td>
                            <td><?= htmlspecialchars($summary['situation']['lastPlay']['scoreValue'] ?? 'null') ?></td>
                        </tr>
                        <tr>
                            <td>Athletes Involved</td>
                            <td>
                                <?php if (isset($summary['situation']['lastPlay']['athletesInvolved'])): ?>
                                    <span class="success">✓ <?= count($summary['situation']['lastPlay']['athletesInvolved']) ?> athletes</span>
                                <?php else: ?>
                                    <span class="error">✗ NOT FOUND</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                    
                    <?php if (isset($summary['situation']['lastPlay']['athletesInvolved'])): ?>
                        <h3>D. Athletes in Last Play</h3>
                        <pre><?= htmlspecialchars(json_encode($summary['situation']['lastPlay']['athletesInvolved'], JSON_PRETTY_PRINT)) ?></pre>
                    <?php endif; ?>
                <?php endif; ?>
            <?php else: ?>
                <p class="error">❌ No 'situation' field found in summary</p>
            <?php endif; ?>
            
            <h3>E. Scoring Plays</h3>
            <?php if (isset($summary['scoringPlays']) && count($summary['scoringPlays']) > 0): ?>
                <p class="success">Found <?= count($summary['scoringPlays']) ?> scoring plays</p>
                
                <h4>Latest Scoring Play Structure:</h4>
                <?php 
                $latestScore = end($summary['scoringPlays']);
                ?>
                <table>
                    <tr>
                        <th>Field</th>
                        <th>Value</th>
                    </tr>
                    <tr>
                        <td>Play ID</td>
                        <td><?= htmlspecialchars($latestScore['id'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Type</td>
                        <td><?= htmlspecialchars($latestScore['type']['text'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Score Value</td>
                        <td><?= htmlspecialchars($latestScore['scoreValue'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Text</td>
                        <td><?= htmlspecialchars($latestScore['text'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Has 'athletes' field?</td>
                        <td>
                            <?php if (isset($latestScore['athletes'])): ?>
                                <span class="success">✓ YES (<?= count($latestScore['athletes']) ?> athletes)</span>
                            <?php else: ?>
                                <span class="error">✗ NO</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <td>Has 'players' field?</td>
                        <td>
                            <?php if (isset($latestScore['players'])): ?>
                                <span class="success">✓ YES (<?= count($latestScore['players']) ?> players)</span>
                            <?php else: ?>
                                <span class="error">✗ NO</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <td>Has 'athletesInvolved' field?</td>
                        <td>
                            <?php if (isset($latestScore['athletesInvolved'])): ?>
                                <span class="success">✓ YES (<?= count($latestScore['athletesInvolved']) ?> athletes)</span>
                            <?php else: ?>
                                <span class="error">✗ NO</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
                
                <h4>Raw Latest Scoring Play Data:</h4>
                <pre><?= htmlspecialchars(json_encode($latestScore, JSON_PRETTY_PRINT)) ?></pre>
                
                <h4>All Scoring Plays (last 3):</h4>
                <?php 
                $lastThree = array_slice($summary['scoringPlays'], -3);
                foreach ($lastThree as $i => $play): 
                ?>
                    <div style="border: 1px solid #ccc; padding: 10px; margin: 10px 0;">
                        <strong>Play <?= $i + 1 ?>:</strong> <?= htmlspecialchars($play['text'] ?? '') ?>
                        <br>Score Value: <?= htmlspecialchars($play['scoreValue'] ?? 'null') ?>
                        <br>Athletes: <?= isset($play['athletes']) ? count($play['athletes']) . ' found' : 'NOT FOUND' ?>
                        <br>Players: <?= isset($play['players']) ? count($play['players']) . ' found' : 'NOT FOUND' ?>
                    </div>
                <?php endforeach; ?>
                
            <?php else: ?>
                <p class="error">❌ No scoring plays found</p>
            <?php endif; ?>
        </div>
        
    <?php else: ?>
        <div class="section">
            <p class="error">❌ Failed to fetch summary data</p>
        </div>
    <?php endif; ?>
    
    <?php if ($pbp): ?>
        
        <!-- Play-by-Play Analysis -->
        <div class="section">
            <h2>2. Play-by-Play Endpoint Analysis</h2>
            
            <h3>A. Latest Scoring Play from PBP</h3>
            <?php
            // Find latest scoring play in PBP
            $latestScoringPlay = null;
            $allPlays = [];
            
            // Collect all plays
            foreach ($pbp['drives']['previous'] ?? [] as $drive) {
                foreach ($drive['plays'] ?? [] as $play) {
                    $allPlays[] = $play;
                }
            }
            foreach ($pbp['drives']['current']['plays'] ?? [] as $play) {
                $allPlays[] = $play;
            }
            
            // Find latest scoring play
            for ($i = count($allPlays) - 1; $i >= 0; $i--) {
                $play = $allPlays[$i];
                if (($play['scoringPlay'] ?? false) && ($play['scoreValue'] ?? 0) > 0) {
                    $text = strtolower($play['text'] ?? '');
                    if (!str_contains($text, 'extra point') && !str_contains($text, 'two-point')) {
                        $latestScoringPlay = $play;
                        break;
                    }
                }
            }
            
            if ($latestScoringPlay): ?>
                <table>
                    <tr>
                        <th>Field</th>
                        <th>Value</th>
                    </tr>
                    <tr>
                        <td>Play ID</td>
                        <td><?= htmlspecialchars($latestScoringPlay['id'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Text</td>
                        <td><?= htmlspecialchars($latestScoringPlay['text'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Score Value</td>
                        <td><?= htmlspecialchars($latestScoringPlay['scoreValue'] ?? 'null') ?></td>
                    </tr>
                    <tr>
                        <td>Has 'participants'?</td>
                        <td>
                            <?php if (isset($latestScoringPlay['participants'])): ?>
                                <span class="success">✓ YES (<?= count($latestScoringPlay['participants']) ?> participants)</span>
                            <?php else: ?>
                                <span class="error">✗ NO</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <td>Has 'athletesInvolved'?</td>
                        <td>
                            <?php if (isset($latestScoringPlay['athletesInvolved'])): ?>
                                <span class="success">✓ YES (<?= count($latestScoringPlay['athletesInvolved']) ?> athletes)</span>
                            <?php else: ?>
                                <span class="error">✗ NO</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
                
                <h4>Raw Play Data:</h4>
                <pre><?= htmlspecialchars(json_encode($latestScoringPlay, JSON_PRETTY_PRINT)) ?></pre>
                
            <?php else: ?>
                <p class="error">❌ No scoring plays found in play-by-play data</p>
            <?php endif; ?>
            
        </div>
    <?php else: ?>
        <div class="section">
            <p class="error">❌ Failed to fetch play-by-play data</p>
        </div>
    <?php endif; ?>
    
    <!-- Summary -->
    <div class="section">
        <h2>3. Data Availability Summary</h2>
        <table>
            <tr>
                <th>Data Source</th>
                <th>Available?</th>
                <th>Has Player Data?</th>
                <th>Notes</th>
            </tr>
            <tr>
                <td>summary.situation.lastPlay</td>
                <td><?= isset($summary['situation']['lastPlay']) ? '✓' : '✗' ?></td>
                <td><?= isset($summary['situation']['lastPlay']['athletesInvolved']) ? '✓' : '✗' ?></td>
                <td>Best source when available</td>
            </tr>
            <tr>
                <td>summary.scoringPlays</td>
                <td><?= isset($summary['scoringPlays']) && count($summary['scoringPlays']) > 0 ? '✓' : '✗' ?></td>
                <td>
                    <?php 
                    $hasAthletes = false;
                    if (isset($summary['scoringPlays'])) {
                        $latest = end($summary['scoringPlays']);
                        $hasAthletes = isset($latest['athletes']) || isset($latest['players']);
                    }
                    echo $hasAthletes ? '✓' : '✗';
                    ?>
                </td>
                <td>Often missing athlete details</td>
            </tr>
            <tr>
                <td>playbyplay.drives.plays</td>
                <td><?= !empty($allPlays) ? '✓' : '✗' ?></td>
                <td>
                    <?php 
                    $hasParticipants = false;
                    if ($latestScoringPlay) {
                        $hasParticipants = isset($latestScoringPlay['participants']);
                    }
                    echo $hasParticipants ? '✓' : '✗';
                    ?>
                </td>
                <td>Usually has participants</td>
            </tr>
        </table>
    </div>
    
    <!-- Raw JSON Toggle -->
    <div class="section">
        <h2>4. Raw JSON Data</h2>
        <button onclick="document.getElementById('raw-summary').style.display='block'">Show Summary JSON</button>
        <button onclick="document.getElementById('raw-pbp').style.display='block'">Show PBP JSON</button>
        
        <div id="raw-summary" style="display:none;">
            <h3>Raw Summary JSON</h3>
            <pre><?= htmlspecialchars(json_encode($summary, JSON_PRETTY_PRINT)) ?></pre>
        </div>
        
        <div id="raw-pbp" style="display:none;">
            <h3>Raw Play-by-Play JSON</h3>
            <pre><?= htmlspecialchars(json_encode($pbp, JSON_PRETTY_PRINT)) ?></pre>
        </div>
    </div>
    
</body>
</html>