<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../../../config.php'; // Include the configuration file

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Set timezone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp = date('Y-m-d H:i:s');

// Ensure database columns exist
$check_column = "SHOW COLUMNS FROM `sports_list` LIKE 'lastUpdatedTime'";
$column_result = mysqli_query($con, $check_column);
if (mysqli_num_rows($column_result) == 0) {
    $add_column = "ALTER TABLE `sports_list` ADD COLUMN `lastUpdatedTime` DATETIME DEFAULT NULL";
    mysqli_query($con, $add_column);
}

// Check for broadcast column and add if needed
$check_broadcast = "SHOW COLUMNS FROM `sports_list` LIKE 'broadcast'";
$broadcast_result = mysqli_query($con, $check_broadcast);
if (mysqli_num_rows($broadcast_result) == 0) {
    $add_broadcast = "ALTER TABLE `sports_list` ADD COLUMN `broadcast` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_broadcast);
}

// Check if we should call APIs - separate checks for today and tomorrow
$current_time_utc = gmdate("Y-m-d\TH:i:s\Z");
$current_timestamp_unix = time();
$hour = date('H');

// Define today's boundaries
$today_start = strtotime("today midnight");
$today_end = strtotime("today 23:59:59");
$tomorrow_start = strtotime("tomorrow midnight");
$tomorrow_end = strtotime("tomorrow 23:59:59");

// Check TODAY's games
$should_call_today = false;
$today_games_info = [];

$check_today_query = "SELECT Game_Date, Game_Played, Team_1_Abbrev, Team_2_Abbrev 
                      FROM sports_list 
                      WHERE Conference = 'NBA' 
                      AND Game_Date >= FROM_UNIXTIME($today_start)
                      AND Game_Date <= FROM_UNIXTIME($today_end)";
$check_today_result = mysqli_query($con, $check_today_query);

if ($check_today_result) {
    while ($row = mysqli_fetch_assoc($check_today_result)) {
        $game_date = $row['Game_Date'];
        $game_status = $row['Game_Played'];
        
        if (!empty($game_date)) {
            $game_timestamp = strtotime($game_date);
            $time_until_game = $game_timestamp - $current_timestamp_unix;
            $time_since_game = $current_timestamp_unix - $game_timestamp;
            
            // Check if we should call today's API:
            if ($time_until_game <= 600 && $time_until_game >= -600) {
                $should_call_today = true;
                $today_games_info[] = "Starting soon: " . $row['Team_2_Abbrev'] . " @ " . $row['Team_1_Abbrev'];
            }
            if ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
                $should_call_today = true;
                $today_games_info[] = "Live: " . $row['Team_2_Abbrev'] . " @ " . $row['Team_1_Abbrev'];
            }
            if ($time_since_game >= 0 && $time_since_game <= 18000) {
                $should_call_today = true;
                if ($game_status == 'GAME_DONE') {
                    $today_games_info[] = "Recently ended: " . $row['Team_2_Abbrev'] . " @ " . $row['Team_1_Abbrev'];
                }
            }
        }
    }
}

// Check TOMORROW's games (only after 1 PM, and with rate limiting)
$should_call_tomorrow = false;
$tomorrow_games_info = [];

// Rate limiting for tomorrow's API - only call at specific times
// Check at 1:00-1:05 PM and 8:00-8:05 PM (5-minute windows to account for cron timing)
$current_minute = date('i');
$is_allowed_time = (($hour == 13 && $current_minute <= 5) || // 1:00-1:05 PM
                    ($hour == 20 && $current_minute <= 5));  // 8:00-8:05 PM

if ($hour >= 13 && $is_allowed_time) {
    // Check if tomorrow has any games scheduled
    $check_tomorrow_query = "SELECT Game_Date, Game_Played, Team_1_Abbrev, Team_2_Abbrev 
                            FROM sports_list 
                            WHERE Conference = 'NBA' 
                            AND Game_Date >= FROM_UNIXTIME($tomorrow_start)
                            AND Game_Date <= FROM_UNIXTIME($tomorrow_end)";
    $check_tomorrow_result = mysqli_query($con, $check_tomorrow_query);
    
    if ($check_tomorrow_result) {
        $has_tomorrow_games = false;
        $tomorrow_game_count = 0;
        
        while ($row = mysqli_fetch_assoc($check_tomorrow_result)) {
            if (!empty($row['Game_Date'])) {
                $has_tomorrow_games = true;
                $tomorrow_game_count++;
                $tomorrow_games_info[] = $row['Team_2_Abbrev'] . " @ " . $row['Team_1_Abbrev'];
            }
        }
        
        // Only call tomorrow's API if there are games tomorrow
        // OR if we need to clear out old game data (no games tomorrow but slots have old data)
        if ($has_tomorrow_games || $tomorrow_game_count == 0) {
            $should_call_tomorrow = true;
            echo "<!-- Debug: Tomorrow check triggered at " . $hour . ":" . str_pad($current_minute, 2, '0', STR_PAD_LEFT) . " -->\n";
        }
    }
}

echo "<!-- Debug: Today API needed = " . ($should_call_today ? 'YES' : 'NO') . " -->\n";
echo "<!-- Debug: Tomorrow API needed = " . ($should_call_tomorrow ? 'YES' : 'NO') . " -->\n";
echo "<!-- Debug: Current time = " . $hour . ":" . str_pad($current_minute, 2, '0', STR_PAD_LEFT) . " -->\n";
if (!empty($today_games_info)) {
    echo "<!-- Debug: Today's triggers: " . implode(', ', $today_games_info) . " -->\n";
}
if (!empty($tomorrow_games_info)) {
    echo "<!-- Debug: Tomorrow's games: " . implode(', ', $tomorrow_games_info) . " -->\n";
}

// If neither today nor tomorrow needs updating, exit early
if (!$should_call_today && !$should_call_tomorrow) {
    // Find when the next API call should happen
    $next_api_call_time = null;
    $next_api_call_reason = '';
    $next_game_info = '';
    
    // Check all games for next trigger
    $next_check_query = "SELECT Game_Date, Team_1_Abbrev, Team_2_Abbrev, Game_Time 
                         FROM sports_list 
                         WHERE Conference = 'NBA' 
                         AND Game_Date IS NOT NULL 
                         AND Game_Date != ''
                         AND Game_Date > NOW()
                         ORDER BY Game_Date ASC";
    $next_check_result = mysqli_query($con, $next_check_query);
    
    if ($next_check_result) {
        while ($row = mysqli_fetch_assoc($next_check_result)) {
            $game_date = $row['Game_Date'];
            $game_timestamp = strtotime($game_date);
            $api_trigger_time = $game_timestamp - 600; // 10 minutes before game
            
            if ($api_trigger_time > $current_timestamp_unix) {
                $next_api_call_time = $api_trigger_time;
                $next_api_call_reason = '10 minutes before tip-off';
                $next_game_info = $row['Team_2_Abbrev'] . ' @ ' . $row['Team_1_Abbrev'] . ' at ' . $row['Game_Time'];
                break;
            }
        }
    }
    
    // Check when tomorrow's API will be allowed next
    $next_tomorrow_check = null;
    if ($hour < 13) {
        $next_tomorrow_check = "1:00 PM (tomorrow's games check)";
    } elseif ($hour >= 13 && $hour < 20) {
        $next_tomorrow_check = "8:00 PM (tomorrow's games check)";
    } elseif ($hour >= 20) {
        $next_tomorrow_check = "1:00 PM tomorrow (tomorrow's games check)";
    }
    
    $time_until_next = '';
    if ($next_api_call_time) {
        $seconds_until = $next_api_call_time - $current_timestamp_unix;
        $hours = floor($seconds_until / 3600);
        $minutes = floor(($seconds_until % 3600) / 60);
        
        if ($hours > 0) {
            $time_until_next = $hours . ' hours, ' . $minutes . ' minutes';
        } else {
            $time_until_next = $minutes . ' minutes';
        }
    }
    
    echo "<!DOCTYPE html>
    <html>
    <head>
        <title>NBA Games - No Update Needed</title>
        <style>
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                margin: 20px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                display: flex;
                justify-content: center;
                align-items: center;
            }
            .message-box {
                background: white;
                padding: 40px;
                border-radius: 16px;
                box-shadow: 0 4px 15px rgba(0,0,0,0.2);
                text-align: center;
                max-width: 700px;
            }
            .message-box h2 {
                color: #2c3e50;
                margin-bottom: 20px;
            }
            .message-box p {
                color: #7f8c8d;
                line-height: 1.6;
            }
            .time {
                color: #667eea;
                font-weight: bold;
                margin-top: 20px;
            }
            .next-update {
                background: linear-gradient(135deg, #f8f9fa, #e9ecef);
                padding: 20px;
                border-radius: 12px;
                margin-top: 20px;
                text-align: left;
            }
            .next-update h3 {
                color: #2c3e50;
                margin: 0 0 10px 0;
                font-size: 16px;
            }
            .status-idle {
                display: inline-block;
                background: #ff9800;
                color: white;
                padding: 4px 12px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: bold;
                margin-bottom: 20px;
            }
            .schedule-info {
                background: #e3f2fd;
                padding: 15px;
                border-radius: 8px;
                margin: 15px 0;
                text-align: left;
            }
        </style>
    </head>
    <body>
        <div class='message-box'>
            <h2>🏀 NBA API Update Status</h2>
            <span class='status-idle'>SYSTEM IDLE</span>
            <p>No API calls needed at this time.</p>
            
            <div class='schedule-info'>
                <strong>ESPN API Call Schedule:</strong>
                <ul style='color: #7f8c8d; margin: 10px 0;'>
                    <li><strong>Today's Games:</strong> Called when games are starting (10 min before), live, or recently ended (up to 5 hours)</li>
                    <li><strong>Tomorrow's Games:</strong> Called only at 1:00 PM and 8:00 PM Eastern (max 2 times per day)</li>
                </ul>
            </div>";
    
    if ($next_api_call_time) {
        echo "<div class='next-update'>
                <h3>Next Game-Based API Call</h3>
                <p><strong>In:</strong> $time_until_next</p>
                <p><strong>Reason:</strong> $next_api_call_reason</p>
                <p><strong>Game:</strong> $next_game_info</p>
              </div>";
    }
    
    if ($next_tomorrow_check) {
        echo "<div class='next-update'>
                <h3>Next Tomorrow Check</h3>
                <p><strong>Scheduled at:</strong> $next_tomorrow_check</p>
                <p style='font-size: 12px; color: #95a5a6;'>Tomorrow's games are checked twice daily to minimize API calls</p>
              </div>";
    }
    
    echo "<p class='time'>Last Check: $time_updated Eastern</p>
            <p style='margin-top: 20px; font-size: 14px; color: #95a5a6;'>
                This conserves API resources and respects ESPN's rate limits.
            </p>
        </div>
    </body>
    </html>";
    
    // Still update the timestamp to show the script ran
    $sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NBA'";
    mysqli_query($con, $sql);
    
    // Close connection and exit
    mysqli_close($con);
    exit();
}

// Process API calls based on what's needed
echo "<!-- API call status: Today=" . ($should_call_today ? 'YES' : 'NO') . ", Tomorrow=" . ($should_call_tomorrow ? 'YES' : 'NO') . " -->\n";

// Update main timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NBA'";
mysqli_query($con, $sql);

/**
 * Convert hex color to RGB
 */
function hexToRgb($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) != 6) return "255,255,255";
    
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    // Force to white if too dark
    if ($r <= 16 && $g <= 16 && $b <= 16) {
        $r = $g = $b = 255;
    }
    
    return "$r,$g,$b";
}

/**
 * Extract game data from competition object
 */
function extractGameData($event, $game_num) {
    $gameData = [
        'game_num' => $game_num,
        'clock' => '0:00',
        'period' => 0,
        'game_status' => 'NOT_STARTED',
        'game_time' => '',
        'game_date' => '',
        'broadcast' => '',
        'score_value' => 'NO-SCORE',
        'team1' => [],
        'team2' => [],
        'last_play' => '',
        'who_has_possession' => '',
        'drive_desc' => '',
        'favorite' => [],
        'over_under' => '',
        'player_stats' => [],
        'trigger_reason' => ''
    ];
    
    $competition = $event['competitions'][0] ?? [];
    
    // Get game timing and status
    $status = $competition['status'] ?? [];
    $gameData['clock'] = $status['displayClock'] ?? '0:00';
    $gameData['period'] = $status['period'] ?? 0;
    
    // Determine game status
    $statusType = $status['type']['name'] ?? '';
    $statusState = $status['type']['state'] ?? '';
    
    if ($statusState == 'pre') {
        $gameData['game_status'] = 'NOT_STARTED';
    } elseif ($statusState == 'in') {
        if ($gameData['period'] == 2 && $gameData['clock'] == '0:00') {
            $gameData['game_status'] = 'HALF_TIME';
        } else {
            $gameData['game_status'] = 'IN_SESSION';
        }
    } elseif ($statusState == 'post') {
        $gameData['game_status'] = 'GAME_DONE';
    }
    
    // Handle overtime
    if ($gameData['period'] > 4) {
        $gameData['drive_desc'] = "OT " . ($gameData['period'] - 4);
    }
    
    // Get game time
    $gameData['game_date'] = $event['date'] ?? '';
    $gameData['game_time'] = date("g:i a", strtotime($gameData['game_date']));
    
    // Extract broadcast information
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcasts = [];
        foreach ($competition['broadcasts'] as $broadcast) {
            if (isset($broadcast['names']) && is_array($broadcast['names'])) {
                $broadcasts = array_merge($broadcasts, $broadcast['names']);
            }
        }
        $gameData['broadcast'] = implode(', ', array_slice($broadcasts, 0, 2)); // Take first 2 broadcasts
    }
    
    // Get teams (home is index 0, away is index 1 in the API)
    $competitors = $competition['competitors'] ?? [];
    
    foreach ($competitors as $competitor) {
        $teamData = [
            'id' => $competitor['team']['id'] ?? '',
            'name' => $competitor['team']['name'] ?? '',
            'location' => $competitor['team']['location'] ?? '',
            'abbreviation' => $competitor['team']['abbreviation'] ?? '',
            'score' => $competitor['score'] ?? '0',
            'record' => '',
            'color' => '',
            'color_rgb' => '255,255,255',
            'leaders' => []
        ];
        
        // Get team colors
        $color = $competitor['team']['color'] ?? '000000';
        $altColor = $competitor['team']['alternateColor'] ?? '';
        
        if ($color == '000000' && !empty($altColor)) {
            $color = $altColor;
        }
        
        $teamData['color'] = $color;
        $teamData['color_rgb'] = hexToRgb($color);
        
        // Get team record
        $records = $competitor['records'] ?? [];
        if (!empty($records)) {
            foreach ($records as $record) {
                if (($record['type'] ?? $record['name'] ?? '') == 'overall' || 
                    ($record['type'] ?? $record['name'] ?? '') == 'total') {
                    $teamData['record'] = $record['summary'] ?? '';
                    break;
                }
            }
            if (empty($teamData['record']) && !empty($records[0]['summary'])) {
                $teamData['record'] = $records[0]['summary'];
            }
        }
        
        // Get team leaders for player stats
        $leaders = $competitor['leaders'] ?? [];
        foreach ($leaders as $leader) {
            if ($leader['name'] == 'points' || $leader['name'] == 'rebounds') {
                $leaderInfo = $leader['leaders'][0] ?? [];
                if (!empty($leaderInfo)) {
                    $teamData['leaders'][$leader['name']] = [
                        'value' => $leaderInfo['displayValue'] ?? '',
                        'athlete' => $leaderInfo['athlete']['displayName'] ?? ''
                    ];
                }
            }
        }
        
        // Assign to team1 (home) or team2 (away)
        if ($competitor['homeAway'] == 'home') {
            $gameData['team1'] = $teamData;
        } else {
            $gameData['team2'] = $teamData;
        }
    }
    
    // Get in-game data if the game is active
    if (in_array($gameData['game_status'], ['IN_SESSION', 'HALF_TIME'])) {
        $situation = $competition['situation'] ?? [];
        $lastPlay = $situation['lastPlay'] ?? [];
        
        if (!empty($lastPlay)) {
            $gameData['last_play'] = $lastPlay['text'] ?? '';
            $scoreValue = $lastPlay['scoreValue'] ?? 0;
            
            // Check for scoring plays (3-pointers, 2-pointers, free throws)
            if ($scoreValue > 0) {
                $gameData['score_value'] = 'SCORE';
            }
        }
        
        // Determine possession
        if (!empty($situation['possession'])) {
            $possessionTeam = $situation['possession'] ?? '';
            if ($possessionTeam == $gameData['team1']['id']) {
                $gameData['who_has_possession'] = $gameData['team1']['abbreviation'];
            } elseif ($possessionTeam == $gameData['team2']['id']) {
                $gameData['who_has_possession'] = $gameData['team2']['abbreviation'];
            }
        }
    }
    
    // Format player stats for database
    if (!empty($gameData['team1']['leaders'])) {
        $stats1 = [];
        if (isset($gameData['team1']['leaders']['points'])) {
            $leader = $gameData['team1']['leaders']['points'];
            $lastName = explode(' ', $leader['athlete']);
            $lastName = end($lastName);
            $stats1[] = '_' . strtoupper($lastName) . ': ' . $leader['value'] . ' PTS';
        }
        if (isset($gameData['team1']['leaders']['rebounds'])) {
            $leader = $gameData['team1']['leaders']['rebounds'];
            $lastName = explode(' ', $leader['athlete']);
            $lastName = end($lastName);
            $stats1[] = '_' . strtoupper($lastName) . ': ' . $leader['value'] . ' RBD';
        }
        $gameData['player_stats']['home'] = implode(', ', $stats1);
    }
    
    if (!empty($gameData['team2']['leaders'])) {
        $stats2 = [];
        if (isset($gameData['team2']['leaders']['points'])) {
            $leader = $gameData['team2']['leaders']['points'];
            $lastName = explode(' ', $leader['athlete']);
            $lastName = end($lastName);
            $stats2[] = '!' . strtoupper($lastName) . ': ' . $leader['value'] . ' PTS';
        }
        if (isset($gameData['team2']['leaders']['rebounds'])) {
            $leader = $gameData['team2']['leaders']['rebounds'];
            $lastName = explode(' ', $leader['athlete']);
            $lastName = end($lastName);
            $stats2[] = '!' . strtoupper($lastName) . ': ' . $leader['value'] . ' RBD';
        }
        $gameData['player_stats']['away'] = implode(', ', $stats2);
    }
    
    // Get betting data
    $odds = $competition['odds'][0] ?? [];
    if (!empty($odds)) {
        $gameData['over_under'] = $odds['overUnder'] ?? '';
        
        // Determine favorite from details field
        if (isset($odds['details'])) {
            $details = $odds['details'];
            if (strpos($details, $gameData['team1']['abbreviation']) !== false) {
                $gameData['favorite']['team'] = $gameData['team1']['abbreviation'];
                // Extract spread from details
                preg_match('/[\-\+]?\d+\.?\d*/', $details, $matches);
                $gameData['favorite']['spread'] = $matches[0] ?? '';
            } elseif (strpos($details, $gameData['team2']['abbreviation']) !== false) {
                $gameData['favorite']['team'] = $gameData['team2']['abbreviation'];
                preg_match('/[\-\+]?\d+\.?\d*/', $details, $matches);
                $gameData['favorite']['spread'] = $matches[0] ?? '';
            }
        }
    }
    
    return $gameData;
}

// Initialize games arrays
$games_processed = [];
$games_for_tomorrow = [];
$current_date = date("Ymd");
$tomorrow_date = date("Ymd", strtotime("+1 day"));

// CALL TODAY'S API if needed
if ($should_call_today) {
    echo "<!-- Calling TODAY's games API -->\n";
    $api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/nba/scoreboard?dates=" . $current_date;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if (!$err) {
        $data = json_decode($response, true);
        if (!empty($data)) {
            $events = $data['events'] ?? [];
            
            foreach ($events as $index => $event) {
                $game_num = $index + 1;
                $gameData = extractGameData($event, $game_num);
                
                // Determine why this game triggered the API call
                $trigger_reason = '';
                if (!empty($gameData['game_date'])) {
                    $game_timestamp = strtotime($gameData['game_date']);
                    $time_until_game = $game_timestamp - $current_timestamp_unix;
                    $time_since_game = $current_timestamp_unix - $game_timestamp;
                    
                    if ($gameData['game_status'] == 'IN_SESSION' || $gameData['game_status'] == 'HALF_TIME') {
                        $trigger_reason = 'LIVE';
                    } elseif ($time_until_game <= 600 && $time_until_game >= -600) {
                        $trigger_reason = 'STARTING SOON';
                    } elseif ($time_since_game >= 0 && $time_since_game <= 18000 && $gameData['game_status'] == 'GAME_DONE') {
                        $hours_ago = round($time_since_game / 3600, 1);
                        $trigger_reason = 'ENDED ' . $hours_ago . 'h AGO';
                    }
                }
                
                $gameData['trigger_reason'] = $trigger_reason;
                
                // Update database for today's games
                $update_data = [
                    'Clock' => mysqli_real_escape_string($con, $gameData['clock']),
                    'Period' => intval($gameData['period']),
                    'Game_Played' => mysqli_real_escape_string($con, $gameData['game_status']),
                    'Game_Time' => mysqli_real_escape_string($con, $gameData['game_time']),
                    'Game_Date' => mysqli_real_escape_string($con, $gameData['game_date']),
                    'broadcast' => mysqli_real_escape_string($con, $gameData['broadcast']),
                    'scoreValue' => mysqli_real_escape_string($con, $gameData['score_value']),
                    'Team_1_Name' => mysqli_real_escape_string($con, $gameData['team1']['name']),
                    'Team_2_Name' => mysqli_real_escape_string($con, $gameData['team2']['name']),
                    'Team_1_Record' => mysqli_real_escape_string($con, $gameData['team1']['record']),
                    'Team_2_Record' => mysqli_real_escape_string($con, $gameData['team2']['record']),
                    'Team_1_Score' => intval($gameData['team1']['score']),
                    'Team_2_Score' => intval($gameData['team2']['score']),
                    'Team_1_Location' => mysqli_real_escape_string($con, $gameData['team1']['location']),
                    'Team_2_Location' => mysqli_real_escape_string($con, $gameData['team2']['location']),
                    'Team_1_Abbrev' => mysqli_real_escape_string($con, $gameData['team1']['abbreviation']),
                    'Team_2_Abbrev' => mysqli_real_escape_string($con, $gameData['team2']['abbreviation']),
                    'Team_1_Color' => '"' . mysqli_real_escape_string($con, $gameData['team1']['color_rgb']) . '"',
                    'Team_2_Color' => '"' . mysqli_real_escape_string($con, $gameData['team2']['color_rgb']) . '"',
                    'Who_Has_Ball' => mysqli_real_escape_string($con, $gameData['who_has_possession']),
                    'lastPlay' => mysqli_real_escape_string($con, $gameData['last_play']),
                    'OverUnder' => mysqli_real_escape_string($con, $gameData['over_under']),
                    'Favorite_Team' => mysqli_real_escape_string($con, $gameData['favorite']['team'] ?? ''),
                    'Favorite_Points' => mysqli_real_escape_string($con, $gameData['favorite']['spread'] ?? ''),
                    'lastUpdatedTime' => $current_timestamp
                ];
                
                // Add player stats if available
                if (isset($gameData['player_stats']['home'])) {
                    $update_data['Ru1'] = mysqli_real_escape_string($con, $gameData['player_stats']['home']);
                }
                if (isset($gameData['player_stats']['away'])) {
                    $update_data['Ru2'] = mysqli_real_escape_string($con, $gameData['player_stats']['away']);
                }
                
                // Extract RGB values for separate columns
                $rgb1 = explode(',', $gameData['team1']['color_rgb']);
                $update_data['Team_1_Color_R'] = $rgb1[0] ?? '255';
                $update_data['Team_1_Color_G'] = $rgb1[1] ?? '255';
                $update_data['Team_1_Color_B'] = $rgb1[2] ?? '255';
                
                $rgb2 = explode(',', $gameData['team2']['color_rgb']);
                $update_data['Team_2_Color_R'] = $rgb2[0] ?? '255';
                $update_data['Team_2_Color_G'] = $rgb2[1] ?? '255';
                $update_data['Team_2_Color_B'] = $rgb2[2] ?? '255';
                
                // Build UPDATE query
                $set_clauses = [];
                foreach ($update_data as $field => $value) {
                    $set_clauses[] = "$field='$value'";
                }
                
                $query = "UPDATE sports_list SET " . implode(', ', $set_clauses) . 
                         " WHERE Game = $game_num AND Conference = 'NBA'";
                
                mysqli_query($con, $query);
                $games_processed[] = $gameData;
            }
        }
    }
}

// CALL TOMORROW'S API if needed (separate from today)
if ($should_call_tomorrow) {
    echo "<!-- Calling TOMORROW's games API (scheduled check at " . $hour . ":00) -->\n";
    $api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/nba/scoreboard?dates=" . $tomorrow_date;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if (!$err) {
        $data = json_decode($response, true);
        if (!empty($data)) {
            $events = $data['events'] ?? [];
            $start_num = count($games_processed) + 1;
            
            foreach ($events as $index => $event) {
                $game_num = $start_num + $index;
                
                // Only process if we have slots available (max 20 games)
                if ($game_num <= 20) {
                    $gameData = extractGameData($event, $game_num);
                    $gameData['trigger_reason'] = 'TOMORROW';
                    
                    // Update database for tomorrow's games
                    $update_data = [
                        'Clock' => mysqli_real_escape_string($con, $gameData['clock']),
                        'Period' => intval($gameData['period']),
                        'Game_Played' => mysqli_real_escape_string($con, $gameData['game_status']),
                        'Game_Time' => mysqli_real_escape_string($con, $gameData['game_time']),
                        'Game_Date' => mysqli_real_escape_string($con, $gameData['game_date']),
                        'broadcast' => mysqli_real_escape_string($con, $gameData['broadcast']),
                        'scoreValue' => mysqli_real_escape_string($con, $gameData['score_value']),
                        'Team_1_Name' => mysqli_real_escape_string($con, $gameData['team1']['name']),
                        'Team_2_Name' => mysqli_real_escape_string($con, $gameData['team2']['name']),
                        'Team_1_Record' => mysqli_real_escape_string($con, $gameData['team1']['record']),
                        'Team_2_Record' => mysqli_real_escape_string($con, $gameData['team2']['record']),
                        'Team_1_Score' => intval($gameData['team1']['score']),
                        'Team_2_Score' => intval($gameData['team2']['score']),
                        'Team_1_Location' => mysqli_real_escape_string($con, $gameData['team1']['location']),
                        'Team_2_Location' => mysqli_real_escape_string($con, $gameData['team2']['location']),
                        'Team_1_Abbrev' => mysqli_real_escape_string($con, $gameData['team1']['abbreviation']),
                        'Team_2_Abbrev' => mysqli_real_escape_string($con, $gameData['team2']['abbreviation']),
                        'Team_1_Color' => '"' . mysqli_real_escape_string($con, $gameData['team1']['color_rgb']) . '"',
                        'Team_2_Color' => '"' . mysqli_real_escape_string($con, $gameData['team2']['color_rgb']) . '"',
                        'Who_Has_Ball' => mysqli_real_escape_string($con, $gameData['who_has_possession']),
                        'lastPlay' => mysqli_real_escape_string($con, $gameData['last_play']),
                        'OverUnder' => mysqli_real_escape_string($con, $gameData['over_under']),
                        'Favorite_Team' => mysqli_real_escape_string($con, $gameData['favorite']['team'] ?? ''),
                        'Favorite_Points' => mysqli_real_escape_string($con, $gameData['favorite']['spread'] ?? ''),
                        'lastUpdatedTime' => $current_timestamp
                    ];
                    
                    // Extract RGB values
                    $rgb1 = explode(',', $gameData['team1']['color_rgb']);
                    $update_data['Team_1_Color_R'] = $rgb1[0] ?? '255';
                    $update_data['Team_1_Color_G'] = $rgb1[1] ?? '255';
                    $update_data['Team_1_Color_B'] = $rgb1[2] ?? '255';
                    
                    $rgb2 = explode(',', $gameData['team2']['color_rgb']);
                    $update_data['Team_2_Color_R'] = $rgb2[0] ?? '255';
                    $update_data['Team_2_Color_G'] = $rgb2[1] ?? '255';
                    $update_data['Team_2_Color_B'] = $rgb2[2] ?? '255';
                    
                    // Build UPDATE query
                    $set_clauses = [];
                    foreach ($update_data as $field => $value) {
                        $set_clauses[] = "$field='$value'";
                    }
                    
                    $query = "UPDATE sports_list SET " . implode(', ', $set_clauses) . 
                             " WHERE Game = $game_num AND Conference = 'NBA'";
                    
                    mysqli_query($con, $query);
                    $games_for_tomorrow[] = $gameData;
                }
            }
        }
    }
}

// Combine all games for display
$all_games = array_merge($games_processed, $games_for_tomorrow);

// If no games were processed but we made API calls, create empty display
if (empty($all_games) && ($should_call_today || $should_call_tomorrow)) {
    echo "<!-- No games found in API responses -->\n";
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>NBA Games Database Update</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .header {
            text-align: center;
            color: white;
            margin-bottom: 30px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        .update-time {
            text-align: center;
            color: white;
            font-size: 14px;
            margin-bottom: 20px;
            opacity: 0.9;
        }
        
        /* Database Table Styles */
        .database-table {
            width: 100%;
            background: white;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            border-radius: 12px;
            overflow: hidden;
        }
        .database-table table {
            width: 100%;
            border-collapse: collapse;
            font-size: 12px;
        }
        .database-table th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 6px;
            text-align: left;
            font-size: 11px;
            position: sticky;
            top: 0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .database-table td {
            padding: 8px 6px;
            border-bottom: 1px solid #eee;
            font-size: 11px;
        }
        .database-table tr:hover {
            background-color: #f8f9ff;
        }
        .status-not-started { 
            color: #95a5a6;
            font-weight: 600;
        }
        .status-in-session { 
            color: #27ae60; 
            font-weight: bold;
            animation: pulse 2s infinite;
        }
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
        .status-half-time { 
            color: #f39c12;
            font-weight: 600;
        }
        .status-game-done { 
            color: #34495e;
            font-weight: 600;
        }
        .score-indicator {
            background: linear-gradient(135deg, #ff6b6b 0%, #ff5252 100%);
            color: white;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: bold;
            display: inline-block;
        }
        .broadcast-cell {
            background: #e8f4fd;
            font-weight: bold;
            color: #1976d2;
        }
        
        /* Card Grid Styles */
        .games-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .game-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            padding: 20px;
            position: relative;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .game-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.2);
        }
        .game-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
        }
        .game-status {
            font-size: 12px;
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: bold;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .game-status.status-not-started {
            background: #ecf0f1;
            color: #7f8c8d;
        }
        .game-status.status-in-session {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            color: white;
        }
        .game-status.status-half-time {
            background: linear-gradient(135deg, #f39c12, #f1c40f);
            color: white;
        }
        .game-status.status-game-done {
            background: #34495e;
            color: white;
        }
        .teams-container {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 20px 0;
        }
        .team {
            flex: 1;
            text-align: center;
        }
        .team-name {
            font-weight: bold;
            margin-bottom: 8px;
            font-size: 16px;
            color: #2c3e50;
        }
        .team-score {
            font-size: 36px;
            font-weight: bold;
            margin: 10px 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .team-record {
            font-size: 12px;
            color: #7f8c8d;
        }
        .vs {
            padding: 0 20px;
            color: #bdc3c7;
            font-size: 14px;
            font-weight: 600;
        }
        .game-info {
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            padding: 12px;
            border-radius: 12px;
            margin-top: 15px;
            font-size: 13px;
        }
        .game-info-row {
            display: flex;
            justify-content: space-between;
            margin: 8px 0;
        }
        .game-info-label {
            color: #666;
            font-weight: 600;
        }
        .last-play {
            background: linear-gradient(135deg, #e8f5e9, #c8e6c9);
            padding: 12px;
            border-radius: 12px;
            margin-top: 12px;
            font-size: 13px;
            border-left: 4px solid #4caf50;
        }
        .player-stats {
            background: #fff3e0;
            padding: 10px;
            border-radius: 8px;
            margin-top: 10px;
            font-size: 12px;
        }
        .summary-section {
            background: white;
            padding: 25px;
            border-radius: 16px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            margin-top: 30px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        .stat-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            color: white;
        }
        .stat-value {
            font-size: 32px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        .stat-label {
            font-size: 13px;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
    </style>
</head>
<body>
    <h1 class="header">🏀 NBA Games Database Update</h1>
    <div class="update-time">Last Updated: <?= $time_updated ?> Eastern</div>
    
    <?php 
    // Count trigger reasons
    $live_count = 0;
    $starting_soon_count = 0;
    $recently_ended_count = 0;
    
    foreach ($all_games as $game) {
        if (!empty($game['trigger_reason'])) {
            if ($game['trigger_reason'] == 'LIVE') $live_count++;
            elseif ($game['trigger_reason'] == 'STARTING SOON') $starting_soon_count++;
            elseif (strpos($game['trigger_reason'], 'ENDED') !== false) $recently_ended_count++;
        }
    }
    
    if ($live_count > 0 || $starting_soon_count > 0 || $recently_ended_count > 0): ?>
    <div style="background: white; padding: 20px; border-radius: 12px; margin: 20px auto; max-width: 600px; box-shadow: 0 4px 15px rgba(0,0,0,0.2);">
        <h3 style="color: #2c3e50; margin: 0 0 15px 0;">📊 API Call Triggered By:</h3>
        <div style="display: flex; gap: 15px; justify-content: center; flex-wrap: wrap;">
            <?php if ($live_count > 0): ?>
            <div style="background: #4caf50; color: white; padding: 10px 20px; border-radius: 8px;">
                <div style="font-size: 24px; font-weight: bold;"><?= $live_count ?></div>
                <div style="font-size: 12px;">LIVE GAMES</div>
            </div>
            <?php endif; ?>
            <?php if ($starting_soon_count > 0): ?>
            <div style="background: #2196f3; color: white; padding: 10px 20px; border-radius: 8px;">
                <div style="font-size: 24px; font-weight: bold;"><?= $starting_soon_count ?></div>
                <div style="font-size: 12px;">STARTING SOON</div>
            </div>
            <?php endif; ?>
            <?php if ($recently_ended_count > 0): ?>
            <div style="background: #ff9800; color: white; padding: 10px 20px; border-radius: 8px;">
                <div style="font-size: 24px; font-weight: bold;"><?= $recently_ended_count ?></div>
                <div style="font-size: 12px;">RECENTLY ENDED</div>
            </div>
            <?php endif; ?>
        </div>
        <p style="margin-top: 15px; color: #7f8c8d; font-size: 14px; text-align: center;">
            ESPN API called successfully. Next check in 1 minute.
        </p>
    </div>
    <?php endif; ?>
    
    <!-- Detailed Database Table -->
    <h2 style="color: white; margin-top: 30px;">Database Update Details</h2>
    <div class="database-table">
        <table>
            <thead>
                <tr>
                    <th>Game</th>
                    <th>Status</th>
                    <th>Time</th>
                    <th>Date</th>
                    <th>Broadcast</th>
                    <th>Away Team</th>
                    <th>Away Score</th>
                    <th>Away Record</th>
                    <th>Home Team</th>
                    <th>Home Score</th>
                    <th>Home Record</th>
                    <th>Period</th>
                    <th>Clock</th>
                    <th>Score Event</th>
                    <th>Last Play</th>
                    <th>Possession</th>
                    <th>Favorite</th>
                    <th>Spread</th>
                    <th>O/U</th>
                    <th>Updated</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($all_games as $game): ?>
                <tr>
                    <td><?= $game['game_num'] ?></td>
                    <td class="status-<?= strtolower(str_replace('_', '-', $game['game_status'])) ?>">
                        <?= $game['game_status'] ?>
                    </td>
                    <td><?= $game['game_time'] ?></td>
                    <td><?= date('m/d', strtotime($game['game_date'])) ?></td>
                    <td class="broadcast-cell"><?= $game['broadcast'] ?: '-' ?></td>
                    <td><strong><?= $game['team2']['abbreviation'] ?></strong></td>
                    <td><strong><?= $game['team2']['score'] ?></strong></td>
                    <td><?= $game['team2']['record'] ?></td>
                    <td><strong><?= $game['team1']['abbreviation'] ?></strong></td>
                    <td><strong><?= $game['team1']['score'] ?></strong></td>
                    <td><?= $game['team1']['record'] ?></td>
                    <td><?= $game['period'] ?: '-' ?></td>
                    <td><?= $game['clock'] ?></td>
                    <td>
                        <?php if ($game['score_value'] == 'SCORE'): ?>
                            <span class="score-indicator">SCORE</span>
                        <?php else: ?>
                            <?= $game['score_value'] ?>
                        <?php endif; ?>
                    </td>
                    <td style="max-width: 200px; overflow: hidden; text-overflow: ellipsis;" 
                        title="<?= htmlspecialchars($game['last_play']) ?>">
                        <?= !empty($game['last_play']) ? 
                            htmlspecialchars(substr($game['last_play'], 0, 50)) . 
                            (strlen($game['last_play']) > 50 ? '...' : '') : '-' ?>
                    </td>
                    <td><?= $game['who_has_possession'] ?: '-' ?></td>
                    <td><?= $game['favorite']['team'] ?? '-' ?></td>
                    <td><?= $game['favorite']['spread'] ?? '-' ?></td>
                    <td><?= $game['over_under'] ?: '-' ?></td>
                    <td style="font-size: 9px;">Now</td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Game Cards Grid -->
    <h2 style="color: white; margin-top: 30px;">Game Overview</h2>
    <div class="games-grid">
        <?php foreach ($all_games as $game): ?>
        <div class="game-card">
            <div class="game-header">
                <span>Game #<?= $game['game_num'] ?></span>
                <div>
                    <?php if (!empty($game['trigger_reason'])): ?>
                    <span style="background: #4caf50; color: white; padding: 4px 8px; border-radius: 12px; font-size: 10px; margin-right: 8px;">
                        <?= $game['trigger_reason'] ?>
                    </span>
                    <?php endif; ?>
                    <span class="game-status status-<?= strtolower(str_replace('_', '-', $game['game_status'])) ?>">
                        <?= $game['game_status'] ?>
                    </span>
                </div>
            </div>
            
            <div class="teams-container">
                <div class="team">
                    <div class="team-name"><?= $game['team2']['abbreviation'] ?></div>
                    <div class="team-score"><?= $game['team2']['score'] ?></div>
                    <div class="team-record"><?= $game['team2']['record'] ?></div>
                </div>
                <div class="vs">@</div>
                <div class="team">
                    <div class="team-name"><?= $game['team1']['abbreviation'] ?></div>
                    <div class="team-score"><?= $game['team1']['score'] ?></div>
                    <div class="team-record"><?= $game['team1']['record'] ?></div>
                </div>
            </div>
            
            <?php if (in_array($game['game_status'], ['IN_SESSION', 'HALF_TIME'])): ?>
            <div class="game-info">
                <div class="game-info-row">
                    <span class="game-info-label">Quarter:</span>
                    <span><?= $game['period'] > 4 ? 'OT ' . ($game['period'] - 4) : 'Q' . $game['period'] ?></span>
                </div>
                <div class="game-info-row">
                    <span class="game-info-label">Clock:</span>
                    <span><?= $game['clock'] ?></span>
                </div>
                <?php if ($game['broadcast']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">TV:</span>
                    <span style="font-weight: bold; color: #1976d2;"><?= $game['broadcast'] ?></span>
                </div>
                <?php endif; ?>
                <?php if ($game['who_has_possession']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Possession:</span>
                    <span>🏀 <?= $game['who_has_possession'] ?></span>
                </div>
                <?php endif; ?>
            </div>
            
            <?php if ($game['last_play']): ?>
            <div class="last-play">
                <strong>Last Play:</strong> <?= htmlspecialchars($game['last_play']) ?>
            </div>
            <?php endif; ?>
            
            <?php 
            // Show player stats if available
            $hasStats = false;
            if (!empty($game['team1']['leaders']) || !empty($game['team2']['leaders'])): 
                $hasStats = true;
            ?>
            <div class="player-stats">
                <strong>Top Performers:</strong><br>
                <?php if (!empty($game['team1']['leaders']['points'])): ?>
                    <div>🏠 <?= $game['team1']['leaders']['points']['athlete'] ?>: 
                         <?= $game['team1']['leaders']['points']['value'] ?> pts</div>
                <?php endif; ?>
                <?php if (!empty($game['team2']['leaders']['points'])): ?>
                    <div>✈️ <?= $game['team2']['leaders']['points']['athlete'] ?>: 
                         <?= $game['team2']['leaders']['points']['value'] ?> pts</div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php else: ?>
            <div class="game-info">
                <div class="game-info-row">
                    <span class="game-info-label">Tip-off:</span>
                    <span><?= $game['game_time'] ?></span>
                </div>
                <?php if ($game['broadcast']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">TV:</span>
                    <span style="font-weight: bold; color: #1976d2;"><?= $game['broadcast'] ?></span>
                </div>
                <?php endif; ?>
                <?php if (!empty($game['favorite']['team'])): ?>
                <div class="game-info-row">
                    <span class="game-info-label">Line:</span>
                    <span><?= $game['favorite']['team'] ?> <?= $game['favorite']['spread'] ?></span>
                </div>
                <?php endif; ?>
                <?php if ($game['over_under']): ?>
                <div class="game-info-row">
                    <span class="game-info-label">O/U:</span>
                    <span><?= $game['over_under'] ?></span>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>
    </div>
    
    <div class="summary-section">
        <h2>Update Summary</h2>
        <div class="stats-grid">
            <div class="stat-box">
                <div class="stat-value"><?= count($all_games) ?></div>
                <div class="stat-label">Total Games</div>
            </div>
            <div class="stat-box">
                <div class="stat-value">
                    <?= count(array_filter($all_games, function($g) { return $g['game_status'] == 'IN_SESSION'; })) ?>
                </div>
                <div class="stat-label">Live Games</div>
            </div>
            <div class="stat-box">
                <div class="stat-value">
                    <?= count(array_filter($all_games, function($g) { return $g['game_status'] == 'GAME_DONE'; })) ?>
                </div>
                <div class="stat-label">Final</div>
            </div>
            <div class="stat-box">
                <div class="stat-value">
                    <?= count(array_filter($all_games, function($g) { return $g['game_status'] == 'NOT_STARTED'; })) ?>
                </div>
                <div class="stat-label">Upcoming</div>
            </div>
        </div>
        
        <div style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 12px;">
            <h3 style="margin: 0 0 10px 0; color: #2c3e50;">Database Information</h3>
            <ul style="margin: 0; padding-left: 20px; color: #555;">
                <li>Database: <?= $database_gscroll ?></li>
                <li>Table: sports_list</li>
                <li>Conference: NBA</li>
                <li>API Source: ESPN NBA Scoreboard</li>
                <li>Broadcast Information: Included for all games</li>
                <li>Update Frequency: Every minute (when games are active)</li>
                <li>Update Time: <?= $time_updated ?></li>
            </ul>
        </div>
    </div>
</body>
</html>

<?php
// Close database connection
mysqli_close($con);

// Call external API endpoint
$url = SERVER_URL . "Scrolling/App_Output/API/Games/NBA_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    echo '<!-- External API error: ' . curl_error($ch) . ' -->';
}
curl_close($ch);
?>