<?php
/**
 * API Update Cron Script
 * Curls all PHP files in the standings directory and generates a report
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);
set_time_limit(300); // 5 minutes
ini_set('memory_limit', '256M');

// Set timezone
date_default_timezone_set('America/New_York');

require_once '../../../config.php';

// Build base URL from SERVER_URL
$baseUrl = rtrim(SERVER_URL, '/') . '/updateAPIs/standings/';

echo "Base URL: " . $baseUrl . "<br><br>";

// List of PHP files to curl
$apiFiles = [
    'EPL.php',
    'LaLiga.php',
    'MLBScoreBoard.php',
    'MLS.php',
    'NBAScoreBoard.php',
    'NCAA_BB.php',
    'NCAAFScoreBoard.php',
    'NFLScoreBoard.php',
    'NHLScoreBoard.php',
    'WNBAScoreBoard.php',
    'Update_NASCAR_Leaderboard.php',
    'NCAAB.php'
];

// DEBUG: Show total files
echo "<div style='background: yellow; padding: 10px; margin: 10px 0;'>";
echo "DEBUG: Total files in array: " . count($apiFiles) . "<br>";
echo "Files: " . implode(', ', $apiFiles) . "<br>";
echo "</div>";

// Results array to store execution details
$results = [];

// Function to make curl request
function curlApi($url) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    
    $startTime = microtime(true);
    $response = curl_exec($ch);
    $endTime = microtime(true);
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'success' => ($httpCode == 200 && !$error),
        'http_code' => $httpCode,
        'error' => $error,
        'execution_time' => round(($endTime - $startTime), 3),
        'response_size' => strlen($response)
    ];
}

// DEBUG: Track processing
echo "<div style='background: lightblue; padding: 10px; margin: 10px 0;'>";
echo "DEBUG: Starting to process files...<br>";

$fileCount = 0;

// Process each API file
foreach ($apiFiles as $index => $file) {
    $fileCount++;
    echo "Processing file #" . ($index + 1) . ": " . $file . " ... ";
    
    $url = $baseUrl . $file;
    $startTime = date('Y-m-d H:i:s');
    
    // Make the curl request
    $result = curlApi($url);
    
    // Store results
    $results[] = [
        'file' => $file,
        'url' => $url,
        'timestamp' => $startTime,
        'success' => $result['success'],
        'http_code' => $result['http_code'],
        'error' => $result['error'],
        'execution_time' => $result['execution_time'],
        'response_size' => $result['response_size']
    ];
    
    echo "Done (HTTP: " . $result['http_code'] . ", Time: " . $result['execution_time'] . "s)<br>";
    
    // Add a slight pause between requests (0.5 seconds)
    usleep(500000);
}

// Add the special Standings.php endpoint at the end
echo "<br><strong>Processing additional endpoint...</strong><br>";
$standingsUrl = 'https://gscroll.us/Scrolling/App_Output/API/Standings/Standings.php';
echo "Processing: Standings.php (full URL) ... ";

$startTime = date('Y-m-d H:i:s');
$result = curlApi($standingsUrl);

// Store results for the special endpoint
$results[] = [
    'file' => 'Standings.php (App_Output)',
    'url' => $standingsUrl,
    'timestamp' => $startTime,
    'success' => $result['success'],
    'http_code' => $result['http_code'],
    'error' => $result['error'],
    'execution_time' => $result['execution_time'],
    'response_size' => $result['response_size']
];

echo "Done (HTTP: " . $result['http_code'] . ", Time: " . $result['execution_time'] . "s)<br>";
$fileCount++;

echo "<br>DEBUG: Processed " . $fileCount . " files total<br>";
echo "DEBUG: Results array has " . count($results) . " entries<br>";
echo "</div>";

// DEBUG: Check if NCAAB.php is in results
echo "<div style='background: pink; padding: 10px; margin: 10px 0;'>";
echo "DEBUG: Checking for NCAAB.php in results:<br>";
$ncaabFound = false;
foreach ($results as $r) {
    if ($r['file'] == 'NCAAB.php') {
        $ncaabFound = true;
        echo "FOUND NCAAB.php in results - HTTP: " . $r['http_code'] . ", Success: " . ($r['success'] ? 'YES' : 'NO') . "<br>";
    }
}
if (!$ncaabFound) {
    echo "NCAAB.php NOT FOUND in results array!<br>";
}

echo "<br>All files in results:<br>";
foreach ($results as $i => $r) {
    echo ($i + 1) . ". " . $r['file'] . "<br>";
}
echo "</div>";

// Calculate summary statistics
$totalFiles = count($results);
$successCount = array_sum(array_column($results, 'success'));
$failureCount = $totalFiles - $successCount;
$totalExecutionTime = array_sum(array_column($results, 'execution_time'));

// DEBUG: Summary stats
echo "<div style='background: lightgreen; padding: 10px; margin: 10px 0;'>";
echo "DEBUG SUMMARY:<br>";
echo "- Expected files: " . (count($apiFiles) + 1) . " (including Standings.php)<br>";
echo "- Actual results: " . $totalFiles . "<br>";
echo "- Successful: " . $successCount . "<br>";
echo "- Failed: " . $failureCount . "<br>";
echo "- Total execution time: " . $totalExecutionTime . "s<br>";
echo "</div>";

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>API Update Report - <?= date('Y-m-d H:i:s') ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            min-height: 100vh;
            padding: 20px;
            color: #333;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            padding: 40px;
            backdrop-filter: blur(10px);
        }
        
        h1 {
            text-align: center;
            color: #2a5298;
            margin-bottom: 10px;
            font-size: 2.5em;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .timestamp {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 1.1em;
        }
        
        .summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .summary-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px;
            border-radius: 15px;
            text-align: center;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        
        .summary-card:hover {
            transform: translateY(-5px);
        }
        
        .summary-card.success {
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
        }
        
        .summary-card.failure {
            background: linear-gradient(135deg, #eb3349 0%, #f45c43 100%);
        }
        
        .summary-card.time {
            background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
        }
        
        .summary-number {
            font-size: 3em;
            font-weight: bold;
            margin-bottom: 10px;
        }
        
        .summary-label {
            font-size: 1.1em;
            opacity: 0.9;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        th {
            background: linear-gradient(135deg, #2a5298 0%, #1e3c72 100%);
            color: white;
            padding: 20px 15px;
            text-align: left;
            font-weight: 600;
            font-size: 1.1em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        td {
            padding: 15px;
            border-bottom: 1px solid #eee;
            transition: background-color 0.3s ease;
        }
        
        tr:hover td {
            background-color: #f8f9fa;
        }
        
        tr:last-child td {
            border-bottom: none;
        }
        
        /* Highlight the special Standings.php row */
        tr.special-endpoint td {
            background-color: #fff9e6;
        }
        
        tr.special-endpoint:hover td {
            background-color: #fff3cc;
        }
        
        .status {
            display: inline-block;
            padding: 6px 16px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.9em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status.success {
            background: #38ef7d;
            color: #0d7f3f;
        }
        
        .status.failure {
            background: #ff6b6b;
            color: #8b0000;
        }
        
        .http-code {
            font-weight: bold;
            color: #2a5298;
            font-size: 1.1em;
        }
        
        .execution-time {
            color: #4facfe;
            font-weight: 600;
        }
        
        .file-name {
            font-weight: 600;
            color: #2a5298;
        }
        
        .file-name.special {
            color: #ff9800;
        }
        
        .error-message {
            color: #ff6b6b;
            font-size: 0.9em;
            font-style: italic;
        }
        
        .response-size {
            color: #666;
            font-size: 0.9em;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 20px;
            }
            
            h1 {
                font-size: 1.8em;
            }
            
            table {
                font-size: 0.9em;
            }
            
            th, td {
                padding: 10px 5px;
            }
            
            .summary {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📊 API Update Report</h1>
        <p class="timestamp">Generated: <?= date('l, F j, Y - g:i:s A') ?></p>
        
        <div class="summary">
            <div class="summary-card">
                <div class="summary-number"><?= $totalFiles ?></div>
                <div class="summary-label">Total APIs</div>
            </div>
            <div class="summary-card success">
                <div class="summary-number"><?= $successCount ?></div>
                <div class="summary-label">Successful</div>
            </div>
            <div class="summary-card failure">
                <div class="summary-number"><?= $failureCount ?></div>
                <div class="summary-label">Failed</div>
            </div>
            <div class="summary-card time">
                <div class="summary-number"><?= number_format($totalExecutionTime, 1) ?>s</div>
                <div class="summary-label">Total Time</div>
            </div>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>#</th>
                    <th>API File</th>
                    <th>Status</th>
                    <th>HTTP Code</th>
                    <th>Execution Time</th>
                    <th>Response Size</th>
                    <th>Timestamp</th>
                    <th>Error Details</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($results as $index => $result): ?>
                <?php 
                    // Check if this is the special Standings.php endpoint
                    $isSpecial = strpos($result['file'], 'App_Output') !== false;
                ?>
                <tr <?= $isSpecial ? 'class="special-endpoint"' : '' ?>>
                    <td><?= $index + 1 ?></td>
                    <td class="file-name <?= $isSpecial ? 'special' : '' ?>"><?= htmlspecialchars($result['file']) ?></td>
                    <td>
                        <span class="status <?= $result['success'] ? 'success' : 'failure' ?>">
                            <?= $result['success'] ? '✓ Success' : '✗ Failed' ?>
                        </span>
                    </td>
                    <td class="http-code"><?= $result['http_code'] ?: 'N/A' ?></td>
                    <td class="execution-time"><?= $result['execution_time'] ?>s</td>
                    <td class="response-size"><?= $result['response_size'] > 0 ? number_format($result['response_size'] / 1024, 2) . ' KB' : 'N/A' ?></td>
                    <td><?= date('H:i:s', strtotime($result['timestamp'])) ?></td>
                    <td class="error-message"><?= $result['error'] ?: '-' ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</body>
</html>