<?php
/**
 * Optimized WNBA Time Processor with Auto-Update Logic
 * 
 * This script processes WNBA games, converts UTC to Eastern time,
 * and triggers API updates when games are in session.
 * 
 * Key improvements:
 * - Better error handling and logging
 * - More efficient database operations
 * - Cleaner code structure with classes
 * - Proper cURL handling with timeout and error management
 * - Status-based API triggering
 * 
 * Created: 2025-03-04
 * Optimized: 2025-05-24
 */

class WNBAProcessor {
    private $db;
    private $timezone;
    private $processed_games = [];
    private $api_called = false;
    private $api_result = '';
    private $api_error = '';
    private $api_execution_time = 0;
    
    // Configuration
    public const API_URL = 'https://gscroll.us/updateAPIs/get_WNBA.php';
    private const CURL_TIMEOUT = 30;
    private const GAME_DURATION_HOURS = 4;
    private const START_WARNING_MINUTES = 5;
    
    public function __construct($db_config) {
        // Set timezone
        $this->timezone = new DateTimeZone('America/New_York');
        date_default_timezone_set('America/New_York');
        
        // Connect to database
        $this->connectDatabase($db_config);
    }
    
    private function connectDatabase($config) {
        $this->db = new mysqli(
            $config['host'], 
            $config['username'], 
            $config['password'], 
            $config['database']
        );
        
        if ($this->db->connect_error) {
            throw new Exception("Database connection failed: " . $this->db->connect_error);
        }
        
        $this->db->set_charset("utf8");
    }
    
    public function processGames() {
        try {
            $games = $this->fetchWNBAGames();
            $current_time = new DateTime('now', $this->timezone);
            $needs_api_update = false;
            
            foreach ($games as $game) {
                $processed_game = $this->processGame($game, $current_time);
                if ($processed_game) {
                    $this->processed_games[] = $processed_game;
                    
                    // Check if we need to call API
                    if ($processed_game['in_session'] || $processed_game['about_to_start']) {
                        $needs_api_update = true;
                    }
                }
            }
            
            // Call API if any game is in session or about to start
            if ($needs_api_update) {
                $this->callUpdateAPI();
            }
            
        } catch (Exception $e) {
            error_log("Error processing WNBA games: " . $e->getMessage());
            throw $e;
        }
    }
    
    private function fetchWNBAGames() {
        $query = "SELECT Game, Team_1_Name, Team_2_Name, Game_Date, Game_Played, Game_Time 
                  FROM sports_list 
                  WHERE Conference = 'WNBA' 
                  AND Game_Date IS NOT NULL 
                  AND Game_Date != ''
                  ORDER BY Game_Date ASC";
        
        $result = $this->db->query($query);
        
        if (!$result) {
            throw new Exception("Error fetching WNBA games: " . $this->db->error);
        }
        
        return $result->fetch_all(MYSQLI_ASSOC);
    }
    
    private function processGame($game, $current_time) {
        try {
            // Parse game time
            $game_time_utc = new DateTime($game['Game_Date']);
            $game_time_eastern = clone $game_time_utc;
            $game_time_eastern->setTimezone($this->timezone);
            
            // Calculate time differences
            $time_diff_seconds = $current_time->getTimestamp() - $game_time_eastern->getTimestamp();
            $time_diff_minutes = round($time_diff_seconds / 60);
            
            // Determine game status flags
            $about_to_start = ($time_diff_minutes >= -self::START_WARNING_MINUTES && $time_diff_minutes <= 0);
            $in_progress = ($time_diff_minutes > 0 && $time_diff_minutes <= (self::GAME_DURATION_HOURS * 60));
            $in_session = ($game['Game_Played'] === 'IN_SESSION' || $in_progress);
            
            // Update game time in database if needed
            $this->updateGameTime($game['Game'], $game_time_eastern);
            
            return [
                'game_id' => $game['Game'],
                'team1' => $game['Team_1_Name'],
                'team2' => $game['Team_2_Name'],
                'status' => $game['Game_Played'],
                'utc_time' => $game['Game_Date'],
                'eastern_time' => $game_time_eastern->format('m/d/Y | g:i A T'),
                'time_diff_readable' => $this->formatTimeDifference($time_diff_seconds),
                'time_diff_minutes' => $time_diff_minutes,
                'about_to_start' => $about_to_start,
                'in_progress' => $in_progress,
                'in_session' => $in_session,
                'needs_update' => ($about_to_start || $in_session)
            ];
            
        } catch (Exception $e) {
            error_log("Error processing game #{$game['Game']}: " . $e->getMessage());
            return null;
        }
    }
    
    private function formatTimeDifference($seconds) {
        $abs_seconds = abs($seconds);
        $hours = floor($abs_seconds / 3600);
        $minutes = floor(($abs_seconds % 3600) / 60);
        
        $formatted = $hours . "h " . $minutes . "m";
        
        return $seconds > 0 ? $formatted . " ago" : "in " . $formatted;
    }
    
    private function updateGameTime($game_id, $game_time_eastern) {
        $formatted_time = $game_time_eastern->format('g:i a');
        $stmt = $this->db->prepare("UPDATE sports_list SET Game_Time = ? WHERE Game = ? AND Conference = 'WNBA'");
        
        if ($stmt) {
            $stmt->bind_param('si', $formatted_time, $game_id);
            $stmt->execute();
            $stmt->close();
        }
    }
    
    private function callUpdateAPI() {
        $start_time = microtime(true);
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => self::API_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => self::CURL_TIMEOUT,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_USERAGENT => 'WNBA-Processor/1.0',
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
        ]);
        
        $this->api_result = curl_exec($ch);
        
        if (curl_errno($ch)) {
            $this->api_error = "cURL Error: " . curl_error($ch);
            error_log($this->api_error);
        } else {
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            if ($http_code !== 200) {
                $this->api_error = "HTTP Error: " . $http_code;
                error_log($this->api_error);
            } else {
                $this->api_called = true;
            }
        }
        
        $this->api_execution_time = round(microtime(true) - $start_time, 2);
        curl_close($ch);
    }
    
    public function getProcessedGames() {
        return $this->processed_games;
    }
    
    public function getAPIStatus() {
        return [
            'called' => $this->api_called,
            'result' => $this->api_result,
            'error' => $this->api_error,
            'execution_time' => $this->api_execution_time
        ];
    }
    
    public function getStatistics() {
        $about_to_start = array_filter($this->processed_games, fn($g) => $g['about_to_start']);
        $in_progress = array_filter($this->processed_games, fn($g) => $g['in_progress']);
        $in_session = array_filter($this->processed_games, fn($g) => $g['in_session']);
        $needs_update = array_filter($this->processed_games, fn($g) => $g['needs_update']);
        
        return [
            'total_games' => count($this->processed_games),
            'about_to_start' => count($about_to_start),
            'in_progress' => count($in_progress),
            'in_session' => count($in_session),
            'needs_update' => count($needs_update)
        ];
    }
    
    public function __destruct() {
        if ($this->db) {
            $this->db->close();
        }
    }
}

// Helper function for status CSS classes
function getStatusClass($game) {
    if ($game['status'] === 'IN_SESSION' || $game['in_session']) {
        return 'status-in-session';
    } elseif ($game['status'] === 'GAME_DONE') {
        return 'status-game-done';
    } elseif ($game['status'] === 'CANCELED') {
        return 'status-canceled';
    } elseif ($game['status'] === 'POSTPONED') {
        return 'status-postponed';
    } elseif ($game['about_to_start']) {
        return 'status-about-to-start';
    } else {
        return 'status-not-started';
    }
}

// Initialize and run processor
try {
    // Load configuration
    require_once '../../config.php';
    
    // Error reporting for development
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    
    // Database configuration
    $db_config = [
        'host' => $host_gscroll,
        'username' => $username_gscroll,
        'password' => $password_gscroll,
        'database' => $database_gscroll
    ];
    
    // Create processor and run
    $processor = new WNBAProcessor($db_config);
    $processor->processGames();
    
    // Get results
    $games = $processor->getProcessedGames();
    $api_status = $processor->getAPIStatus();
    $stats = $processor->getStatistics();
    $current_time = (new DateTime('now'))->format('Y-m-d H:i:s');
    
} catch (Exception $e) {
    die("Fatal error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WNBA Time Processor - Optimized</title>
    <style>
        * { box-sizing: border-box; }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            margin: 0;
            padding: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        
        .container { max-width: 1200px; margin: 0 auto; }
        
        h1 {
            text-align: center;
            color: #2c3e50;
            margin-bottom: 10px;
            font-size: 2.5em;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
        }
        
        .timestamp {
            text-align: center;
            color: #7f8c8d;
            margin-bottom: 30px;
            font-size: 1.1em;
        }
        
        .card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.1);
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .stat-item {
            text-align: center;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            font-weight: bold;
        }
        
        .stat-number { font-size: 2em; display: block; }
        .stat-label { font-size: 0.9em; opacity: 0.9; }
        
        .api-status {
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .api-success {
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
            color: white;
        }
        
        .api-no-update {
            background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
            color: white;
        }
        
        .api-error {
            background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
            color: white;
        }
        
        .games-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .game-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 16px rgba(0,0,0,0.1);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        
        .game-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(0,0,0,0.15);
        }
        
        .game-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #ecf0f1;
        }
        
        .game-time {
            font-weight: 600;
            color: #2c3e50;
            font-size: 0.95em;
        }
        
        .teams {
            font-size: 1.2em;
            font-weight: bold;
            margin-bottom: 15px;
            text-align: center;
            color: #2c3e50;
        }
        
        .vs { color: #7f8c8d; font-weight: normal; margin: 0 10px; }
        
        .game-meta {
            display: flex;
            justify-content: space-between;
            font-size: 0.85em;
            color: #7f8c8d;
        }
        
        .status {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: bold;
            font-size: 0.8em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-in-session { background: #f39c12; color: white; }
        .status-not-started { background: #3498db; color: white; }
        .status-game-done { background: #95a5a6; color: white; }
        .status-about-to-start { background: #27ae60; color: white; }
        .status-canceled { background: #e74c3c; color: white; }
        .status-postponed { background: #e67e22; color: white; }
        
        .flag-true { color: #27ae60; font-weight: bold; }
        .flag-false { color: #e74c3c; }
        
        .table-container {
            overflow-x: auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 16px rgba(0,0,0,0.1);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.9em;
        }
        
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 10px;
            text-align: left;
            font-weight: 600;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        td {
            padding: 12px 10px;
            border-bottom: 1px solid #ecf0f1;
        }
        
        tr:nth-child(even) { background: #f8f9fa; }
        tr:hover { background: #e3f2fd; }
        
        .code-block {
            background: #2c3e50;
            color: #ecf0f1;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 0.85em;
            max-height: 200px;
            overflow-y: auto;
            margin-top: 10px;
        }
        
        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .games-grid { grid-template-columns: 1fr; }
            h1 { font-size: 2em; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🏀 WNBA Time Processor</h1>
        <p class="timestamp">Current Eastern Time: <?= $current_time ?></p>
        
        <div class="card">
            <h2>📊 Summary Statistics</h2>
            <div class="stats-grid">
                <div class="stat-item">
                    <span class="stat-number"><?= $stats['total_games'] ?></span>
                    <span class="stat-label">Total Games</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?= $stats['about_to_start'] ?></span>
                    <span class="stat-label">Starting Soon</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?= $stats['in_progress'] ?></span>
                    <span class="stat-label">In Progress</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?= $stats['in_session'] ?></span>
                    <span class="stat-label">In Session</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?= $stats['needs_update'] ?></span>
                    <span class="stat-label">Need Updates</span>
                </div>
            </div>
            
            <div class="api-status <?= $api_status['called'] ? 'api-success' : ($stats['needs_update'] > 0 ? 'api-error' : 'api-no-update') ?>">
                <h3>🔄 API Update Status</h3>
                <?php if ($api_status['called']): ?>
                    <p><strong>✅ API Successfully Called!</strong></p>
                    <p>URL: <?= $processor::API_URL ?></p>
                    <p>Execution time: <?= $api_status['execution_time'] ?> seconds</p>
                    <?php if ($api_status['error']): ?>
                        <p><strong>⚠️ Warning:</strong> <?= htmlspecialchars($api_status['error']) ?></p>
                    <?php endif; ?>
                    <?php if (strlen($api_status['result']) > 0): ?>
                        <p>Response (<?= strlen($api_status['result']) ?> characters):</p>
                        <div class="code-block"><?= htmlspecialchars(substr($api_status['result'], 0, 500)) ?><?= strlen($api_status['result']) > 500 ? '...' : '' ?></div>
                    <?php endif; ?>
                <?php elseif ($stats['needs_update'] > 0): ?>
                    <p><strong>❌ API Call Failed!</strong></p>
                    <?php if ($api_status['error']): ?>
                        <p><strong>Error:</strong> <?= htmlspecialchars($api_status['error']) ?></p>
                    <?php endif; ?>
                <?php else: ?>
                    <p><strong>ℹ️ No API Update Needed</strong></p>
                    <p>No games are currently in session or about to start.</p>
                <?php endif; ?>
            </div>
        </div>
        
        <?php 
        $active_games = array_filter($games, fn($g) => $g['needs_update']);
        if (count($active_games) > 0): 
        ?>
        <div class="card">
            <h2>🔥 Active Games (Triggering Updates)</h2>
            <div class="games-grid">
                <?php foreach ($active_games as $game): ?>
                    <div class="game-card">
                        <div class="game-header">
                            <div class="game-time"><?= $game['eastern_time'] ?></div>
                            <span class="status <?= getStatusClass($game) ?>">
                                <?= $game['about_to_start'] ? 'Starting Soon' : ($game['in_session'] ? 'In Session' : $game['status']) ?>
                            </span>
                        </div>
                        <div class="teams">
                            <?= htmlspecialchars($game['team1']) ?>
                            <span class="vs">vs</span>
                            <?= htmlspecialchars($game['team2']) ?>
                        </div>
                        <div class="game-meta">
                            <span>Game #<?= $game['game_id'] ?></span>
                            <span><?= $game['time_diff_readable'] ?></span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <h2>📋 All Games</h2>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Game #</th>
                            <th>Teams</th>
                            <th>Status</th>
                            <th>Eastern Time</th>
                            <th>Time Difference</th>
                            <th>Triggers Update</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($games as $game): ?>
                            <tr>
                                <td><?= $game['game_id'] ?></td>
                                <td><?= htmlspecialchars($game['team1']) ?> vs. <?= htmlspecialchars($game['team2']) ?></td>
                                <td>
                                    <span class="status <?= getStatusClass($game) ?>">
                                        <?= $game['status'] ?>
                                    </span>
                                </td>
                                <td><?= $game['eastern_time'] ?></td>
                                <td><?= $game['time_diff_readable'] ?></td>
                                <td class="<?= $game['needs_update'] ? 'flag-true' : 'flag-false' ?>">
                                    <?= $game['needs_update'] ? '✅ Yes' : '❌ No' ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</body>
</html>