<?php
//************************************************//
// CFB Score Updater - Complete Version with Debug + Broadcast
// Simple API logic + Beautiful HTML + Scoring Pause + DEBUG OUTPUT
// Modified to handle PAT (scoreVal = 1) without overwriting TD info
// Now includes lastPlayer tracking fields AND broadcast information
//************************************************//

set_time_limit(300);
ini_set('memory_limit', '256M');
error_reporting(E_ALL);
ini_set('display_errors', 1);

// CFB Conference mappings
$conferences = [
    'TOP-25' => null,
    'SEC' => 8,
    'BIG-10' => 5,
    'PAC-12' => 9,
    'ACC' => 1,
    'BIG-12' => 4,
    'MW' => 17,
    'MAC' => 15,
    'CUSA' => 12,
    'IVY' => 22,
    'Sun-Belt' => 37,
    'AAC' => 151
];

require_once '../../config.php';
session_start();
ob_start();

// Beautiful HTML header
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CFB Score Updater - Live Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: #2c3e50;
            font-size: 32px;
            margin-bottom: 15px;
        }
        
        .time-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }
        
        .time-box {
            background: #f8f9fa;
            padding: 10px 15px;
            border-radius: 8px;
            border-left: 4px solid #667eea;
        }
        
        .debug-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            margin: 15px;
            border-radius: 8px;
            font-family: monospace;
            font-size: 12px;
        }
        
        .debug-box h4 {
            color: #856404;
            margin-bottom: 10px;
        }
        
        .performance-metrics {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        
        .metrics-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }
        
        .metric-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            border-radius: 10px;
            text-align: center;
        }
        
        .metric-card .value {
            font-size: 28px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .metric-card .label {
            font-size: 12px;
            text-transform: uppercase;
            opacity: 0.9;
        }
        
        .conference-card {
            background: white;
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        
        .conference-header {
            padding: 20px;
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            color: white;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .conference-header h2 {
            font-size: 24px;
        }
        
        .status-badge {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: bold;
        }
        
        .badge-active {
            background: #00c853;
            color: white;
        }
        
        .badge-upcoming {
            background: #2196f3;
            color: white;
        }
        
        .badge-skipped {
            background: #9e9e9e;
            color: white;
        }
        
        .game-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .game-table thead {
            background: #f5f5f5;
        }
        
        .game-table th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #2c3e50;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid #e0e0e0;
        }
        
        .game-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 14px;
        }
        
        .game-table tr:hover {
            background: #f9f9f9;
        }
        
        .game-row-active {
            background: #e8f5e9 !important;
            border-left: 4px solid #4caf50;
        }
        
        .game-row-scoring {
            background: #fff3e0 !important;
            border-left: 4px solid #ff9800;
            animation: pulse 2s infinite;
        }
        
        .game-row-paused {
            background: #fce4ec !important;
            border-left: 4px solid #e91e63;
        }
        
        @keyframes pulse {
            0% { background: #fff3e0; }
            50% { background: #ffe0b2; }
            100% { background: #fff3e0; }
        }
        
        .status-pill {
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: bold;
            display: inline-block;
        }
        
        .pill-active {
            background: #4caf50;
            color: white;
        }
        
        .pill-halftime {
            background: #ff9800;
            color: white;
        }
        
        .pill-final {
            background: #9e9e9e;
            color: white;
        }
        
        .pill-upcoming {
            background: #2196f3;
            color: white;
        }
        
        .score {
            font-size: 18px;
            font-weight: bold;
            color: #2c3e50;
        }
        
        .score-winning {
            color: #4caf50;
        }
        
        .score-losing {
            color: #f44336;
        }
        
        .info-box {
            background: #e3f2fd;
            border-left: 4px solid #2196f3;
            padding: 15px;
            margin: 15px;
            border-radius: 4px;
        }
        
        .summary {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-top: 30px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        
        .summary h2 {
            color: #2c3e50;
            margin-bottom: 20px;
            font-size: 28px;
        }
        
        .broadcast-cell {
            background: #e3f2fd;
            font-weight: bold;
        }
    </style>
</head>
<body>
<div class="container">';

// Set timezone
date_default_timezone_set('America/New_York');
$start_time = microtime(true);
$current_time_et = date('m/d/Y | g:i:s A');
$current_datetime = date('Y-m-d H:i:s');
$current_time_utc = gmdate('Y-m-d\TH:i:s\Z');

// Header
echo '<div class="header">
    <h1>�9�8 CFB Score Updater - Live Dashboard</h1>
    <div class="time-info">
        <div class="time-box">
            <label>Eastern Time</label>
            <div class="value">' . $current_time_et . '</div>
        </div>
        <div class="time-box">
            <label>UTC Time</label>
            <div class="value">' . $current_time_utc . '</div>
        </div>
        <div class="time-box">
            <label>Script Started</label>
            <div class="value">' . date('g:i:s A') . '</div>
        </div>
    </div>
</div>';

// Initialize counters
$total_api_calls = 0;
$total_games_updated = 0;
$total_games_checked = 0;
$total_games_paused = 0;
$conferences_with_active_games = [];
$conferences_skipped = [];
$games_with_pauses = [];

// Performance metrics placeholder
echo '<div class="performance-metrics">
    <div class="metrics-grid">
        <div class="metric-card">
            <div class="value" id="api-calls">0</div>
            <div class="label">API Calls</div>
        </div>
        <div class="metric-card">
            <div class="value" id="games-updated">0</div>
            <div class="label">Games Updated</div>
        </div>
        <div class="metric-card">
            <div class="value" id="active-games">0</div>
            <div class="label">Active Conferences</div>
        </div>
        <div class="metric-card">
            <div class="value" id="memory-usage">0 MB</div>
            <div class="label">Memory Used</div>
        </div>
    </div>
</div>';

// Database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die('<div class="error-box">Database connection failed: ' . mysqli_connect_error() . '</div>');
}

// API UPDATE SUMMARY - Check each conference individually
echo '<div class="conference-card" style="background: #e3f2fd; border-left: 5px solid #2196f3;">
    <div class="conference-header" style="background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);">
        <h2>�9�1 Conference Update Schedule</h2>
    </div>
    <div style="padding: 20px;">';

echo '<table class="game-table">
<thead>
    <tr>
        <th>Conference</th>
        <th>Current Status</th>
        <th>Next Update In</th>
        <th>Update Time (ET)</th>
        <th>Game Triggering Update</th>
        <th>Game Time</th>
    </tr>
</thead>
<tbody>';

$current_timestamp = time();
$current_time_utc = gmdate('Y-m-d\TH:i:s\Z');
$any_games_live = false;

foreach ($conferences as $conf_name => $conf_id) {
    echo '<tr>';
    echo '<td><strong>' . $conf_name . '</strong></td>';
    
    // Check for live games in THIS conference only
    $live_query = "SELECT COUNT(*) as live_count, 
                   GROUP_CONCAT(CONCAT(Team_2_Abbrev, ' @ ', Team_1_Abbrev) SEPARATOR ', ') as live_games
                   FROM sports_list 
                   WHERE Conference = '$conf_name' 
                   AND Game_Played IN ('IN_SESSION', 'HALF_TIME')";
    $live_result = mysqli_query($con, $live_query);
    $live_data = mysqli_fetch_assoc($live_result);
    $live_count = $live_data['live_count'] ?? 0;
    $live_games = $live_data['live_games'] ?? '';
    
    if ($live_count > 0) {
        $any_games_live = true;
        echo '<td><span class="status-pill pill-active">UPDATING</span></td>';
        echo '<td style="color: #4caf50; font-weight: bold;">NOW</td>';
        echo '<td style="color: #4caf50;">Currently Active</td>';
        echo '<td>' . $live_games . '</td>';
        echo '<td>LIVE</td>';
    } else {
        // Find next game for THIS conference that will trigger update (10 min before)
        $next_game_query = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Time
                           FROM sports_list 
                           WHERE Conference = '$conf_name' 
                           AND Game_Date > '$current_time_utc'
                           AND Game_Played = 'NOT_STARTED'
                           ORDER BY Game_Date ASC
                           LIMIT 1";
        
        $next_result = mysqli_query($con, $next_game_query);
        
        if ($next_result && mysqli_num_rows($next_result) > 0) {
            $next_game = mysqli_fetch_assoc($next_result);
            $game_timestamp = strtotime($next_game['Game_Date']);
            $update_timestamp = $game_timestamp - 600; // 10 minutes before
            $seconds_until_update = $update_timestamp - $current_timestamp;
            
            echo '<td><span class="status-pill pill-upcoming">SCHEDULED</span></td>';
            
            if ($seconds_until_update > 0) {
                $hours = floor($seconds_until_update / 3600);
                $minutes = floor(($seconds_until_update % 3600) / 60);
                
                if ($hours > 24) {
                    $days = floor($hours / 24);
                    $hours = $hours % 24;
                    echo '<td>' . $days . 'd ' . $hours . 'h ' . $minutes . 'm</td>';
                } elseif ($hours > 0) {
                    echo '<td style="color: #2196f3; font-weight: bold;">' . $hours . 'h ' . $minutes . 'm</td>';
                } else {
                    echo '<td style="color: #ff9800; font-weight: bold;">' . $minutes . ' minutes</td>';
                }
                
                $update_time_et = date('m/d g:i A', $update_timestamp);
                echo '<td>' . $update_time_et . '</td>';
                echo '<td>' . $next_game['Team_2_Abbrev'] . ' @ ' . $next_game['Team_1_Abbrev'] . '</td>';
                echo '<td>' . $next_game['Game_Time'] . '</td>';
            } else {
                echo '<td style="color: #f44336;">OVERDUE</td>';
                echo '<td>Should be updating</td>';
                echo '<td>' . $next_game['Team_2_Abbrev'] . ' @ ' . $next_game['Team_1_Abbrev'] . '</td>';
                echo '<td>' . $next_game['Game_Time'] . '</td>';
            }
        } else {
            echo '<td><span class="status-pill pill-final">NO GAMES</span></td>';
            echo '<td style="color: #9e9e9e;">-</td>';
            echo '<td style="color: #9e9e9e;">-</td>';
            echo '<td style="color: #9e9e9e;">No upcoming games</td>';
            echo '<td style="color: #9e9e9e;">-</td>';
        }
    }
    
    echo '</tr>';
}

echo '</tbody></table>';

// Summary statistics
echo '<div style="margin-top: 20px; padding: 15px; background: white; border-radius: 8px;">';

if ($any_games_live) {
    echo '<div style="padding: 10px; background: #c8e6c9; border-radius: 5px; margin-bottom: 10px;">';
    echo '<strong style="color: #2e7d32;">�0�8 SYSTEM ACTIVE - Games are currently live</strong>';
    echo '</div>';
} else {
    echo '<div style="padding: 10px; background: #ffccbc; border-radius: 5px; margin-bottom: 10px;">';
    echo '<strong style="color: #bf360c;">�9�2 SYSTEM IDLE - No games currently live</strong>';
    echo '</div>';
}

// Find the very next update across all conferences
$next_update_query = "SELECT Conference, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Time
                      FROM sports_list 
                      WHERE Game_Date > '$current_time_utc'
                      AND Game_Played = 'NOT_STARTED'
                      AND Conference IN ('" . implode("','", array_keys($conferences)) . "')
                      ORDER BY Game_Date ASC
                      LIMIT 1";

$next_update_result = mysqli_query($con, $next_update_query);
if ($next_update_result && mysqli_num_rows($next_update_result) > 0) {
    $next_update = mysqli_fetch_assoc($next_update_result);
    $next_timestamp = strtotime($next_update['Game_Date']);
    $update_starts = $next_timestamp - 600;
    $seconds_until = $update_starts - $current_timestamp;
    
    if ($seconds_until > 0) {
        $hours = floor($seconds_until / 3600);
        $minutes = floor(($seconds_until % 3600) / 60);
        
        echo '<div style="padding: 10px; background: #e3f2fd; border-radius: 5px;">';
        echo '<strong>Next Update Starts:</strong><br>';
        echo 'In <span style="font-size: 1.2em; color: #1976d2; font-weight: bold;">';
        if ($hours > 0) {
            echo $hours . ' hours, ' . $minutes . ' minutes';
        } else {
            echo $minutes . ' minutes';
        }
        echo '</span><br>';
        echo 'At ' . date('g:i A', $update_starts) . ' Eastern (10 min before ' . 
             $next_update['Team_2_Abbrev'] . ' @ ' . $next_update['Team_1_Abbrev'] . ' in ' . 
             $next_update['Conference'] . ')';
        echo '</div>';
    }
}

echo '</div>';
echo '</div></div>';

// Check/add lastUpdatedTime column
$check_column = "SHOW COLUMNS FROM `sports_list` LIKE 'lastUpdatedTime'";
$column_result = mysqli_query($con, $check_column);
if (mysqli_num_rows($column_result) == 0) {
    $add_column = "ALTER TABLE `sports_list` ADD COLUMN `lastUpdatedTime` DATETIME DEFAULT NULL";
    mysqli_query($con, $add_column);
}

// Check/add broadcast column
$check_broadcast = "SHOW COLUMNS FROM `sports_list` LIKE 'broadcast'";
$broadcast_result = mysqli_query($con, $check_broadcast);
if (mysqli_num_rows($broadcast_result) == 0) {
    $add_broadcast = "ALTER TABLE `sports_list` ADD COLUMN `broadcast` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_broadcast);
}

// Helper function for pause logic - Modified to ignore PAT
function shouldPauseUpdate($game_num, $conference, $con, $scoreVal) {
    // Never pause for PAT (scoreVal = 1)
    if ($scoreVal == 1) {
        return array('pause' => false, 'time_remaining' => 0);
    }
    
    $check_query = "SELECT scoreValue, lastUpdatedTime FROM sports_list 
                    WHERE Game = $game_num AND Conference = '$conference'";
    $result = mysqli_query($con, $check_query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $currentScoreValue = $row['scoreValue'] ?? '';
        $lastUpdatedTime = $row['lastUpdatedTime'] ?? null;
        
        if (!empty($currentScoreValue) && 
            $currentScoreValue != 'NO-SCORE' && 
            $currentScoreValue != 'PAT' &&
            $currentScoreValue != ' ' &&
            !empty($lastUpdatedTime)) {
            
            $lastUpdateTimestamp = strtotime($lastUpdatedTime);
            $currentTimestamp = time();
            $timeDifference = $currentTimestamp - $lastUpdateTimestamp;
            
            if ($timeDifference < 90) {
                return array(
                    'pause' => true,
                    'time_remaining' => 90 - $timeDifference
                );
            }
        }
    }
    
    return array('pause' => false, 'time_remaining' => 0);
}

function getGameStatus($status_data) {
    $type_id = $status_data['type']['id'] ?? '';
    $completed = $status_data['type']['completed'] ?? false;
    $state = $status_data['type']['state'] ?? '';
    
    // Handle overtime
    if ($type_id == '3' || $completed === true || $state == 'post') {
        return 'GAME_DONE';
    } elseif ($type_id == '2' || $state == 'in') {
        $period = $status_data['period'] ?? 0;
        $clock = $status_data['displayClock'] ?? '';
        if ($period == 2 && $clock == '0:00') {
            return 'HALF_TIME';
        }
        // Overtime is still IN_SESSION
        return 'IN_SESSION';
    }
    
    return 'NOT_STARTED';
}

function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

function parseTeamColor($team) {
    $color_hex = $team['color'] ?? '000000';
    if ($color_hex == '000000') {
        $color_hex = $team['alternateColor'] ?? '000000';
    }
    $color_hex = clean($color_hex);
    
    $split = str_split($color_hex, 2);
    $r = hexdec($split[0] ?? '00');
    $g = hexdec($split[1] ?? '00');
    $b = hexdec($split[2] ?? '00');
    
    $r = addZeros($r);
    $g = addZeros($g);
    $b = addZeros($b);
    
    if ($r < 30 && $g < 30 && $b < 30) {
        $r = 255;
        $g = 255;
        $b = 255;
    }
    
    return [
        'rgb' => sprintf('"%s,%s,%s"', $r, $g, $b),
        'r' => $r,
        'g' => $g,
        'b' => $b
    ];
}

// MAIN LOOP - Process each conference
foreach ($conferences as $conference_name => $conference_id) {
    
    echo '<div class="conference-card">';
    echo '<div class="conference-header">';
    echo '<h2>' . $conference_name . '</h2>';
    echo '<div class="conference-status">';
    
    // SIMPLE CHECK: Does this conference need updating?
    $current_timestamp = time();
    $five_hours_ago = gmdate('Y-m-d\TH:i:s\Z', $current_timestamp - (5 * 3600));
    $ten_minutes_from_now = gmdate('Y-m-d\TH:i:s\Z', $current_timestamp + 600);
    
    // DEBUG OUTPUT
    echo '<div class="debug-box">';
    echo '<h4>�9�3 DEBUG: ' . $conference_name . '</h4>';
    echo '<strong>Time Windows:</strong><br>';
    echo 'Current UTC: ' . gmdate('Y-m-d\TH:i:s\Z') . '<br>';
    echo 'Current ET: ' . date('m/d/Y g:i:s A') . '<br>';
    echo 'Five Hours Ago: ' . $five_hours_ago . '<br>';
    echo 'Ten Minutes From Now: ' . $ten_minutes_from_now . '<br>';
    echo '<hr style="margin: 10px 0;">';
    
    $check_query = "
        SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played, 
               Team_1_Score, Team_2_Score, Clock, Period
        FROM sports_list 
        WHERE Conference = '$conference_name'
        AND (
            Game_Played IN ('IN_SESSION', 'HALF_TIME')
            OR (Game_Played = 'NOT_STARTED' AND 
                Game_Date >= '$five_hours_ago' AND 
                Game_Date <= '$ten_minutes_from_now')
        )
    ";
    
    // Show the query
    echo '<strong>Check Query:</strong><br>';
    echo '<pre style="background: #f5f5f5; padding: 10px; overflow-x: auto;">' . htmlspecialchars($check_query) . '</pre>';
    
    $check_result = mysqli_query($con, $check_query);
    $games_needing_update = [];
    
    if ($check_result && mysqli_num_rows($check_result) > 0) {
        echo '<strong>Games Found Needing Update:</strong><br>';
        while ($row = mysqli_fetch_assoc($check_result)) {
            $games_needing_update[] = $row;
            
            $game_timestamp = strtotime($row['Game_Date']);
            $minutes_diff = ($game_timestamp - $current_timestamp) / 60;
            
            echo '�6�1 ' . $row['Team_2_Abbrev'] . ' @ ' . $row['Team_1_Abbrev'] . 
                 ' | Date: ' . $row['Game_Date'] . 
                 ' | Status: ' . $row['Game_Played'] . 
                 ' | Minutes from now: ' . round($minutes_diff, 1) . '<br>';
        }
    } else {
        echo '<strong style="color: #d32f2f;">No games found needing update!</strong><br>';
    }
    
    // Additional debug: Show ALL games for this conference
    echo '<hr style="margin: 10px 0;">';
    echo '<strong>All Games in Database for ' . $conference_name . ':</strong><br>';
    
    $all_debug_query = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played, Game_Time
                        FROM sports_list 
                        WHERE Conference = '$conference_name'
                        ORDER BY Game_Date ASC";
    $all_debug_result = mysqli_query($con, $all_debug_query);
    
    while ($debug_row = mysqli_fetch_assoc($all_debug_result)) {
        $game_timestamp = strtotime($debug_row['Game_Date']);
        $minutes_until = ($game_timestamp - $current_timestamp) / 60;
        
        $should_update = false;
        $reason = '';
        
        if ($debug_row['Game_Played'] == 'IN_SESSION' || $debug_row['Game_Played'] == 'HALF_TIME') {
            $should_update = true;
            $reason = 'LIVE/HALFTIME';
        } elseif ($debug_row['Game_Played'] == 'NOT_STARTED') {
            if ($minutes_until <= 10 && $minutes_until >= -300) {
                $should_update = true;
                $reason = 'WITHIN WINDOW';
            } elseif ($minutes_until > 10) {
                $reason = 'TOO EARLY (' . round($minutes_until - 10) . ' min until window)';
            } else {
                $reason = 'TOO LATE (started ' . round(abs($minutes_until)) . ' min ago)';
            }
        } else {
            $reason = 'GAME DONE';
        }
        
        $color = $should_update ? 'green' : 'gray';
        echo '<span style="color: ' . $color . ';">Game ' . $debug_row['Game'] . ': ' . 
             $debug_row['Team_2_Abbrev'] . ' @ ' . $debug_row['Team_1_Abbrev'] . 
             ' (' . $debug_row['Game_Time'] . ' ET) - ' . 
             'Status: ' . $debug_row['Game_Played'] . ' - ' . 
             'Minutes: ' . round($minutes_until, 1) . ' - ' . 
             ($should_update ? '�7�7 SHOULD UPDATE' : '�7�1 ' . $reason) . '</span><br>';
    }
    
    echo '</div>';
    
    echo 'Games needing update: ' . count($games_needing_update) . '<br>';
    
    if (count($games_needing_update) == 0) {
        // NO UPDATE NEEDED
        echo '<span class="status-badge badge-skipped">SKIPPED</span>';
        echo '</div></div>';
        echo '<div class="info-box">No active games or games starting within 10 minutes.</div>';
        
        // Still show database games for reference
        $all_games_query = "SELECT * FROM sports_list WHERE Conference = '$conference_name' ORDER BY Game ASC";
        $all_games_result = mysqli_query($con, $all_games_query);
        
        if ($all_games_result && mysqli_num_rows($all_games_result) > 0) {
            echo '<table class="game-table">
            <thead>
                <tr>
                    <th>Game</th>
                    <th>Matchup</th>
                    <th>Status</th>
                    <th>Score</th>
                    <th>Clock</th>
                    <th>Game Time (ET)</th>
                    <th>Broadcast</th>
                </tr>
            </thead>
            <tbody>';
            
            while ($row = mysqli_fetch_assoc($all_games_result)) {
                echo '<tr>';
                echo '<td>' . $row['Game'] . '</td>';
                echo '<td><strong>' . $row['Team_2_Abbrev'] . '</strong> @ <strong>' . $row['Team_1_Abbrev'] . '</strong></td>';
                
                echo '<td>';
                if ($row['Game_Played'] == 'IN_SESSION') {
                    echo '<span class="status-pill pill-active">LIVE</span>';
                } elseif ($row['Game_Played'] == 'HALF_TIME') {
                    echo '<span class="status-pill pill-halftime">HALFTIME</span>';
                } elseif ($row['Game_Played'] == 'GAME_DONE') {
                    echo '<span class="status-pill pill-final">FINAL</span>';
                } else {
                    $game_et = date('g:i a', strtotime($row['Game_Date']));
                    echo '<span class="status-pill pill-upcoming">' . $game_et . '</span>';
                }
                echo '</td>';
                
                echo '<td>';
                if ($row['Game_Played'] != 'NOT_STARTED') {
                    echo '<span class="score">' . $row['Team_1_Score'] . ' - ' . $row['Team_2_Score'] . '</span>';
                } else {
                    echo '<span class="score">-</span>';
                }
                echo '</td>';
                
                echo '<td>' . ($row['Clock'] ?: '-') . '</td>';
                echo '<td>' . $row['Game_Time'] . '</td>';
                echo '<td class="broadcast-cell">' . ($row['broadcast'] ?? '-') . '</td>';
                echo '</tr>';
            }
            
            echo '</tbody></table>';
        }
        
        echo '</div>';
        $conferences_skipped[] = $conference_name;
        continue;
    }
    
    // THIS CONFERENCE NEEDS UPDATING!
    $active_count = 0;
    $upcoming_count = 0;
    
    foreach ($games_needing_update as $game) {
        if ($game['Game_Played'] == 'IN_SESSION' || $game['Game_Played'] == 'HALF_TIME') {
            $active_count++;
        } else {
            $upcoming_count++;
        }
    }
    
    if ($active_count > 0) {
        echo '<span class="status-badge badge-active">' . $active_count . ' ACTIVE</span>';
    }
    if ($upcoming_count > 0) {
        echo '<span class="status-badge badge-upcoming">' . $upcoming_count . ' STARTING</span>';
    }
    echo '</div></div>';
    
    // Build API URL
    if ($conference_name == "TOP-25") {
        $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard";
    } else {
        $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard?groups=" . $conference_id;
    }
    
    echo '<div class="debug-box">';
    echo '<h4>API Call</h4>';
    echo 'URL: <a href="' . $api_url . '" target="_blank">' . $api_url . '</a><br>';
    echo '</div>';
    
    // Make API call
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $result = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if ($err) {
        echo '<div class="error-box">API Error: ' . htmlspecialchars($err) . '</div></div>';
        continue;
    }
    
    $total_api_calls++;
    
    $data = json_decode($result, true);
    if (!$data || !isset($data['events'])) {
        echo '<div class="error-box">Invalid JSON response</div></div>';
        continue;
    }
    
    $events = $data['events'];
    $games_count = count($events);
    $total_games_checked += $games_count;
    
    echo '<div class="debug-box">';
    echo '<h4>API Response</h4>';
    echo 'Games returned: ' . $games_count . '<br>';
    echo '</div>';
    
    // Game table with all features
    echo '<table class="game-table">
    <thead>
        <tr>
            <th>Game</th>
            <th>Matchup</th>
            <th>Status</th>
            <th>Score</th>
            <th>Clock</th>
            <th>Period</th>
            <th>Broadcast</th>
            <th>Down & Distance</th>
            <th>Ball Position</th>
            <th>Score Type</th>
            <th>Last Player</th>
            <th>Update Status</th>
        </tr>
    </thead>
    <tbody>';
    
    // Process each game from API
    for ($i = 0; $i < $games_count; $i++) {
        $event = $events[$i];
        $competition = $event['competitions'][0] ?? [];
        
        $game_date = $event['date'] ?? '';
        $status = $competition['status'] ?? [];
        $clock = $status['displayClock'] ?? '0:00';
        $period = $status['period'] ?? 0;
        $game_status = getGameStatus($status);
        
        $game_time_et = date("g:i a", strtotime($game_date));
        
        // Extract broadcast information
        $broadcast = '';
        if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
            $broadcast_data = $competition['broadcasts'][0] ?? [];
            if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
                $broadcast = $broadcast_data['names'][0] ?? '';
            }
        }
        
        // Get competitors
        $competitors = $competition['competitors'] ?? [];
        $team1 = null;
        $team2 = null;
        
        foreach ($competitors as $competitor) {
            if ($competitor['homeAway'] == 'home') {
                $team1 = $competitor;
            } else {
                $team2 = $competitor;
            }
        }
        
        if (!$team1 || !$team2) {
            continue;
        }
        
        // Extract all team data
        $team1_data = $team1['team'] ?? [];
        $team2_data = $team2['team'] ?? [];
        
        $team1_name = $team1_data['name'] ?? '';
        $team2_name = $team2_data['name'] ?? '';
        $team1_location = $team1_data['location'] ?? '';
        $team2_location = $team2_data['location'] ?? '';
        $team1_abbrev = $team1_data['abbreviation'] ?? '';
        $team2_abbrev = $team2_data['abbreviation'] ?? '';
        $team1_id = $team1_data['id'] ?? '';
        $team2_id = $team2_data['id'] ?? '';
        
        if ($team1_abbrev == "TA&M") $team1_abbrev = "TAMU";
        if ($team2_abbrev == "TA&M") $team2_abbrev = "TAMU";
        
        // Get records
        $team1_records = $team1['records'] ?? [];
        $team2_records = $team2['records'] ?? [];
        $team1_record = '0-0';
        $team2_record = '0-0';
        
        foreach ($team1_records as $record) {
            if ($record['type'] == 'total') {
                $team1_record = $record['summary'] ?? '0-0';
                break;
            }
        }
        
        foreach ($team2_records as $record) {
            if ($record['type'] == 'total') {
                $team2_record = $record['summary'] ?? '0-0';
                break;
            }
        }
        
        // Get scores
        $team1_score = ($game_status != 'NOT_STARTED') ? ($team1['score'] ?? '0') : '0';
        $team2_score = ($game_status != 'NOT_STARTED') ? ($team2['score'] ?? '0') : '0';
        
        // Get colors
        $team1_colors = parseTeamColor($team1_data);
        $team2_colors = parseTeamColor($team2_data);
        
        // Get odds
        $odds = $competition['odds'][0] ?? [];
        $favorite_team = '';
        $favorite_points = '';
        $overUnder = '';
        
        if (!empty($odds)) {
            $overUnder = $odds['overUnder'] ?? '';
            $details = $odds['details'] ?? '';
            
            if (strpos($details, $team1_abbrev) !== false) {
                $favorite_team = $team1_abbrev;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            } elseif (strpos($details, $team2_abbrev) !== false) {
                $favorite_team = $team2_abbrev;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            }
        }
        
        // Get situation data for live games
        $situation = $competition['situation'] ?? [];
        $who_has_ball = '';
        $short_distance_text = '';
        $pos_text = '';
        $is_red_zone = '';
        $score_value = '';
        $lastPlayerID = '';
        $lastPlayerName = '';
        $lastPlayerPos = '';
        $lastPlayerTeamID = '';
        $lastPlayerJersey = '';
        $driveDescription = '';
        
        $scoreVal = 0;
        
        if ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            $lastPlay = $situation['lastPlay'] ?? [];
            $lastPlayText = $lastPlay['text'] ?? '';
            $scoreVal = $lastPlay['scoreValue'] ?? 0;
            
            // Extract player information from athletesInvolved
            if (isset($lastPlay['athletesInvolved'][0])) {
                $firstPlayer = $lastPlay['athletesInvolved'][0];
                $lastPlayerID = $firstPlayer['id'] ?? '';
                $lastPlayerName = $firstPlayer['fullName'] ?? $firstPlayer['displayName'] ?? '';
                $lastPlayerPos = $firstPlayer['position'] ?? '';
                $lastPlayerJersey = $firstPlayer['jersey'] ?? '';
                $lastPlayerTeamID = $firstPlayer['team']['id'] ?? '';
            }
            
            // Extract drive description if available
            if (isset($lastPlay['drive']['description'])) {
                $driveDescription = strip_tags(trim($lastPlay['drive']['description']));
                if (strlen($driveDescription) > 100) {
                    $driveDescription = substr($driveDescription, 0, 97) . '...';
                }
            }
            
            // Determine score type - MODIFIED to recognize PAT
            if ($scoreVal == 0) {
                $score_value = "NO-SCORE";
            } elseif ($scoreVal == 1) {
                $score_value = "PAT";  // Changed from TOUCHDOWN to PAT
            } elseif ($scoreVal == 2) {
                $score_value = "TWO-PT";
            } elseif ($scoreVal == 3) {
                $score_value = "FIELD-GOAL";
            } elseif ($scoreVal == 6) {
                $score_value = "TOUCHDOWN";
            }
            
            // Get possession
            $possession_team_id = $lastPlay['team']['id'] ?? '';
            if ($possession_team_id == $team1_id) {
                $who_has_ball = $team1_abbrev;
            } elseif ($possession_team_id == $team2_id) {
                $who_has_ball = $team2_abbrev;
            }
            
            // Get down and distance
            $short_distance_text = $situation['shortDownDistanceText'] ?? '';
            $downDistanceText = $situation['downDistanceText'] ?? '';
            
            if ($downDistanceText) {
                if (strpos($downDistanceText, ' at ') !== false) {
                    $pos_text = substr($downDistanceText, strpos($downDistanceText, ' at ') + 4);
                    $pos_text = str_replace(' ', '-', trim($pos_text));
                }
            }
            
            // Red zone
            $is_red_zone = ($situation['isRedZone'] ?? false) ? 'true' : 'false';
            
            // Clear fields if scoring play (but NOT for PAT)
            if ($scoreVal > 1) {  // Changed from > 0 to > 1
                $short_distance_text = ' ';
                $pos_text = ' ';
                $is_red_zone = ' ';
            }
        } else {
            // Not in session - clear all in-game fields
            $lastPlayerID = '';
            $lastPlayerName = '';
            $lastPlayerPos = '';
            $lastPlayerTeamID = '';
            $lastPlayerJersey = '';
            $driveDescription = '';
            $who_has_ball = '';
            $short_distance_text = '';
            $pos_text = '';
            $is_red_zone = '';
            $score_value = '';
        }
        
        // Get player stats
        $leaders = $competition['leaders'] ?? [];
        
        $rusher1 = '';
        $rusher2 = '';
        $QB1 = '';
        $QB2 = '';
        $Rec1 = '';
        $Rec2 = '';
        
        if ($game_status == 'NOT_STARTED') {
            $rusher1 = '_TBD';
            $rusher2 = '!TBD';
            $QB1 = '_TBD';
            $QB2 = '!TBD';
            $Rec1 = '_TBD';
            $Rec2 = '!TBD';
        } else {
            foreach ($leaders as $leader) {
                if ($leader['name'] == 'rushingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $rusher1 = $signal . $fullName . ': ' . $displayValue;
                    $rusher2 = $rusher1;
                }
                
                if ($leader['name'] == 'passingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $QB1 = $signal . $fullName . ': ' . $displayValue;
                    $QB2 = $QB1;
                }
                
                if ($leader['name'] == 'receivingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $Rec1 = $signal . $fullName . ': ' . $displayValue;
                    $Rec2 = $Rec1;
                }
            }
        }
        
        // Clean special characters
        $team1_location = str_replace("'", "", $team1_location);
        $team2_location = str_replace("'", "", $team2_location);
        $team1_name = str_replace("'", "", $team1_name);
        $team2_name = str_replace("'", "", $team2_name);
        $pos_text = str_replace("&", "A", $pos_text);
        
        // CHECK FOR PAUSE (90 second scoring pause) - Modified to pass scoreVal
        $game_num = $i + 1;
        $pauseInfo = shouldPauseUpdate($game_num, $conference_name, $con, $scoreVal);
        $isPaused = $pauseInfo['pause'];
        $timeRemaining = $pauseInfo['time_remaining'];
        
        // Check if new score (but not PAT)
        $isNewScore = false;
        if ($scoreVal > 1 && ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME')) {
            $check_score = "SELECT scoreValue FROM sports_list WHERE Game = $game_num AND Conference = '$conference_name'";
            $score_result = mysqli_query($con, $check_score);
            if ($score_result && mysqli_num_rows($score_result) > 0) {
                $current_db_score = mysqli_fetch_assoc($score_result)['scoreValue'] ?? '';
                if ($current_db_score != $score_value) {
                    $isNewScore = true;
                }
            }
        }
        
        // Determine row class for highlighting
        $row_class = '';
        if ($isNewScore) {
            $row_class = 'game-row-scoring';
        } elseif ($isPaused) {
            $row_class = 'game-row-paused';
        } elseif ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            $row_class = 'game-row-active';
        }
        
        // Display in table
        echo '<tr class="' . $row_class . '">';
        echo '<td>' . $game_num . '</td>';
        echo '<td><strong>' . $team2_abbrev . '</strong> @ <strong>' . $team1_abbrev . '</strong></td>';
        
        // Status
        echo '<td>';
        if ($game_status == 'IN_SESSION') {
            if ($period > 4) {
                echo '<span class="status-pill pill-active">OT' . ($period - 4) . '</span>';
            } else {
                echo '<span class="status-pill pill-active">LIVE</span>';
            }
        } elseif ($game_status == 'HALF_TIME') {
            echo '<span class="status-pill pill-halftime">HALFTIME</span>';
        } elseif ($game_status == 'GAME_DONE') {
            echo '<span class="status-pill pill-final">FINAL</span>';
        } else {
            echo '<span class="status-pill pill-upcoming">' . $game_time_et . '</span>';
        }
        echo '</td>';
        
        // Score
        echo '<td>';
        if ($game_status != 'NOT_STARTED') {
            $team1_score_int = intval($team1_score);
            $team2_score_int = intval($team2_score);
            
            if ($team1_score_int > $team2_score_int) {
                echo '<span class="score score-winning">' . $team1_score . '</span> - ';
                echo '<span class="score score-losing">' . $team2_score . '</span>';
            } elseif ($team2_score_int > $team1_score_int) {
                echo '<span class="score score-losing">' . $team1_score . '</span> - ';
                echo '<span class="score score-winning">' . $team2_score . '</span>';
            } else {
                echo '<span class="score">' . $team1_score . ' - ' . $team2_score . '</span>';
            }
        } else {
            echo '<span class="score">-</span>';
        }
        echo '</td>';
        
        // Clock
        echo '<td>' . ($clock ?: '-') . '</td>';
        
        // Period
        echo '<td>';
        if ($period > 0) {
            if ($period <= 4) {
                echo 'Q' . $period;
            } else {
                echo 'OT' . ($period - 4);
            }
        } else {
            echo '-';
        }
        echo '</td>';
        
        // Broadcast
        echo '<td class="broadcast-cell">' . ($broadcast ?: '-') . '</td>';
        
        // Down & Distance
        echo '<td>';
        if (!empty($short_distance_text) && $short_distance_text != ' ') {
            echo $short_distance_text;
            if ($is_red_zone == 'true') {
                echo ' <span style="color: #f44336; font-weight: bold;">RZ</span>';
            }
        } else {
            echo '-';
        }
        echo '</td>';
        
        // Ball Position
        echo '<td>';
        if (!empty($pos_text) && $pos_text != ' ') {
            echo $pos_text;
        } else {
            echo '-';
        }
        echo '</td>';
        
        // Score Type
        echo '<td>';
        if ($score_value && $score_value != 'NO-SCORE' && $score_value != ' ') {
            if ($score_value == 'TOUCHDOWN') {
                echo '<span style="background: #4caf50; color: white; padding: 2px 8px; border-radius: 4px; font-weight: bold;">TD</span>';
            } elseif ($score_value == 'FIELD-GOAL') {
                echo '<span style="background: #2196f3; color: white; padding: 2px 8px; border-radius: 4px; font-weight: bold;">FG</span>';
            } elseif ($score_value == 'TWO-PT') {
                echo '<span style="background: #ff9800; color: white; padding: 2px 8px; border-radius: 4px; font-weight: bold;">2PT</span>';
            } elseif ($score_value == 'PAT') {
                echo '<span style="background: #9e9e9e; color: white; padding: 2px 8px; border-radius: 4px; font-weight: bold;">PAT</span>';
            } else {
                echo $score_value;
            }
        } else {
            echo '-';
        }
        echo '</td>';
        
        // Last Player
        echo '<td>';
        if (!empty($lastPlayerName)) {
            echo $lastPlayerName;
            if (!empty($lastPlayerJersey)) {
                echo ' #' . $lastPlayerJersey;
            }
        } else {
            echo '-';
        }
        echo '</td>';
        
        // Update Status
        echo '<td>';
        if ($scoreVal == 1) {
            echo '<span style="background: #9e9e9e; color: white; padding: 4px 8px; border-radius: 4px;">PAT - LIMITED UPDATE</span>';
        } elseif ($isPaused || $isNewScore) {
            if ($isNewScore) {
                echo '<span style="background: #e91e63; color: white; padding: 4px 8px; border-radius: 4px; font-weight: bold;">NEW SCORE - PAUSED 90s</span>';
            } else {
                echo '<span style="background: #ff5722; color: white; padding: 4px 8px; border-radius: 4px;">PAUSED ' . $timeRemaining . 's</span>';
            }
        } elseif ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            echo '<span style="background: #4caf50; color: white; padding: 4px 8px; border-radius: 4px;">UPDATING</span>';
        } else {
            echo '<span style="color: #9e9e9e;">-</span>';
        }
        echo '</td>';
        
        echo '</tr>';
        
        // Track paused games (but not PAT)
        if (($isPaused || $isNewScore) && $scoreVal != 1) {
            $games_with_pauses[] = [
                'conference' => $conference_name,
                'game' => $game_num,
                'teams' => "{$team2_abbrev} @ {$team1_abbrev}",
                'score' => "{$team1_score} - {$team2_score}",
                'scoreValue' => $score_value,
                'time_remaining' => $isNewScore ? 90 : $timeRemaining
            ];
            $total_games_paused++;
        }
        
        // Skip old games
        $gameDateTime = strtotime($game_date);
        $fiveDaysAgo = strtotime('-5 days');
        
        if ($gameDateTime < $fiveDaysAgo) {
            continue;
        }
        
        // UPDATE DATABASE - MODIFIED FOR PAT HANDLING AND NEW FIELDS
        if (!$isPaused) {
            $db_datetime = date('Y-m-d H:i:s');
            
            // Special handling for PAT (scoreVal = 1)
            if ($scoreVal == 1) {
                // For PAT: Only update scores, clock, period, game status
                // DO NOT update scoreValue, lastUpdatedTime, or any play information
                $query = "
                    UPDATE sports_list
                    SET
                        Team_1_Score = '" . addslashes($team1_score) . "',
                        Team_2_Score = '" . addslashes($team2_score) . "',
                        Clock = '" . addslashes($clock) . "',
                        Period = '" . addslashes($period) . "',
                        Game_Played = '" . addslashes($game_status) . "'
                    WHERE
                        Game = $game_num
                        AND Conference = '$conference_name'
                ";
            } else {
                // Full update for all other situations - NOW INCLUDING NEW FIELDS AND BROADCAST
                $query = "
                    UPDATE sports_list
                    SET
                        Clock = '" . addslashes($clock) . "',
                        Period = '" . addslashes($period) . "',
                        Game_Played = '" . addslashes($game_status) . "',
                        Game_Time = '" . addslashes($game_time_et) . "',
                        Game_Date = '" . addslashes($game_date) . "',
                        broadcast = '" . addslashes($broadcast) . "',
                        scoreValue = '" . addslashes($score_value) . "',
                        lastUpdatedTime = '" . addslashes($db_datetime) . "',
                        
                        Team_1_Name = '" . addslashes($team1_name) . "',
                        Team_1_Record = '" . addslashes($team1_record) . "',
                        Team_1_Score = '" . addslashes($team1_score) . "',
                        Team_1_Location = '" . addslashes($team1_location) . "',
                        Team_1_Abbrev = '" . addslashes($team1_abbrev) . "',
                        Team_1_Color = '" . addslashes($team1_colors['rgb']) . "',
                        Team_1_Color_R = '" . addslashes($team1_colors['r']) . "',
                        Team_1_Color_G = '" . addslashes($team1_colors['g']) . "',
                        Team_1_Color_B = '" . addslashes($team1_colors['b']) . "',
                        
                        Team_2_Name = '" . addslashes($team2_name) . "',
                        Team_2_Record = '" . addslashes($team2_record) . "',
                        Team_2_Score = '" . addslashes($team2_score) . "',
                        Team_2_Location = '" . addslashes($team2_location) . "',
                        Team_2_Abbrev = '" . addslashes($team2_abbrev) . "',
                        Team_2_Color = '" . addslashes($team2_colors['rgb']) . "',
                        Team_2_Color_R = '" . addslashes($team2_colors['r']) . "',
                        Team_2_Color_G = '" . addslashes($team2_colors['g']) . "',
                        Team_2_Color_B = '" . addslashes($team2_colors['b']) . "',
                        
                        Who_Has_Ball = '" . addslashes($who_has_ball) . "',
                        Distance = '" . addslashes($short_distance_text) . "',
                        Possesion = '" . addslashes($pos_text) . "',
                        Favorite_Team = '" . addslashes($favorite_team) . "',
                        Favorite_Points = '" . addslashes($favorite_points) . "',
                        OverUnder = '" . addslashes($overUnder) . "',
                        Red_Zone = '" . addslashes($is_red_zone) . "',
                        Ru1 = '" . addslashes($rusher1) . "',
                        Ru2 = '" . addslashes($rusher2) . "',
                        QB1 = '" . addslashes($QB1) . "',
                        QB2 = '" . addslashes($QB2) . "',
                        Rec1 = '" . addslashes($Rec1) . "',
                        Rec2 = '" . addslashes($Rec2) . "',
                        
                        lastPlay = '" . addslashes($lastPlayText) . "',
                        lastPlayerID = '" . addslashes($lastPlayerID) . "',
                        lastPlayerName = '" . addslashes($lastPlayerName) . "',
                        lastPlayerPos = '" . addslashes($lastPlayerPos) . "',
                        lastPlayerTeamID = '" . addslashes($lastPlayerTeamID) . "',
                        jersey = '" . addslashes($lastPlayerJersey) . "',
                        driveDesc = '" . addslashes($driveDescription) . "'
                    WHERE
                        Game = $game_num
                        AND Conference = '$conference_name'
                ";
            }
            
            mysqli_query($con, $query);
            $total_games_updated++;
        } else {
            // Partial update during pause
            $query = "
                UPDATE sports_list
                SET
                    Team_1_Score = '" . addslashes($team1_score) . "',
                    Team_2_Score = '" . addslashes($team2_score) . "'
                WHERE
                    Game = $game_num
                    AND Conference = '$conference_name'
            ";
            
            mysqli_query($con, $query);
        }
    }
    
    echo '</tbody></table>';
    
    // Update API timestamp
    $time_updated = date('m/d/Y | g:iA');
    $sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'CFB-$conference_name'";
    mysqli_query($con, $sql);
    
    if ($active_count > 0) {
        $conferences_with_active_games[] = $conference_name;
    }
    
    echo '</div>';
    
    // Update metrics
    echo '<script>
        document.getElementById("api-calls").innerHTML = "' . $total_api_calls . '";
        document.getElementById("games-updated").innerHTML = "' . $total_games_updated . '";
        document.getElementById("active-games").innerHTML = "' . count($conferences_with_active_games) . '";
        document.getElementById("memory-usage").innerHTML = "' . round(memory_get_usage() / 1024 / 1024, 2) . ' MB";
    </script>';
    
    flush();
    ob_flush();
}

// Display paused games if any
if (count($games_with_pauses) > 0) {
    echo '<div class="conference-card" style="border-left: 4px solid #e91e63;">
        <div class="conference-header" style="background: linear-gradient(135deg, #e91e63 0%, #ad1457 100%);">
            <h2>�7�2�1�5 Paused Score Updates</h2>
        </div>
        <table class="game-table">
        <thead>
            <tr>
                <th>Conference</th>
                <th>Game</th>
                <th>Matchup</th>
                <th>Score</th>
                <th>Score Type</th>
                <th>Resume Time</th>
            </tr>
        </thead>
        <tbody>';
    
    foreach ($games_with_pauses as $pausedGame) {
        $resume_time = date('g:i:s A', time() + $pausedGame['time_remaining']);
        echo '<tr>';
        echo '<td>' . $pausedGame['conference'] . '</td>';
        echo '<td>' . $pausedGame['game'] . '</td>';
        echo '<td><strong>' . $pausedGame['teams'] . '</strong></td>';
        echo '<td><span class="score">' . $pausedGame['score'] . '</span></td>';
        echo '<td>';
        if ($pausedGame['scoreValue'] == 'TOUCHDOWN') {
            echo '<span style="background: #4caf50; color: white; padding: 2px 8px; border-radius: 4px;">TD</span>';
        } else {
            echo $pausedGame['scoreValue'];
        }
        echo '</td>';
        echo '<td><strong>' . $resume_time . '</strong> (' . $pausedGame['time_remaining'] . 's)</td>';
        echo '</tr>';
    }
    
    echo '</tbody></table>
    </div>';
}

// Summary
$execution_time = round(microtime(true) - $start_time, 2);

echo '<div class="summary">
    <h2>�9�6 Execution Summary</h2>
    <div class="metrics-grid">
        <div class="metric-card">
            <div class="value">' . $execution_time . 's</div>
            <div class="label">Total Time</div>
        </div>
        <div class="metric-card">
            <div class="value">' . $total_api_calls . '</div>
            <div class="label">API Calls</div>
        </div>
        <div class="metric-card">
            <div class="value">' . $total_games_checked . '</div>
            <div class="label">Games Checked</div>
        </div>
        <div class="metric-card">
            <div class="value">' . $total_games_updated . '</div>
            <div class="label">Games Updated</div>
        </div>
        <div class="metric-card">
            <div class="value">' . $total_games_paused . '</div>
            <div class="label">Paused Updates</div>
        </div>
        <div class="metric-card">
            <div class="value">' . round(memory_get_peak_usage() / 1024 / 1024, 2) . ' MB</div>
            <div class="label">Peak Memory</div>
        </div>
    </div>';

if (!empty($conferences_with_active_games)) {
    echo '<div style="margin-top: 20px;">
        <h3>Conferences with Active Games</h3>
        <p>' . implode(', ', $conferences_with_active_games) . '</p>
    </div>';
}

if (!empty($conferences_skipped)) {
    echo '<div style="margin-top: 20px;">
        <h3>Conferences Skipped</h3>
        <p>' . implode(', ', $conferences_skipped) . '</p>
    </div>';
}

echo '<div style="margin-top: 20px;">
    <h3>Broadcast Information</h3>
    <p>Broadcast/channel data has been added and will be displayed for all games with available information.</p>
</div>';

echo '</div>';

// Call output APIs if games were updated
if ($total_games_updated > 0) {
    $url = SERVER_URL . "Scrolling/App_Output/API/Games/NCAAF_Games.php";
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_exec($ch);
    curl_close($ch);
    
    $url = SERVER_URL . "Scrolling/App_Output/API/Games/Scoring_ALERT.php";
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_exec($ch);
    curl_close($ch);
}

echo '</div></body></html>';

mysqli_close($con);
session_write_close();

$html_output = ob_get_clean();
echo $html_output;

?>