<?php
/**
 * NCAA Baseball Time Processor with Auto-Update Logic
 * 
 * This script finds all NCAA Baseball games in the database,
 * converts UTC game times to Eastern time, calculates time differences,
 * and sets debug flags based on proximity to game start/end times.
 * 
 * If any game has a debug flag set to true, it will trigger an update
 * by calling the get_NCAABase.php script via cURL.
 * 
 * Created: 2025-03-04
 */

// Updated config path
require_once '../../config.php';

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone to Eastern Time
date_default_timezone_set('America/New_York');

// Connect to the database
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Array to hold processed games
$processed_games = array();
$any_debug_flag_true = false; // Flag to track if any game needs updating

// Get current time in Eastern Time
$current_time_eastern = new DateTime('now');
$current_time_formatted = $current_time_eastern->format('Y-m-d H:i:s');

// Query to get all NCAA Baseball games
$query = "SELECT * FROM sports_list WHERE Conference = 'NCAABASE'";
$result = mysqli_query($con, $query);

if (!$result) {
    die("Error fetching games: " . mysqli_error($con));
}

// Iterate through each game
while ($game = mysqli_fetch_assoc($result)) {
    $game_id = $game['Game'];
    $game_date_utc = $game['Game_Date'];
    $game_status = $game['Game_Played'];
    
    // Skip if no game date
    if (empty($game_date_utc)) {
        continue;
    }
    
    // Process the UTC date string
    try {
        // Create DateTime object from the UTC time
        $game_time_utc = new DateTime($game_date_utc);
        
        // Convert to Eastern Time
        $game_time_utc->setTimezone(new DateTimeZone('America/New_York'));
        $game_time_eastern = $game_time_utc->format('Y-m-d H:i:s');
        $game_time_display = $game_time_utc->format('m/d/Y | g:i A T');
        
        // Calculate time difference in minutes
        $current_time = new DateTime('now');
        $time_diff = $current_time->getTimestamp() - $game_time_utc->getTimestamp();
        $time_diff_minutes = round($time_diff / 60);
        
        // Calculate time difference in a readable format
        if ($time_diff > 0) {
            // Game time is in the past
            $hours = floor(abs($time_diff) / 3600);
            $minutes = floor((abs($time_diff) % 3600) / 60);
            $time_diff_readable = $hours . "h " . $minutes . "m ago";
        } else {
            // Game time is in the future
            $hours = floor(abs($time_diff) / 3600);
            $minutes = floor((abs($time_diff) % 3600) / 60);
            $time_diff_readable = "in " . $hours . "h " . $minutes . "m";
        }
        
        // Determine if game is about to start (within 5 minutes)
        $about_to_start = false;
        if ($time_diff_minutes > -5 && $time_diff_minutes <= 0) {
            $about_to_start = true;
        }
        
        // Determine if game is in progress
        $in_progress = false;
        if ($time_diff_minutes > 0 && $time_diff_minutes <= 240) { // 4 hours = 240 minutes
            $in_progress = true;
        }
        
        // Set the debug flag
        $debug_flag = false;
        if ($about_to_start || $in_progress) {
            $debug_flag = true;
            $any_debug_flag_true = true; // Set the global flag
        }
        
        // Update Game_Time with Eastern time for display purposes
        // and use existing fields rather than creating new ones
        $update_query = "UPDATE sports_list SET 
            Game_Time = '" . mysqli_real_escape_string($con, $game_time_utc->format('g:i a')) . "'
            WHERE Game = $game_id AND Conference = 'NCAABASE'";
        
        $update_result = mysqli_query($con, $update_query);
        
        if (!$update_result) {
            echo "Error updating game #$game_id: " . mysqli_error($con) . "<br>";
        }
        
        // Add to processed games array
        $processed_games[] = array(
            'game_id' => $game_id,
            'team1' => $game['Team_1_Name'],
            'team2' => $game['Team_2_Name'],
            'status' => $game_status,
            'utc_time' => $game_date_utc,
            'eastern_time' => $game_time_display,
            'time_diff' => $time_diff_readable,
            'time_diff_minutes' => $time_diff_minutes,
            'about_to_start' => $about_to_start,
            'in_progress' => $in_progress,
            'debug_flag' => $debug_flag
        );
        
    } catch (Exception $e) {
        echo "Error processing game #$game_id: " . $e->getMessage() . "<br>";
    }
}

// If any game has debug flag set to true, trigger the update script via cURL
$curl_result = '';
$curl_error = '';
$curl_time = '';

if ($any_debug_flag_true) {
    $curl_start_time = microtime(true);
    
    // Initialize cURL session
    $ch = curl_init();
    
    // Set cURL options
    curl_setopt($ch, CURLOPT_URL, SERVER_URL . "updateAPIs/get_NCAABase.php");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Execute cURL session
    $curl_result = curl_exec($ch);
    
    // Check for errors
    if (curl_errno($ch)) {
        $curl_error = "cURL Error: " . curl_error($ch);
    }
    
    // Calculate execution time
    $curl_time = round(microtime(true) - $curl_start_time, 2);
    
    // Close cURL session
    curl_close($ch);
}

// Close the database connection
mysqli_close($con);

// Function to classify game status for display
function getStatusClass($status, $about_to_start, $in_progress) {
    if ($status == 'IN_SESSION' || $in_progress) {
        return 'status-in-session';
    } else if ($status == 'GAME_DONE') {
        return 'status-game-done';
    } else if ($status == 'CANCELED') {
        return 'status-canceled';
    } else if ($status == 'POSTPONED') {
        return 'status-postponed';
    } else if ($about_to_start) {
        return 'status-about-to-start';
    } else {
        return 'status-not-started';
    }
}

// Output HTML
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>NCAA Baseball Time Processor</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f5f5f5;
        }
        
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #4CAF50;
        }
        
        .summary {
            background-color: white;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }
        
        .curl-status {
            margin-top: 20px;
            background-color: #e8f5e9;
            padding: 15px;
            border-radius: 5px;
            border-left: 5px solid #4CAF50;
        }
        
        .curl-error {
            background-color: #ffebee;
            border-left: 5px solid #f44336;
        }
        
        .games-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.15);
            background-color: white;
        }
        
        .games-table th {
            background-color: #4CAF50;
            color: white;
            font-weight: bold;
            padding: 12px 15px;
            text-align: left;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .games-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #dddddd;
            font-size: 0.9em;
        }
        
        .games-table tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        
        .games-table tr:hover {
            background-color: #f1f1f1;
        }
        
        .status {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-weight: bold;
            text-align: center;
            min-width: 100px;
        }
        
        .status-in-session {
            background-color: #FFC107;
            color: #333;
        }
        
        .status-not-started {
            background-color: #2196F3;
            color: white;
        }
        
        .status-game-done {
            background-color: #9E9E9E;
            color: white;
        }
        
        .status-about-to-start {
            background-color: #4CAF50;
            color: white;
        }
        
        .status-canceled {
            background-color: #F44336;
            color: white;
        }
        
        .status-postponed {
            background-color: #FF9800;
            color: white;
        }
        
        .flag-true {
            background-color: #4CAF50;
            color: white;
            padding: 2px 8px;
            border-radius: 4px;
        }
        
        .flag-false {
            background-color: #F44336;
            color: white;
            padding: 2px 8px;
            border-radius: 4px;
        }
        
        .timestamp {
            text-align: center;
            margin-bottom: 10px;
            color: #666;
            font-style: italic;
        }
        
        .section {
            margin-top: 30px;
            background-color: white;
            padding: 15px;
            border-radius: 5px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }
        
        .game-card {
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 15px;
            margin-bottom: 15px;
            background-color: white;
            display: flex;
            flex-direction: column;
        }
        
        .game-header {
            display: flex;
            justify-content: space-between;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
            margin-bottom: 10px;
        }
        
        .game-time {
            font-weight: bold;
            color: #555;
        }
        
        .game-status {
            margin-left: auto;
        }
        
        .teams {
            display: flex;
            justify-content: space-between;
            font-size: 1.2em;
            margin-bottom: 10px;
        }
        
        .team {
            display: flex;
            align-items: center;
        }
        
        .team-name {
            margin-right: 10px;
            font-weight: bold;
        }
        
        .team-score {
            font-size: 1.2em;
            font-weight: bold;
            min-width: 30px;
            text-align: center;
        }
        
        .game-meta {
            display: flex;
            justify-content: space-between;
            font-size: 0.9em;
            color: #666;
        }
        
        .code-block {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 4px;
            font-family: monospace;
            white-space: pre-wrap;
            max-height: 200px;
            overflow-y: auto;
            margin-top: 10px;
            border: 1px solid #ddd;
        }
    </style>
</head>
<body>
    <h1>NCAA Baseball Time Processor</h1>
    <p class="timestamp">Current Eastern Time: <?php echo $current_time_formatted; ?></p>
    
    <div class="summary">
        <h2>Summary</h2>
        <p>
            Processed <?php echo count($processed_games); ?> NCAA Baseball games.<br>
            <?php 
                $about_to_start_count = array_filter($processed_games, function($game) { return $game['about_to_start']; });
                $in_progress_count = array_filter($processed_games, function($game) { return $game['in_progress']; });
                $debug_flag_count = array_filter($processed_games, function($game) { return $game['debug_flag']; });
                
                echo count($about_to_start_count) . " games about to start.<br>";
                echo count($in_progress_count) . " games in progress.<br>";
                echo count($debug_flag_count) . " games with debug flag set to true.";
            ?>
        </p>
        
        <?php if ($any_debug_flag_true): ?>
        <div class="curl-status <?php echo !empty($curl_error) ? 'curl-error' : ''; ?>">
            <h3>Update Triggered</h3>
            <p>Debug flag is TRUE for at least one game. Update API has been called.</p>
            <p>Execution time: <?php echo $curl_time; ?> seconds</p>
            
            <?php if (!empty($curl_error)): ?>
                <p><strong>Error:</strong> <?php echo $curl_error; ?></p>
            <?php endif; ?>
            
            <?php if (strlen($curl_result) > 500): ?>
                <p>API response is too long to display completely (<?php echo strlen($curl_result); ?> characters).</p>
                <div class="code-block"><?php echo htmlspecialchars(substr($curl_result, 0, 500) . '...'); ?></div>
            <?php elseif (!empty($curl_result)): ?>
                <div class="code-block"><?php echo htmlspecialchars($curl_result); ?></div>
            <?php endif; ?>
        </div>
        <?php else: ?>
        <div class="curl-status">
            <h3>No Update Needed</h3>
            <p>No games currently have debug flag set to TRUE. API update was not triggered.</p>
        </div>
        <?php endif; ?>
    </div>
    
    <?php if (count($debug_flag_count) > 0): ?>
    <div class="section">
        <h2>Games with Debug Flag = TRUE</h2>
        <div class="card-container">
            <?php foreach ($debug_flag_count as $game): ?>
                <div class="game-card">
                    <div class="game-header">
                        <div class="game-time"><?php echo $game['eastern_time']; ?> (<?php echo $game['time_diff']; ?>)</div>
                        <div class="game-status">
                            <span class="status <?php echo getStatusClass($game['status'], $game['about_to_start'], $game['in_progress']); ?>">
                                <?php 
                                    if ($game['about_to_start']) echo "STARTING SOON";
                                    else if ($game['in_progress']) echo "IN PROGRESS";
                                    else echo $game['status'];
                                ?>
                            </span>
                        </div>
                    </div>
                    <div class="teams">
                        <div class="team">
                            <div class="team-name"><?php echo htmlspecialchars($game['team1']); ?></div>
                        </div>
                        <div class="team">
                            <div class="team-name"><?php echo htmlspecialchars($game['team2']); ?></div>
                        </div>
                    </div>
                    <div class="game-meta">
                        <div>Game #<?php echo $game['game_id']; ?></div>
                        <div>Debug Flag: <span class="flag-<?php echo $game['debug_flag'] ? 'true' : 'false'; ?>"><?php echo $game['debug_flag'] ? 'TRUE' : 'FALSE'; ?></span></div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>
    
    <h2>All Processed Games</h2>
    <table class="games-table">
        <thead>
            <tr>
                <th>Game #</th>
                <th>Teams</th>
                <th>Status</th>
                <th>UTC Time</th>
                <th>Eastern Time</th>
                <th>Time Difference</th>
                <th>About to Start</th>
                <th>In Progress</th>
                <th>Debug Flag</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($processed_games as $game): ?>
                <tr>
                    <td><?php echo $game['game_id']; ?></td>
                    <td><?php echo htmlspecialchars($game['team1']); ?> vs. <?php echo htmlspecialchars($game['team2']); ?></td>
                    <td>
                        <span class="status <?php echo getStatusClass($game['status'], $game['about_to_start'], $game['in_progress']); ?>">
                            <?php echo $game['status']; ?>
                        </span>
                    </td>
                    <td><?php echo $game['utc_time']; ?></td>
                    <td><?php echo $game['eastern_time']; ?></td>
                    <td><?php echo $game['time_diff']; ?></td>
                    <td class="flag-<?php echo $game['about_to_start'] ? 'true' : 'false'; ?>">
                        <?php echo $game['about_to_start'] ? 'Yes' : 'No'; ?>
                    </td>
                    <td class="flag-<?php echo $game['in_progress'] ? 'true' : 'false'; ?>">
                        <?php echo $game['in_progress'] ? 'Yes' : 'No'; ?>
                    </td>
                    <td class="flag-<?php echo $game['debug_flag'] ? 'true' : 'false'; ?>">
                        <?php echo $game['debug_flag'] ? 'True' : 'False'; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</body>
</html>