<?php
/**
 * NCAA Basketball Time Processor with Auto-Update Logic
 * 
 * This script finds all NCAA Basketball games in the database,
 * converts UTC game times to Eastern time, calculates time differences,
 * and sets debug flags based on proximity to game start/end times.
 * 
 * If any game has a debug flag set to true, it will trigger an update
 * by calling the Basketball.php script via cURL.
 * 
 * Created: 2025-03-04
 */

// Updated config path
require_once '../../config.php';

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone to Eastern Time
date_default_timezone_set('America/New_York');

// Connect to the database
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Array to hold processed games
$processed_games = array();
$any_debug_flag_true = false; // Flag to track if any game needs updating

// Get current time in Eastern Time
$current_time_eastern = new DateTime('now');
$current_time_formatted = $current_time_eastern->format('Y-m-d H:i:s');

// Query to get all NCAA Basketball games
$query = "SELECT * FROM sports_list WHERE Conference = 'NCAAB'";
$result = mysqli_query($con, $query);

if (!$result) {
    die("Error fetching games: " . mysqli_error($con));
}

// Iterate through each game
while ($game = mysqli_fetch_assoc($result)) {
    $game_id = $game['Game'];
    $game_date_utc = $game['Game_Date'];
    $game_status = $game['Game_Played'];
    
    // Skip if no game date
    if (empty($game_date_utc)) {
        continue;
    }
    
    // Process the UTC date string
    try {
        // Create DateTime object from the UTC time
        $game_time_utc = new DateTime($game_date_utc);
        
        // Convert to Eastern Time
        $game_time_utc->setTimezone(new DateTimeZone('America/New_York'));
        $game_time_eastern = $game_time_utc->format('Y-m-d H:i:s');
        $game_time_display = $game_time_utc->format('m/d/Y | g:i A T');
        
        // Calculate time difference in minutes
        $current_time = new DateTime('now');
        $time_diff = $current_time->getTimestamp() - $game_time_utc->getTimestamp();
        $time_diff_minutes = round($time_diff / 60);
        
        // Calculate time difference in a readable format
        if ($time_diff > 0) {
            // Game time is in the past
            $hours = floor(abs($time_diff) / 3600);
            $minutes = floor((abs($time_diff) % 3600) / 60);
            $time_diff_readable = $hours . "h " . $minutes . "m ago";
        } else {
            // Game time is in the future
            $hours = floor(abs($time_diff) / 3600);
            $minutes = floor((abs($time_diff) % 3600) / 60);
            $time_diff_readable = "in " . $hours . "h " . $minutes . "m";
        }
        
        // Determine if game is about to start (within 5 minutes)
        $about_to_start = false;
        if ($time_diff_minutes > -5 && $time_diff_minutes <= 0) {
            $about_to_start = true;
        }
        
        // Determine if game is in progress
        $in_progress = false;
        if ($time_diff_minutes > 0 && $time_diff_minutes <= 240) { // 4 hours = 240 minutes
            $in_progress = true;
        }
        
        // Set the debug flag
        $debug_flag = false;
        if ($about_to_start || $in_progress) {
            $debug_flag = true;
            $any_debug_flag_true = true; // Set the global flag
        }
        
        // Update Game_Time with Eastern time for display purposes
        // and use existing fields rather than creating new ones
        $update_query = "UPDATE sports_list SET 
            Game_Time = '" . mysqli_real_escape_string($con, $game_time_utc->format('g:i a')) . "'
            WHERE Game = $game_id AND Conference = 'Basketball'";
        
        $update_result = mysqli_query($con, $update_query);
        
        if (!$update_result) {
            echo "Error updating game #$game_id: " . mysqli_error($con) . "<br>";
        }
        
        // Add to processed games array
        $processed_games[] = array(
            'game_id' => $game_id,
            'team1' => $game['Team_1_Name'],
            'team2' => $game['Team_2_Name'],
            'status' => $game_status,
            'utc_time' => $game_date_utc,
            'eastern_time' => $game_time_display,
            'time_diff' => $time_diff_readable,
            'time_diff_minutes' => $time_diff_minutes,
            'about_to_start' => $about_to_start,
            'in_progress' => $in_progress,
            'debug_flag' => $debug_flag
        );
        
    } catch (Exception $e) {
        echo "Error processing game #$game_id: " . $e->getMessage() . "<br>";
    }
}

// If any game has debug flag set to true, trigger the update script via cURL
$curl_result = '';
$curl_error = '';
$curl_time = '';

if ($any_debug_flag_true) {
    $curl_start_time = microtime(true);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, SERVER_URL . "updateAPIs/get_NCAAB.php");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    $curl_result = curl_exec($ch);
    
    echo "HELLO FUCKER";
}


// Close the database connection
mysqli_close($con);


// Output HTML
?>
