<?php
//************************************************//
// NHL Score Updater - Optimized Version with Score Change Detection
// Only calls API when active games exist or games starting within 10 minutes
// Caches database state before API calls for comparison
// Only calls summary API when scores change
// Protects GOAL entries from being overwritten
//************************************************//

set_time_limit(300);
ini_set('memory_limit', '256M');
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../../config.php';

// Beautiful HTML header
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>NHL Score Updater - Optimized Score Detection</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background: linear-gradient(135deg, #1e88e5 0%, #0d47a1 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        .header {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            margin-bottom: 30px;
        }

        .header h1 {
            color: #0d47a1;
            font-size: 32px;
            margin-bottom: 15px;
        }

        .time-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .time-box {
            background: #e3f2fd;
            padding: 10px 15px;
            border-radius: 8px;
            border-left: 4px solid #1e88e5;
        }

        .debug-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            border-radius: 8px;
            font-family: monospace;
            font-size: 12px;
        }

        .debug-box h4 {
            color: #856404;
            margin-bottom: 10px;
        }

        .score-change-card {
            background: white;
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
            border-left: 5px solid #ff5722;
        }

        .score-change-header {
            padding: 20px;
            background: linear-gradient(135deg, #ff5722 0%, #d84315 100%);
            color: white;
        }

        .score-change-detected {
            background: #ffebee;
            border: 2px solid #ff5252;
            padding: 15px;
            margin: 15px;
            border-radius: 8px;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0% { border-color: #ff5252; }
            50% { border-color: #ff1744; }
            100% { border-color: #ff5252; }
        }

        .goal-detection-card {
            background: white;
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
            border-left: 5px solid #ff5722;
        }

        .goal-detection-header {
            padding: 20px;
            background: linear-gradient(135deg, #ff5722 0%, #d84315 100%);
            color: white;
        }

        .goal-alert {
            background: #ffebee;
            border: 2px solid #ff5252;
            padding: 15px;
            margin: 15px;
            border-radius: 8px;
            animation: pulse 2s infinite;
        }

        .update-check-card {
            background: white;
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }

        .update-check-header {
            padding: 20px;
            background: linear-gradient(135deg, #1e88e5 0%, #1976d2 100%);
            color: white;
        }

        .status-badge {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: bold;
            display: inline-block;
        }

        .badge-active {
            background: #00c853;
            color: white;
        }

        .badge-skip {
            background: #9e9e9e;
            color: white;
        }

        .game-table {
            width: 100%;
            border-collapse: collapse;
        }

        .game-table thead {
            background: #f5f5f5;
        }

        .game-table th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #0d47a1;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid #e0e0e0;
        }

        .game-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 14px;
        }

        .game-row-active {
            background: #e8f5e9 !important;
            border-left: 4px solid #4caf50;
        }

        .game-row-paused {
            background: #fce4ec !important;
            border-left: 4px solid #e91e63;
        }

        .game-row-goal {
            background: #fff3cd !important;
            border-left: 4px solid #ffc107;
        }

        .status-pill {
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: bold;
            display: inline-block;
        }

        .pill-active {
            background: #4caf50;
            color: white;
        }

        .pill-overtime {
            background: #ff9800;
            color: white;
        }

        .pill-shootout {
            background: #9c27b0;
            color: white;
        }

        .pill-final {
            background: #9e9e9e;
            color: white;
        }

        .pill-upcoming {
            background: #2196f3;
            color: white;
        }

        .goal-indicator {
            background: #ff5252;
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
            display: inline-block;
            animation: pulse 2s infinite;
        }

        .info-box {
            background: #e3f2fd;
            border-left: 4px solid #2196f3;
            padding: 15px;
            margin: 15px;
            border-radius: 4px;
        }

        .skip-notice {
            background: #ffebee;
            border-left: 4px solid #f44336;
            padding: 15px;
            margin: 15px;
            border-radius: 4px;
            font-weight: bold;
            color: #b71c1c;
        }

        .summary {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-top: 30px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }

        .cache-info {
            background: #f5f5f5;
            border: 1px solid #ddd;
            padding: 10px;
            margin: 10px 0;
            border-radius: 5px;
        }

        .score-comparison-table {
            width: 100%;
            margin: 15px 0;
            border-collapse: collapse;
        }

        .score-comparison-table td, .score-comparison-table th {
            padding: 8px;
            border: 1px solid #ddd;
            text-align: center;
        }

        .score-comparison-table th {
            background: #f5f5f5;
        }

        .score-changed {
            background: #ffeb3b !important;
            font-weight: bold;
        }
    </style>
</head>
<body>
<div class="container">';

// Set timezone
date_default_timezone_set('America/New_York');
$start_time = microtime(true);
$current_time_et = date('m/d/Y | g:i:s A');
$current_timestamp = time();
$current_datetime = date('Y-m-d H:i:s');
$time_updated = date('m/d/Y | g:iA');

// Header
echo '<div class="header">
    <h1>🏒 NHL Score Updater - Optimized Score Detection</h1>
    <div class="time-info">
        <div class="time-box">
            <label>Eastern Time</label>
            <div class="value">' . $current_time_et . '</div>
        </div>
        <div class="time-box">
            <label>Script Started</label>
            <div class="value">' . date('g:i:s A') . '</div>
        </div>
        <div class="time-box">
            <label>Season</label>
            <div class="value">2024-25</div>
        </div>
    </div>
</div>';

// Database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die('<div class="error-box">Database connection failed: ' . mysqli_connect_error() . '</div>');
}

// Check/add required columns
$columns_to_check = ['eventID', 'lastUpdatedTime', 'broadcast', 'scoreValue', 'lastPlay',
                     'lastPlayerID', 'lastPlayerName', 'lastPlayerPos',
                     'lastPlayerTeamID', 'jersey', 'driveDesc'];

foreach ($columns_to_check as $column) {
    $check_column = "SHOW COLUMNS FROM `sports_list` LIKE '$column'";
    $column_result = mysqli_query($con, $check_column);
    if (mysqli_num_rows($column_result) == 0) {
        if ($column == 'lastUpdatedTime') {
            $type = 'DATETIME';
        } elseif ($column == 'broadcast') {
            $type = 'VARCHAR(50)';
        } elseif ($column == 'eventID') {
            $type = 'VARCHAR(50)';
        } else {
            $type = 'VARCHAR(255)';
        }
        $add_column = "ALTER TABLE `sports_list` ADD COLUMN `$column` $type DEFAULT NULL";
        mysqli_query($con, $add_column);
    }
}

// =======================
// CACHE CURRENT DATABASE STATE
// =======================
echo '<div class="update-check-card">
    <div class="update-check-header">
        <h2>💾 Caching Current Database State</h2>
    </div>
    <div style="padding: 20px;">';

// Cache ALL games from database before any API calls
// First check if eventID column exists
$eventID_exists = false;
$check_eventID = "SHOW COLUMNS FROM `sports_list` LIKE 'eventID'";
$eventID_result = mysqli_query($con, $check_eventID);
if (mysqli_num_rows($eventID_result) > 0) {
    $eventID_exists = true;
}

// Build cache query based on whether eventID exists
if ($eventID_exists) {
    $cache_query = "
        SELECT Game, eventID, Team_1_Abbrev, Team_2_Abbrev, Team_1_Score, Team_2_Score,
               Game_Played, scoreValue, lastUpdatedTime, Period, Clock, lastPlay,
               lastPlayerName, lastPlayerID
        FROM sports_list
        WHERE Conference = 'NHL'
        ORDER BY Game ASC
    ";
} else {
    $cache_query = "
        SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Team_1_Score, Team_2_Score,
               Game_Played, scoreValue, lastUpdatedTime, Period, Clock, lastPlay,
               lastPlayerName, lastPlayerID
        FROM sports_list
        WHERE Conference = 'NHL'
        ORDER BY Game ASC
    ";
}

$cache_result = mysqli_query($con, $cache_query);
$cached_database_state = [];

if ($cache_result && mysqli_num_rows($cache_result) > 0) {
    while ($row = mysqli_fetch_assoc($cache_result)) {
        $cached_database_state[$row['Game']] = [
            'eventID' => $eventID_exists ? ($row['eventID'] ?? '') : '',
            'away_team' => $row['Team_2_Abbrev'],
            'home_team' => $row['Team_1_Abbrev'],
            'away_score' => intval($row['Team_2_Score']),
            'home_score' => intval($row['Team_1_Score']),
            'status' => $row['Game_Played'],
            'scoreValue' => $row['scoreValue'],
            'lastUpdated' => $row['lastUpdatedTime'],
            'period' => $row['Period'],
            'clock' => $row['Clock'],
            'lastPlay' => $row['lastPlay'],
            'lastPlayerName' => $row['lastPlayerName'],
            'lastPlayerID' => $row['lastPlayerID']
        ];
    }
    echo '<div class="cache-info">
        ✅ Cached ' . count($cached_database_state) . ' games from database<br>
        <small>This cached state will be used for comparisons after API call</small>
    </div>';
} else {
    echo '<div class="cache-info">
        ⚠️ No games found in database to cache
    </div>';
}

echo '</div></div>';

// =======================
// CHECK IF ANY GAMES NEED UPDATING
// =======================
echo '<div class="update-check-card">
    <div class="update-check-header">
        <h2>🔍 Checking for Games to Update</h2>
    </div>
    <div style="padding: 20px;">';

$five_hours_ago = date('Y-m-d H:i:s', $current_timestamp - (5 * 3600));
$ten_minutes_from_now = date('Y-m-d H:i:s', $current_timestamp + 600);

// Check for active games or games starting soon
$check_query = "
    SELECT COUNT(*) as count,
           SUM(CASE WHEN Game_Played IN ('IN_SESSION', 'HALF_TIME') THEN 1 ELSE 0 END) as active_count,
           SUM(CASE WHEN Game_Played = 'NOT_STARTED'
                    AND Game_Date >= '$five_hours_ago'
                    AND Game_Date <= '$ten_minutes_from_now'
                    THEN 1 ELSE 0 END) as upcoming_count
    FROM sports_list
    WHERE Conference = 'NHL'
";

$check_result = mysqli_query($con, $check_query);
$game_counts = mysqli_fetch_assoc($check_result);

$active_games = intval($game_counts['active_count']);
$upcoming_games = intval($game_counts['upcoming_count']);
$games_needing_update = $active_games + $upcoming_games;

echo '<div class="info-box">
    <strong>Active Games (IN_SESSION/HALF_TIME):</strong> ' . $active_games . '<br>
    <strong>Games Starting Soon (within 10 min):</strong> ' . $upcoming_games . '<br>
    <strong>Total Games Needing Update:</strong> ' . $games_needing_update . '<br>
</div>';

// =======================
// DECISION: SKIP OR PROCEED WITH API CALL
// =======================
if ($games_needing_update == 0) {
    echo '<div class="skip-notice">
        ⚠️ <strong>NO API CALL REQUIRED</strong><br>
        No active games or games starting within 10 minutes.<br>
        Script will exit without making any API calls.
    </div>';

    echo '</div></div>';

    echo '<div class="summary">
        <h2>📊 Execution Summary</h2>
        <p><strong>Status:</strong> No API calls made - no games require updating</p>
        <p><strong>Execution Time:</strong> ' . round(microtime(true) - $start_time, 2) . ' seconds</p>
        <p><strong>Cached Games:</strong> ' . count($cached_database_state) . '</p>
        <p><strong>API Calls Saved:</strong> All (100% efficiency)</p>
        <p><strong>Next Check:</strong> Script will run again in next scheduled interval</p>
    </div>';

    // Update the API timestamp even when skipped
    $sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NHL'";
    mysqli_query($con, $sql);

    mysqli_close($con);
    echo '</div></body></html>';
    exit();
}

echo '<div class="info-box" style="background: #c8e6c9; border-color: #4caf50;">
    ✅ <strong>PROCEEDING WITH API CALLS</strong><br>
    Found ' . $games_needing_update . ' game(s) requiring updates
</div>';

echo '</div></div>';

// =======================
// HELPER FUNCTIONS
// =======================

// Helper function to convert hex to RGB
function hexToRgb($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) != 6) return "255,255,255";

    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));

    // Force to white if too dark
    if ($r <= 16 && $g <= 16 && $b <= 16) {
        $r = $g = $b = 255;
    }

    return "$r,$g,$b";
}

// Function to check for recent goals via Summary API - ONLY called when scores change
function checkRecentGoalsOnScoreChange($eventId, $con, $current_timestamp, $game_num, $old_scores, $new_scores) {
    $summary_url = "http://site.api.espn.com/apis/site/v2/sports/hockey/nhl/summary?event=" . $eventId;

    echo '<div class="debug-box" style="border-color: #ff5722; background: #ffebee;">';
    echo '<h4>🚨 SCORE CHANGE DETECTED - Fetching Goal Details for Game ' . $game_num . '</h4>';
    echo '<strong>Score Changed From:</strong> ' . $old_scores['away_score'] . '-' . $old_scores['home_score'] .
         ' → ' . $new_scores['away'] . '-' . $new_scores['home'] . '<br>';
    echo '<strong>Event ID:</strong> ' . $eventId . '<br>';
    echo '<strong>API URL:</strong> <a href="' . $summary_url . '" target="_blank">' . $summary_url . '</a><br>';

    // Try to fetch summary data WITHOUT proxy
    $maxRetries = 3;
    $success = false;
    $response = false;

    for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
        $curl = curl_init($summary_url);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => "Mozilla/5.0 (compatible; GlanceBot/1.0)"
            // NO PROXY
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($response && $httpCode === 200) {
            $success = true;
            echo '<span style="color: green;">✓ Summary API fetched successfully</span><br>';
            break;
        }

        if ($attempt < $maxRetries) {
            sleep(1);
        }
    }

    if (!$success) {
        echo '<span style="color: red;">Failed to fetch summary data after ' . $maxRetries . ' attempts</span><br>';
        echo '</div>';
        return null;
    }

    $summary_data = json_decode($response, true);

    if (empty($summary_data)) {
        echo '<span style="color: red;">Failed to parse summary JSON</span><br>';
        echo '</div>';
        return null;
    }

    // Extract the most recent goal information
    $goal_info = [
        'scoreValue' => 'GOAL',
        'lastPlay' => '',
        'lastPlayerName' => '',
        'lastPlayerID' => '',
        'period' => 0,
        'clock' => ''
    ];

    // Check for scoring plays
    if (isset($summary_data['scoringPlays']) && !empty($summary_data['scoringPlays'])) {
        $most_recent_goal = end($summary_data['scoringPlays']);

        if ($most_recent_goal) {
            $goal_info['lastPlay'] = $most_recent_goal['text'] ?? '';
            $goal_info['period'] = $most_recent_goal['period']['number'] ?? 0;
            $goal_info['clock'] = $most_recent_goal['clock']['displayValue'] ?? '';

            // Get scorer information
            if (isset($most_recent_goal['participants'])) {
                foreach ($most_recent_goal['participants'] as $participant) {
                    if ($participant['type'] == 'scorer') {
                        $goal_info['lastPlayerID'] = $participant['athlete']['id'] ?? '';
                        $goal_info['lastPlayerName'] = $participant['athlete']['displayName'] ?? '';
                        break;
                    }
                }
            }

            echo '<div class="goal-alert">';
            echo '🚨 <strong>GOAL DETECTED!</strong><br>';
            echo 'Scorer: ' . ($goal_info['lastPlayerName'] ?: 'Unknown') . '<br>';
            echo 'Period: ' . $goal_info['period'] . ' - ' . $goal_info['clock'] . '<br>';
            echo 'Description: ' . substr($goal_info['lastPlay'], 0, 100) . '...<br>';
            echo '</div>';
        }
    }

    echo '</div>';

    return $goal_info;
}

// Extract game data function
function extractGameData($event, $game_num) {
    $gameData = [
        'game_num' => $game_num,
        'event_id' => $event['id'] ?? '',
        'clock' => '0:00',
        'period' => 0,
        'game_status' => 'NOT_STARTED',
        'game_time' => '',
        'game_date' => '',
        'broadcast' => '',
        'score_value' => 'NO-SCORE',
        'team1' => [],
        'team2' => [],
        'last_play' => '',
        'last_player' => [],
        'who_has_possession' => '',
        'drive_desc' => '',
        'favorite' => [],
        'over_under' => ''
    ];

    $competition = $event['competitions'][0] ?? [];

    // Get game timing and status
    $status = $competition['status'] ?? [];
    $gameData['clock'] = $status['displayClock'] ?? '0:00';
    $gameData['period'] = $status['period'] ?? 0;

    // Determine game status
    $statusType = $status['type']['name'] ?? '';
    switch ($statusType) {
        case 'STATUS_SCHEDULED':
            $gameData['game_status'] = 'NOT_STARTED';
            break;
        case 'STATUS_IN_PROGRESS':
            $gameData['game_status'] = 'IN_SESSION';
            break;
        case 'STATUS_END_PERIOD':
            $gameData['game_status'] = 'HALF_TIME';
            break;
        case 'STATUS_FINAL':
            $gameData['game_status'] = 'GAME_DONE';
            break;
    }

    // Get game time
    $gameData['game_date'] = $event['date'] ?? '';
    $gameData['game_time'] = date("g:i a", strtotime($gameData['game_date']));

    // Extract broadcast information
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $gameData['broadcast'] = $broadcast_data['names'][0] ?? '';
        }
    }

    // Get teams
    $competitors = $competition['competitors'] ?? [];

    foreach ($competitors as $competitor) {
        $teamData = [
            'id' => $competitor['team']['id'] ?? '',
            'name' => $competitor['team']['name'] ?? '',
            'location' => $competitor['team']['location'] ?? '',
            'abbreviation' => $competitor['team']['abbreviation'] ?? '',
            'score' => $competitor['score'] ?? '0',
            'record' => '',
            'color' => '',
            'color_rgb' => '255,255,255'
        ];

        // Get team colors
        $color = $competitor['team']['color'] ?? '000000';
        $altColor = $competitor['team']['alternateColor'] ?? '';

        if ($color == '000000' && !empty($altColor)) {
            $color = $altColor;
        }

        $teamData['color'] = $color;
        $teamData['color_rgb'] = hexToRgb($color);

        // Get team record
        $records = $competitor['records'] ?? [];
        foreach ($records as $record) {
            if ($record['type'] == 'ytd' || $record['name'] == 'YTD') {
                $teamData['record'] = $record['summary'] ?? '';
                break;
            }
        }

        // Assign to team1 (home) or team2 (away)
        if ($competitor['homeAway'] == 'home') {
            $gameData['team1'] = $teamData;
        } else {
            $gameData['team2'] = $teamData;
        }
    }

    // Get in-game data if the game is active
    if (in_array($gameData['game_status'], ['IN_SESSION', 'HALF_TIME'])) {
        $situation = $competition['situation'] ?? [];
        $lastPlay = $situation['lastPlay'] ?? [];

        if (!empty($lastPlay)) {
            $gameData['last_play'] = $lastPlay['text'] ?? '';
            $gameData['score_value'] = ($lastPlay['scoreValue'] ?? 0) == 1 ? 'GOAL' : 'NO-SCORE';

            // Get player involved
            $athletes = $lastPlay['athletesInvolved'] ?? [];
            if (!empty($athletes[0])) {
                $athlete = $athletes[0];
                $gameData['last_player'] = [
                    'id' => $athlete['id'] ?? '',
                    'name' => $athlete['displayName'] ?? $athlete['fullName'] ?? '',
                    'position' => $athlete['position'] ?? '',
                    'jersey' => $athlete['jersey'] ?? '',
                    'team_id' => $athlete['team']['id'] ?? ''
                ];
            }

            // Check for goal in text
            if (stripos($gameData['last_play'], 'goal') !== false && stripos($gameData['last_play'], 'goalie') === false) {
                $gameData['score_value'] = 'GOAL';
            }
        }

        // Set period description
        if ($gameData['period'] <= 3) {
            $gameData['drive_desc'] = "Period " . $gameData['period'];
        } elseif ($gameData['period'] == 4) {
            $gameData['drive_desc'] = "Overtime";
        } elseif ($gameData['period'] == 5) {
            $gameData['drive_desc'] = "Shootout";
        }
    }

    return $gameData;
}

// =======================
// MAIN API CALL (WITHOUT PROXY)
// =======================
$api_url = "http://site.api.espn.com/apis/site/v2/sports/hockey/nhl/scoreboard";

echo '<div class="debug-box">';
echo '<h4>🔍 Main Scoreboard API Request</h4>';
echo '<strong>API URL:</strong> <a href="' . $api_url . '" target="_blank">' . $api_url . '</a><br>';
echo '<strong>Request Time:</strong> ' . date('g:i:s A') . '<br>';

$maxRetries = 3;
$success = false;
$lastError = '';

for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
    echo "Attempt $attempt... ";

    $curl = curl_init($api_url);
    curl_setopt_array($curl, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => "Mozilla/5.0 (compatible; GlanceBot/1.0)"
        // NO PROXY
    ]);

    $response = curl_exec($curl);
    $lastError = curl_error($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    if ($response && $httpCode === 200 && strlen($response) > 100) {
        $success = true;
        echo "✓ Success<br>";
        break;
    }

    echo "✗ Failed (HTTP $httpCode)<br>";
    if ($attempt < $maxRetries) sleep(1);
}

if (!$success) {
    die("<div class='error-box'>❌ API call failed after $maxRetries attempts.<br>Last error: "
        . htmlspecialchars($lastError) . "</div></div></body></html>");
}

$data = json_decode($response, true);
if (empty($data)) {
    die('<div class="error-box">Failed to parse API response (invalid JSON)</div></div></body></html>');
}

echo '</div>';

// =======================
// PROCESS GAMES AND DETECT SCORE CHANGES
// =======================

$events = $data['events'] ?? [];
$games_with_score_changes = [];
$games_processed = [];

echo '<div class="score-change-card">';
echo '<div class="score-change-header">';
echo '<h2>📊 Score Comparison & Change Detection</h2>';
echo '</div>';
echo '<div style="padding: 20px;">';

echo '<table class="score-comparison-table">';
echo '<thead><tr><th>Game #</th><th>Teams</th><th>Old Away</th><th>New Away</th><th>Old Home</th><th>New Home</th><th>Change?</th></tr></thead>';
echo '<tbody>';

foreach ($events as $index => $event) {
    $game_num = $index + 1;
    $gameData = extractGameData($event, $game_num);

    // Get cached data for this game
    $cached_data = $cached_database_state[$game_num] ?? null;

    // Store new scores
    $new_away_score = intval($gameData['team2']['score']);
    $new_home_score = intval($gameData['team1']['score']);

    // Check for score change
    $score_changed = false;
    if ($cached_data && ($gameData['game_status'] == 'IN_SESSION' || $gameData['game_status'] == 'HALF_TIME')) {
        if ($cached_data['away_score'] != $new_away_score || $cached_data['home_score'] != $new_home_score) {
            $score_changed = true;
            // Use eventID from cache if available, otherwise use from API
            $event_id_to_use = !empty($cached_data['eventID']) ? $cached_data['eventID'] : $gameData['event_id'];
            $games_with_score_changes[$game_num] = [
                'event_id' => $event_id_to_use,
                'old_data' => $cached_data,
                'new_scores' => ['away' => $new_away_score, 'home' => $new_home_score],
                'teams' => $gameData['team2']['abbreviation'] . ' @ ' . $gameData['team1']['abbreviation']
            ];
        }

        echo '<tr>';
        echo '<td>' . $game_num . '</td>';
        echo '<td>' . $gameData['team2']['abbreviation'] . ' @ ' . $gameData['team1']['abbreviation'] . '</td>';
        echo '<td>' . $cached_data['away_score'] . '</td>';
        echo '<td class="' . ($cached_data['away_score'] != $new_away_score ? 'score-changed' : '') . '">' . $new_away_score . '</td>';
        echo '<td>' . $cached_data['home_score'] . '</td>';
        echo '<td class="' . ($cached_data['home_score'] != $new_home_score ? 'score-changed' : '') . '">' . $new_home_score . '</td>';
        echo '<td>' . ($score_changed ? '<span style="color: red; font-weight: bold;">🚨 GOAL!</span>' : '✓ No change') . '</td>';
        echo '</tr>';
    }

    $gameData['score_changed'] = $score_changed;
    $gameData['cached_data'] = $cached_data;
    $games_processed[] = $gameData;
}

echo '</tbody></table>';

if (count($games_with_score_changes) > 0) {
    echo '<div class="score-change-detected">';
    echo '<h3>🚨 ' . count($games_with_score_changes) . ' Game(s) with Score Changes Detected!</h3>';
    echo 'Summary API will be called ONLY for these games to get goal details.<br>';
    echo '</div>';
} else {
    echo '<div class="info-box">No score changes detected - Summary API calls will be SKIPPED</div>';
}

echo '</div></div>';

// =======================
// CALL SUMMARY API ONLY FOR GAMES WITH SCORE CHANGES
// =======================

if (count($games_with_score_changes) > 0) {
    echo '<div class="goal-detection-card">';
    echo '<div class="goal-detection-header">';
    echo '<h2>🎯 Fetching Goal Details (Only for Score Changes)</h2>';
    echo '</div>';
    echo '<div style="padding: 20px;">';

    foreach ($games_with_score_changes as $game_num => $change_info) {
        // Call checkRecentGoalsOnScoreChange to get goal details
        $goal_details = checkRecentGoalsOnScoreChange(
            $change_info['event_id'],
            $con,
            $current_timestamp,
            $game_num,
            $change_info['old_data'],
            $change_info['new_scores']
        );

        // Attach goal details to the game data
        if ($goal_details) {
            foreach ($games_processed as &$game) {
                if ($game['game_num'] == $game_num) {
                    $game['goal_details'] = $goal_details;
                    break;
                }
            }
            unset($game); // Break the reference
        }
    }

    echo '</div></div>';
}

// =======================
// UPDATE DATABASE WITH PROTECTION FOR GOAL ENTRIES
// =======================

echo '<div class="update-check-card">';
echo '<div class="update-check-header" style="background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%);">';
echo '<h2>✅ Database Updates</h2>';
echo '</div>';
echo '<div style="padding: 20px;">';

$updated_count = 0;
$goal_detected_count = 0;
$protected_goals = 0;

foreach ($games_processed as $game) {
    // Skip games that don't need updating
    if ($game['game_status'] != 'IN_SESSION' &&
        $game['game_status'] != 'HALF_TIME' &&
        !isset($games_with_score_changes[$game['game_num']])) {
        continue;
    }

    $game_num = $game['game_num'];
    $cached_data = $game['cached_data'];

    // PROTECTION: Check if there's already a GOAL in the database
    $has_existing_goal = false;
    if ($cached_data && $cached_data['scoreValue'] == 'GOAL') {
        $has_existing_goal = true;
        $protected_goals++;
        echo '<div class="info-box" style="background: #fff3cd; border-color: #ffc107;">
            🛡️ Game ' . $game_num . ' has GOAL protected - will not update scoreValue, lastPlay, or player info
        </div>';
    }

    // Build update query
    $update_data = [
        'eventID' => mysqli_real_escape_string($con, $game['event_id']),
        'Clock' => mysqli_real_escape_string($con, $game['clock']),
        'Period' => intval($game['period']),
        'Game_Played' => mysqli_real_escape_string($con, $game['game_status']),
        'Game_Time' => mysqli_real_escape_string($con, $game['game_time']),
        'Game_Date' => mysqli_real_escape_string($con, $game['game_date']),
        'broadcast' => mysqli_real_escape_string($con, $game['broadcast']),
        'Team_1_Name' => mysqli_real_escape_string($con, $game['team1']['name']),
        'Team_2_Name' => mysqli_real_escape_string($con, $game['team2']['name']),
        'Team_1_Record' => mysqli_real_escape_string($con, $game['team1']['record']),
        'Team_2_Record' => mysqli_real_escape_string($con, $game['team2']['record']),
        'Team_1_Score' => intval($game['team1']['score']),
        'Team_2_Score' => intval($game['team2']['score']),
        'Team_1_Location' => mysqli_real_escape_string($con, $game['team1']['location']),
        'Team_2_Location' => mysqli_real_escape_string($con, $game['team2']['location']),
        'Team_1_Abbrev' => mysqli_real_escape_string($con, $game['team1']['abbreviation']),
        'Team_2_Abbrev' => mysqli_real_escape_string($con, $game['team2']['abbreviation']),
        'Team_1_Color' => '"' . mysqli_real_escape_string($con, $game['team1']['color_rgb']) . '"',
        'Team_2_Color' => '"' . mysqli_real_escape_string($con, $game['team2']['color_rgb']) . '"',
        'lastUpdatedTime' => $current_datetime
    ];

    // Extract RGB values
    $rgb1 = explode(',', $game['team1']['color_rgb']);
    $update_data['Team_1_Color_R'] = $rgb1[0] ?? '255';
    $update_data['Team_1_Color_G'] = $rgb1[1] ?? '255';
    $update_data['Team_1_Color_B'] = $rgb1[2] ?? '255';

    $rgb2 = explode(',', $game['team2']['color_rgb']);
    $update_data['Team_2_Color_R'] = $rgb2[0] ?? '255';
    $update_data['Team_2_Color_G'] = $rgb2[1] ?? '255';
    $update_data['Team_2_Color_B'] = $rgb2[2] ?? '255';

    // Only update scoreValue and player info if:
    // 1. There's NO existing GOAL in the database
    // 2. We have new goal details to add
    if (!$has_existing_goal) {
        if (isset($game['goal_details']) && $game['goal_details']) {
            // New goal detected from score change
            $goal_data = $game['goal_details'];
            $update_data['scoreValue'] = 'GOAL';
            $update_data['lastPlay'] = mysqli_real_escape_string($con, $goal_data['lastPlay']);
            $update_data['lastPlayerName'] = mysqli_real_escape_string($con, $goal_data['lastPlayerName']);
            $update_data['lastPlayerID'] = mysqli_real_escape_string($con, $goal_data['lastPlayerID']);
            $update_data['driveDesc'] = mysqli_real_escape_string($con,
                'Period ' . $goal_data['period'] . ' - ' . $goal_data['clock']);
            $goal_detected_count++;
        } else {
            // Regular update - no goal
            $update_data['scoreValue'] = 'NO-SCORE';
            $update_data['lastPlay'] = mysqli_real_escape_string($con, $game['last_play']);
            $update_data['lastPlayerName'] = mysqli_real_escape_string($con, $game['last_player']['name'] ?? '');
            $update_data['lastPlayerID'] = mysqli_real_escape_string($con, $game['last_player']['id'] ?? '');
            $update_data['driveDesc'] = mysqli_real_escape_string($con, $game['drive_desc']);
        }
    }

    // Build and execute UPDATE query
    $set_clauses = [];
    foreach ($update_data as $field => $value) {
        $set_clauses[] = "$field='$value'";
    }

    $query = "UPDATE sports_list SET " . implode(', ', $set_clauses) .
             " WHERE Game = " . $game_num . " AND Conference = 'NHL'";

    $result = mysqli_query($con, $query);

    if ($result) {
        $updated_count++;
        echo '✅ Game ' . $game_num . ' updated successfully';
        if (isset($game['goal_details']) && !$has_existing_goal) {
            echo ' - <span class="goal-indicator">GOAL ADDED</span>';
        } elseif ($has_existing_goal) {
            echo ' - <span style="background: #ffc107; color: #333; padding: 2px 6px; border-radius: 3px;">GOAL PROTECTED</span>';
        }
        echo '<br>';
    } else {
        echo '❌ Game ' . $game_num . ' update failed: ' . mysqli_error($con) . '<br>';
    }
}

echo '</div></div>';

// =======================
// EXECUTION SUMMARY
// =======================

$execution_time = round(microtime(true) - $start_time, 2);
$summary_api_calls = count($games_with_score_changes);
$api_efficiency = $active_games > 0
    ? round((1 - ($summary_api_calls / $active_games)) * 100, 1)
    : 100;

echo '<div class="summary">
    <h2>📊 Execution Summary</h2>
    <p><strong>Execution Time:</strong> ' . $execution_time . ' seconds</p>
    <p><strong>Games Cached:</strong> ' . count($cached_database_state) . '</p>
    <p><strong>Active Games Monitored:</strong> ' . $active_games . '</p>
    <p><strong>Games Updated:</strong> ' . $updated_count . '</p>
    <p><strong>Goals Detected:</strong> ' . $goal_detected_count . '</p>
    <p><strong>Goals Protected:</strong> ' . $protected_goals . '</p>
    <p><strong>Main API Calls:</strong> 1</p>
    <p><strong>Summary API Calls:</strong> ' . $summary_api_calls . ' (only for score changes)</p>
    <p><strong>API Efficiency:</strong> ' . $api_efficiency . '% reduction in Summary API calls</p>
    <p><strong>Database Updated:</strong> ' . $time_updated . '</p>
</div>';

// Call external APIs
$url = "https://gscroll.us/Scrolling/App_Output/API/Games/NHL_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_exec($ch);
curl_close($ch);

$url = "https://gscroll.us/Scrolling/App_Output/API/Games/Scoring_ALERT.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_exec($ch);
curl_close($ch);

// Update API timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NHL'";
mysqli_query($con, $sql);

// Close database connection
mysqli_close($con);

echo '</div></body></html>';

?>