<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once '../config.php';

// Remaining F1 2025 races with UTC times
$remaining_races_2025 = [
    ['name' => 'Italian Grand Prix', 'date' => '2025-09-07 14:00:00'],
    ['name' => 'Azerbaijan Grand Prix', 'date' => '2025-09-21 12:00:00'],
    ['name' => 'Singapore Grand Prix', 'date' => '2025-10-05 13:00:00'],
    ['name' => 'United States Grand Prix', 'date' => '2025-10-19 20:00:00'],
    ['name' => 'Mexico City Grand Prix', 'date' => '2025-10-26 20:00:00'],
    ['name' => 'Brazilian Grand Prix', 'date' => '2025-11-09 17:00:00'],
    ['name' => 'Las Vegas Grand Prix', 'date' => '2025-11-23 04:00:00'],
    ['name' => 'Qatar Grand Prix', 'date' => '2025-11-30 16:00:00'],
    ['name' => 'Abu Dhabi Grand Prix', 'date' => '2025-12-07 13:00:00']
];

// Function to check if we should perform API call
function shouldPerformAPICall($con, $remaining_races) {
    $now = new DateTime("now", new DateTimeZone("UTC"));
    
    // 1. Check database for any IN_SESSION races
    $check_session_query = "SELECT COUNT(*) as count FROM race_list_F1 WHERE status LIKE '%IN_PROGRESS%' OR status LIKE '%IN_SESSION%'";
    $result = $con->query($check_session_query);
    if ($result) {
        $row = $result->fetch_assoc();
        if ($row['count'] > 0) {
            echo "<p style='color:green;'>Found IN_SESSION/IN_PROGRESS race in database. Proceeding with API call.</p>";
            return true;
        }
    }
    
    // 2. Check if any race is starting within 5 minutes
    foreach ($remaining_races as $race) {
        $race_time = new DateTime($race['date'], new DateTimeZone("UTC"));
        $time_diff = ($race_time->getTimestamp() - $now->getTimestamp()) / 60; // Difference in minutes
        
        // Check if race is within 5 minutes of starting (or up to 30 minutes after start)
        if ($time_diff >= -30 && $time_diff <= 5) {
            echo "<p style='color:orange;'>Race '{$race['name']}' is within 5 minutes of start time (or recently started). Proceeding with API call.</p>";
            return true;
        }
    }
    
    // 3. Check if it's race weekend (Friday to Sunday) for any upcoming race
    foreach ($remaining_races as $race) {
        $race_time = new DateTime($race['date'], new DateTimeZone("UTC"));
        $race_weekend_start = clone $race_time;
        $race_weekend_start->modify('-2 days'); // Friday
        $race_weekend_end = clone $race_time;
        $race_weekend_end->modify('+1 day'); // Day after race
        
        if ($now >= $race_weekend_start && $now <= $race_weekend_end) {
            echo "<p style='color:blue;'>Currently in race weekend for '{$race['name']}'. Proceeding with API call.</p>";
            return true;
        }
    }
    
    echo "<p style='color:red;'>No active sessions, no races starting soon, and not in race weekend. Skipping API call.</p>";
    return false;
}

function strip_accents($str) {
    return iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $str);
}

// Connect to database
$con = new mysqli($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if ($con->connect_error) {
    die("Connection failed: " . $con->connect_error);
}

// Check if we should perform the API call
if (!shouldPerformAPICall($con, $remaining_races_2025)) {
    // Log the check without performing API call
    date_default_timezone_set('America/New_York');
    $time_checked = date('m/d/Y | g:iA');
    echo "<p>Last checked: $time_checked - No API call needed.</p>";
    

    
    $con->close();
    exit();
}

// ============ ORIGINAL CODE CONTINUES HERE (only executes if check passes) ============

// 1. Update API_DB timestamp
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$update_stmt = $con->prepare("UPDATE API_DB SET TIME_UPDATED = ? WHERE API = 'F1'");
$update_stmt->bind_param("s", $time_updated);
$update_stmt->execute();
$update_stmt->close();

// 2. Fetch API data
$scoreboard_url = "https://site.api.espn.com/apis/site/v2/sports/racing/f1/scoreboard";
$curl = curl_init($scoreboard_url);
curl_setopt_array($curl, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
]);
$scoreboard_response = curl_exec($curl);
if (curl_errno($curl)) {
    die("cURL error: " . curl_error($curl));
}
curl_close($curl);
$scoreboard_data = json_decode($scoreboard_response, true);
if (!$scoreboard_data) {
    die("Error decoding JSON.");
}

// 3. Process events based on current date and today's race logic
$now = new DateTime("now", new DateTimeZone("UTC"));
$today = $now->format('Y-m-d');
$events = $scoreboard_data['events'] ?? [];
$calendarEvents = $scoreboard_data['leagues'][0]['calendar'] ?? [];
$selectedEvent = null;
$competitionsToProcess = [];
$latestCompletedRace = null;

// NEW: If today's race is active (season type 2 and day date is today), try to pick an event with an IN_PROGRESS competition.
if (isset($scoreboard_data['day']['date']) && $scoreboard_data['day']['date'] === $today 
    && isset($scoreboard_data['season']['type']) && $scoreboard_data['season']['type'] == 2) {
    foreach ($events as $event) {
         foreach ($event['competitions'] as $competition) {
             $statusName = strtoupper($competition['status']['type']['name'] ?? '');
             if (strpos($statusName, 'IN_PROGRESS') !== false) {
                 $selectedEvent = $event;
                 $competitionsToProcess = $event['competitions'];
                 break 2;
             }
         }
    }
    if (!$selectedEvent && count($events) > 0) {
         $selectedEvent = $events[0];
         $competitionsToProcess = $selectedEvent['competitions'];
         echo "<p style='color:blue;'>Using default today's event: " . htmlspecialchars($selectedEvent['name']) . "</p>";
    }
}

// Fallback to existing logic if today's race was not selected
if (!$selectedEvent) {
    foreach ($events as $event) {
        $competitions = $event['competitions'] ?? [];
        foreach ($competitions as $competition) {
            $date = new DateTime($competition['date'] ?? '', new DateTimeZone("UTC"));
            $isMainRace = ($competition['type']['id'] === "3");
            $completed = $competition['status']['type']['completed'] ?? false;
            
            // Track latest completed race
            if ($completed && $date < $now && $isMainRace) {
                if (!$latestCompletedRace || $date > $latestCompletedRace['date']) {
                    $latestCompletedRace = [
                        'event' => $event,
                        'competition' => $competition,
                        'date' => $date
                    ];
                }
            }
        }
    }
    
    // Determine event to focus on from calendar events
    $twoDaysFromNow = (clone $now)->modify('+2 days');
    foreach ($calendarEvents as $calEvent) {
        $startDate = new DateTime($calEvent['startDate'] ?? '', new DateTimeZone("UTC"));
        $endDate = new DateTime($calEvent['endDate'] ?? '', new DateTimeZone("UTC"));
        
        if ($startDate <= $twoDaysFromNow && $endDate >= $now) {
            foreach ($events as $event) {
                if ($event['id'] === str_replace('http://sports.core.api.espn.pvt/v2/sports/racing/leagues/f1/events/', '', $calEvent['event']['$ref'])) {
                    $selectedEvent = $event;
                    $competitionsToProcess = $event['competitions'];
                    break 2;
                }
            }
        }
    }
    
    // Fallback to latest completed race if still not found
    if (!$selectedEvent && $latestCompletedRace) {
        $selectedEvent = $latestCompletedRace['event'];
        $competitionsToProcess = [$latestCompletedRace['competition']];
        echo "<p style='color:green;'>Using latest completed race: " . htmlspecialchars($selectedEvent['name']) . "</p>";
    } elseif (!$selectedEvent) {
        $upcoming = array_filter($calendarEvents, fn($e) => new DateTime($e['startDate'], new DateTimeZone("UTC")) > $now);
        if (!empty($upcoming)) {
            usort($upcoming, fn($a, $b) => strtotime($a['startDate']) - strtotime($b['startDate']));
            $selectedEvent = $events[array_search($upcoming[0]['event']['$ref'], array_column($events, '$ref'))] ?? null;
            $competitionsToProcess = $selectedEvent['competitions'] ?? [];
            echo "<p style='color:red;'>No completed race or upcoming within 2 days. Using next event: " . htmlspecialchars($upcoming[0]['label']) . "</p>";
        } else {
            die("No suitable events found.");
        }
    }
}

// Build a combined team mapping for the selected event (from all competitions)
$teamMapping = [];
foreach ($selectedEvent['competitions'] as $comp) {
    foreach ($comp['competitors'] as $c) {
         if (isset($c['team']['displayName'])) {
             $teamMapping[$c['id']] = $c['team']['displayName'];
         }
    }
}

// FIXED: Choose a single competition to process with this priority:
// 1. If a competition with type.id "3" (final race) exists, use that (regardless of status)
// 2. Else, if a competition is IN_PROGRESS, use that.
// 3. Otherwise, default to the first competition.
$competitionToProcess = null;
foreach ($competitionsToProcess as $comp) {
    if ($comp['type']['id'] === "3") {
         $competitionToProcess = $comp;
         break;
    }
}
if (!$competitionToProcess) {
    foreach ($competitionsToProcess as $comp) {
         $stat = strtoupper($comp['status']['type']['name'] ?? '');
         if (strpos($stat, 'IN_PROGRESS') !== false) {
              $competitionToProcess = $comp;
              break;
         }
    }
}
if (!$competitionToProcess && count($competitionsToProcess) > 0) {
    $competitionToProcess = $competitionsToProcess[0];
}

// Extract event details
$eventId = $selectedEvent['id'] ?? '';
$raceTitle = strip_accents($selectedEvent['name'] ?? 'Unknown Race');
$raceTimeUTC = $selectedEvent['date'] ?? '';
$dt = new DateTime($raceTimeUTC, new DateTimeZone("UTC"));
$dt->setTimezone(new DateTimeZone('America/New_York'));
$raceTimeLocal = $dt->format('Y-m-d H:i:s');
$countdown = ($dt > $now) ? (($dt->getTimestamp() - $now->getTimestamp()) / 86400) . " days remaining" : "Event is in the past";

// 5. Process the chosen competition and update the database (limit to 16 racers)
// Create table if it does not exist and add the new lap column (as VARCHAR)
$con->query("CREATE TABLE IF NOT EXISTS race_list_F1 (
    unique_key INT,
    rankValue INT,
    race_time DATETIME,
    race_title VARCHAR(255),
    status VARCHAR(50),
    event_in VARCHAR(50),
    racer_name VARCHAR(100),
    athlete_id INT,
    race_id INT,
    League VARCHAR(10),
    car_type VARCHAR(50),
    session_type VARCHAR(10),
    lap VARCHAR(10) DEFAULT '',
    racer_number INT DEFAULT 0,
    PRIMARY KEY (athlete_id, session_type, event_in)
)");

// Instead of deleting only the current event rows, clear the entire table so that only the new 16 rows remain.
$con->query("TRUNCATE TABLE race_list_F1");

// Get competitors from the selected competition, sort and limit to 16
$status = strtoupper($competitionToProcess['status']['type']['name'] ?? 'SCHEDULED');
$sessionType = $competitionToProcess['type']['id'] ?? 'Unknown';
$competitors = $competitionToProcess['competitors'] ?? [];
usort($competitors, fn($a, $b) => ($a['order'] ?? 999) - ($b['order'] ?? 999));
$competitors = array_slice($competitors, 0, 16);  // Limit to 16 racers

// Add team info for each competitor
foreach ($competitors as &$comp) {
    $comp['team'] = ['displayName' => $teamMapping[$comp['id']] ?? 'TBD'];
}
unset($comp);

// Determine the lap value: if the session is "3" or the competition is in progress, then use the status period as lap.
if ($sessionType === "3" || $status === "STATUS_IN_PROGRESS") {
    $lap = $competitionToProcess['status']['period'] ?? '';
} else {
    $lap = '';
}

$insert_sql = "INSERT INTO race_list_F1 (unique_key, rankValue, race_time, race_title, status, event_in, racer_name, athlete_id, race_id, League, car_type, session_type, lap) 
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
               ON DUPLICATE KEY UPDATE rankValue = VALUES(rankValue), race_time = VALUES(race_time), status = VALUES(status), car_type = VALUES(car_type), lap = VALUES(lap)";
$stmt = $con->prepare($insert_sql);

$counter = 1;
$con->begin_transaction();
foreach ($competitors as $comp) {
    $rankValue = $comp['order'] ?? $counter;
    $racer_name = strip_accents($comp['athlete']['shortName'] ?? $comp['athlete']['fullName'] ?? '');
    $athlete_id = $comp['id'] ?? -$counter;
    $car_type = strip_accents($comp['team']['displayName'] ?? 'TBD');
    $race_id = $counter;
    $League = 'F1';

    // Bind 13 parameters. Data types: i (int), i (int), s (string), s, s, s, s, i, i, s, s, s, s.
    $stmt->bind_param("iissssssiisss", 
        $counter, 
        $rankValue, 
        $raceTimeLocal, 
        $raceTitle, 
        $status, 
        $eventId,
        $racer_name, 
        $athlete_id, 
        $race_id, 
        $League, 
        $car_type, 
        $sessionType,
        $lap
    );
    
    if (!$stmt->execute()) {
        echo "<p>Error inserting row $counter: " . $stmt->error . "</p>";
    }
    $counter++;
}
$con->commit();
$stmt->close();

// 6. Call external scripts
$curl = curl_init('https://gscroll.us/Scrolling/Headshots/getF1Headshots.php');
curl_setopt_array($curl, [CURLOPT_RETURNTRANSFER => true, CURLOPT_FOLLOWLOCATION => true, CURLOPT_TIMEOUT => 30]);
echo "<pre>Headshots API response: " . curl_exec($curl) . "</pre>";
curl_close($curl);

$curl = curl_init('https://gscroll.us/Scrolling/App_Output/API/F1/wrapperF1.php');
curl_setopt_array($curl, [CURLOPT_RETURNTRANSFER => true, CURLOPT_FOLLOWLOCATION => true, CURLOPT_TIMEOUT => 30]);
echo "<pre>Image API response: " . curl_exec($curl) . "</pre>";
curl_close($curl);

// Log successful API call
echo "<p style='color:green;'>API call completed successfully at $time_updated</p>";
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>F1 Race Data Debug</title>
    <style>
        body { font-family: Arial, sans-serif; }
        table { border-collapse: collapse; width: 90%; margin: 20px auto; }
        th, td { border: 1px solid #333; padding: 8px; text-align: center; }
        th { background-color: #f0f0f0; }
        pre { background: #f5f5f5; padding: 10px; margin: 10px; }
    </style>
</head>
<body>
    <h1>F1 Debug Information</h1>
    <h2>Selected Race: <?= htmlspecialchars($raceTitle) ?></h2>
    <p>Event ID: <?= htmlspecialchars($eventId) ?> | Time: <?= htmlspecialchars($raceTimeLocal) ?> | Countdown: <?= htmlspecialchars($countdown) ?></p>
    <h2>Race Data (Max 16 Racers)</h2>
    <table>
        <tr>
            <th>unique_key</th>
            <th>rankValue</th>
            <th>race_time</th>
            <th>race_title</th>
            <th>racer_name</th>
            <th>race_id</th>
            <th>League</th>
            <th>status</th>
            <th>athlete_id</th>
            <th>event_in</th>
            <th>car_type</th>
            <th>session_type</th>
            <th>lap</th>
            <th>racer_number</th>
        </tr>
        <?php
        $result = $con->query("SELECT * FROM race_list_F1 WHERE event_in = '$eventId' ORDER BY session_type, rankValue");
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['unique_key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['rankValue']) . "</td>";
            echo "<td>" . htmlspecialchars($row['race_time']) . "</td>";
            echo "<td>" . htmlspecialchars($row['race_title']) . "</td>";
            echo "<td>" . htmlspecialchars($row['racer_name']) . "</td>";
            echo "<td>" . htmlspecialchars($row['race_id']) . "</td>";
            echo "<td>" . htmlspecialchars($row['League']) . "</td>";
            echo "<td>" . htmlspecialchars($row['status']) . "</td>";
            echo "<td>" . htmlspecialchars($row['athlete_id']) . "</td>";
            echo "<td>" . htmlspecialchars($row['event_in']) . "</td>";
            echo "<td>" . htmlspecialchars($row['car_type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['session_type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['lap']) . "</td>";
            echo "<td>" . htmlspecialchars($row['racer_number']) . "</td>";
            echo "</tr>";
        }
        $con->close();
        ?>
    </table>
</body>
</html>