<?php
//************************************************//
// FIXED VERSION - Using proper JSON parsing
// Date: 11/03/2025
//************************************************//

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../config.php'; // Include the configuration file

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone and format it
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

// Update the API timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'MMA'";
if (mysqli_query($con, $sql)) {
    if (mysqli_affected_rows($con) > 0) {
        echo "Record updated successfully for API: MMA<br>";
    }
} else {
    echo "Error executing query: " . mysqli_error($con) . "<br>";
}

// Function to fetch API data with odds
function fetch_api_odds($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return null;
}

// Function to fetch last play details for finished fights
function fetch_fight_details($fight_event_ID, $compt_event_ID) {
    $api_call = "https://sports.core.api.espn.com/v2/sports/mma/leagues/ufc/events/" . $fight_event_ID. "/competitions/" . $compt_event_ID. "/status";
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_call,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $result = curl_exec($curl);
    curl_close($curl);
    
    if ($result) {
        $data = json_decode($result, true);
        if (isset($data['type']['shortDisplayName'])) {
            return str_replace('/', '-', $data['type']['shortDisplayName']);
        }
    }
    return '';
}

// Main API call to get all MMA data
$api_call = "https://site.api.espn.com/apis/site/v2/sports/mma/ufc/scoreboard";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error #:" . $err);
}

// Parse JSON properly
$data = json_decode($result, true);

if (!$data || !isset($data['events'])) {
    die("Failed to parse API response or no events found");
}

// Clean up database
$query = "DELETE FROM fight_list WHERE fight_id >= 0";
mysqli_query($con, $query);

// Prepare insert for up to 30 fights
$values = [];
for ($i = 0; $i <= 30; $i++) {
    $values[] = "($i)";
}
$query = "INSERT INTO fight_list (fight_id) VALUES " . implode(", ", $values);
mysqli_query($con, $query);

// HTML debug output start
$html_output = "<!DOCTYPE html>
<html>
<head>
    <title>MMA Fight Debug Output</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        table { border-collapse: collapse; width: 100%; margin-bottom: 30px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #4CAF50; color: white; }
        tr:nth-child(even) { background-color: #f2f2f2; }
        .found { background-color: #90EE90 !important; }
        .missing { background-color: #FFB6C1 !important; }
        .debug { background-color: #f0f0f0; padding: 10px; margin: 10px 0; border: 1px solid #ccc; }
        pre { background-color: #f8f8f8; padding: 10px; overflow-x: auto; }
    </style>
</head>
<body>
    <h1>MMA Fight Data Debug - " . $time_updated . "</h1>";

// Track all fights for debugging
$all_fights = [];
$fight_counter = 0;

// Process each event
foreach ($data['events'] as $event) {
    if (!isset($event['competitions'])) {
        continue;
    }
    
    $event_name = isset($event['name']) ? $event['name'] : 'Unknown Event';
    $event_name = str_replace("UFC Fight Night: ", "", $event_name);
    
    $html_output .= "<h2>Event: $event_name</h2>";
    $html_output .= "<table>";
    $html_output .= "<tr><th>Fight #</th><th>Status</th><th>Weight Class</th><th>Fighter 1</th><th>Record 1</th><th>ML1</th><th>Fighter 2</th><th>Record 2</th><th>ML2</th><th>Start Time</th><th>Round/Time</th><th>Result</th></tr>";
    
    // Process each competition (fight) in the event
    foreach ($event['competitions'] as $competition) {
        $fight_counter++;
        
        // Extract IDs
        $fight_event_ID = isset($event['id']) ? $event['id'] : '';
        $compt_event_ID = isset($competition['id']) ? $competition['id'] : '';
        
        // Extract status
        $status = isset($competition['status']['type']['name']) ? $competition['status']['type']['name'] : '';
        $round_info = '';
        $time_info = '';
        $lastPlay = '';
        
        // Handle in-progress fights
        if ($status == "STATUS_IN_PROGRESS_2" && isset($competition['status']['type']['shortDetail'])) {
            $details = $competition['status']['type']['shortDetail'];
            if (strpos($details, ', ') !== false) {
                list($round_info, $time_info) = explode(', ', $details);
            }
        }
        
        // Handle finished fights
        if ($status == "STATUS_FINAL") {
            $lastPlay = fetch_fight_details($fight_event_ID, $compt_event_ID);
        }
        
        // Extract weight class
        $weight_class = '';
        if (isset($competition['notes'][0]['text'])) {
            $weight_class = $competition['notes'][0]['text'];
        }
        
        // Initialize moneyline variables
        $ML1 = 'NULL';
        $ML2 = 'NULL';
        
        // Fetch odds for scheduled fights
        if ($status == "STATUS_SCHEDULED") {
            $odds_url = "http://sports.core.api.espn.com/v2/sports/mma/leagues/ufc/events/{$fight_event_ID}/competitions/{$compt_event_ID}/odds?lang=en&region=us";
            $odds_data = fetch_api_odds($odds_url);
            
            if ($odds_data && isset($odds_data['items'][0])) {
                $odds = $odds_data['items'][0];
                
                if (isset($odds['homeAthleteOdds']['moneyLine'])) {
                    $ML1 = $odds['homeAthleteOdds']['moneyLine'];
                }
                if (isset($odds['awayAthleteOdds']['moneyLine'])) {
                    $ML2 = $odds['awayAthleteOdds']['moneyLine'];
                }
            }
        }
        
        // Process competitors
        $fighter_data = [
            1 => ['name' => '', 'record' => '', 'winner' => false, 'order' => 0],
            2 => ['name' => '', 'record' => '', 'winner' => false, 'order' => 0]
        ];
        
        if (isset($competition['competitors']) && is_array($competition['competitors'])) {
            foreach ($competition['competitors'] as $idx => $competitor) {
                // Determine which fighter position based on order
                $order = isset($competitor['order']) ? $competitor['order'] : ($idx + 1);
                $fighter_num = $order == 1 ? 1 : 2;
                
                // Extract fighter name
                if (isset($competitor['athlete']['shortName'])) {
                    $name = $competitor['athlete']['shortName'];
                    
                    // Process names with dots (like initials)
                    if (strpos($name, '.') !== false) {
                        list($initial, $rest) = explode('.', $name, 2);
                        $initial = trim($initial);
                        $rest = trim($rest);
                        
                        if (strpos($rest, ' ') !== false) {
                            $words = explode(' ', $rest);
                            $rest = end($words);
                        }
                        $name = $initial . '.' . $rest;
                    }
                    
                    $fighter_data[$fighter_num]['name'] = $name;
                }
                
                // Extract record
                if (isset($competitor['records'][0]['summary'])) {
                    $fighter_data[$fighter_num]['record'] = $competitor['records'][0]['summary'];
                }
                
                // Extract winner status
                $fighter_data[$fighter_num]['winner'] = isset($competitor['winner']) ? $competitor['winner'] : false;
                $fighter_data[$fighter_num]['order'] = $order;
            }
        }
        
        // Extract start time
        $start_time = isset($competition['startDate']) ? substr($competition['startDate'], 0, 20) : '';
        
        // Clean strings for database
        $clean_data = [
            'status' => mysqli_real_escape_string($con, $status),
            'weight_class' => mysqli_real_escape_string($con, $weight_class),
            'fighter1_name' => mysqli_real_escape_string($con, preg_replace('/[^\x20-\x7E]/', '', $fighter_data[1]['name'])),
            'fighter1_winner' => $fighter_data[1]['winner'] ? 'true' : 'false',
            'fighter1_record' => mysqli_real_escape_string($con, $fighter_data[1]['record']),
            'fighter2_name' => mysqli_real_escape_string($con, preg_replace('/[^\x20-\x7E]/', '', $fighter_data[2]['name'])),
            'fighter2_winner' => $fighter_data[2]['winner'] ? 'true' : 'false',
            'fighter2_record' => mysqli_real_escape_string($con, $fighter_data[2]['record']),
            'start_time' => mysqli_real_escape_string($con, $start_time),
            'event_name' => mysqli_real_escape_string($con, $event_name),
            'round' => mysqli_real_escape_string($con, $round_info),
            'time' => mysqli_real_escape_string($con, $time_info),
            'lastPlay' => mysqli_real_escape_string($con, $lastPlay)
        ];
        
        // Store fight data
        $all_fights[] = [
            'fighter1' => $fighter_data[1]['name'],
            'fighter2' => $fighter_data[2]['name'],
            'status' => $status,
            'weight_class' => $weight_class
        ];
        
        // Determine row class based on whether this is the missing fight
        $row_class = '';
        if (strpos($fighter_data[1]['name'], 'McVey') !== false || strpos($fighter_data[2]['name'], 'Valentin') !== false) {
            $row_class = 'found';
        }
        
        // Add to HTML table
        $html_output .= "<tr class='$row_class'>
            <td>$fight_counter</td>
            <td>{$clean_data['status']}</td>
            <td>{$clean_data['weight_class']}</td>
            <td>{$clean_data['fighter1_name']}</td>
            <td>{$clean_data['fighter1_record']}</td>
            <td>$ML1</td>
            <td>{$clean_data['fighter2_name']}</td>
            <td>{$clean_data['fighter2_record']}</td>
            <td>$ML2</td>
            <td>{$clean_data['start_time']}</td>
            <td>{$clean_data['round']} {$clean_data['time']}</td>
            <td>{$clean_data['lastPlay']}</td>
        </tr>";
        
        // Update database - handle main event (fight_id 0) and regular fights
        if ($fight_counter == count($event['competitions'])) {
            // This is the main event, update fight_id 0
            $query = "UPDATE fight_list SET 
                status='{$clean_data['status']}', 
                weight_class='{$clean_data['weight_class']}', 
                short_name_1='{$clean_data['fighter1_name']}', 
                winner_fighter_1='{$clean_data['fighter1_winner']}', 
                record_fighter_1='{$clean_data['fighter1_record']}', 
                short_name_2='{$clean_data['fighter2_name']}', 
                winner_fighter_2='{$clean_data['fighter2_winner']}', 
                record_fighter_2='{$clean_data['fighter2_record']}', 
                fight_start_time='{$clean_data['start_time']}', 
                FT_='{$clean_data['event_name']}', 
                round='{$clean_data['round']}', 
                time='{$clean_data['time']}', 
                lastPlay='{$clean_data['lastPlay']}', 
                ML1=$ML1, 
                ML2=$ML2 
                WHERE fight_id='0'";
            
            mysqli_query($con, $query);
        }
        
        // Update regular fight
        $query = "UPDATE fight_list SET 
            status='{$clean_data['status']}', 
            weight_class='{$clean_data['weight_class']}', 
            short_name_1='{$clean_data['fighter1_name']}', 
            winner_fighter_1='{$clean_data['fighter1_winner']}', 
            record_fighter_1='{$clean_data['fighter1_record']}', 
            short_name_2='{$clean_data['fighter2_name']}', 
            winner_fighter_2='{$clean_data['fighter2_winner']}', 
            record_fighter_2='{$clean_data['fighter2_record']}', 
            fight_start_time='{$clean_data['start_time']}', 
            FT_='{$clean_data['event_name']}', 
            round='{$clean_data['round']}', 
            time='{$clean_data['time']}', 
            lastPlay='{$clean_data['lastPlay']}', 
            ML1=$ML1, 
            ML2=$ML2 
            WHERE fight_id='$fight_counter'";
        
        mysqli_query($con, $query);
    }
    
    $html_output .= "</table>";
}

// Add debug information
$html_output .= "<div class='debug'>
    <h3>Debug Information</h3>
    <p><strong>Total fights processed:</strong> $fight_counter</p>
    <p><strong>Looking for:</strong> Robert Valentin vs Jackson McVey</p>";

// Check if we found the missing fight
$found_valentin = false;
$found_mcvey = false;
foreach ($all_fights as $fight) {
    if (strpos($fight['fighter1'], 'Valentin') !== false || strpos($fight['fighter2'], 'Valentin') !== false) {
        $found_valentin = true;
    }
    if (strpos($fight['fighter1'], 'McVey') !== false || strpos($fight['fighter2'], 'McVey') !== false) {
        $found_mcvey = true;
    }
}

if ($found_valentin && $found_mcvey) {
    $html_output .= "<p style='color: green; font-weight: bold;'>✓ Found the Valentin vs McVey fight!</p>";
} else {
    $html_output .= "<p style='color: red; font-weight: bold;'>✗ Missing: ";
    if (!$found_valentin) $html_output .= "Robert Valentin ";
    if (!$found_mcvey) $html_output .= "Jackson McVey";
    $html_output .= "</p>";
}

$html_output .= "</div>";

// Add raw JSON for debugging
$html_output .= "<div class='debug'>
    <h3>Raw JSON Structure (first event, first 2 competitions)</h3>
    <pre>";
if (isset($data['events'][0])) {
    $debug_data = [
        'event_name' => $data['events'][0]['name'],
        'event_id' => $data['events'][0]['id'],
        'competitions_count' => count($data['events'][0]['competitions']),
        'first_two_competitions' => array_slice($data['events'][0]['competitions'], 0, 2)
    ];
    $html_output .= htmlspecialchars(json_encode($debug_data, JSON_PRETTY_PRINT));
}
$html_output .= "</pre></div>";

$html_output .= "</body></html>";

// Save HTML output
file_put_contents('mma_debug_output.html', $html_output);
echo "Debug HTML saved to mma_debug_output.html<br>";

// Call the headshot script
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => 'https://gscroll.us/Scrolling/Headshots/getMMAHeadshots.php',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
]);
$response = curl_exec($curl);
curl_close($curl);
echo "Headshots script response: " . $response . "<br>";

// Call the wrapper script
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => 'https://gscroll.us/Scrolling/App_Output/API/MMA/wrapperMMA.php',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
]);
$response = curl_exec($curl);
curl_close($curl);
echo "Wrapper script response: " . $response . "<br>";

echo "Processing complete! Total fights: $fight_counter<br>";

?>