<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database connection
require_once '../config.php';
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

if (!$con) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Function to normalize fighter names to match database format
function normalize_fighter_name($name) {
    if (!$name) return $name;
    
    // If the name contains a dot, process the part after the dot
    if (strpos($name, '.') !== false) {
        // Split at the first dot into two parts
        list($initial, $rest) = explode('.', $name, 2);
        $initial = trim($initial);
        $rest = trim($rest);
        
        // If the portion after the dot contains a space, take only the last word
        if (strpos($rest, ' ') !== false) {
            $words = explode(' ', $rest);
            $rest = end($words);
        }
        
        // Reassemble the name without space after period
        $name = $initial . '.' . $rest;
    }
    
    // Remove non-ASCII characters (like accents)
    $name = preg_replace('/[^\x20-\x7E]/', '', $name);
    
    return $name;
}

// Function to make curl requests
function fetch_api($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return null;
}

// Get the scoreboard
$scoreboard_url = "https://site.api.espn.com/apis/site/v2/sports/mma/ufc/scoreboard";
$scoreboard_data = fetch_api($scoreboard_url);

if (!$scoreboard_data || !isset($scoreboard_data['events'])) {
    die("Failed to fetch scoreboard data");
}

// Debug: Uncomment to see raw structure
// echo "<pre>"; print_r($scoreboard_data); echo "</pre>"; die();

// Find the most recent event
// Logic: Get the event that either just happened (within last 24-48 hours) or is upcoming
$most_recent_event = null;
$current_time = time();

foreach ($scoreboard_data['events'] as $event) {
    if (isset($event['competitions'][0])) {
        $competition = $event['competitions'][0];
        
        // Get start date
        $start_date = isset($competition['startDate']) ? strtotime($competition['startDate']) : null;
        
        if ($start_date) {
            // If event is within last 48 hours or upcoming, consider it
            $time_diff = $current_time - $start_date;
            
            if ($time_diff >= -604800 && $time_diff <= 172800) { // Within 7 days future to 48 hours past
                if (!$most_recent_event) {
                    $most_recent_event = [
                        'competition' => $competition,
                        'event_data' => $event
                    ];
                } else {
                    $prev_start = strtotime($most_recent_event['competition']['startDate']);
                    if (abs($current_time - $start_date) < abs($current_time - $prev_start)) {
                        $most_recent_event = [
                            'competition' => $competition,
                            'event_data' => $event
                        ];
                    }
                }
            }
        }
    }
}

if (!$most_recent_event) {
    die("No recent UFC event found");
}

// Extract event ID from the $ref URL - try multiple possible locations
$event_ref = null;
if (isset($most_recent_event['competition']['event']['$ref'])) {
    $event_ref = $most_recent_event['competition']['event']['$ref'];
} elseif (isset($most_recent_event['event_data']['id'])) {
    $event_id = $most_recent_event['event_data']['id'];
} elseif (isset($most_recent_event['event_data']['uid'])) {
    // Parse from UID like "s:270~l:1~e:600056045"
    $uid = $most_recent_event['event_data']['uid'];
    if (preg_match('/~e:(\d+)/', $uid, $matches)) {
        $event_id = $matches[1];
    }
}

// If we have event_ref, extract ID from URL
if ($event_ref && !isset($event_id)) {
    preg_match('/events\/(\d+)/', $event_ref, $matches);
    $event_id = isset($matches[1]) ? $matches[1] : null;
}

if (!isset($event_id) || !$event_id) {
    // Debug output to see the actual structure
    echo "<h3>Debug: Could not extract event ID. Here's the event structure:</h3>";
    echo "<pre>";
    print_r($most_recent_event);
    echo "</pre>";
    die("Could not extract event ID from event data");
}

$event_label = isset($most_recent_event['competition']['label']) ? $most_recent_event['competition']['label'] : 'UFC Event';
$event_date = isset($most_recent_event['competition']['startDate']) ? date('F j, Y', strtotime($most_recent_event['competition']['startDate'])) : '';

// Get all competitions for this event
$event_url = "http://sports.core.api.espn.com/v2/sports/mma/leagues/ufc/events/{$event_id}?lang=en&region=us";
$event_data = fetch_api($event_url);

if (!$event_data || !isset($event_data['competitions'])) {
    die("Failed to fetch event data for event ID: {$event_id}");
}

// Get all competition IDs
$competition_ids = [];
foreach ($event_data['competitions'] as $comp) {
    if (isset($comp['$ref'])) {
        preg_match('/competitions\/(\d+)/', $comp['$ref'], $matches);
        if (isset($matches[1])) {
            $competition_ids[] = $matches[1];
        }
    }
}

// Fetch odds for each competition
$all_odds = [];
foreach ($competition_ids as $comp_id) {
    $odds_url = "http://sports.core.api.espn.com/v2/sports/mma/leagues/ufc/events/{$event_id}/competitions/{$comp_id}/odds?lang=en&region=us";
    $odds_data = fetch_api($odds_url);
    
    if ($odds_data && isset($odds_data['items']) && count($odds_data['items']) > 0) {
        // Get the first provider's odds
        $odds = $odds_data['items'][0];
        
        $ml1 = null;
        $ml2 = null;
        $fighter1_name = null;
        $fighter2_name = null;
        
        // Extract moneyline for away athlete (fighter 1)
        if (isset($odds['awayAthleteOdds']['moneyLine'])) {
            $ml1 = $odds['awayAthleteOdds']['moneyLine'];
        }
        
        // Extract moneyline for home athlete (fighter 2)
        if (isset($odds['homeAthleteOdds']['moneyLine'])) {
            $ml2 = $odds['homeAthleteOdds']['moneyLine'];
        }
        
        // Get fighter 1 name (away athlete)
        if (isset($odds['awayAthleteOdds']['athlete']['$ref'])) {
            $athlete_data = fetch_api($odds['awayAthleteOdds']['athlete']['$ref']);
            if ($athlete_data && isset($athlete_data['shortName'])) {
                $fighter1_name = normalize_fighter_name($athlete_data['shortName']);
            }
        }
        
        // Get fighter 2 name (home athlete)
        if (isset($odds['homeAthleteOdds']['athlete']['$ref'])) {
            $athlete_data = fetch_api($odds['homeAthleteOdds']['athlete']['$ref']);
            if ($athlete_data && isset($athlete_data['shortName'])) {
                $fighter2_name = normalize_fighter_name($athlete_data['shortName']);
            }
        }
        
        $all_odds[] = [
            'competition_id' => $comp_id,
            'ml1' => $ml1,
            'ml2' => $ml2,
            'fighter1_name' => $fighter1_name,
            'fighter2_name' => $fighter2_name,
            'details' => isset($odds['details']) ? $odds['details'] : 'N/A',
            'provider' => isset($odds['provider']['name']) ? $odds['provider']['name'] : 'Unknown'
        ];
    }
}

// Update database with moneyline odds - match by fighter names
$matched_count = 0;
$unmatched_fights = [];

foreach ($all_odds as $key => $odds) {
    if ($odds['fighter1_name'] && $odds['fighter2_name']) {
        $ml1_value = $odds['ml1'] !== null ? $odds['ml1'] : 'NULL';
        $ml2_value = $odds['ml2'] !== null ? $odds['ml2'] : 'NULL';
        
        $fighter1 = mysqli_real_escape_string($con, $odds['fighter1_name']);
        $fighter2 = mysqli_real_escape_string($con, $odds['fighter2_name']);
        
        // Check if this fight exists in database (either order)
        $check_query = "SELECT fight_id, short_name_1, short_name_2 FROM `fight_list` 
                       WHERE (`short_name_1` = '{$fighter1}' AND `short_name_2` = '{$fighter2}')
                       OR (`short_name_1` = '{$fighter2}' AND `short_name_2` = '{$fighter1}')";
        $check_result = mysqli_query($con, $check_query);
        
        if ($check_result && mysqli_num_rows($check_result) > 0) {
            $row = mysqli_fetch_assoc($check_result);
            
            // Determine if we need to swap ML1 and ML2 based on database order
            if ($row['short_name_1'] == $fighter1) {
                // Order matches - ML1 goes to fighter1, ML2 to fighter2
                $update_query = "UPDATE `fight_list` SET `ML1` = {$ml1_value}, `ML2` = {$ml2_value} WHERE `fight_id` = {$row['fight_id']}";
            } else {
                // Order reversed - swap the moneylines
                $update_query = "UPDATE `fight_list` SET `ML1` = {$ml2_value}, `ML2` = {$ml1_value} WHERE `fight_id` = {$row['fight_id']}";
            }
            
            mysqli_query($con, $update_query);
            $matched_count++;
            
            // Add match info to the odds array for display
            $all_odds[$key]['matched_fight_id'] = $row['fight_id'];
            $all_odds[$key]['db_fighter1'] = $row['short_name_1'];
            $all_odds[$key]['db_fighter2'] = $row['short_name_2'];
        } else {
            // No match found
            $unmatched_fights[] = $odds;
        }
    }
}

// Close database connection
mysqli_close($con);


// Output HTML table
?>
<!DOCTYPE html>
<html>
<head>
    <title>UFC Odds - <?php echo htmlspecialchars($event_label); ?></title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 3px solid #d41a1a;
            padding-bottom: 10px;
        }
        .event-info {
            margin-bottom: 20px;
            color: #666;
            font-size: 14px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th {
            background-color: #d41a1a;
            color: white;
            padding: 12px;
            text-align: left;
            font-weight: bold;
        }
        td {
            padding: 12px;
            border-bottom: 1px solid #ddd;
        }
        tr:hover {
            background-color: #f9f9f9;
        }
        .no-odds {
            text-align: center;
            padding: 40px;
            color: #999;
            font-style: italic;
        }
        .footer {
            margin-top: 20px;
            text-align: center;
            color: #999;
            font-size: 12px;
        }
        .match-success {
            color: green;
            font-weight: bold;
        }
        .match-failed {
            color: red;
            font-weight: bold;
        }
        .warning-box {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>UFC Fight Odds</h1>
        <div class="event-info">
            <strong><?php echo htmlspecialchars($event_label); ?></strong><br>
            <?php echo htmlspecialchars($event_date); ?><br>
            Event ID: <?php echo htmlspecialchars($event_id); ?><br>
            <span class="match-success">✓ Matched: <?php echo $matched_count; ?> fights</span>
            <?php if (count($unmatched_fights) > 0): ?>
                <span class="match-failed"> | ✗ Unmatched: <?php echo count($unmatched_fights); ?> fights</span>
            <?php endif; ?>
        </div>
        
        <?php if (count($unmatched_fights) > 0): ?>
        <div class="warning-box">
            <strong>⚠️ Unmatched Fights (not in database):</strong><br>
            <?php foreach ($unmatched_fights as $fight): ?>
                <div style="margin: 5px 0;">
                    <?php echo htmlspecialchars($fight['fighter1_name']); ?> vs <?php echo htmlspecialchars($fight['fighter2_name']); ?>
                    (ML: <?php echo $fight['ml1']; ?> / <?php echo $fight['ml2']; ?>)
                </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <?php if (count($all_odds) > 0): ?>
        <table>
            <thead>
                <tr>
                    <th>Status</th>
                    <th>Fight #</th>
                    <th>Fighter 1</th>
                    <th>ML1</th>
                    <th>Fighter 2</th>
                    <th>ML2</th>
                    <th>Details</th>
                    <th>Provider</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($all_odds as $index => $odds): ?>
                <tr style="<?php echo isset($odds['matched_fight_id']) ? '' : 'background-color: #ffe6e6;'; ?>">
                    <td>
                        <?php if (isset($odds['matched_fight_id'])): ?>
                            <span class="match-success">✓</span>
                        <?php else: ?>
                            <span class="match-failed">✗</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo isset($odds['matched_fight_id']) ? $odds['matched_fight_id'] : '-'; ?></td>
                    <td><?php echo htmlspecialchars($odds['fighter1_name'] ?? 'N/A'); ?></td>
                    <td><strong><?php echo $odds['ml1'] !== null ? $odds['ml1'] : 'N/A'; ?></strong></td>
                    <td><?php echo htmlspecialchars($odds['fighter2_name'] ?? 'N/A'); ?></td>
                    <td><strong><?php echo $odds['ml2'] !== null ? $odds['ml2'] : 'N/A'; ?></strong></td>
                    <td><?php echo htmlspecialchars($odds['details']); ?></td>
                    <td><?php echo htmlspecialchars($odds['provider']); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php else: ?>
        <div class="no-odds">No odds available for this event</div>
        <?php endif; ?>
        
        <div class="footer">
            Last updated: <?php echo date('F j, Y g:i A'); ?>
        </div>
    </div>
</body>
</html>