<?php

require_once '../config.php'; // Include the configuration file

// ========== SMART UPDATE CONTROL ==========
$UPDATE_ANYWAY = isset($_GET['UPDATE_ANYWAY']) && $_GET['UPDATE_ANYWAY'] === 'true';

// Helper function to parse ISO Zulu time to ET
function parseIsoZuluToET($isoStr) {
    if (!$isoStr) return null;
    try {
        $utc = new DateTime($isoStr, new DateTimeZone('UTC'));
        $utc->setTimezone(new DateTimeZone('America/New_York'));
        return $utc;
    } catch (Exception $e) {
        return null;
    }
}

// Establish database connection for preflight check
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

// ======== PREFLIGHT: proceed only within [start-10m, start+5h] OR UPDATE_ANYWAY ========
$preflight = [
    'update_anyway' => (bool)$UPDATE_ANYWAY,
    'window_active' => false,
    'starts_within_10' => false,
    'proceed' => false,
    'notes' => [],
    'next_start_rows' => [],
    'next_start_closest' => null,
];

if ($UPDATE_ANYWAY) {
    $preflight['proceed'] = true;
    $preflight['notes'][] = 'Proceeding due to UPDATE_ANYWAY=true.';
} else {
    $nowET = new DateTime('now', new DateTimeZone('America/New_York'));

    // Pull Champions League game times from DB
    $sql = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played
            FROM sports_list
            WHERE League='Champion' AND Game_Date IS NOT NULL
            ORDER BY Game ASC";
    
    if ($rs = mysqli_query($con, $sql)) {
        while ($row = mysqli_fetch_assoc($rs)) {
            $startET = parseIsoZuluToET($row['Game_Date'] ?? null);
            if (!$startET) continue;

            $matchup = trim(($row['Team_2_Abbrev'] ?? '') . ' @ ' . ($row['Team_1_Abbrev'] ?? ''));
            $diffSec = $startET->getTimestamp() - $nowET->getTimestamp();
            
            if ($diffSec >= 0 && $diffSec <= 600) {
                $preflight['starts_within_10'] = true;
            }

            // Track for UI
            $preflight['next_start_rows'][] = [
                'game'    => (int)$row['Game'],
                'matchup' => $matchup,
                'startET' => $startET,
                'diffSec' => $diffSec,
            ];
            
            if ($preflight['next_start_closest'] === null ||
                $diffSec < $preflight['next_start_closest']['diffSec']) {
                $preflight['next_start_closest'] = [
                    'game'    => (int)$row['Game'],
                    'matchup' => $matchup,
                    'startET' => $startET,
                    'diffSec' => $diffSec,
                ];
            }

            // Window rule: start-10m .. start+5h
            $windowStart = (clone $startET)->modify('-10 minutes');
            $windowEnd   = (clone $startET)->modify('+5 hours');
            if ($nowET >= $windowStart && $nowET <= $windowEnd) {
                $preflight['window_active'] = true;
            }
        }
        mysqli_free_result($rs);
    }

    $preflight['proceed'] = $preflight['window_active'];
    if ($preflight['proceed']) {
        $preflight['notes'][] = 'Proceeding: within 10m-before to 5h-after window.';
    } else {
        $preflight['notes'][] = 'Blocked: outside all game windows.';
    }
}

// ===== If NOT proceeding, show status page and exit =====
if (!$preflight['proceed']) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
      <title>UEFA Champions League Updater — Preflight (No Run)</title>
      <style>
        body{font-family:Arial,sans-serif;background:#0b1320;color:#e6eefc;padding:24px}
        .card{background:#0f1a2b;border:1px solid #1e2b44;border-radius:8px;padding:16px;max-width:880px;margin:auto}
        .pill{display:inline-block;padding:4px 8px;border-radius:999px;font-size:12px;margin-right:6px;color:#fff}
        .pill-blue{background:#1e88e5}
        .pill-grey{background:#607d8b}
        .pill-orange{background:#ef6c00}
        .kv{display:grid;grid-template-columns:180px 1fr;gap:8px;margin-top:12px}
        code{background:#1e2b44;padding:2px 6px;border-radius:4px}
        table{width:100%;border-collapse:collapse;margin-top:10px}
        th,td{padding:6px 4px;border-bottom:1px solid #1e2b44;text-align:left;font-size:12px}
      </style>
    </head>
    <body>
      <div class="card">
        <h2 style="margin:0 0 8px 0;">UEFA Champions League Updater — Preflight</h2>
        <div>
          <span class="pill pill-blue">No Run</span>
          <?php if ($preflight['update_anyway']): ?>
            <span class="pill pill-orange">UPDATE_ANYWAY=true</span>
          <?php else: ?>
            <span class="pill pill-grey">Normal Mode</span>
          <?php endif; ?>
        </div>
        <div class="kv">
          <div><strong>Time (ET)</strong></div><div><?php echo htmlspecialchars($time_updated); ?></div>
          <div><strong>Starts ≤ 10 min?</strong></div><div><?php echo $preflight['starts_within_10'] ? 'Yes' : 'No'; ?></div>
        </div>

        <?php
        $closest = $preflight['next_start_closest'] ?? null;
        if ($closest && $closest['diffSec'] > 0) {
          $secs = (int)$closest['diffSec'];
          $mins = intdiv($secs, 60);
          $hrs  = intdiv($mins, 60);
          $mins = $mins % 60;
          $startStr = $closest['startET']->format('Y-m-d g:i A \E\T');
        ?>
        <div class="kv" style="margin-top:8px;">
          <div><strong>Closest UCL start</strong></div>
          <div>
            <?php echo htmlspecialchars($closest['matchup']); ?> —
            <strong><?php echo htmlspecialchars($startStr); ?></strong>
            (starts in <strong><?php echo $hrs; ?>h <?php echo $mins; ?>m</strong>)
          </div>
        </div>
        <?php } ?>

        <?php if (!empty($preflight['notes'])): ?>
          <div style="margin-top:10px"><strong>Notes</strong>
            <ul style="margin:6px 0 0 18px;">
              <?php foreach ($preflight['notes'] as $n): ?>
                <li><?php echo htmlspecialchars($n); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
        <?php endif; ?>

        <?php if (!empty($preflight['next_start_rows'])): ?>
          <div style="margin-top:10px">
            <strong>Upcoming UEFA Champions League matches (ET)</strong>
            <table>
              <thead>
                <tr><th>Game</th><th>Matchup</th><th>Start (ET)</th><th>Starts In</th></tr>
              </thead>
              <tbody>
                <?php foreach ($preflight['next_start_rows'] as $row):
                  if ($row['diffSec'] < 0) continue; // Skip past games
                  $secs = (int)$row['diffSec'];
                  $mins = intdiv($secs, 60);
                  $hrs  = intdiv($mins, 60);
                  $mins = $mins % 60;
                  $startStr = $row['startET']->format('Y-m-d g:i A \E\T');
                ?>
                <tr>
                  <td><?php echo (int)$row['game']; ?></td>
                  <td><?php echo htmlspecialchars($row['matchup']); ?></td>
                  <td><?php echo htmlspecialchars($startStr); ?></td>
                  <td><?php echo $hrs; ?>h <?php echo $mins; ?>m</td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

        <div style="margin-top:10px">
          Tip: append <code>?UPDATE_ANYWAY=true</code> to force an update (ignores time window).
        </div>
      </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    die();
}

// ==================== MAIN UPDATE LOGIC STARTS HERE ====================

// Update API timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'Champion'";
if (mysqli_query($con, $sql)) {
    if (mysqli_affected_rows($con) > 0) {
        echo "Record updated successfully for API: Champion<br><br>";
    }
}

//************************************************//
// Helper Functions
//************************************************//

function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

function parseHexColor($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    return [
        'r' => addZeros($r),
        'g' => addZeros($g),
        'b' => addZeros($b),
        'rgb' => sprintf('"%s,%s,%s"', addZeros($r), addZeros($g), addZeros($b))
    ];
}

//************************************************//
// API Call
//************************************************//

$api_url = "https://site.api.espn.com/apis/site/v2/sports/soccer/uefa.champions/scoreboard";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$response = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error: " . $err);
}

// Parse JSON response
$data = json_decode($response, true);

if (!$data) {
    die("Failed to parse API response");
}

// Start HTML output
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>UEFA Champions League - Database Update</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        h1 {
            color: #333;
            text-align: center;
            border-bottom: 3px solid #667eea;
            padding-bottom: 15px;
            margin-bottom: 30px;
        }
        .info-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 8px;
            font-size: 14px;
        }
        .info-box strong {
            color: #ffd700;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            background: white;
        }
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 8px;
            text-align: left;
            font-size: 13px;
            font-weight: 600;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        td {
            padding: 10px 8px;
            border-bottom: 1px solid #e0e0e0;
            font-size: 12px;
        }
        tr:hover {
            background-color: #f5f5ff;
        }
        .status-live {
            background: #ff5722;
            color: white;
            padding: 3px 8px;
            border-radius: 4px;
            font-weight: bold;
            font-size: 11px;
        }
        .status-done {
            background: #9e9e9e;
            color: white;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 11px;
        }
        .status-scheduled {
            background: #2196f3;
            color: white;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 11px;
        }
        .status-halftime {
            background: #ff9800;
            color: white;
            padding: 3px 8px;
            border-radius: 4px;
            font-weight: bold;
            font-size: 11px;
        }
        .team-name {
            font-weight: 600;
            color: #333;
        }
        .score {
            font-size: 16px;
            font-weight: bold;
            color: #333;
        }
        .color-box {
            display: inline-block;
            width: 20px;
            height: 20px;
            border-radius: 3px;
            border: 1px solid #ccc;
            vertical-align: middle;
            margin-right: 5px;
        }
        .success-msg {
            background: #4caf50;
            color: white;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }
        .error-msg {
            background: #f44336;
            color: white;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }
        .sql-query {
            background: #f5f5f5;
            padding: 10px;
            margin: 5px 0;
            border-left: 4px solid #667eea;
            font-family: 'Courier New', monospace;
            font-size: 11px;
            word-break: break-all;
            display: none;
        }
        .show-sql {
            cursor: pointer;
            color: #667eea;
            text-decoration: underline;
            font-size: 12px;
        }
        .update-mode {
            display: inline-block;
            background: <?php echo $UPDATE_ANYWAY ? '#ff9800' : '#4caf50'; ?>;
            color: white;
            padding: 5px 10px;
            border-radius: 5px;
            margin-left: 10px;
            font-size: 12px;
        }
    </style>
</head>
<body>
<div class="container">
    <h1>⚽ UEFA Champions League - Database Update
        <span class="update-mode">
            <?php echo $UPDATE_ANYWAY ? 'FORCED UPDATE' : 'NORMAL MODE'; ?>
        </span>
    </h1>
    
    <div class="info-box">
        <strong>API Source:</strong> ESPN API - UEFA Champions League<br>
        <strong>Last Updated:</strong> <?php echo $time_updated; ?><br>
        <strong>Total Games:</strong> <?php echo count($data['events']); ?><br>
        <strong>Database:</strong> <?php echo $database_gscroll; ?><br>
        <?php if ($UPDATE_ANYWAY): ?>
        <strong>⚠️ Update Mode:</strong> FORCED (UPDATE_ANYWAY=true)
        <?php endif; ?>
    </div>

    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Game Time</th>
                <th>Status</th>
                <th>Home Team</th>
                <th>Score</th>
                <th>Away Team</th>
                <th>Period</th>
                <th>Clock</th>
                <th>Venue</th>
                <th>Broadcast</th>
                <th>Records</th>
                <th>Colors</th>
                <th>DB Update</th>
            </tr>
        </thead>
        <tbody>
<?php

// Process each game
$gameCounter = 0;
$successCount = 0;
$errorCount = 0;
$allQueries = [];
$has_active_games = false; // Track if we find any active games
$has_upcoming_games = false; // Track if games are starting soon

foreach ($data['events'] as $event) {
    $gameCounter++;
    
    // Extract competition data
    $competition = $event['competitions'][0];
    $status = $competition['status'];
    
    // Extract teams
    $homeTeam = null;
    $awayTeam = null;
    
    foreach ($competition['competitors'] as $competitor) {
        if ($competitor['homeAway'] == 'home') {
            $homeTeam = $competitor;
        } else {
            $awayTeam = $competitor;
        }
    }
    
    // Game Status
    $clock = isset($status['displayClock']) ? $status['displayClock'] : '0\'';
    $clockNumeric = preg_replace('/[^0-9]/', '', $clock); // Extract numeric value
    $period = isset($status['period']) ? $status['period'] : 0;
    $gameTime = $event['date'];
    $gameTimeFormatted = date("g:i A", strtotime($gameTime));
    $gameDateFormatted = date("m/d/Y", strtotime($gameTime));
    
    // Determine game status
    $gameStatus = 'NOT_STARTED';
    if ($status['type']['completed']) {
        $gameStatus = 'GAME_DONE';
    } elseif ($status['type']['state'] == 'in') {
        if (strpos($status['type']['name'], 'HALFTIME') !== false) {
            $gameStatus = 'HALF_TIME';
            $has_active_games = true; // Game is active
        } else {
            $gameStatus = 'IN_SESSION';
            $has_active_games = true; // Game is active
        }
    } else {
        // Check if game starts within 10 minutes
        $gameStartTime = new DateTime($gameTime, new DateTimeZone('UTC'));
        $nowTime = new DateTime('now', new DateTimeZone('UTC'));
        $diffSeconds = $gameStartTime->getTimestamp() - $nowTime->getTimestamp();
        if ($diffSeconds >= 0 && $diffSeconds <= 600) {
            $has_upcoming_games = true;
        }
    }
    
    // Extract broadcast information
    $broadcast = '';
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $broadcast = $broadcast_data['names'][0] ?? '';
        }
    }
    
    // Team 1 (Home) Data
    $team1Name = $homeTeam['team']['displayName'];
    $team1Abbrev = $homeTeam['team']['abbreviation'];
    $team1Location = $homeTeam['team']['location'];
    $team1Score = $homeTeam['score'];
    $team1Record = isset($homeTeam['records'][0]) ? $homeTeam['records'][0]['summary'] : 'N/A';
    $team1Color = parseHexColor($homeTeam['team']['color']);
    $team1ID = $homeTeam['team']['id'];
    
    // Team 2 (Away) Data
    $team2Name = $awayTeam['team']['displayName'];
    $team2Abbrev = $awayTeam['team']['abbreviation'];
    $team2Location = $awayTeam['team']['location'];
    $team2Score = $awayTeam['score'];
    $team2Record = isset($awayTeam['records'][0]) ? $awayTeam['records'][0]['summary'] : 'N/A';
    $team2Color = parseHexColor($awayTeam['team']['color']);
    $team2ID = $awayTeam['team']['id'];
    
    // Use alternate color if main color is black
    if ($homeTeam['team']['color'] == '000000' && isset($homeTeam['team']['alternateColor'])) {
        $team1Color = parseHexColor($homeTeam['team']['alternateColor']);
    }
    if ($awayTeam['team']['color'] == '000000' && isset($awayTeam['team']['alternateColor'])) {
        $team2Color = parseHexColor($awayTeam['team']['alternateColor']);
    }
    
    // Venue
    $venue = isset($competition['venue']['fullName']) ? $competition['venue']['fullName'] : 'TBD';
    
    // Check for recent scoring (similar to EPL logic)
    $scoreValue = 'NO-SCORE';
    $whoHasBall = '';
    
    if ($gameStatus === 'IN_SESSION') {
        $details = $competition['details'] ?? [];
        $currentClockMinutes = intval($clockNumeric);
        
        foreach ($details as $detail) {
            if (isset($detail['scoringPlay']) && $detail['scoringPlay']) {
                $playClockValue = $detail['clock']['displayValue'] ?? '';
                $playClockMinutes = intval(preg_replace('/[^0-9]/', '', $playClockValue));
                
                // Check if goal was within last 2 minutes
                if ($currentClockMinutes - $playClockMinutes <= 2) {
                    $scoreValue = 'GOAL';
                    $scoringTeamId = $detail['team']['id'] ?? '';
                    if ($scoringTeamId == $homeTeam['team']['id']) {
                        $whoHasBall = $team1Abbrev;
                    } elseif ($scoringTeamId == $awayTeam['team']['id']) {
                        $whoHasBall = $team2Abbrev;
                    }
                    break;
                }
            }
        }
    }
    
    // Additional fields (set defaults for this example)
    $distance = '';
    $possession = '';
    $favoriteTeam = '';
    $favoritePoints = '';
    $lastPlay = '';
    $overUnder = '';
    $redZone = '';
    
    // Build the UPDATE query
    $query = "UPDATE sports_list SET 
        Clock = '".mysqli_real_escape_string($con, $clockNumeric)."',
        Period = '".mysqli_real_escape_string($con, $period)."',
        Game_Played = '".mysqli_real_escape_string($con, $gameStatus)."',
        Game_Time = '".mysqli_real_escape_string($con, $gameTimeFormatted)."',
        Game_Date = '".mysqli_real_escape_string($con, $gameTime)."',
        scoreValue = '".mysqli_real_escape_string($con, $scoreValue)."',
        Team_1_Name = '".mysqli_real_escape_string($con, $team1Name)."',
        Team_2_Name = '".mysqli_real_escape_string($con, $team2Name)."',
        Team_1_Record = '".mysqli_real_escape_string($con, $team1Record)."',
        Team_2_Record = '".mysqli_real_escape_string($con, $team2Record)."',
        Team_1_Score = '".mysqli_real_escape_string($con, $team1Score)."',
        Team_2_Score = '".mysqli_real_escape_string($con, $team2Score)."',
        Team_1_Location = '".mysqli_real_escape_string($con, $team1Location)."',
        Team_2_Location = '".mysqli_real_escape_string($con, $team2Location)."',
        Team_1_Abbrev = '".mysqli_real_escape_string($con, $team1Abbrev)."',
        Team_1_Color = '".mysqli_real_escape_string($con, $team1Color['rgb'])."',
        Team_1_Color_R = '".mysqli_real_escape_string($con, $team1Color['r'])."',
        Team_1_Color_G = '".mysqli_real_escape_string($con, $team1Color['g'])."',
        Team_1_Color_B = '".mysqli_real_escape_string($con, $team1Color['b'])."',
        Team_2_Abbrev = '".mysqli_real_escape_string($con, $team2Abbrev)."',
        Team_2_Color = '".mysqli_real_escape_string($con, $team2Color['rgb'])."',
        Team_2_Color_R = '".mysqli_real_escape_string($con, $team2Color['r'])."',
        Team_2_Color_G = '".mysqli_real_escape_string($con, $team2Color['g'])."',
        Team_2_Color_B = '".mysqli_real_escape_string($con, $team2Color['b'])."',
        Broadcast = '".mysqli_real_escape_string($con, $broadcast)."',
        Who_Has_Ball = '".mysqli_real_escape_string($con, $whoHasBall)."',
        Distance = '".mysqli_real_escape_string($con, $distance)."',
        Possesion = '".mysqli_real_escape_string($con, $possession)."',
        Favorite_Team = '".mysqli_real_escape_string($con, $favoriteTeam)."',
        Favorite_Points = '".mysqli_real_escape_string($con, $favoritePoints)."',
        lastPlay = '".mysqli_real_escape_string($con, $lastPlay)."',
        OverUnder = '".mysqli_real_escape_string($con, $overUnder)."',
        Red_Zone = '".mysqli_real_escape_string($con, $redZone)."'
        WHERE Game = $gameCounter AND League = 'Champion'";
    
    // Store query for display
    $allQueries[] = $query;
    
    // Execute query
    $queryResult = mysqli_query($con, $query);
    $dbStatus = '';
    
    if ($queryResult) {
        $successCount++;
        $dbStatus = '<span class="success-msg">✓</span>';
        echo "<!-- Game $gameCounter updated: $team2Abbrev @ $team1Abbrev - $team2Score-$team1Score ($gameStatus) -->\n";
    } else {
        $errorCount++;
        $dbStatus = '<span class="error-msg">✗ ' . mysqli_error($con) . '</span>';
    }
    
    // Determine status class
    $statusClass = 'status-scheduled';
    if ($gameStatus == 'GAME_DONE') $statusClass = 'status-done';
    elseif ($gameStatus == 'IN_SESSION') $statusClass = 'status-live';
    elseif ($gameStatus == 'HALF_TIME') $statusClass = 'status-halftime';
    
    // Output table row
    ?>
    <tr>
        <td><?php echo $gameCounter; ?></td>
        <td><?php echo $gameDateFormatted . '<br>' . $gameTimeFormatted; ?></td>
        <td><span class="<?php echo $statusClass; ?>"><?php echo $gameStatus; ?></span></td>
        <td>
            <span class="color-box" style="background: #<?php echo str_pad($homeTeam['team']['color'], 6, '0'); ?>"></span>
            <span class="team-name"><?php echo $team1Name; ?></span><br>
            <small><?php echo $team1Abbrev; ?></small>
        </td>
        <td class="score"><?php echo $team1Score; ?> - <?php echo $team2Score; ?></td>
        <td>
            <span class="color-box" style="background: #<?php echo str_pad($awayTeam['team']['color'], 6, '0'); ?>"></span>
            <span class="team-name"><?php echo $team2Name; ?></span><br>
            <small><?php echo $team2Abbrev; ?></small>
        </td>
        <td><?php echo $period; ?></td>
        <td><?php echo $clock; ?></td>
        <td><?php echo substr($venue, 0, 30); ?><?php echo strlen($venue) > 30 ? '...' : ''; ?></td>
        <td><?php echo $broadcast ? $broadcast : 'N/A'; ?></td>
        <td><?php echo $team1Record; ?><br><?php echo $team2Record; ?></td>
        <td>
            H: <?php echo $team1Color['r'].','.$team1Color['g'].','.$team1Color['b']; ?><br>
            A: <?php echo $team2Color['r'].','.$team2Color['g'].','.$team2Color['b']; ?>
        </td>
        <td>
            <?php echo $dbStatus; ?>
            <span class="show-sql" onclick="toggleSQL(<?php echo $gameCounter; ?>)">View SQL</span>
            <div id="sql-<?php echo $gameCounter; ?>" class="sql-query">
                <?php echo htmlspecialchars($query); ?>
            </div>
        </td>
    </tr>
    <?php
}
?>
        </tbody>
    </table>
    
    <div class="info-box" style="margin-top: 30px; background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);">
        <strong>Update Summary:</strong><br>
        ✅ Successful Updates: <?php echo $successCount; ?><br>
        ❌ Failed Updates: <?php echo $errorCount; ?><br>
        📊 Total Games Processed: <?php echo $gameCounter; ?><br>
        ⏰ Completed at: <?php echo $time_updated; ?>
    </div>
    
    <?php
    // ========== SOCCER GAMES UPDATE TRIGGER (AFTER DATABASE UPDATE) ==========
    // Curl Soccer_Games.php if UPDATE_ANYWAY=true OR games starting within 10 min OR games in session
    $should_update_soccer = $UPDATE_ANYWAY || $has_upcoming_games || $has_active_games;
    
    if ($should_update_soccer) {
        echo '<div class="info-box" style="margin-top: 20px; background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);">';
        echo '<strong>⚽ Soccer_Games.php Update Trigger</strong><br>';
        echo '===============================================<br>';
        
        // Log the trigger reason
        $trigger_reasons = [];
        if ($UPDATE_ANYWAY) $trigger_reasons[] = "UPDATE_ANYWAY=true";
        if ($has_upcoming_games) $trigger_reasons[] = "games starting within 10 minutes";
        if ($has_active_games) $trigger_reasons[] = "games currently in session";
        echo 'Trigger reason(s): ' . implode(", ", $trigger_reasons) . '<br>';
        echo 'Calling Soccer_Games.php...<br><br>';
        
        $soccer_url = "https://gscroll.us/Scrolling/App_Output/API/Games/Soccer_Games.php";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $soccer_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);
        
        $start_time = microtime(true);
        $soccer_result = curl_exec($ch);
        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 2);
        
        $soccer_err = curl_error($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $content_length = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        curl_close($ch);
        
        if ($soccer_err) {
            echo '<span style="color:#ffcdd2;">❌ Soccer_Games.php curl FAILED!</span><br>';
            echo 'Error: ' . $soccer_err . '<br>';
        } else {
            echo '<span style="color:#c8e6c9;">✅ Soccer_Games.php called SUCCESSFULLY!</span><br>';
            echo 'HTTP Status Code: ' . $http_code . '<br>';
            echo 'Response Size: ' . number_format($content_length) . ' bytes<br>';
            echo 'Execution Time: ' . $execution_time . ' seconds<br>';
            
            if ($http_code == 200) {
                echo '<span style="color:#c8e6c9;">Status: SUCCESS - Soccer games updated!</span><br>';
            } else {
                echo '<span style="color:#fff3e0;">Warning: Unexpected HTTP code ' . $http_code . '</span><br>';
            }
        }
        echo '===============================================';
        echo '</div>';
    } else {
        echo '<div class="info-box" style="margin-top: 20px; background: linear-gradient(135deg, #9e9e9e 0%, #757575 100%);">';
        echo '<strong>Soccer_Games.php Update Status</strong><br>';
        echo 'Update NOT triggered<br>';
        echo 'Reason: No active games, no upcoming games within 10 min, and UPDATE_ANYWAY not set';
        echo '</div>';
    }
    
    // Close database connection
    mysqli_close($con);
    ?>
    
    <script>
    function toggleSQL(id) {
        var element = document.getElementById('sql-' + id);
        if (element.style.display === 'none' || element.style.display === '') {
            element.style.display = 'block';
        } else {
            element.style.display = 'none';
        }
    }
    
    // Auto-refresh page every 60 seconds (optional)
    // setTimeout(function() { location.reload(); }, 60000);
    </script>
</div>
</body>
</html>