<?php

require_once '../config.php'; // Include the configuration file

// ========== SMART UPDATE CONTROL ==========
$UPDATE_ANYWAY = isset($_GET['UPDATE_ANYWAY']) && $_GET['UPDATE_ANYWAY'] === 'true';

// Helper function to parse ISO Zulu time to ET
function parseIsoZuluToET($isoStr) {
    if (!$isoStr) return null;
    try {
        $utc = new DateTime($isoStr, new DateTimeZone('UTC'));
        $utc->setTimezone(new DateTimeZone('America/New_York'));
        return $utc;
    } catch (Exception $e) {
        return null;
    }
}

// Establish database connection for preflight check
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

// ======== PREFLIGHT: proceed only within [start-10m, start+5h] OR UPDATE_ANYWAY ========
$preflight = [
    'update_anyway' => (bool)$UPDATE_ANYWAY,
    'window_active' => false,
    'starts_within_10' => false,
    'proceed' => false,
    'notes' => [],
    'next_start_rows' => [],
    'next_start_closest' => null,
];

if ($UPDATE_ANYWAY) {
    $preflight['proceed'] = true;
    $preflight['notes'][] = 'Proceeding due to UPDATE_ANYWAY=true.';
} else {
    $nowET = new DateTime('now', new DateTimeZone('America/New_York'));

    // Pull EPL game times from DB
    $sql = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played
            FROM sports_list
            WHERE League='EPL' AND Game_Date IS NOT NULL
            ORDER BY Game ASC";
    
    if ($rs = mysqli_query($con, $sql)) {
        while ($row = mysqli_fetch_assoc($rs)) {
            $startET = parseIsoZuluToET($row['Game_Date'] ?? null);
            if (!$startET) continue;

            $matchup = trim(($row['Team_2_Abbrev'] ?? '') . ' @ ' . ($row['Team_1_Abbrev'] ?? ''));
            $diffSec = $startET->getTimestamp() - $nowET->getTimestamp();
            
            if ($diffSec >= 0 && $diffSec <= 600) {
                $preflight['starts_within_10'] = true;
            }

            // Track for UI
            $preflight['next_start_rows'][] = [
                'game'    => (int)$row['Game'],
                'matchup' => $matchup,
                'startET' => $startET,
                'diffSec' => $diffSec,
            ];
            
            if ($preflight['next_start_closest'] === null ||
                $diffSec < $preflight['next_start_closest']['diffSec']) {
                $preflight['next_start_closest'] = [
                    'game'    => (int)$row['Game'],
                    'matchup' => $matchup,
                    'startET' => $startET,
                    'diffSec' => $diffSec,
                ];
            }

            // Window rule: start-10m .. start+5h
            $windowStart = (clone $startET)->modify('-10 minutes');
            $windowEnd   = (clone $startET)->modify('+5 hours');
            if ($nowET >= $windowStart && $nowET <= $windowEnd) {
                $preflight['window_active'] = true;
            }
        }
        mysqli_free_result($rs);
    }

    $preflight['proceed'] = $preflight['window_active'];
    if ($preflight['proceed']) {
        $preflight['notes'][] = 'Proceeding: within 10m-before to 5h-after window.';
    } else {
        $preflight['notes'][] = 'Blocked: outside all game windows.';
    }
}

// ===== If NOT proceeding, show status page and exit =====
if (!$preflight['proceed']) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
      <title>EPL Updater — Preflight (No Run)</title>
      <style>
        body{font-family:Arial,sans-serif;background:#0b1320;color:#e6eefc;padding:24px}
        .card{background:#0f1a2b;border:1px solid #1e2b44;border-radius:8px;padding:16px;max-width:880px;margin:auto}
        .pill{display:inline-block;padding:4px 8px;border-radius:999px;font-size:12px;margin-right:6px;color:#fff}
        .pill-blue{background:#1e88e5}
        .pill-grey{background:#607d8b}
        .pill-orange{background:#ef6c00}
        .kv{display:grid;grid-template-columns:180px 1fr;gap:8px;margin-top:12px}
        code{background:#1e2b44;padding:2px 6px;border-radius:4px}
        table{width:100%;border-collapse:collapse;margin-top:10px}
        th,td{padding:6px 4px;border-bottom:1px solid #1e2b44;text-align:left;font-size:12px}
      </style>
    </head>
    <body>
      <div class="card">
        <h2 style="margin:0 0 8px 0;">EPL Updater — Preflight</h2>
        <div>
          <span class="pill pill-blue">No Run</span>
          <?php if ($preflight['update_anyway']): ?>
            <span class="pill pill-orange">UPDATE_ANYWAY=true</span>
          <?php else: ?>
            <span class="pill pill-grey">Normal Mode</span>
          <?php endif; ?>
        </div>
        <div class="kv">
          <div><strong>Time (ET)</strong></div><div><?php echo htmlspecialchars($time_updated); ?></div>
          <div><strong>Starts ≤ 10 min?</strong></div><div><?php echo $preflight['starts_within_10'] ? 'Yes' : 'No'; ?></div>
        </div>

        <?php
        $closest = $preflight['next_start_closest'] ?? null;
        if ($closest && $closest['diffSec'] > 0) {
          $secs = (int)$closest['diffSec'];
          $mins = intdiv($secs, 60);
          $hrs  = intdiv($mins, 60);
          $mins = $mins % 60;
          $startStr = $closest['startET']->format('Y-m-d g:i A \E\T');
        ?>
        <div class="kv" style="margin-top:8px;">
          <div><strong>Closest EPL start</strong></div>
          <div>
            <?php echo htmlspecialchars($closest['matchup']); ?> —
            <strong><?php echo htmlspecialchars($startStr); ?></strong>
            (starts in <strong><?php echo $hrs; ?>h <?php echo $mins; ?>m</strong>)
          </div>
        </div>
        <?php } ?>

        <?php if (!empty($preflight['notes'])): ?>
          <div style="margin-top:10px"><strong>Notes</strong>
            <ul style="margin:6px 0 0 18px;">
              <?php foreach ($preflight['notes'] as $n): ?>
                <li><?php echo htmlspecialchars($n); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
        <?php endif; ?>

        <?php if (!empty($preflight['next_start_rows'])): ?>
          <div style="margin-top:10px">
            <strong>Upcoming EPL matches (ET)</strong>
            <table>
              <thead>
                <tr><th>Game</th><th>Matchup</th><th>Start (ET)</th><th>Starts In</th></tr>
              </thead>
              <tbody>
                <?php foreach ($preflight['next_start_rows'] as $row):
                  if ($row['diffSec'] < 0) continue; // Skip past games
                  $secs = (int)$row['diffSec'];
                  $mins = intdiv($secs, 60);
                  $hrs  = intdiv($mins, 60);
                  $mins = $mins % 60;
                  $startStr = $row['startET']->format('Y-m-d g:i A \E\T');
                ?>
                <tr>
                  <td><?php echo (int)$row['game']; ?></td>
                  <td><?php echo htmlspecialchars($row['matchup']); ?></td>
                  <td><?php echo htmlspecialchars($startStr); ?></td>
                  <td><?php echo $hrs; ?>h <?php echo $mins; ?>m</td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

        <div style="margin-top:10px">
          Tip: append <code>?UPDATE_ANYWAY=true</code> to force an update (ignores time window).
        </div>
      </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    die();
}

// ==================== MAIN UPDATE LOGIC STARTS HERE ====================

// Update API_DB timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'EPL'";
if (mysqli_query($con, $sql)) {
    if (mysqli_affected_rows($con) > 0) {
        echo "Record updated successfully for API<br>";
    } else {
        echo "No records updated. Please check if the API entry exists.<br>";
    }
} else {
    echo "Error executing query: " . mysqli_error($con) . "<br>";
}

// Helper functions
function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

// API CALL
$api_call = "http://site.api.espn.com/apis/site/v2/sports/soccer/eng.1/scoreboard";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    echo "cURL Error #:" . $err;
    exit;
}

// Parse JSON response
$data = json_decode($result, true);
if (!$data) {
    echo "Failed to parse JSON response";
    exit;
}

// Process each event/game
$events = $data['events'] ?? [];
$gameNumber = 1;
$has_active_games = false; // Track if we find any active games
$has_upcoming_games = false; // Track if games are starting soon

foreach ($events as $event) {
    // Get competition data
    $competition = $event['competitions'][0] ?? null;
    if (!$competition) continue;
    
    // Get status info
    $status = $competition['status'] ?? [];
    $clock = $status['displayClock'] ?? '0\'';
    $period = $status['period'] ?? 0;
    
    // Parse clock to numeric value (remove quotes and non-numeric chars)
    $clockNumeric = preg_replace('/[^0-9]/', '', $clock);
    
    // Determine game state
    $gameDate = $competition['date'] ?? '';
    $completed = $status['type']['completed'] ?? false;
    $state = $status['type']['state'] ?? '';
    $statusName = $status['type']['name'] ?? '';
    
    if ($state === 'pre' || !$completed && $state !== 'in') {
        $game_being_played = 'NOT_STARTED';
        
        // Check if game starts within 10 minutes
        if ($gameDate) {
            $gameStartTime = new DateTime($gameDate, new DateTimeZone('UTC'));
            $nowTime = new DateTime('now', new DateTimeZone('UTC'));
            $diffSeconds = $gameStartTime->getTimestamp() - $nowTime->getTimestamp();
            if ($diffSeconds >= 0 && $diffSeconds <= 600) {
                $has_upcoming_games = true;
            }
        }
    } elseif ($completed || $state === 'post') {
        $game_being_played = 'GAME_DONE';
    } elseif ($statusName === 'STATUS_HALFTIME') {
        $game_being_played = 'HALF_TIME';
        $has_active_games = true; // Game is active
    } else {
        $game_being_played = 'IN_SESSION';
        $has_active_games = true; // Game is active
    }
    
    // Get game time
    $gameTimeET = '';
    if ($gameDate) {
        $dt = new DateTime($gameDate, new DateTimeZone('UTC'));
        $dt->setTimezone(new DateTimeZone('America/New_York'));
        $gameTimeET = $dt->format('g:i a');
    }
    
    // Extract broadcast information
    $broadcast = '';
    if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
        $broadcast_data = $competition['broadcasts'][0] ?? [];
        if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
            $broadcast = $broadcast_data['names'][0] ?? '';
        }
    }
    
    // Get competitors
    $competitors = $competition['competitors'] ?? [];
    $homeTeam = null;
    $awayTeam = null;
    
    foreach ($competitors as $competitor) {
        if ($competitor['homeAway'] === 'home') {
            $homeTeam = $competitor;
        } else {
            $awayTeam = $competitor;
        }
    }
    
    if (!$homeTeam || !$awayTeam) continue;
    
    // Extract team data
    $team1_name = clean($homeTeam['team']['displayName'] ?? '');
    $team2_name = clean($awayTeam['team']['displayName'] ?? '');
    
    $team1_abbrev = $homeTeam['team']['abbreviation'] ?? '';
    $team2_abbrev = $awayTeam['team']['abbreviation'] ?? '';
    
    $team1_location = clean($homeTeam['team']['location'] ?? '');
    $team2_location = clean($awayTeam['team']['location'] ?? '');
    
    // Get scores
    $team1_score = ($game_being_played !== 'NOT_STARTED') ? ($homeTeam['score'] ?? '0') : '0';
    $team2_score = ($game_being_played !== 'NOT_STARTED') ? ($awayTeam['score'] ?? '0') : '0';
    
    // Get records
    $team1_record = $homeTeam['records'][0]['summary'] ?? '';
    $team2_record = $awayTeam['records'][0]['summary'] ?? '';
    
    // Get colors and convert to RGB
    $team1_color_hex = $homeTeam['team']['color'] ?? '000000';
    if ($team1_color_hex === '000000') {
        $team1_color_hex = $homeTeam['team']['alternateColor'] ?? '000000';
    }
    $team1_rgb = hexToRgb($team1_color_hex);
    $team1_color_r = $team1_rgb['r'];
    $team1_color_g = $team1_rgb['g'];
    $team1_color_b = $team1_rgb['b'];
    $team1_color_rgb = '"' . $team1_rgb['rgb'] . '"';
    
    $team2_color_hex = $awayTeam['team']['color'] ?? '000000';
    if ($team2_color_hex === '000000') {
        $team2_color_hex = $awayTeam['team']['alternateColor'] ?? '000000';
    }
    $team2_rgb = hexToRgb($team2_color_hex);
    $team2_color_r = $team2_rgb['r'];
    $team2_color_g = $team2_rgb['g'];
    $team2_color_b = $team2_rgb['b'];
    $team2_color_rgb = '"' . $team2_rgb['rgb'] . '"';
    
    // Check for recent scoring (within last 2 minutes)
    $score_value = 'NO-SCORE';
    $who_has_ball = '';
    
    if ($game_being_played === 'IN_SESSION') {
        $details = $competition['details'] ?? [];
        $currentClockMinutes = intval(preg_replace('/[^0-9]/', '', $clock));
        
        foreach ($details as $detail) {
            if (isset($detail['scoringPlay']) && $detail['scoringPlay']) {
                $playClockValue = $detail['clock']['displayValue'] ?? '';
                $playClockMinutes = intval(preg_replace('/[^0-9]/', '', $playClockValue));
                
                // Check if goal was within last 2 minutes
                if ($currentClockMinutes - $playClockMinutes <= 2) {
                    $score_value = 'GOAL';
                    $scoringTeamId = $detail['team']['id'] ?? '';
                    if ($scoringTeamId == $homeTeam['team']['id']) {
                        $who_has_ball = $team1_abbrev;
                    } elseif ($scoringTeamId == $awayTeam['team']['id']) {
                        $who_has_ball = $team2_abbrev;
                    }
                    break; // Found most recent goal
                }
            }
        }
    }
    
    // Prepare SQL update
    $query = "UPDATE sports_list SET 
        Clock='" . mysqli_real_escape_string($con, $clockNumeric) . "',
        Period='" . mysqli_real_escape_string($con, $period) . "',
        Game_Played='" . mysqli_real_escape_string($con, $game_being_played) . "',
        Game_Time='" . mysqli_real_escape_string($con, $gameTimeET) . "',
        Game_Date='" . mysqli_real_escape_string($con, $gameDate) . "',
        scoreValue='" . mysqli_real_escape_string($con, $score_value) . "',
        Team_1_Name='" . mysqli_real_escape_string($con, $team1_name) . "',
        Team_2_Name='" . mysqli_real_escape_string($con, $team2_name) . "',
        Team_1_Record='" . mysqli_real_escape_string($con, $team1_record) . "',
        Team_2_Record='" . mysqli_real_escape_string($con, $team2_record) . "',
        Team_1_Score='" . mysqli_real_escape_string($con, $team1_score) . "',
        Team_2_Score='" . mysqli_real_escape_string($con, $team2_score) . "',
        Team_1_Location='" . mysqli_real_escape_string($con, $team1_location) . "',
        Team_2_Location='" . mysqli_real_escape_string($con, $team2_location) . "',
        Team_1_Abbrev='" . mysqli_real_escape_string($con, $team1_abbrev) . "',
        Team_1_Color='" . mysqli_real_escape_string($con, $team1_color_rgb) . "',
        Team_1_Color_R='" . mysqli_real_escape_string($con, $team1_color_r) . "',
        Team_1_Color_G='" . mysqli_real_escape_string($con, $team1_color_g) . "',
        Team_1_Color_B='" . mysqli_real_escape_string($con, $team1_color_b) . "',
        Team_2_Abbrev='" . mysqli_real_escape_string($con, $team2_abbrev) . "',
        Team_2_Color='" . mysqli_real_escape_string($con, $team2_color_rgb) . "',
        Team_2_Color_R='" . mysqli_real_escape_string($con, $team2_color_r) . "',
        Team_2_Color_G='" . mysqli_real_escape_string($con, $team2_color_g) . "',
        Team_2_Color_B='" . mysqli_real_escape_string($con, $team2_color_b) . "',
        Broadcast='" . mysqli_real_escape_string($con, $broadcast) . "',
        Who_Has_Ball='" . mysqli_real_escape_string($con, $who_has_ball) . "'
        WHERE Game = $gameNumber AND League = 'EPL'";
    
    // Execute update
    if (mysqli_query($con, $query)) {
        echo "Game $gameNumber updated: $team2_abbrev @ $team1_abbrev - $team2_score-$team1_score ($game_being_played)<br>";
    } else {
        echo "Error updating game $gameNumber: " . mysqli_error($con) . "<br>";
    }
    
    $gameNumber++;
}

// Helper function to convert hex to RGB
function hexToRgb($hex) {
    $hex = str_replace(['#', '"'], '', $hex);
    if (strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    if (strlen($hex) != 6) {
        return array('r' => '000', 'g' => '000', 'b' => '000', 'rgb' => '000,000,000');
    }
    $r = addZeros(hexdec(substr($hex, 0, 2)));
    $g = addZeros(hexdec(substr($hex, 2, 2)));
    $b = addZeros(hexdec(substr($hex, 4, 2)));
    return array('r' => $r, 'g' => $g, 'b' => $b, 'rgb' => "$r,$g,$b");
}

echo "<br>EPL update complete at $time_updated<br>";

// ========== SOCCER GAMES UPDATE TRIGGER (AFTER DATABASE UPDATE) ==========
// Curl Soccer_Games.php if UPDATE_ANYWAY=true OR games starting within 10 min OR games in session
$should_update_soccer = $UPDATE_ANYWAY || $has_upcoming_games || $has_active_games;

if ($should_update_soccer) {
    echo "<br>===============================================<br>";
    echo "<strong>Soccer_Games.php Update Trigger</strong><br>";
    echo "===============================================<br>";
    
    // Log the trigger reason
    $trigger_reasons = [];
    if ($UPDATE_ANYWAY) $trigger_reasons[] = "UPDATE_ANYWAY=true";
    if ($has_upcoming_games) $trigger_reasons[] = "games starting within 10 minutes";
    if ($has_active_games) $trigger_reasons[] = "games currently in session";
    echo "Trigger reason(s): " . implode(", ", $trigger_reasons) . "<br>";
    echo "Calling Soccer_Games.php...<br>";
    
    $soccer_url = "https://gscroll.us/Scrolling/App_Output/API/Games/Soccer_Games.php";
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $soccer_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
    ]);
    
    $start_time = microtime(true);
    $soccer_result = curl_exec($ch);
    $end_time = microtime(true);
    $execution_time = round($end_time - $start_time, 2);
    
    $soccer_err = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $content_length = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
    curl_close($ch);
    
    if ($soccer_err) {
        echo "<span style='color:red;'>❌ Soccer_Games.php curl FAILED!</span><br>";
        echo "Error: " . $soccer_err . "<br>";
    } else {
        echo "<span style='color:green;'>✅ Soccer_Games.php called SUCCESSFULLY!</span><br>";
        echo "HTTP Status Code: " . $http_code . "<br>";
        echo "Response Size: " . number_format($content_length) . " bytes<br>";
        echo "Execution Time: " . $execution_time . " seconds<br>";
        
        // Show first 200 chars of response for debugging (optional)
        if ($http_code == 200) {
            echo "<span style='color:green;'>Status: SUCCESS - Soccer games updated!</span><br>";
        } else {
            echo "<span style='color:orange;'>Warning: Unexpected HTTP code " . $http_code . "</span><br>";
        }
    }
    echo "===============================================<br>";
} else {
    echo "<br>===============================================<br>";
    echo "Soccer_Games.php update NOT triggered<br>";
    echo "Reason: No active games, no upcoming games within 10 min, and UPDATE_ANYWAY not set<br>";
    echo "===============================================<br>";
}

mysqli_close($con);

echo "<br>All operations complete!";

?>