<?php

require_once '../config.php'; // Include the configuration file

// ========== SMART UPDATE CONTROL ==========
$UPDATE_ANYWAY = isset($_GET['UPDATE_ANYWAY']) && $_GET['UPDATE_ANYWAY'] === 'true';

// Helper function to parse ISO Zulu time to ET
function parseIsoZuluToET($isoStr) {
    if (!$isoStr) return null;
    try {
        $utc = new DateTime($isoStr, new DateTimeZone('UTC'));
        $utc->setTimezone(new DateTimeZone('America/New_York'));
        return $utc;
    } catch (Exception $e) {
        return null;
    }
}

// Establish database connection for preflight check
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Get current time in Eastern Time Zone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

// ======== PREFLIGHT: proceed only within [start-10m, start+5h] OR UPDATE_ANYWAY ========
$preflight = [
    'update_anyway' => (bool)$UPDATE_ANYWAY,
    'window_active' => false,
    'starts_within_10' => false,
    'proceed' => false,
    'notes' => [],
    'next_start_rows' => [],
    'next_start_closest' => null,
];

if ($UPDATE_ANYWAY) {
    $preflight['proceed'] = true;
    $preflight['notes'][] = 'Proceeding due to UPDATE_ANYWAY=true.';
} else {
    $nowET = new DateTime('now', new DateTimeZone('America/New_York'));

    // Pull La Liga game times from DB
    $sql = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played
            FROM sports_list
            WHERE League='LaLiga' AND Game_Date IS NOT NULL
            ORDER BY Game ASC";
    
    if ($rs = mysqli_query($con, $sql)) {
        while ($row = mysqli_fetch_assoc($rs)) {
            $startET = parseIsoZuluToET($row['Game_Date'] ?? null);
            if (!$startET) continue;

            $matchup = trim(($row['Team_2_Abbrev'] ?? '') . ' @ ' . ($row['Team_1_Abbrev'] ?? ''));
            $diffSec = $startET->getTimestamp() - $nowET->getTimestamp();
            
            if ($diffSec >= 0 && $diffSec <= 600) {
                $preflight['starts_within_10'] = true;
            }

            // Track for UI
            $preflight['next_start_rows'][] = [
                'game'    => (int)$row['Game'],
                'matchup' => $matchup,
                'startET' => $startET,
                'diffSec' => $diffSec,
            ];
            
            if ($preflight['next_start_closest'] === null ||
                $diffSec < $preflight['next_start_closest']['diffSec']) {
                $preflight['next_start_closest'] = [
                    'game'    => (int)$row['Game'],
                    'matchup' => $matchup,
                    'startET' => $startET,
                    'diffSec' => $diffSec,
                ];
            }

            // Window rule: start-10m .. start+5h
            $windowStart = (clone $startET)->modify('-10 minutes');
            $windowEnd   = (clone $startET)->modify('+5 hours');
            if ($nowET >= $windowStart && $nowET <= $windowEnd) {
                $preflight['window_active'] = true;
            }
        }
        mysqli_free_result($rs);
    }

    $preflight['proceed'] = $preflight['window_active'];
    if ($preflight['proceed']) {
        $preflight['notes'][] = 'Proceeding: within 10m-before to 5h-after window.';
    } else {
        $preflight['notes'][] = 'Blocked: outside all game windows.';
    }
}

// ===== If NOT proceeding, show status page and exit =====
if (!$preflight['proceed']) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
      <title>La Liga Updater — Preflight (No Run)</title>
      <style>
        body{font-family:Arial,sans-serif;background:#0b1320;color:#e6eefc;padding:24px}
        .card{background:#0f1a2b;border:1px solid #1e2b44;border-radius:8px;padding:16px;max-width:880px;margin:auto}
        .pill{display:inline-block;padding:4px 8px;border-radius:999px;font-size:12px;margin-right:6px;color:#fff}
        .pill-blue{background:#1e88e5}
        .pill-grey{background:#607d8b}
        .pill-orange{background:#ef6c00}
        .kv{display:grid;grid-template-columns:180px 1fr;gap:8px;margin-top:12px}
        code{background:#1e2b44;padding:2px 6px;border-radius:4px}
        table{width:100%;border-collapse:collapse;margin-top:10px}
        th,td{padding:6px 4px;border-bottom:1px solid #1e2b44;text-align:left;font-size:12px}
      </style>
    </head>
    <body>
      <div class="card">
        <h2 style="margin:0 0 8px 0;">La Liga Updater — Preflight</h2>
        <div>
          <span class="pill pill-blue">No Run</span>
          <?php if ($preflight['update_anyway']): ?>
            <span class="pill pill-orange">UPDATE_ANYWAY=true</span>
          <?php else: ?>
            <span class="pill pill-grey">Normal Mode</span>
          <?php endif; ?>
        </div>
        <div class="kv">
          <div><strong>Time (ET)</strong></div><div><?php echo htmlspecialchars($time_updated); ?></div>
          <div><strong>Starts ≤ 10 min?</strong></div><div><?php echo $preflight['starts_within_10'] ? 'Yes' : 'No'; ?></div>
        </div>

        <?php
        $closest = $preflight['next_start_closest'] ?? null;
        if ($closest && $closest['diffSec'] > 0) {
          $secs = (int)$closest['diffSec'];
          $mins = intdiv($secs, 60);
          $hrs  = intdiv($mins, 60);
          $mins = $mins % 60;
          $startStr = $closest['startET']->format('Y-m-d g:i A \E\T');
        ?>
        <div class="kv" style="margin-top:8px;">
          <div><strong>Closest La Liga start</strong></div>
          <div>
            <?php echo htmlspecialchars($closest['matchup']); ?> —
            <strong><?php echo htmlspecialchars($startStr); ?></strong>
            (starts in <strong><?php echo $hrs; ?>h <?php echo $mins; ?>m</strong>)
          </div>
        </div>
        <?php } ?>

        <?php if (!empty($preflight['notes'])): ?>
          <div style="margin-top:10px"><strong>Notes</strong>
            <ul style="margin:6px 0 0 18px;">
              <?php foreach ($preflight['notes'] as $n): ?>
                <li><?php echo htmlspecialchars($n); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
        <?php endif; ?>

        <?php if (!empty($preflight['next_start_rows'])): ?>
          <div style="margin-top:10px">
            <strong>Upcoming La Liga matches (ET)</strong>
            <table>
              <thead>
                <tr><th>Game</th><th>Matchup</th><th>Start (ET)</th><th>Starts In</th></tr>
              </thead>
              <tbody>
                <?php foreach ($preflight['next_start_rows'] as $row):
                  if ($row['diffSec'] < 0) continue; // Skip past games
                  $secs = (int)$row['diffSec'];
                  $mins = intdiv($secs, 60);
                  $hrs  = intdiv($mins, 60);
                  $mins = $mins % 60;
                  $startStr = $row['startET']->format('Y-m-d g:i A \E\T');
                ?>
                <tr>
                  <td><?php echo (int)$row['game']; ?></td>
                  <td><?php echo htmlspecialchars($row['matchup']); ?></td>
                  <td><?php echo htmlspecialchars($startStr); ?></td>
                  <td><?php echo $hrs; ?>h <?php echo $mins; ?>m</td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

        <div style="margin-top:10px">
          Tip: append <code>?UPDATE_ANYWAY=true</code> to force an update (ignores time window).
        </div>
      </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    die();
}

// ==================== MAIN UPDATE LOGIC STARTS HERE ====================

// Update API timestamp
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'LaLiga'";
if (mysqli_query($con, $sql)) {
    if (mysqli_affected_rows($con) > 0) {
        echo "Record updated successfully for API: LaLiga<br>";
    } else {
        echo "No records updated. Please check if the API entry exists.<br>";
    }
} else {
    echo "Error executing query: " . mysqli_error($con) . "<br>";
}

//************************************************//
// HELPER FUNCTIONS
//************************************************//

// function to clear out special chars 
function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

// function to add leading zeros
function addZeros($rgb) {
    $rgb = str_pad($rgb, 3, '0', STR_PAD_LEFT);
    return $rgb;
}

// function to get strings in between strings 
function getBetween($string, $start = "", $end = ""){
    if (strpos($string, $start)) {
        $startCharCount = strpos($string, $start) + strlen($start);
        $firstSubStr = substr($string, $startCharCount, strlen($string));
        $endCharCount = strpos($firstSubStr, $end);
        if ($endCharCount == 0) {
            $endCharCount = strlen($firstSubStr);
        }
        return substr($firstSubStr, 0, $endCharCount);
    } else {
        return '';
    }
}

// function to convert time 
function dateToTimezone($timeZone = 'UTC', $dateTimeUTC = null, $dateFormat = 'Y-m-d H:i:s'){
    $dateTimeUTC = $dateTimeUTC ? $dateTimeUTC : date("Y-m-d H:i:s");
    $date = new DateTime($dateTimeUTC, new DateTimeZone('UTC'));
    $date->setTimeZone(new DateTimeZone($timeZone));
    return $date->format($dateFormat);
}

// function to find the instance a word shows up
function strposX($haystack, $needle, $number = 0) {
    return strpos($haystack, $needle,
        $number > 1 ?
        strposX($haystack, $needle, $number - 1) + strlen($needle) : 0
    );
}

//************************************************//
// API CALL
//************************************************//

$api_call = "http://site.api.espn.com/apis/site/v2/sports/soccer/esp.1/scoreboard";

// Start API call
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    echo "cURL Error #:" . $err;
    exit;
}

//************************************************//
// HOW MANY GAMES ARE IN THE API CURRENTLY
//************************************************//

$loop_length = substr_count($result, '"homeAway"', 0) / 2; // total number of games
$active_games = substr_count($result, '"shortDownDistanceText"', 0);
$current_active_games = 0;

// Track if we have active or upcoming games
$has_active_games = false;
$has_upcoming_games = false;

echo "<!DOCTYPE html>";
echo "<html><head><title>La Liga Update Results</title>";
echo "<style>";
echo "body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }";
echo ".update-mode { background: " . ($UPDATE_ANYWAY ? '#ff9800' : '#4caf50') . "; color: white; padding: 5px 10px; border-radius: 5px; display: inline-block; }";
echo ".container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }";
echo ".game-update { margin: 10px 0; padding: 10px; background: #f9f9f9; border-left: 4px solid #ea0028; }";
echo ".success { color: green; } .error { color: red; }";
echo ".soccer-update { margin-top: 20px; padding: 15px; background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%); color: white; border-radius: 8px; }";
echo "</style></head><body>";
echo "<div class='container'>";
echo "<h1>⚽ La Liga Database Update <span class='update-mode'>" . ($UPDATE_ANYWAY ? 'FORCED UPDATE' : 'NORMAL MODE') . "</span></h1>";
echo "<p><strong>Time Updated:</strong> $time_updated</p>";
echo "<p><strong>Total Games:</strong> $loop_length</p>";
if ($UPDATE_ANYWAY) {
    echo "<p style='color: #ff9800;'><strong>⚠️ Update Mode:</strong> FORCED (UPDATE_ANYWAY=true)</p>";
}
echo "<hr>";

//************************************************//
// START PARSING DATA
//************************************************//

for ($i = 1; $i <= $loop_length; $i++) {
    
    // Initialize variables
    $score_value = 'NO-SCORE';
    $who_has_ball = '';
    $short_distance_text = '';
    $pos_text = '';
    $favorite_team = '';
    $favorite_points = '';
    $lastPlay = '';
    $overUnder = '';
    $is_red_zone = '';
    
    // Get game clock time
    $start_suffix = '"displayClock":"';
    $end_suffix = ',"period":';
    $index_start = strposX($result, $start_suffix, $i*2) + 16;
    $index_end = strposX($result, $end_suffix, $i*2) - $index_start - 1;
    $clock = substr($result, $index_start, $index_end);
    $clock = clean($clock);
    
    if ($clock > 100) {
        $clock = substr($clock, 0, 2) + substr($clock, 2, 3);
    }
    
    if (strlen($clock) > 10) {
        $clock_value1 = substr($clock, 0, 4);
        $clock = $clock_value1;
        $clock = preg_replace("/[^0-9]/", "", $clock);
    }

    // Get period
    $start_suffix = '"periods":';
    $index_start = strposX($result, $start_suffix, $i) + strlen($start_suffix);
    $index_end = 1;
    $quarter = substr($result, $index_start, $index_end);

    // Get game time
    $start_suffix = '"attendance":';
    $index_start = strposX($result, $start_suffix, $i) - 19;
    $index_end = 18;
    $game_time = substr($result, $index_start, $index_end);
    $game_time = str_replace('"', "", $game_time);

    // Determine game status
    $over_time_flag = FALSE;
    if ($game_time > gmdate("Y-m-d\TH:i:s\Z") & $quarter == 0 | $quarter == 0 & $clock == "0.0") {
        $game_being_played = 'NOT_STARTED';
        
        // Check if game starts within 10 minutes
        try {
            $gameStartTime = new DateTime($game_time, new DateTimeZone('UTC'));
            $nowTime = new DateTime('now', new DateTimeZone('UTC'));
            $diffSeconds = $gameStartTime->getTimestamp() - $nowTime->getTimestamp();
            if ($diffSeconds >= 0 && $diffSeconds <= 600) {
                $has_upcoming_games = true;
            }
        } catch (Exception $e) {
            // Handle date parsing error
        }
    } elseif ($game_time < gmdate("Y-m-d\TH:i:s\Z")) {
        if ($quarter == 4 & $clock == "0.0" || ($quarter == 5 & $clock == "0.0")) {
            $game_being_played = 'GAME_DONE';
        } elseif ($quarter == 2 & $clock == "0.0") {
            $game_being_played = 'HALF_TIME';
            $has_active_games = true;
        } else {
            $game_being_played = 'IN_SESSION';
            $has_active_games = true;
        }
    }

    // Check game state more specifically
    $start_suffix = '"displayClock":"';
    $end_suffix = '"shortDetail":"';
    $index_start = strposX($result, $start_suffix, $i*2) + 16;
    $index_end = strposX($result, $end_suffix, $i*2) - $index_start;
    $check_game = substr($result, $index_start, $index_end);

    if (strpos($check_game, 'completed":true') > 10) {
        $game_being_played = 'GAME_DONE';
    }
    if (strpos($check_game, 'completed":false') > 10) {
        $game_being_played = 'NOT_STARTED';
    }
    if (strpos($check_game, ',"state":"in","completed":false,') > 10) {
        $game_being_played = 'IN_SESSION';
        $has_active_games = true;
    }
    if (strpos($check_game, '"name":"STATUS_HALFTIME"') > 10) {
        $game_being_played = 'HALF_TIME';
        $has_active_games = true;
    }

    $game_time_2 = date("g:i a", strtotime($game_time) - 14400); // Eastern time

    // Extract team data
    $start_suffix = '"homeAway"';
    $index_start = strposX($result, $start_suffix, $i*2-1);
    $index_end = strposX($result, $start_suffix, $i*2) - $index_start;
    $team1_result = substr($result, $index_start, $index_end);

    $index_start = strposX($result, $start_suffix, $i*2);
    $index_end = strposX($result, $start_suffix, $i*2+1) - $index_start;
    $team2_result = substr($result, $index_start, $index_end);

    if ($i == $loop_length) {
        $start_suffix = '"homeAway"';
        $index_start = strposX($result, $start_suffix, $i*2-1);
        $index_end = strposX($result, $start_suffix, $i*2) - $index_start;
        $team1_result = substr($result, $index_start, $index_end);

        $index_start = strposX($result, $start_suffix, $i*2);
        $team2_result = substr($result, $index_start);
    }

    // Extract broadcast information (NEW)
    $broadcast = '';
    $broadcast_pattern = '/"broadcasts":\[.*?"names":\["([^"]+)"/';
    $game_section_start = strposX($result, '"competitions"', $i);
    $game_section_end = ($i < $loop_length) ? strposX($result, '"competitions"', $i+1) : strlen($result);
    $game_section = substr($result, $game_section_start, $game_section_end - $game_section_start);
    
    if (preg_match($broadcast_pattern, $game_section, $broadcast_matches)) {
        $broadcast = $broadcast_matches[1];
    }

    // Get Team Names
    $start_suffix = '"displayName":';
    $end_suffix = ',"shortDisplayName":';
    $team_name1_start = strposX($team1_result, $start_suffix, 1) + strlen($start_suffix);
    $team_name1_end = strposX($team1_result, $end_suffix, 1) - $team_name1_start;
    $team_name1 = substr($team1_result, $team_name1_start, $team_name1_end);
    $team_name1 = clean($team_name1);

    $team_name2_start = strposX($team2_result, $start_suffix, 1) + strlen($start_suffix);
    $team_name2_end = strposX($team2_result, $end_suffix, 1) - $team_name2_start;
    $team_name2 = substr($team2_result, $team_name2_start, $team_name2_end);
    $team_name2 = clean($team_name2);

    // Get Team Records
    $start_suffix = '"total","summary":';
    $end_suffix = ',"abbreviation":';
    $team1_record_start = strposX($team1_result, $start_suffix, 1) + strlen($start_suffix);
    $team1_record_end = strposX($team1_result, $end_suffix, 1) - $team1_record_start - 1;
    $team1_record = substr($team1_result, $team1_record_start, $team1_record_end);
    $team1_record = clean($team1_record);
    if (strlen($team1_record) > 20) {
        $team1_record = '';
    }

    $team2_record_start = strposX($team2_result, $start_suffix, 1) + strlen($start_suffix);
    $team2_record_end = strposX($team2_result, $end_suffix, 1) - $team2_record_start - 1;
    $team2_record = substr($team2_result, $team2_record_start, $team2_record_end);
    $team2_record = clean($team2_record);
    if (strlen($team2_record) > 20) {
        $team2_record = '';
    }

    // Get Scores
    if ($game_being_played == 'GAME_DONE' | $game_being_played == 'IN_SESSION' | $game_being_played == 'HALF_TIME') {
        $start_suffix = ',"score":"';
        $end_suffix = ',"records":[{';
        $team1_score_start = strposX($team1_result, $start_suffix, 1) + strlen(',"score":"');
        $team1_score_end = strposX($team1_result, $end_suffix, 1) - $team1_score_start;
        $team1_score = substr($team1_result, $team1_score_start, $team1_score_end);
        $team1_score = clean($team1_score);

        $team2_score_start = strposX($team2_result, $start_suffix, 1) + strlen(',"score":"');
        $team2_score_end = strposX($team2_result, $end_suffix, 1) - $team2_score_start;
        $team2_score = substr($team2_result, $team2_score_start, $team2_score_end);
        $team2_score = clean($team2_score);
    } else {
        $team1_score = 0;
        $team2_score = 0;
    }

    if (strlen($team1_score) > 10) $team1_score = 0;
    if (strlen($team2_score) > 10) $team2_score = 0;

    // Get Team Locations
    $start_suffix = '"location":"';
    $end_suffix = ',"color":';
    $location_team1_start = strposX($team1_result, $start_suffix, 1) + strlen($start_suffix);
    $location_team1_end = strposX($team1_result, $end_suffix, 1) - $location_team1_start;
    $location_team1 = substr($team1_result, $location_team1_start, $location_team1_end);
    $location_team1 = clean($location_team1);

    $location_team2_start = strposX($team2_result, $start_suffix, 1) + strlen($start_suffix);
    $location_team2_end = strposX($team2_result, $end_suffix, 1) - $location_team2_start;
    $location_team2 = substr($team2_result, $location_team2_start, $location_team2_end);
    $location_team2 = clean($location_team2);

    // Get Abbreviations
    $value_start = '"abbreviation":';
    $value_end = ',"displayName":"';
    $start_suffix = strposX($team1_result, $value_start, 2) + strlen($value_start);
    $end_suffix = strposX($team1_result, $value_end, 1) - $start_suffix;
    $team_abbrev_1 = substr($team1_result, $start_suffix, $end_suffix);
    $team_abbrev_1 = str_replace('"', "", $team_abbrev_1);

    $start_suffix = strposX($team2_result, $value_start, 2) + strlen($value_start);
    $end_suffix = strposX($team2_result, $value_end, 1) - $start_suffix;
    $team_abbrev_2 = substr($team2_result, $start_suffix, $end_suffix);
    $team_abbrev_2 = str_replace('"', "", $team_abbrev_2);

    // Get Colors
    $value_start = ',"color":"';
    $value_end = ',"alternateColor":';
    $start_suffix = strposX($team1_result, $value_start, 1) + strlen($value_start);
    $end_suffix = strposX($team1_result, $value_end, 1) - $start_suffix;
    $team1_color_hex = substr($team1_result, $start_suffix, $end_suffix);
    $team1_color_hex = clean($team1_color_hex);

    $split = str_split($team1_color_hex, 2);
    $r = hexdec($split[0]);
    $g = hexdec($split[1]);
    $b = hexdec($split[2]);

    $r = addZeros($r);
    $g = addZeros($g);
    $b = addZeros($b);

    if ($team1_color_hex == "000000") {
        $value_start = ',"alternateColor":';
        $value_end = ',"isActive"';
        $start_suffix = strposX($team1_result, $value_start, 1) + strlen($value_start);
        $end_suffix = strposX($team1_result, $value_end, 1) - $start_suffix;
        $team1_color_hex = substr($team1_result, $start_suffix, $end_suffix);
        $team1_color_hex = clean($team1_color_hex);
        
        $split = str_split($team1_color_hex, 2);
        $r = hexdec($split[0]);
        $g = hexdec($split[1]);
        $b = hexdec($split[2]);
        
        $r = addZeros($r);
        $g = addZeros($g);
        $b = addZeros($b);
    }

    $team1_color_r = $r;
    $team1_color_g = $g;
    $team1_color_b = $b;
    $team1_color_rgb = sprintf('"%s,%s,%s"', $r, $g, $b);

    // Team 2 colors
    $value_start = ',"color":"';
    $value_end = ',"alternateColor":';
    $start_suffix = strposX($team2_result, $value_start, 1) + strlen($value_start);
    $end_suffix = strposX($team2_result, $value_end, 1) - $start_suffix;
    $team2_color_hex = substr($team2_result, $start_suffix, $end_suffix);
    $team2_color_hex = clean($team2_color_hex);

    $split = str_split($team2_color_hex, 2);
    $r = hexdec($split[0]);
    $g = hexdec($split[1]);
    $b = hexdec($split[2]);

    $r = addZeros($r);
    $g = addZeros($g);
    $b = addZeros($b);

    if ($team2_color_hex == "000000") {
        $value_start = ',"alternateColor":';
        $value_end = ',"isActive"';
        $start_suffix = strposX($team2_result, $value_start, 1) + strlen($value_start);
        $end_suffix = strposX($team2_result, $value_end, 1) - $start_suffix;
        $team2_color_hex = substr($team2_result, $start_suffix, $end_suffix);
        $team2_color_hex = clean($team2_color_hex);
        
        $split = str_split($team2_color_hex, 2);
        $r = hexdec($split[0]);
        $g = hexdec($split[1]);
        $b = hexdec($split[2]);
        
        $r = addZeros($r);
        $g = addZeros($g);
        $b = addZeros($b);
    }

    $team2_color_r = $r;
    $team2_color_g = $g;
    $team2_color_b = $b;
    $team2_color_rgb = sprintf('"%s,%s,%s"', $r, $g, $b);

    // Get Team IDs
    $value_start = ',"team":{"id":';
    $value_end = ',"uid":"s';
    $start_suffix = strposX($team1_result, $value_start, 1) + strlen($value_start);
    $end_suffix = strposX($team1_result, $value_end, 1) - $start_suffix;
    $team1_ID = substr($team1_result, $start_suffix, $end_suffix);
    $team1_ID = clean($team1_ID);

    $start_suffix = strposX($team2_result, $value_start, $i) + strlen($value_start);
    $end_suffix = strposX($team2_result, $value_end, $i) - $start_suffix;
    $team2_ID = substr($team2_result, $start_suffix, $end_suffix);
    $team2_ID = clean($team2_ID);

    // Check for Score Values (Goals within last 2 minutes)
    $loop_length_SV = substr_count($result, '"scoreValue"', 0);

    if ($game_being_played == 'IN_SESSION') {
        for ($j = 1; $j <= $loop_length_SV; $j++) {
            // Reinitialize flags
            $team_ID_FLAG = 'FALSE';
            $time_since_score_FLAG = 'FALSE';
            $was_there_a_score_FLAG = 'FALSE';
            $scoreValue_Flag = 'FALSE';
            
            $value_start = '},"scoreValue"';
            $start_suffix = strposX($result, $value_start, $j) - 50;
            $scoreValueString = substr($result, $start_suffix, 150);

            $team_id_temp = getBetween($scoreValueString, $start = ',"team":{"id":', $end = '"},"scoreValue"');
            $team_id_temp = clean($team_id_temp);
            
            // Team ID check
            if ($team_id_temp == $team1_ID || $team_id_temp == $team2_ID) {
                $team_ID_FLAG = 'TRUE';
            }

            if ($team_id_temp == $team1_ID) {
                $who_has_ball = $team_abbrev_1;
            }
            if ($team_id_temp == $team2_ID) {
                $who_has_ball = $team_abbrev_2;
            }

            // Score time check
            $displayClock_temp = getBetween($scoreValueString, $start = ',"displayValue":"', $end = ',"team":{"id');
            $displayClock_temp = clean($displayClock_temp);

            if ($clock - $displayClock_temp <= 2) {
                $time_since_score_FLAG = 'TRUE';
            }
            
            $scoreValue_temp = getBetween($scoreValueString, $start = ',"scoreValue":', $end = ',"scoringPlay"');

            if ($scoreValue_temp == 1) {
                $was_there_a_score_FLAG = 'TRUE';
            }

            // Must meet all requirements
            if ($team_ID_FLAG == 'TRUE' && $time_since_score_FLAG == 'TRUE' && $was_there_a_score_FLAG == 'TRUE') {
                $scoreValue_Flag = 'TRUE';
            } else {
                $scoreValue_Flag = 'FALSE';
            }

            if ($scoreValue_Flag == 'TRUE') {
                $score_value = 'GOAL';
            } else {
                $score_value = 'NO-SCORE';
            }
        }
    }

    // Build the UPDATE query
    $query = "UPDATE sports_list SET 
        Clock='" . mysqli_real_escape_string($con, $clock) . "',
        Period='" . mysqli_real_escape_string($con, $quarter) . "',
        Game_Played='" . mysqli_real_escape_string($con, $game_being_played) . "',
        Game_Time='" . mysqli_real_escape_string($con, $game_time_2) . "',
        Game_Date='" . mysqli_real_escape_string($con, $game_time) . "',
        scoreValue='" . mysqli_real_escape_string($con, $score_value) . "',
        Team_1_Name='" . mysqli_real_escape_string($con, $team_name1) . "',
        Team_2_Name='" . mysqli_real_escape_string($con, $team_name2) . "',
        Team_1_Record='" . mysqli_real_escape_string($con, $team1_record) . "',
        Team_2_Record='" . mysqli_real_escape_string($con, $team2_record) . "',
        Team_1_Score='" . mysqli_real_escape_string($con, $team1_score) . "',
        Team_2_Score='" . mysqli_real_escape_string($con, $team2_score) . "',
        Team_1_Location='" . mysqli_real_escape_string($con, $location_team1) . "',
        Team_2_Location='" . mysqli_real_escape_string($con, $location_team2) . "',
        Team_1_Abbrev='" . mysqli_real_escape_string($con, $team_abbrev_1) . "',
        Team_1_Color='" . mysqli_real_escape_string($con, $team1_color_rgb) . "',
        Team_1_Color_R='" . mysqli_real_escape_string($con, $team1_color_r) . "',
        Team_1_Color_G='" . mysqli_real_escape_string($con, $team1_color_g) . "',
        Team_1_Color_B='" . mysqli_real_escape_string($con, $team1_color_b) . "',
        Team_2_Abbrev='" . mysqli_real_escape_string($con, $team_abbrev_2) . "',
        Team_2_Color='" . mysqli_real_escape_string($con, $team2_color_rgb) . "',
        Team_2_Color_R='" . mysqli_real_escape_string($con, $team2_color_r) . "',
        Team_2_Color_G='" . mysqli_real_escape_string($con, $team2_color_g) . "',
        Team_2_Color_B='" . mysqli_real_escape_string($con, $team2_color_b) . "',
        Broadcast='" . mysqli_real_escape_string($con, $broadcast) . "',
        Who_Has_Ball='" . mysqli_real_escape_string($con, $who_has_ball) . "',
        Distance='" . mysqli_real_escape_string($con, $short_distance_text) . "',
        Possesion='" . mysqli_real_escape_string($con, $pos_text) . "',
        Favorite_Team='" . mysqli_real_escape_string($con, $favorite_team) . "',
        Favorite_Points='" . mysqli_real_escape_string($con, $favorite_points) . "',
        lastPlay='" . mysqli_real_escape_string($con, $lastPlay) . "',
        OverUnder='" . mysqli_real_escape_string($con, $overUnder) . "',
        Red_Zone='" . mysqli_real_escape_string($con, $is_red_zone) . "' 
        WHERE Game = $i AND League = 'LaLiga'";

    // Execute query
    $query_run = mysqli_query($con, $query);
    
    // Output result
    echo "<div class='game-update'>";
    echo "<strong>Game $i:</strong> $team_abbrev_2 @ $team_abbrev_1 | ";
    echo "Score: $team2_score - $team1_score | ";
    echo "Status: $game_being_played | ";
    echo "Broadcast: " . ($broadcast ? $broadcast : 'N/A') . " | ";
    
    if ($query_run) {
        echo "<span class='success'>✓ Updated Successfully</span>";
    } else {
        echo "<span class='error'>✗ Update Failed: " . mysqli_error($con) . "</span>";
    }
    echo "</div>";
}

echo "<hr>";
echo "<p><strong>Update Complete at:</strong> $time_updated</p>";

// ========== SOCCER GAMES UPDATE TRIGGER (AFTER DATABASE UPDATE) ==========
// Curl Soccer_Games.php if UPDATE_ANYWAY=true OR games starting within 10 min OR games in session
$should_update_soccer = $UPDATE_ANYWAY || $has_upcoming_games || $has_active_games;

if ($should_update_soccer) {
    echo "<div class='soccer-update'>";
    echo "<strong>⚽ Soccer_Games.php Update Trigger</strong><br>";
    echo "===============================================<br>";
    
    // Log the trigger reason
    $trigger_reasons = [];
    if ($UPDATE_ANYWAY) $trigger_reasons[] = "UPDATE_ANYWAY=true";
    if ($has_upcoming_games) $trigger_reasons[] = "games starting within 10 minutes";
    if ($has_active_games) $trigger_reasons[] = "games currently in session";
    echo "Trigger reason(s): " . implode(", ", $trigger_reasons) . "<br>";
    echo "Calling Soccer_Games.php...<br><br>";
    
    $soccer_url = "https://gscroll.us/Scrolling/App_Output/API/Games/Soccer_Games.php";
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $soccer_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
    ]);
    
    $start_time = microtime(true);
    $soccer_result = curl_exec($ch);
    $end_time = microtime(true);
    $execution_time = round($end_time - $start_time, 2);
    
    $soccer_err = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $content_length = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
    curl_close($ch);
    
    if ($soccer_err) {
        echo "<span style='color:#ffcdd2;'>❌ Soccer_Games.php curl FAILED!</span><br>";
        echo "Error: " . $soccer_err . "<br>";
    } else {
        echo "<span style='color:#c8e6c9;'>✅ Soccer_Games.php called SUCCESSFULLY!</span><br>";
        echo "HTTP Status Code: " . $http_code . "<br>";
        echo "Response Size: " . number_format($content_length) . " bytes<br>";
        echo "Execution Time: " . $execution_time . " seconds<br>";
        
        if ($http_code == 200) {
            echo "<span style='color:#c8e6c9;'>Status: SUCCESS - Soccer games updated!</span><br>";
        } else {
            echo "<span style='color:#fff3e0;'>Warning: Unexpected HTTP code " . $http_code . "</span><br>";
        }
    }
    echo "===============================================";
    echo "</div>";
} else {
    echo "<div class='soccer-update' style='background: linear-gradient(135deg, #9e9e9e 0%, #757575 100%);'>";
    echo "<strong>Soccer_Games.php Update Status</strong><br>";
    echo "Update NOT triggered<br>";
    echo "Reason: No active games, no upcoming games within 10 min, and UPDATE_ANYWAY not set";
    echo "</div>";
}

// Close connection
mysqli_close($con);

echo "</div></body></html>";

?>