<?php
/*************************************************
 * MLB Updater (Simple Rules) + Preflight Status Output
 * - Adds visible preflight reasons (forced / in-session / 10-min window / blackout)
 * - Blackout (2:00–8:59 ET): exit before any DB/API calls
 * - UPDATE_ANYWAY override (URL or constant) still respects blackout
 *************************************************/
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../config.php';
if (!defined('SERVER_URL')) define('SERVER_URL', 'http://localhost/');

// ===== FEATURE FLAGS (unchanged) =====
$PLAY_OFF = true;
$ENABLE_PBP_ENRICHMENT = true;   // still used for DELTA handling
$PAUSE_SECONDS = 150;

// ===== TIME (ET) =====
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp_dt = date('Y-m-d H:i:s');

// ===== UPDATE_ANYWAY (constant or URL) =====
if (!defined('UPDATE_ANYWAY')) { define('UPDATE_ANYWAY', false); }
$UPDATE_ANYWAY = UPDATE_ANYWAY;
if (isset($_GET['UPDATE_ANYWAY'])) {
  $val = strtolower(trim($_GET['UPDATE_ANYWAY']));
  $UPDATE_ANYWAY = in_array($val, ['1','true','yes','on'], true);
}

// ===== PREFLIGHT STATUS STRUCT =====
$preflight = [
  'blackout'        => false,
  'update_anyway'   => (bool)$UPDATE_ANYWAY,
  'in_session_in_db'=> false,
  'starts_within_10'=> false,
  'proceed'         => false,
  'notes'           => [],
];

/**
 * ============================================================
 * =============== CHANGED PREFLIGHT SECTION ==================
 * ============================================================
 * Logic you asked for:
 * - Respect blackout: 2:00–8:59 AM ET
 * - Proceed if UPDATE_ANYWAY OR any IN_SESSION in DB OR NOW in any [start-10m, start+5h] window
 * - Stop if every MLB game is GAME_DONE (unless UPDATE_ANYWAY)
 * - Keep UI fields (starts_within_10 / next_start_* ) populated
 */

// ----- (1) BLACKOUT CHECK: 02:00–08:59 ET -----
$nowET = new DateTime('now', new DateTimeZone('America/New_York'));
$h = (int)$nowET->format('G');   // 0..23
$inBlackout = ($h >= 2 && $h <= 8); // inclusive: 02:00 through 08:59
if ($inBlackout) {
  $preflight['blackout'] = true;
  ?>
  <!DOCTYPE html>
  <html>
  <head>
    <title>MLB Updater — Blackout</title>
    <style>
      body{font-family:Arial, sans-serif;background:#111;color:#fff;padding:24px}
      .card{background:#222;border:1px solid #333;border-radius:8px;padding:16px;max-width:720px}
      .pill{display:inline-block;padding:4px 8px;border-radius:999px;font-size:12px;margin-right:6px}
      .pill-red{background:#b71c1c}
      .pill-grey{background:#424242}
      .row{margin:6px 0}
      code{background:#333;padding:2px 6px;border-radius:4px}
    </style>
  </head>
  <body>
    <div class="card">
      <h2 style="margin:0 0 8px 0;">MLB Updater — Preflight</h2>
      <div class="row">
        <span class="pill pill-red">BLACKOUT WINDOW</span>
        <span class="pill pill-grey">2:00–8:59 AM ET</span>
      </div>
      <div class="row">Time now (ET): <strong><?php echo htmlspecialchars($time_updated); ?></strong></div>
      <div class="row">No DB/API calls were made. This page is just confirming the blackout rule is active.</div>
      <?php if ($UPDATE_ANYWAY): ?>
      <div class="row" style="margin-top:10px">
        ⚠️ <strong>UPDATE_ANYWAY was set</strong> but blackout takes precedence, so the script did not run.
        Try again after 9:00 AM ET with <code>?UPDATE_ANYWAY=true</code> if you want to force an update then.
      </div>
      <?php endif; ?>
    </div>
  </body>
  </html>
  <?php
  die();
}

// ===== DB CONNECT (safe after blackout) =====
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) { die("Connection failed: " . mysqli_connect_error()); }

// ===== PREFLIGHT DECISION (proceed?) =====
if ($UPDATE_ANYWAY) {
  $preflight['proceed'] = true;
  $preflight['notes'][] = 'Proceeding due to UPDATE_ANYWAY=true.';
} else {
  // (a) any IN_SESSION?
  $chk_sql = "SELECT 1 FROM sports_list WHERE Conference='MLB' AND Game_Played='IN_SESSION' LIMIT 1";
  if ($rs = mysqli_query($con, $chk_sql)) {
    if (mysqli_fetch_row($rs)) { $preflight['in_session_in_db'] = true; }
    mysqli_free_result($rs);
  }

  // Helper: parse "YYYY-mm-ddTHH:iiZ" (UTC) into ET (kept as in your code)
  if (!function_exists('parseIsoZuluToET')) {
    function parseIsoZuluToET(?string $raw) : ?DateTime {
      if (!$raw) return null;
      $raw = trim($raw);
      $dt = DateTime::createFromFormat('Y-m-d\TH:i\Z', $raw, new DateTimeZone('UTC'));
      if ($dt === false) {
        try { $dt = new DateTime($raw, new DateTimeZone('UTC')); }
        catch (Exception $e) { return null; }
      }
      $dt->setTimezone(new DateTimeZone('America/New_York'));
      return $dt;
    }
  }

  // (b) WINDOW RULE: any game with NOW in [startET - 10m, startET + 5h] keeps updater running
  $preflight['starts_within_10']   = false;   // for your UI
  $preflight['window_active']      = false;   // NEW: any window hit
  $preflight['all_done']           = false;   // NEW: every game GAME_DONE
  $preflight['next_start_closest'] = null;
  $preflight['next_start_rows']    = [];

  $total_games = 0;
  $done_count  = 0;

  $q = "SELECT Game, Team_1_Abbrev, Team_2_Abbrev, Game_Date, Game_Played
        FROM sports_list
        WHERE Conference='MLB' AND Game_Date IS NOT NULL
        ORDER BY Game ASC";
  if ($r = mysqli_query($con, $q)) {
    while ($row = mysqli_fetch_assoc($r)) {
      $total_games++;
      $status = strtoupper(trim($row['Game_Played'] ?? ''));
      if ($status === 'GAME_DONE') $done_count++;

      $startET = parseIsoZuluToET($row['Game_Date'] ?? null);
      if (!$startET) continue;

      $match = trim(($row['Team_2_Abbrev'] ?? '').' @ '.($row['Team_1_Abbrev'] ?? ''));

      // Populate UI list + starts_within_10 (same as before)
      $diffSec = $startET->getTimestamp() - $nowET->getTimestamp();
      if ($diffSec >= 0 && $diffSec <= 600) $preflight['starts_within_10'] = true;

      $preflight['next_start_rows'][] = [
        'game'    => (int)$row['Game'],
        'matchup' => $match,
        'startET' => $startET,
        'diffSec' => $diffSec,
      ];
      if ($preflight['next_start_closest'] === null ||
          $diffSec < $preflight['next_start_closest']['diffSec']) {
        $preflight['next_start_closest'] = [
          'game'    => (int)$row['Game'],
          'matchup' => $match,
          'startET' => $startET,
          'diffSec' => $diffSec,
        ];
      }

      // Window check: startET - 10 minutes .. startET + 5 hours
      $windowStart = (clone $startET)->modify('-10 minutes');
      $windowEnd   = (clone $startET)->modify('+5 hours');
      if ($nowET >= $windowStart && $nowET <= $windowEnd) {
        $preflight['window_active'] = true;
      }
    }
    mysqli_free_result($r);
  }

  // All-done determination (only if we saw at least one game row)
  if ($total_games > 0 && $done_count === $total_games) {
    $preflight['all_done'] = true;
  }

  // Proceed decision: any IN_SESSION OR any active window; stop only if ALL DONE
  $preflight['proceed'] = ($preflight['in_session_in_db'] || $preflight['window_active']) && !$preflight['all_done'];

  if ($preflight['in_session_in_db']) $preflight['notes'][] = 'Proceeding: DB shows at least one IN_SESSION game.';
  if ($preflight['window_active'])    $preflight['notes'][] = 'Proceeding: within 10m-before to 5h-after window.';
  if ($preflight['all_done'])         $preflight['notes'][] = 'Blocked: all MLB games are GAME_DONE.';
  if (!$preflight['proceed'] && !$preflight['all_done']) {
    $preflight['notes'][] = 'Blocked: no IN_SESSION game and no active game window.';
  }
}
/**
 * ================= END CHANGED PREFLIGHT =====================
 */

// ===== If NOT proceeding, show a small status page and exit =====
if (!$preflight['proceed']) {
  ?>
  <!DOCTYPE html>
  <html>
  <head>
    <title>MLB Updater — Preflight (No Run)</title>
    <style>
      body{font-family:Arial, sans-serif;background:#0b1320;color:#e6eefc;padding:24px}
      .card{background:#0f1a2b;border:1px solid #1e2b44;border-radius:8px;padding:16px;max-width:880px}
      .pill{display:inline-block;padding:4px 8px;border-radius:999px;font-size:12px;margin-right:6px}
      .pill-blue{background:#1e88e5}
      .pill-grey{background:#455a64}
      .kv{display:grid;grid-template-columns:180px 1fr;gap:8px;margin-top:12px}
      code{background:#1e2b44;padding:2px 6px;border-radius:4px}
      ul{margin:8px 0 0 18px}
    </style>
  </head>
  <body>
    <div class="card">
      <h2 style="margin:0 0 8px 0;">MLB Updater — Preflight</h2>
      <div>
        <span class="pill pill-blue">No Run</span>
        <?php if ($preflight['update_anyway']): ?><span class="pill pill-grey">UPDATE_ANYWAY=true</span><?php endif; ?>
      </div>
      <div class="kv">
        <div><strong>Time (ET)</strong></div><div><?php echo htmlspecialchars($time_updated); ?></div>
        <div><strong>IN_SESSION in DB?</strong></div><div><?php echo $preflight['in_session_in_db'] ? 'Yes' : 'No'; ?></div>
        <div><strong>Starts within 10 min?</strong></div><div><?php echo $preflight['starts_within_10'] ? 'Yes' : 'No'; ?></div>
        <div><strong>Forced (UPDATE_ANYWAY)?</strong></div><div><?php echo $preflight['update_anyway'] ? 'Yes' : 'No'; ?></div>
      </div>

      <?php
      $closest = $preflight['next_start_closest'] ?? null;
      if ($closest) {
        $secs = (int)$closest['diffSec'];
        $mins = intdiv($secs, 60);
        $hrs  = intdiv($mins, 60);
        $mins = $mins % 60;
        $startStr = $closest['startET']->format('Y-m-d g:i A \E\T');
      ?>
      <div class="kv" style="margin-top:8px;">
        <div><strong>Closest MLB start</strong></div>
        <div>
          <?php echo htmlspecialchars($closest['matchup']); ?> —
          <strong><?php echo htmlspecialchars($startStr); ?></strong>
          (starts in <strong><?php echo $hrs; ?>h <?php echo $mins; ?>m</strong>)
        </div>
      </div>
      <?php } ?>

      <?php if (!empty($preflight['notes'])): ?>
      <div style="margin-top:10px"><strong>Notes</strong>
        <ul>
          <?php foreach ($preflight['notes'] as $n): ?>
            <li><?php echo htmlspecialchars($n); ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
      <?php endif; ?>

      <?php if (!empty($preflight['next_start_rows'])): ?>
      <div style="margin-top:10px">
        <strong>Upcoming MLB starts (ET)</strong>
        <table style="width:100%;border-collapse:collapse;margin-top:6px;">
          <thead>
            <tr>
              <th style="text-align:left;padding:4px;border-bottom:1px solid #1e2b44;">Game</th>
              <th style="text-align:left;padding:4px;border-bottom:1px solid #1e2b44;">Matchup</th>
              <th style="text-align:left;padding:4px;border-bottom:1px solid #1e2b44;">Start (ET)</th>
              <th style="text-align:left;padding:4px;border-bottom:1px solid #1e2b44;">Starts In</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach (($preflight['next_start_rows'] ?? []) as $row):
              $secs = (int)$row['diffSec'];
              $mins = intdiv($secs, 60);
              $hrs  = intdiv($mins, 60);
              $mins = $mins % 60;
              $startStr = $row['startET']->format('Y-m-d g:i A \E\T');
            ?>
              <tr>
                <td style="padding:4px;border-bottom:1px solid #1e2b44;"><?php echo (int)$row['game']; ?></td>
                <td style="padding:4px;border-bottom:1px solid #1e2b44;"><?php echo htmlspecialchars($row['matchup']); ?></td>
                <td style="padding:4px;border-bottom:1px solid #1e2b44;"><?php echo htmlspecialchars($startStr); ?></td>
                <td style="padding:4px;border-bottom:1px solid #1e2b44;"><?php echo $hrs; ?>h <?php echo $mins; ?>m</td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <?php endif; ?>

      <div style="margin-top:10px">
        Tip: append <code>?UPDATE_ANYWAY=true</code> to force an update (outside blackout).
      </div>
    </div>
  </body>
  </html>
  <?php
  mysqli_close($con);
  die();
}

// ===== From here on, your existing logic remains unchanged =====

// ===== SCHEMA ENSURE =====
$columns_to_check = [
  'scoreValue','lastUpdatedTime','lastPlay','lastPlayerID','lastPlayerName',
  'lastPlayerPos','lastPlayerTeamID','jersey','broadcast','outsB','onBases',
  'team1HE','team2HE','Who_Has_Ball','Favorite_Team','Favorite_Points','OverUnder',
  'Team_1_Color','Team_1_Color_R','Team_1_Color_G','Team_1_Color_B',
  'Team_2_Color','Team_2_Color_R','Team_2_Color_G','Team_2_Color_B'
];
foreach ($columns_to_check as $column) {
  $check = mysqli_query($con, "SHOW COLUMNS FROM sports_list LIKE '$column'");
  if (mysqli_num_rows($check) == 0) {
    $type = 'VARCHAR(255)';
    if ($column === 'lastUpdatedTime') $type = 'DATETIME';
    if ($column === 'broadcast')      $type = 'VARCHAR(50)';
    mysqli_query($con, "ALTER TABLE sports_list ADD COLUMN $column $type DEFAULT NULL");
  }
}

// ===== HELPERS (unchanged) =====
function hexToRgb($hex){
  $hex = str_replace(['#','"'], '', $hex);
  if (strlen($hex) == 3) $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
  if (strlen($hex) != 6) return ['r'=>'000','g'=>'000','b'=>'000','rgb'=>'000,000,000'];
  $r = str_pad(hexdec(substr($hex,0,2)),3,'0',STR_PAD_LEFT);
  $g = str_pad(hexdec(substr($hex,2,2)),3,'0',STR_PAD_LEFT);
  $b = str_pad(hexdec(substr($hex,4,2)),3,'0',STR_PAD_LEFT);
  return ['r'=>$r,'g'=>$g,'b'=>$b,'rgb'=>"$r,$g,$b"];
}
function rgbCss($r,$g,$b){ return 'rgb('.intval($r).','.intval($g).','.intval($b).')'; }

function shouldPauseUpdate($game_num, $con, $pauseSeconds){
  $q = "SELECT scoreValue,lastUpdatedTime FROM sports_list WHERE Game=$game_num AND Conference='MLB'";
  $r = mysqli_query($con, $q);
  if ($r && mysqli_num_rows($r) > 0) {
    $row = mysqli_fetch_assoc($r);
    $sv  = trim($row['scoreValue'] ?? '');
    $lut = $row['lastUpdatedTime'] ?? null;
    if ($sv !== '' && $sv !== 'NO-SCORE' && !empty($lut)) {
      $diff = time() - strtotime($lut);
      if ($diff < $pauseSeconds) return ['pause'=>true,'time_remaining'=>$pauseSeconds-$diff];
    }
  }
  return ['pause'=>false,'time_remaining'=>0];
}

function fetchProxyList(){
  $url = "https://proxy.webshare.io/api/v2/proxy/list/download/jtqnktxhwdjdmbvpokcwllcblcalyuiecmiconkc/-/any/username/direct/-/?plan_id=12029912";
  $ch = curl_init();
  curl_setopt_array($ch, [
    CURLOPT_URL=>$url, CURLOPT_RETURNTRANSFER=>true, CURLOPT_FOLLOWLOCATION=>true,
    CURLOPT_TIMEOUT=>10, CURLOPT_SSL_VERIFYPEER=>false
  ]);
  $response = curl_exec($ch); curl_close($ch);
  if (!$response) return [];
  $out = [];
  foreach (explode("\n", trim($response)) as $line) {
    $line = trim($line); if ($line==='') continue;
    $parts = explode(':', $line);
    if (count($parts)==4) $out[]=['ip'=>$parts[0],'port'=>$parts[1],'username'=>$parts[2],'password'=>$parts[3]];
  }
  return $out;
}
function getRandomProxy($proxies){ if (empty($proxies)) return null; return $proxies[array_rand($proxies)]; }
function callESPNApiWithProxy($url, $proxies, $maxAttempts=6){
  $attempts=0;
  while ($attempts<$maxAttempts && !empty($proxies)) {
    $attempts++;
    $p=getRandomProxy($proxies); if(!$p) break;
    $ch=curl_init();
    curl_setopt_array($ch, [
      CURLOPT_URL=>$url, CURLOPT_RETURNTRANSFER=>true, CURLOPT_FOLLOWLOCATION=>true,
      CURLOPT_TIMEOUT=>10, CURLOPT_CONNECTTIMEOUT=>5, CURLOPT_SSL_VERIFYPEER=>false,
      CURLOPT_PROXY=>$p['ip'].':'.$p['port'], CURLOPT_PROXYUSERPWD=>$p['username'].':'.$p['password'],
      CURLOPT_PROXYTYPE=>CURLPROXY_HTTP
    ]);
    $resp=curl_exec($ch); $code=curl_getinfo($ch, CURLINFO_HTTP_CODE); curl_close($ch);
    if ($resp && $code==200) return ['success'=>true,'response'=>$resp,'attempts'=>$attempts];
    $idx=array_search($p,$proxies); if($idx!==false){ unset($proxies[$idx]); $proxies=array_values($proxies); }
  }
  $ch=curl_init();
  curl_setopt_array($ch, [CURLOPT_URL=>$url, CURLOPT_RETURNTRANSFER=>true, CURLOPT_FOLLOWLOCATION=>true,
    CURLOPT_TIMEOUT=>10, CURLOPT_CONNECTTIMEOUT=>5, CURLOPT_SSL_VERIFYPEER=>false]);
  $resp=curl_exec($ch); $code=curl_getinfo($ch, CURLINFO_HTTP_CODE); curl_close($ch);
  if ($resp && $code==200) return ['success'=>true,'response'=>$resp,'attempts'=>$attempts+1];
  return ['success'=>false,'response'=>null,'attempts'=>$attempts];
}

/** pull PBP info for delta handling */
function getAthleteInfoFromPbp($pbp_data, $team_id, $athlete_id){
  $out=['name'=>'','jersey'=>'','pos'=>''];
  if (empty($athlete_id) || empty($pbp_data)) return $out;
  if (!empty($pbp_data['boxscore']['players'])) {
    foreach ($pbp_data['boxscore']['players'] as $block) {
      if (($block['team']['id'] ?? '') != $team_id) continue;
      foreach (($block['statistics'] ?? []) as $sg) {
        foreach (($sg['athletes'] ?? []) as $ath) {
          $a = $ath['athlete'] ?? [];
          if (($a['id'] ?? '') == $athlete_id) {
            $out['name']   = $a['displayName'] ?? ($a['fullName'] ?? '');
            $out['jersey'] = $a['jersey'] ?? '';
            $pos = $a['position'] ?? [];
            $out['pos'] = is_array($pos) ? ($pos['abbreviation'] ?? '') : $pos;
            return $out;
          }
        }
      }
    }
  }
  if (!empty($pbp_data['rosters'])) {
    foreach ($pbp_data['rosters'] as $r) {
      if (($r['team']['id'] ?? '') != $team_id) continue;
      foreach (($r['roster'] ?? []) as $pl) {
        $a = $pl['athlete'] ?? [];
        if (($a['id'] ?? '') == $athlete_id) {
          $out['name']   = $a['displayName'] ?? ($a['fullName'] ?? '');
          $out['jersey'] = $a['jersey'] ?? '';
          $pos = $a['position'] ?? ($pl['position'] ?? []);
          $out['pos'] = is_array($pos) ? ($pos['abbreviation'] ?? '') : $pos;
          return $out;
        }
      }
    }
  }
  return $out;
}

// ===== FETCH SCOREBOARD =====
$proxies = fetchProxyList();
$scoreboard_url = "http://site.api.espn.com/apis/site/v2/sports/baseball/mlb/scoreboard";
$sbResult = callESPNApiWithProxy($scoreboard_url, $proxies);
if (!$sbResult['success']) { die("Failed to fetch MLB scoreboard after {$sbResult['attempts']} attempts"); }
$data = json_decode($sbResult['response'], true);
if (!$data || !isset($data['events'])) { die("Error parsing MLB scoreboard or no events"); }
$events = $data['events'];
$loop_length = count($events);

// ===== BEFORE SNAPSHOT (DB) for live games =====
$score_snapshot = [];
$snap_q = "SELECT Game, Team_1_Score, Team_2_Score, Game_Played, scoreValue, Team_1_Abbrev, Team_2_Abbrev, lastUpdatedTime, Who_Has_Ball, onBases, outsB, Team_1_Color_R, Team_1_Color_G, Team_1_Color_B, Team_2_Color_R, Team_2_Color_G, Team_2_Color_B FROM sports_list WHERE Conference='MLB' AND Game_Played='IN_SESSION' ORDER BY Game ASC";
$snap_r = mysqli_query($con, $snap_q);
while ($row = mysqli_fetch_assoc($snap_r)) {
  $score_snapshot[$row['Game']] = [
    'home_score'=>(int)$row['Team_1_Score'],
    'away_score'=>(int)$row['Team_2_Score'],
    'game_status'=>$row['Game_Played'],
    'score_value'=>($row['scoreValue'] === null || $row['scoreValue'] === '') ? 'NO-SCORE' : $row['scoreValue'],
    'home_abbr'=>$row['Team_1_Abbrev'] ?: '',
    'away_abbr'=>$row['Team_2_Abbrev'] ?: '',
    'last_updated'=>$row['lastUpdatedTime'],
    'who_ball'=>$row['Who_Has_Ball'] ?: '',
    'onBases'=>$row['onBases'] ?? '0,0,0',
    'outsB'=>$row['outsB'] ?? '0',
    't1R'=>$row['Team_1_Color_R']??'0','t1G'=>$row['Team_1_Color_G']??'0','t1B'=>$row['Team_1_Color_B']??'0',
    't2R'=>$row['Team_2_Color_R']??'0','t2G'=>$row['Team_2_Color_G']??'0','t2B'=>$row['Team_2_Color_B']??'0',
  ];
}
$before_table = [];
foreach ($score_snapshot as $gameNum => $snap) {
  $away = trim($snap['away_abbr'] ?? ''); $home = trim($snap['home_abbr'] ?? '');
  if ($away === '' || $home === '') continue;
  $batColor = 'rgb(144,164,174)';
  if ($snap['who_ball'] === $home) $batColor = rgbCss($snap['t1R'],$snap['t1G'],$snap['t1B']);
  if ($snap['who_ball'] === $away) $batColor = rgbCss($snap['t2R'],$snap['t2G'],$snap['t2B']);
  $before_table[] = [
    'Game'=>$gameNum,
    'Matchup'=>$away.' @ '.$home,
    'Away'=>$snap['away_score'] ?? 0,
    'Home'=>$snap['home_score'] ?? 0,
    'Status'=>$snap['game_status'] ?? '',
    'ScoreVal'=>$snap['score_value'] ?? 'NO-SCORE',
    'onBases'=>$snap['onBases'],
    'outsB'=>$snap['outsB'],
    'batRGB'=>$batColor,
  ];
}

// ===== PARSE EVENTS & WRITE BASE FIELDS =====
$games_data = [];
for ($i=1; $i <= $loop_length; $i++) {
  $event = $events[$i-1];
  if (!$event || !isset($event['competitions'][0])) continue;
  $competition = $event['competitions'][0];
  $status = $competition['status'] ?? [];
  $period = intval($status['period'] ?? 0);
  $statusType = $status['type'] ?? [];

  $game_being_played = 'NOT_STARTED';
  if (!empty($statusType['completed'])) $game_being_played = 'GAME_DONE';
  elseif (($statusType['state'] ?? '') === 'in') $game_being_played = 'IN_SESSION';
  elseif (($statusType['state'] ?? '') === 'post') $game_being_played = 'GAME_DONE';

  $game_time_iso = $event['date'] ?? '';
  $game_time_et = $game_time_iso ? date("g:i a", strtotime($game_time_iso)) : '';

  $broadcast = '';
  if (!empty($competition['broadcasts'][0]['names'][0])) $broadcast = $competition['broadcasts'][0]['names'][0];

  $home = null; $away = null;
  foreach (($competition['competitors'] ?? []) as $c) {
    if (($c['homeAway'] ?? '') === 'home') $home = $c; else $away = $c;
  }

  // Home (Team_1)
  $t1 = $home['team'] ?? [];
  $team_name1 = $t1['name'] ?? '';
  $location_team1 = $t1['location'] ?? '';
  $team_abbrev_1 = $t1['abbreviation'] ?? '';
  $team1_score = $home['score'] ?? '0';
  $team_id_1 = $t1['id'] ?? '';
  $hex1 = $t1['color'] ?? '000000';
  if ($hex1 == '000000' || empty($hex1)) $hex1 = $t1['alternateColor'] ?? '000000';
  $c1 = hexToRgb($hex1);

  // Away (Team_2)
  $t2 = $away['team'] ?? [];
  $team_name2 = $t2['name'] ?? '';
  $location_team2 = $t2['location'] ?? '';
  $team_abbrev_2 = $t2['abbreviation'] ?? '';
  $team2_score = $away['score'] ?? '0';
  $team_id_2 = $t2['id'] ?? '';
  $hex2 = $t2['color'] ?? '000000';
  if ($hex2 == '000000' || empty($hex2)) $hex2 = $t2['alternateColor'] ?? '000000';
  $c2 = hexToRgb($hex2);

 // records / playoff
  $team1_record = '0-0'; 
  $team2_record = '0-0'; 
  $is_playoff_game = false;
  
  // Check for playoff game and extract playoff records
  if ($PLAY_OFF && isset($competition['series']) && ($competition['series']['type'] ?? '') === 'playoff') {
    $is_playoff_game = true;
    $series_competitors = $competition['series']['competitors'] ?? [];
    
    // Collect all teams' wins
    $team_wins = [];
    foreach ($series_competitors as $series_comp) {
      $team_id = strval($series_comp['id'] ?? ''); // Convert to string for consistent comparison
      $wins = intval($series_comp['wins'] ?? 0);
      $team_wins[$team_id] = $wins;
    }
    
    // Calculate playoff records (each team's losses = opponent's wins)
    $playoff_records = [];
    foreach ($series_competitors as $series_comp) {
      $team_id = strval($series_comp['id'] ?? ''); // Convert to string
      $wins = intval($series_comp['wins'] ?? 0);
      
      // Find opponent's wins (which are this team's losses)
      $losses = 0;
      foreach ($team_wins as $tid => $w) {
        if ($tid != $team_id && $tid != '') {  // Skip empty IDs
          $losses = $w;
          break;
        }
      }
      
      $playoff_records[$team_id] = $wins;
    }
    
    // Now assign the playoff records to the correct team variables
    // IMPORTANT: Convert team IDs to strings for comparison
    $team_id_1_str = strval($team_id_1);
    $team_id_2_str = strval($team_id_2);
    
    // Assign records based on team IDs
    if (isset($playoff_records[$team_id_1_str])) {
      $team1_record = $playoff_records[$team_id_1_str];
    }
    if (isset($playoff_records[$team_id_2_str])) {
      $team2_record = $playoff_records[$team_id_2_str];
    }
    
  } else {
    // Regular season records (keep this unchanged)
    foreach (($home['records'] ?? []) as $r) {
      if (($r['type'] ?? '') === 'total') { 
        $team1_record = $r['summary'] ?? '0-0'; 
        break; 
      }
    }
    foreach (($away['records'] ?? []) as $r) {
      if (($r['type'] ?? '') === 'total') { 
        $team2_record = $r['summary'] ?? '0-0'; 
        break; 
      }
    }
  }

  // situation / who bats / bases+outs / lastPlay text from SCOREBOARD
  $who_has_ball=' '; $outsB='0'; $onBases='0,0,0'; $shortDetail='';
  if (!empty($status['type']['shortDetail'])) $shortDetail=$status['type']['shortDetail'];
  elseif (!empty($status['type']['detail']))  $shortDetail=$status['type']['detail'];

  $situation = $competition['situation'] ?? [];
  $situation_lastPlayText = trim($situation['lastPlay']['text'] ?? '');

  if ($game_being_played === 'IN_SESSION') {
    if (preg_match('/(\d+)/', strtolower($shortDetail), $m)) $period = intval($m[1]);
    $sd = strtolower($shortDetail);
    if (strpos($sd,'top') !== false)      $who_has_ball = $team_abbrev_2;
    elseif (strpos($sd,'bot') !== false)  $who_has_ball = $team_abbrev_1;
    elseif (strpos($sd,'mid') !== false)  $who_has_ball = 'mid';
    elseif (strpos($sd,'end') !== false)  $who_has_ball = 'end';

    $outsB = isset($situation['outs']) ? strval((int)$situation['outs']) : '0';
    $onFirst  = (!empty($situation['onFirst'])  && $situation['onFirst']  !== false) ? '1' : '0';
    $onSecond = (!empty($situation['onSecond']) && $situation['onSecond'] !== false) ? '1' : '0';
    $onThird  = (!empty($situation['onThird'])  && $situation['onThird']  !== false) ? '1' : '0';
    $onBases  = $onFirst . ',' . $onSecond . ',' . $onThird;

    if (!empty(($situation['lastPlay']['text'] ?? ''))) {
      $lp_text_low = strtolower($situation['lastPlay']['text']);
      if (strpos($lp_text_low,'middle of') !== false || strpos($lp_text_low,'mid ') !== false) $who_has_ball = 'mid';
      if (strpos($lp_text_low,'end of')   !== false)                                         $who_has_ball = 'end';
    }
  }

  // odds
  $favorite_team=''; $favorite_points=''; $overUnder='';
  if (!empty($competition['odds'][0])) {
    $od = $competition['odds'][0];
    $det = $od['details'] ?? '';
    if ($det) {
      if (strpos($det, $team_abbrev_1) !== false) { $favorite_team=$team_abbrev_1; preg_match('/([+-]?\d+\.?\d*)/',$det,$m); $favorite_points=$m[1] ?? ''; }
      elseif (strpos($det, $team_abbrev_2) !== false) { $favorite_team=$team_abbrev_2; preg_match('/([+-]?\d+\.?\d*)/',$det,$m); $favorite_points=$m[1] ?? ''; }
    }
    $overUnder = $od['overUnder'] ?? '';
  }

  // pause state
  $pauseInfo   = shouldPauseUpdate($i, $con, $PAUSE_SECONDS);
  $isPaused    = $pauseInfo['pause'];
  $timeRemain  = $pauseInfo['time_remaining'];

  // Base DB write (always keep scores, status, odds, bases, colors fresh)
  if (!$isPaused) {
    $team1HE = ($game_being_played==='IN_SESSION' ? (($home['statistics'][0]['displayValue'] ?? '0').','.($home['statistics'][1]['displayValue'] ?? '0')) : '');
    $team2HE = ($game_being_played==='IN_SESSION' ? (($away['statistics'][0]['displayValue'] ?? '0').','.($away['statistics'][1]['displayValue'] ?? '0')) : '');

    $q = "UPDATE sports_list SET
      Clock='0:00',
      Period='".mysqli_real_escape_string($con, $period)."',
      Game_Played='".mysqli_real_escape_string($con, $game_being_played)."',
      Game_Time='".mysqli_real_escape_string($con, $game_time_et)."',
      Game_Date='".mysqli_real_escape_string($con, $game_time_iso)."',
      broadcast='".mysqli_real_escape_string($con, $broadcast)."',
      Team_1_Name='".mysqli_real_escape_string($con, $team_name1)."',
      Team_1_Record='".mysqli_real_escape_string($con, $team1_record)."',
      Team_1_Score='".mysqli_real_escape_string($con, $team1_score)."',
      Team_1_Location='".mysqli_real_escape_string($con, $location_team1)."',
      Team_1_Abbrev='".mysqli_real_escape_string($con, $team_abbrev_1)."',
      Team_1_Color='".mysqli_real_escape_string($con, '\"'.$c1['rgb'].'\"')."',
      Team_1_Color_R='".mysqli_real_escape_string($con, $c1['r'])."',
      Team_1_Color_G='".mysqli_real_escape_string($con, $c1['g'])."',
      Team_1_Color_B='".mysqli_real_escape_string($con, $c1['b'])."',
      Team_2_Name='".mysqli_real_escape_string($con, $team_name2)."',
      Team_2_Record='".mysqli_real_escape_string($con, $team2_record)."',
      Team_2_Score='".mysqli_real_escape_string($con, $team2_score)."',
      Team_2_Location='".mysqli_real_escape_string($con, $location_team2)."',
      Team_2_Abbrev='".mysqli_real_escape_string($con, $team_abbrev_2)."',
      Team_2_Color='".mysqli_real_escape_string($con, '\"'.$c2['rgb'].'\"')."',
      Team_2_Color_R='".mysqli_real_escape_string($con, $c2['r'])."',
      Team_2_Color_G='".mysqli_real_escape_string($con, $c2['g'])."',
      Team_2_Color_B='".mysqli_real_escape_string($con, $c2['b'])."',
      Who_Has_Ball='".mysqli_real_escape_string($con, $who_has_ball)."',
      Favorite_Team='".mysqli_real_escape_string($con, $favorite_team)."',
      Favorite_Points='".mysqli_real_escape_string($con, $favorite_points)."',
      OverUnder='".mysqli_real_escape_string($con, $overUnder)."',
      outsB='".mysqli_real_escape_string($con, $outsB)."',
      onBases='".mysqli_real_escape_string($con, $onBases)."',
      team1HE='".mysqli_real_escape_string($con, $team1HE)."',
      scoreValue = 'NO-SCORE',
      lastPlay = '',
      team2HE='".mysqli_real_escape_string($con, $team2HE)."'
      WHERE Game=$i AND Conference='MLB'";
    mysqli_query($con, $q);
  } else {
    $q = "UPDATE sports_list SET
      Team_1_Score='".mysqli_real_escape_string($con, $team1_score)."',
      Team_2_Score='".mysqli_real_escape_string($con, $team2_score)."',
      Game_Played='".mysqli_real_escape_string($con, $game_being_played)."',
      Period='".mysqli_real_escape_string($con, $period)."',
      outsB='".mysqli_real_escape_string($con, $outsB)."',
      onBases='".mysqli_real_escape_string($con, $onBases)."'
      WHERE Game=$i AND Conference='MLB'";
    mysqli_query($con, $q);
  }

  // Collect for AFTER table & simple pass
  $batRGB='rgb(144,164,174)';
  if ($who_has_ball === $team_abbrev_1) $batRGB = rgbCss($c1['r'],$c1['g'],$c1['b']);
  if ($who_has_ball === $team_abbrev_2) $batRGB = rgbCss($c2['r'],$c2['g'],$c2['b']);

  $games_data[] = [
    'Game'=>$i,
    'event_id'=>$event['id'] ?? '',
    'Game_Played'=>$game_being_played,
    'Period'=>$period,
    'Game_Time'=>$game_time_et,
    'Game_Date'=>$game_time_iso,
    'broadcast'=>$broadcast,
    'Team_1_Abbrev'=>$team_abbrev_1,
    'Team_1_Score'=>$team1_score,
    'Team_1_ID'=>$team_id_1,
    'Team_2_Abbrev'=>$team_abbrev_2,
    'Team_2_Score'=>$team2_score,
    'Team_2_ID'=>$team_id_2,
    'Who_Has_Ball'=>$who_has_ball,
    'outsB'=>$outsB,
    'onBases'=>$onBases,
    'batRGB'=>$batRGB,
    'is_paused'=>$isPaused,
    'time_remaining'=>$timeRemain,
    'is_playoff'=>$is_playoff_game,
    'situation_lastPlayText'=>$situation_lastPlayText
  ];
}

// ===== AFTER table scaffold =====
$after_table = [];
foreach ($games_data as $g) {
  $after_table[$g['Game']] = [
    'Game'=>$g['Game'],
    'Matchup'=>$g['Team_2_Abbrev'].' @ '.$g['Team_1_Abbrev'],
    'Away'=>$g['Team_2_Score'],
    'Home'=>$g['Team_1_Score'],
    'Status'=>$g['Game_Played'],
    'Paused'=>$g['is_paused'],
    'Countdown'=>$g['time_remaining'],
    'ScoreVal'=>'',
    'LastPlay'=>'',
    'Player'=>'',
    'Team'=>'',
    'Note'=>'',
    'onBases'=>$g['onBases'],
    'outsB'=>$g['outsB'],
    'batRGB'=>$g['batRGB']
  ];
}

// ===== DELTA-FIRST SCORING DETECTION (unchanged) =====
$scoring_plays_detected = [];
$scored_this_cycle = [];

foreach ($games_data as &$g) {
  $game_num = $g['Game'];
  $snapshot = $score_snapshot[$game_num] ?? ['home_score'=>0,'away_score'=>0,'game_status'=>'','score_value'=>'NO-SCORE'];

  if (($g['Game_Played'] ?? '') !== 'IN_SESSION') continue;
  if (!empty($g['is_paused'])) continue;

  $new_home = intval($g['Team_1_Score']);
  $new_away = intval($g['Team_2_Score']);
  $old_home = intval($snapshot['home_score']);
  $old_away = intval($snapshot['away_score']);
  $home_diff = $new_home - $old_home;
  $away_diff = $new_away - $old_away;
  if ($home_diff <= 0 && $away_diff <= 0) continue;

  // Optional PBP enrichment
  $pbp_data = null; $pbp_ok = false;
  $event_id = $g['event_id'];
  if ($ENABLE_PBP_ENRICHMENT && !empty($event_id)) {
    $pbp_url = "https://site.api.espn.com/apis/site/v2/sports/baseball/mlb/summary?event=".$event_id;
    $pbpRes = callESPNApiWithProxy($pbp_url, $proxies);
    if ($pbpRes['success']) { $pbp_data = json_decode($pbpRes['response'], true); $pbp_ok = !empty($pbp_data); }
  }

  $sides = [];
  if ($home_diff > 0) $sides[] = ['side'=>'home','delta'=>$home_diff];
  if ($away_diff > 0) $sides[] = ['side'=>'away','delta'=>$away_diff];

  foreach ($sides as $entry) {
    $side  = $entry['side'];
    $delta = max(1, (int)$entry['delta']);
    $abbr    = ($side==='home') ? $g['Team_1_Abbrev'] : $g['Team_2_Abbrev'];
    $team_id = ($side==='home') ? $g['Team_1_ID']     : $g['Team_2_ID'];

    $play_text = $abbr.' scored';
    $scorer_id = ''; $info=['name'=>'','jersey'=>'','pos'=>''];

    $prefix = 'R-'; $count=$delta;

    if ($pbp_ok && !empty($pbp_data['plays'])) {
      $plays = $pbp_data['plays'];
      $target_home = $new_home; $target_away = $new_away;
      $sp = null;
      for ($k=count($plays)-1; $k>=0; $k--) {
        $p = $plays[$k];
        if (!empty($p['scoringPlay'])) {
          $p_home = isset($p['homeScore']) ? (int)$p['homeScore'] : -1;
          $p_away = isset($p['awayScore']) ? (int)$p['awayScore'] : -1;
          $p_team_id = $p['team']['id'] ?? null;
          if (($p_home===$target_home && $p_away===$target_away) ||
              ($p_team_id !== null && (string)$p_team_id === (string)$team_id)) { $sp=$p; break; }
        }
      }
      if ($sp) {
        $play_text = $sp['text'] ?? $play_text;
        foreach (($sp['participants'] ?? []) as $pt) {
          if (($pt['type'] ?? '') === 'batter') { $scorer_id = $pt['athlete']['id'] ?? ''; break; }
        }
        if ($scorer_id) $info = getAthleteInfoFromPbp($pbp_data, $team_id, $scorer_id);
        $is_hr = (stripos($play_text,'home run')!==false || stripos($play_text,'homer')!==false || stripos($play_text,'grand slam')!==false);
        $prefix = $is_hr ? 'H-' : 'R-';
        $count  = isset($sp['scoreValue']) ? (int)$sp['scoreValue'] : $delta;
        if ($count < 1) $count = $delta;
      }
    }

    $enriched = [
      "scoreValue"       => $prefix . $count,
      "lastPlay"         => mysqli_real_escape_string($con, $play_text),
      "lastPlayerID"     => mysqli_real_escape_string($con, $scorer_id),
      "lastPlayerName"   => mysqli_real_escape_string($con, $info['name']),
      "lastPlayerPos"    => mysqli_real_escape_string($con, $info['pos']),
      "jersey"           => mysqli_real_escape_string($con, $info['jersey']),
      "lastPlayerTeamID" => mysqli_real_escape_string($con, $team_id),
      "Who_Has_Ball"     => mysqli_real_escape_string($con, $abbr),
      "lastUpdatedTime"  => $current_timestamp_dt
    ];
    $sets=[]; foreach ($enriched as $f=>$v) { $sets[]="$f='$v'"; }
    $uq = "UPDATE sports_list SET ".implode(', ',$sets)." WHERE Game=$game_num AND Conference='MLB'";
    mysqli_query($con, $uq);

    $scoring_plays_detected[] = [
      'game'=>$game_num,
      'teams'=>$g['Team_2_Abbrev'].' @ '.$g['Team_1_Abbrev'],
      'team'=>$abbr,
      'runs'=>$count,
      'type'=>$enriched['scoreValue'],
      'player_name'=>$info['name'],
      'player_id'=>$scorer_id,
      'jersey'=>$info['jersey'],
      'position'=>$info['pos'],
      'play_text'=>stripslashes($play_text)
    ];
    $scored_this_cycle[$game_num] = true;
  }
}

// ===== SIMPLE PASS: NOT PAUSED & NO DELTA => NO-SCORE + lastPlay =====
foreach ($games_data as $g) {
  $game_num = $g['Game'];
  $paused   = !empty($g['is_paused']);
  $did_score_now = !empty($scored_this_cycle[$game_num]);

  if (!$paused && !$did_score_now) {
    $lp = mysqli_real_escape_string($con, $g['situation_lastPlayText'] ?? '');
    $sql = "UPDATE sports_list SET
              scoreValue='NO-SCORE',
              lastPlay='$lp',
              lastPlayerID='',
              lastPlayerName='',
              lastPlayerPos='',
              jersey='',
              lastPlayerTeamID='',
              Who_Has_Ball='".mysqli_real_escape_string($con, $g['Who_Has_Ball'])."',
              lastUpdatedTime=NULL
            WHERE Game=$game_num AND Conference='MLB'";
    mysqli_query($con, $sql);
  }
}

// ===== AFTER table fill =====
foreach ($after_table as $gnum => &$row) {
  $pq = "SELECT scoreValue,lastPlay,lastPlayerName,jersey,lastPlayerPos FROM sports_list WHERE Game=$gnum AND Conference='MLB' LIMIT 1";
  $pr = mysqli_query($con, $pq);
  if ($pr && ($prow = mysqli_fetch_assoc($pr))) {
    $row['ScoreVal'] = ($prow['scoreValue'] === null || $prow['scoreValue']==='') ? 'NO-SCORE' : $prow['scoreValue'];
    $row['LastPlay'] = $prow['lastPlay'] ?? '';
    $bits = [];
    if (!empty($prow['lastPlayerName'])) $bits[] = $prow['lastPlayerName'];
    if (!empty($prow['jersey']))        $bits[] = '#'.$prow['jersey'];
    if (!empty($prow['lastPlayerPos'])) $bits[] = '('.$prow['lastPlayerPos'].')';
    $row['Player'] = trim(implode(' ', $bits));
  }
}

// ===== UI helpers (unchanged) =====
function renderBasesOuts($onBasesStr, $outsStr, $rgbCss){
  $first=$second=$third='0';
  if ($onBasesStr && strpos($onBasesStr, ',')!==false) {
    $parts = explode(',', $onBasesStr);
    if (count($parts)>=3){ $first=trim($parts[0]); $second=trim($parts[1]); $third=trim($parts[2]); }
  }
  $outs = max(0, min(3, intval($outsStr)));
  $fill = $rgbCss ?: 'rgb(144,164,174)'; $empty='#cfd8dc';
  return '
  <div class="bo-wrap">
    <div class="bo-diamond">
      <span class="base b2" style="background:'.($second==='1'?$fill:$empty).'"></span>
      <span class="base b1" style="background:'.($first==='1'?$fill:$empty).'"></span>
      <span class="base b3" style="background:'.($third==='1'?$fill:$empty).'"></span>
    </div>
    <div class="bo-outs">
      <span class="o '.($outs>=1?'lit':'').'"></span>
      <span class="o '.($outs>=2?'lit':'').'"></span>
      <span class="o '.($outs>=3?'lit':'').'"></span>
    </div>
  </div>';
}
?>
<!DOCTYPE html>
<html>
<head>
  <title>MLB Games Database Update - <?php echo htmlspecialchars($time_updated); ?></title>
  <style>
    body { font-family: Arial, sans-serif; margin:20px; background:#f5f5f5; }
    h1 { color:#333; text-align:center; background:#fff; padding:20px; border-radius:5px; box-shadow:0 2px 4px rgba(0,0,0,.1); }
    .summary, .pause-info { background:#fff; padding:15px; margin:20px 0; border-radius:5px; box-shadow:0 2px 4px rgba(0,0,0,.1); }
    .playoff-mode { background:#ff9800; color:#fff; padding:10px; margin:20px 0; border-radius:5px; text-align:center; font-weight:bold; }
    table { width:100%; border-collapse:collapse; background:#fff; margin:20px 0; box-shadow:0 2px 4px rgba(0,0,0,.1); font-size:11px; }
    th { background:#2c3e50; color:#fff; padding:8px 4px; text-align:left; font-size:10px; position:sticky; top:0; z-index:10; }
    td { padding:6px 4px; border-bottom:1px solid #ddd; font-size:10px; vertical-align: middle; }
    tr:nth-child(even) { background:#f9f9f9; }
    tr:hover { background:#f0f0f0; }
    .paused-row { background:#fff59d !important; }
    .countdown-timer { background:#ff9800; color:#fff; padding:2px 6px; border-radius:3px; font-weight:bold; font-size:11px; display:inline-block; }
    .truncate { max-width:320px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap; }
    .grid-2 { display:grid; grid-template-columns: 1fr 1fr; gap:16px; }
    .bo-wrap { display:flex; align-items:center; gap:10px; }
    .bo-diamond { position:relative; width:32px; height:32px; margin:2px 0 0 2px; }
    .bo-diamond .base { position:absolute; width:12px; height:12px; transform: rotate(45deg); background:#cfd8dc; border:1px solid rgba(0,0,0,.12); }
    .bo-diamond .b2 { left:10px; top:-2px; }
    .bo-diamond .b1 { left:20px; top:8px; }
    .bo-diamond .b3 { left:0px;  top:8px; }
    .bo-outs .o { width:8px; height:8px; border-radius:50%; background:#eceff1; border:1px solid rgba(0,0,0,.15); display:inline-block; }
    .bo-outs .o.lit { background:#d32f2f; }
    .pill { display:inline-block; padding:2px 8px; border-radius:999px; font-size:11px; color:#fff; margin-right:6px; }
    .pill-green { background:#2e7d32; }
    .pill-blue  { background:#1565c0; }
    .pill-grey  { background:#607d8b; }
    .pill-orange{ background:#ef6c00; }
  </style>
</head>
<body>
  <h1>MLB Games Database Update</h1>
  <?php if ($PLAY_OFF): ?>
    <div class="playoff-mode">🏆 PLAYOFF MODE ACTIVE — showing series records</div>
  <?php endif; ?>

  <!-- NEW: Preflight banner -->
  <div class="summary">
    <div style="margin-bottom:6px;">
      <span class="pill <?php echo $preflight['update_anyway'] ? 'pill-orange' : 'pill-grey'; ?>">
        <?php echo $preflight['update_anyway'] ? 'FORCED UPDATE (UPDATE_ANYWAY)' : 'Normal Mode'; ?>
      </span>
      <span class="pill <?php echo $preflight['in_session_in_db'] ? 'pill-green' : 'pill-grey'; ?>">
        IN_SESSION in DB: <?php echo $preflight['in_session_in_db'] ? 'Yes' : 'No'; ?>
      </span>
      <span class="pill <?php echo $preflight['starts_within_10'] ? 'pill-blue' : 'pill-grey'; ?>">
        Starts ≤10 min: <?php echo $preflight['starts_within_10'] ? 'Yes' : 'No'; ?>
      </span>
      <span class="pill pill-green">
        Proceeded: <?php echo $preflight['proceed'] ? 'Yes' : 'No'; ?>
      </span>
    </div>
    <strong>Update Time:</strong> <?php echo htmlspecialchars($time_updated); ?><br>
    <strong>Database:</strong> <?php echo htmlspecialchars($database_gscroll); ?><br>
    <?php if (!empty($preflight['notes'])): ?>
      <div style="margin-top:6px;">
        <em><?php echo htmlspecialchars(implode('  •  ', $preflight['notes'])); ?></em>
      </div>
    <?php endif; ?>
  </div>

  <div class="summary">
    <strong>Total Games (scoreboard):</strong> <?php echo intval($loop_length); ?><br>
    <strong>Mode:</strong> <?php echo $PLAY_OFF ? 'PLAYOFF' : 'Regular Season'; ?>
  </div>

  <?php $paused_count = 0; foreach ($after_table as $r) { if (!empty($r['Paused'])) $paused_count++; } ?>
  <?php if ($paused_count > 0): ?>
  <div class="pause-info">
    <h3 style="margin-top:0;">⏸️ Active Pauses (<?php echo (int)$PAUSE_SECONDS; ?>s window)</h3>
    <table>
      <thead><tr><th>Game</th><th>Matchup</th><th>Countdown</th><th>ScoreVal</th><th>Last Play</th><th>Player</th><th>Bases/Outs</th></tr></thead>
      <tbody>
      <?php foreach ($after_table as $r): if (empty($r['Paused'])) continue; ?>
        <tr>
          <td><?php echo (int)$r['Game']; ?></td>
          <td><?php echo htmlspecialchars($r['Matchup']); ?></td>
          <td><span class="countdown-timer"><?php echo (int)$r['Countdown']; ?>s</span></td>
          <td><?php echo htmlspecialchars($r['ScoreVal'] ?: 'NO-SCORE'); ?></td>
          <td class="truncate" title="<?php echo htmlspecialchars($r['LastPlay']); ?>"><?php echo htmlspecialchars($r['LastPlay']); ?></td>
          <td><?php echo htmlspecialchars($r['Player']); ?></td>
          <td><?php echo renderBasesOuts($r['onBases'], $r['outsB'], $r['batRGB']); ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>
  <?php endif; ?>

  <div class="summary" style="overflow-x:auto;">
    <h3 style="margin-top:0;">🧪 Snapshot vs Update</h3>
    <div class="grid-2">
      <div>
        <h4 style="margin:0 0 8px 0;">Before API (Live Only)</h4>
        <table>
          <thead><tr><th>Game</th><th>Matchup</th><th>Away</th><th>Home</th><th>Status</th><th>ScoreVal</th><th>Bases/Outs</th></tr></thead>
          <tbody>
            <?php if (!empty($before_table)): foreach ($before_table as $r): ?>
              <tr>
                <td><?php echo (int)$r['Game']; ?></td>
                <td><?php echo htmlspecialchars($r['Matchup']); ?></td>
                <td><?php echo htmlspecialchars($r['Away']); ?></td>
                <td><?php echo htmlspecialchars($r['Home']); ?></td>
                <td><?php echo htmlspecialchars($r['Status']); ?></td>
                <td><?php echo htmlspecialchars($r['ScoreVal']); ?></td>
                <td><?php echo renderBasesOuts($r['onBases'], $r['outsB'], $r['batRGB']); ?></td>
              </tr>
            <?php endforeach; else: ?>
              <tr><td colspan="7"><em>No IN_SESSION snapshot rows.</em></td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
      <div>
        <h4 style="margin:0 0 8px 0;">After API</h4>
        <table>
          <thead>
            <tr>
              <th>Game</th><th>Matchup</th><th>Away</th><th>Home</th><th>Status</th>
              <th>Paused?</th><th>Countdown</th><th>ScoreVal</th><th>Last Play</th><th>Player</th><th>Bases/Outs</th><th>Note</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!empty($after_table)): foreach ($after_table as $r):
              $row_class = !empty($r['Paused']) ? 'paused-row' : ''; ?>
              <tr class="<?php echo $row_class; ?>">
                <td><?php echo (int)$r['Game']; ?></td>
                <td><?php echo htmlspecialchars($r['Matchup']); ?></td>
                <td><?php echo htmlspecialchars($r['Away']); ?></td>
                <td><?php echo htmlspecialchars($r['Home']); ?></td>
                <td><?php echo htmlspecialchars($r['Status']); ?></td>
                <td><?php echo !empty($r['Paused']) ? 'YES' : 'No'; ?></td>
                <td><?php echo !empty($r['Paused']) ? ('<span class="countdown-timer">'.(int)$r['Countdown'].'s</span>') : '-'; ?></td>
                <td><?php echo htmlspecialchars($r['ScoreVal'] ?: 'NO-SCORE'); ?></td>
                <td class="truncate" title="<?php echo htmlspecialchars($r['LastPlay']); ?>"><?php echo htmlspecialchars($r['LastPlay']); ?></td>
                <td><?php echo htmlspecialchars($r['Player']); ?></td>
                <td><?php echo renderBasesOuts($r['onBases'], $r['outsB'], $r['batRGB']); ?></td>
                <td><?php echo htmlspecialchars($r['Note']); ?></td>
              </tr>
            <?php endforeach; else: ?>
              <tr><td colspan="12"><em>No update rows.</em></td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

  <div class="summary">
    <h3 style="margin-top:0;">🏟️ Scoring Plays Detected This Update</h3>
    <?php if (!empty($scoring_plays_detected)): ?>
      <table>
        <thead>
          <tr>
            <th>Game</th><th>Matchup</th><th>Team</th><th>Runs</th><th>Type</th>
            <th>Player</th><th>ID</th><th>Jersey</th><th>Pos</th><th>Play Description</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($scoring_plays_detected as $sp): ?>
            <tr>
              <td><?php echo (int)$sp['game']; ?></td>
              <td><?php echo htmlspecialchars($sp['teams']); ?></td>
              <td><strong><?php echo htmlspecialchars($sp['team']); ?></strong></td>
              <td><?php echo (int)$sp['runs']; ?></td>
              <td><strong><?php echo htmlspecialchars($sp['type']); ?></strong></td>
              <td><?php echo htmlspecialchars($sp['player']); ?></td>
              <td><?php echo htmlspecialchars($sp['player_id']); ?></td>
              <td><?php echo htmlspecialchars($sp['jersey']); ?></td>
              <td><?php echo htmlspecialchars($sp['position']); ?></td>
              <td class="truncate" title="<?php echo htmlspecialchars($sp['play_text']); ?>"><?php echo htmlspecialchars($sp['play_text']); ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    <?php else: ?>
      <em>No scoring plays detected this update.</em>
    <?php endif; ?>
  </div>

  <div class="summary">
    <strong>Notes:</strong><br>
    • No-delta & not paused ⇒ we write <code>NO-SCORE</code> and the live situation’s <code>lastPlay</code> text.<br>
    • Paused ⇒ scoreValue/lastPlay untouched.<br>
    • Delta ⇒ same enriched scoring logic as before (PBP-first, then delta).<br>
  </div>
</body>
</html>
<?php
// ===== FINALIZATIONS =====
$sql = "UPDATE API_DB SET TIME_UPDATED = '".mysqli_real_escape_string($con, $time_updated)."' WHERE API = 'MLB'";
mysqli_query($con, $sql);

// touch downstream endpoints (ignore response)
$urls = [
  SERVER_URL . "Scrolling/App_Output/API/Games/MLB_Games.php",
  SERVER_URL . "Scrolling/App_Output/API/Games/Scoring_ALERT.php",
];
foreach ($urls as $url) {
  $ch = curl_init();
  curl_setopt($ch, CURLOPT_URL, $url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
  curl_exec($ch);
  curl_close($ch);
}

mysqli_close($con);
?>
