<?php

require_once '../config.php';
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check for UPDATE_ANYWAY parameter (can be GET or POST)
$UPDATE_ANYWAY = isset($_REQUEST['UPDATE_ANYWAY']) && $_REQUEST['UPDATE_ANYWAY'] == 'true';

// Initialize database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Set timezone
date_default_timezone_set('America/New_York');

// Function to determine if we should update
function shouldUpdate($con, $UPDATE_ANYWAY) {
    // If UPDATE_ANYWAY is true, always update
    if ($UPDATE_ANYWAY) {
        return array(
            'should_update' => true, 
            'reason' => 'UPDATE_ANYWAY parameter is set to true'
        );
    }
    
    // Query all NCAAB games from database
    $query = "SELECT Game_Date, Game_Played FROM sports_list WHERE Conference = 'NCAAB'";
    $result = mysqli_query($con, $query);
    
    if (!$result) {
        // If query fails, update anyway to be safe
        return array(
            'should_update' => true,
            'reason' => 'Database query failed - updating to be safe'
        );
    }
    
    $current_time = time();
    $has_active_game = false;
    $min_time_to_game_start = PHP_INT_MAX;
    $game_count = 0;
    
    while ($row = mysqli_fetch_assoc($result)) {
        $game_count++;
        
        // Check if any game is IN_SESSION or HALF_TIME
        if ($row['Game_Played'] == 'IN_SESSION' || $row['Game_Played'] == 'HALF_TIME') {
            $has_active_game = true;
        }
        
        // Parse game date and calculate time until game start
        if (!empty($row['Game_Date'])) {
            // Convert UTC time string to timestamp
            $game_timestamp = strtotime($row['Game_Date']);
            if ($game_timestamp !== false) {
                $time_until_game = $game_timestamp - $current_time;
                
                // Only consider games that haven't started yet
                if ($time_until_game > 0 && $time_until_game < $min_time_to_game_start) {
                    $min_time_to_game_start = $time_until_game;
                }
            }
        }
    }
    
    // If no games found, update anyway
    if ($game_count == 0) {
        return array(
            'should_update' => true,
            'reason' => 'No NCAAB games found in database - initial load'
        );
    }
    
    // If any game is active, update
    if ($has_active_game) {
        return array(
            'should_update' => true,
            'reason' => 'Active games in progress'
        );
    }
    
    // If within 10 minutes (600 seconds) of any game start, update
    if ($min_time_to_game_start <= 600) {
        $minutes_until = round($min_time_to_game_start / 60, 1);
        return array(
            'should_update' => true,
            'reason' => "Game starting in $minutes_until minutes"
        );
    }
    
    // If we get here: no active games AND no games within 10 minutes
    // DON'T UPDATE
    if ($min_time_to_game_start == PHP_INT_MAX) {
        // No upcoming games found at all
        return array(
            'should_update' => false,
            'reason' => "No active or upcoming games found"
        );
    } else {
        // Games exist but are more than 10 minutes away
        $minutes_until = round($min_time_to_game_start / 60);
        $hours_until = round($min_time_to_game_start / 3600, 1);
        
        if ($hours_until >= 1) {
            return array(
                'should_update' => false,
                'reason' => "No active games, next game starts in $hours_until hours"
            );
        } else {
            return array(
                'should_update' => false,
                'reason' => "No active games, next game starts in $minutes_until minutes"
            );
        }
    }
}

// Check if we should update
$update_check = shouldUpdate($con, $UPDATE_ANYWAY);

// Display update decision
echo "<!-- Update Decision: " . ($update_check['should_update'] ? "YES" : "NO") . " - " . $update_check['reason'] . " -->\n";

// If we shouldn't update, display message and exit
if (!$update_check['should_update']) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>NCAAB Update Skipped</title>
        <style>
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                margin: 20px;
                background: #f5f5f5;
            }
            .header {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 20px;
                border-radius: 10px;
                margin-bottom: 20px;
            }
            .info-box {
                background: #fff3cd;
                color: #856404;
                border: 1px solid #ffeeba;
                padding: 15px;
                border-radius: 5px;
                margin: 20px 0;
            }
            .force-update {
                background: #e3f2fd;
                color: #1976d2;
                border: 1px solid #90caf9;
                padding: 15px;
                border-radius: 5px;
                margin: 20px 0;
            }
            code {
                background: #f5f5f5;
                padding: 2px 5px;
                border-radius: 3px;
                font-family: monospace;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>🏀 NCAAB Update Status</h1>
            <p>Checked at: <?php echo date('m/d/Y | g:iA'); ?></p>
        </div>
        
        <div class="info-box">
            <h2>📊 Update Not Needed</h2>
            <p><strong>Reason:</strong> <?php echo htmlspecialchars($update_check['reason']); ?></p>
            <p>The system has determined that an API update is not necessary at this time.</p>
        </div>
        
        <div class="force-update">
            <h3>Force Update Option</h3>
            <p>To force an update regardless of game status, add <code>UPDATE_ANYWAY=true</code> to the URL:</p>
            <p><code><?php echo $_SERVER['PHP_SELF']; ?>?UPDATE_ANYWAY=true</code></p>
        </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    exit();
}

// If we get here, we should update - proceed with the rest of the script
$time_updated = date('m/d/Y | g:iA');
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NCAAB'";
mysqli_query($con, $sql);

// Helper function to add leading zeros
function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

// Helper function to convert hex to RGB
function hexToRgb($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) == 6) {
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
    } else {
        $r = $g = $b = 0;
    }
    
    // Force white if color is too dark
    if ($r <= 16 && $g <= 16 && $b <= 16) {
        $r = $g = $b = 255;
    }
    
    return array(
        'r' => addZeros($r),
        'g' => addZeros($g),
        'b' => addZeros($b),
        'rgb' => sprintf('"%s,%s,%s"', addZeros($r), addZeros($g), addZeros($b))
    );
}

// Function to process games from API
function processGames($api_url, $con, $startingGameNumber = 1, $dateLabel = '') {
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        return array('error' => "cURL Error #:" . $err, 'games' => array());
    }

    // Parse JSON response
    $data = json_decode($response, true);
    if (!$data || !isset($data['events'])) {
        return array('error' => "Invalid API response", 'games' => array());
    }

    $games = array();
    $gameCounter = $startingGameNumber;

    // Process each event/game
    foreach ($data['events'] as $event) {
        if (!isset($event['competitions'][0])) {
            continue;
        }
        
        $competition = $event['competitions'][0];
        $status = $competition['status'];
        
        // Extract game time and status
        $game_time = $event['date'];
        $game_time_formatted = date("g:i a", strtotime($game_time));
        $clock = isset($status['displayClock']) ? $status['displayClock'] : '0:00';
        $period = isset($status['period']) ? $status['period'] : 0;
        
        // Find home and away teams FIRST to get scores for overtime detection
        $homeTeam = null;
        $awayTeam = null;
        
        foreach ($competition['competitors'] as $competitor) {
            if ($competitor['homeAway'] == 'home') {
                $homeTeam = $competitor;
            } else {
                $awayTeam = $competitor;
            }
        }
        
        if (!$homeTeam || !$awayTeam) {
            continue;
        }
        
        // Get scores early for overtime detection
        $home_score = isset($homeTeam['score']) ? intval($homeTeam['score']) : 0;
        $away_score = isset($awayTeam['score']) ? intval($awayTeam['score']) : 0;
        
        // Determine game status with IMPROVED OVERTIME LOGIC
        $game_being_played = 'NOT_STARTED';
        if ($status['type']['state'] == 'pre') {
            $game_being_played = 'NOT_STARTED';
        } elseif ($status['type']['state'] == 'in') {
            $game_being_played = 'IN_SESSION';
            
            // Check for halftime (end of period 1)
            if ($period == 1 && $clock == "0:00") {
                $game_being_played = 'HALF_TIME';
            }
            
            // Check for potential overtime scenarios
            // Period 2 (or any even period) with 0:00 and tied score means OT is coming
            if ($clock == "0:00" && $home_score == $away_score) {
                // In overtime periods (3+) there might be brief intermissions
                if ($period >= 2) {
                    // Still mark as IN_SESSION since overtime is part of the game
                    $game_being_played = 'IN_SESSION'; 
                }
            }
        } elseif ($status['type']['state'] == 'post') {
            $game_being_played = 'GAME_DONE';
        }
        
        // IMPORTANT: Match your original code structure
        // Team 1 = HOME team (first in your original extraction)
        // Team 2 = AWAY team (second in your original extraction)
        
        // Extract TEAM 1 data (HOME)
        $team1_name = mysqli_real_escape_string($con, $homeTeam['team']['name']);
        $team1_location = mysqli_real_escape_string($con, $homeTeam['team']['location']);
        $team1_abbrev = substr($homeTeam['team']['abbreviation'], 0, 4);
        $team1_score = $home_score; // Already extracted above for overtime detection
        
        // Get team 1 color
        $team1_color_hex = isset($homeTeam['team']['color']) ? $homeTeam['team']['color'] : '000000';
        if ($team1_color_hex == '000000' && isset($homeTeam['team']['alternateColor'])) {
            $team1_color_hex = $homeTeam['team']['alternateColor'];
        }
        $team1_colors = hexToRgb($team1_color_hex);
        
        // Get team 1 record
        $team1_record = '0-0';
        if (isset($homeTeam['records'])) {
            foreach ($homeTeam['records'] as $record) {
                if ($record['name'] == 'overall') {
                    $team1_record = $record['summary'];
                    break;
                }
            }
        }
        
        // Extract TEAM 2 data (AWAY)
        $team2_name = mysqli_real_escape_string($con, $awayTeam['team']['name']);
        $team2_location = mysqli_real_escape_string($con, $awayTeam['team']['location']);
        $team2_abbrev = substr($awayTeam['team']['abbreviation'], 0, 4);
        $team2_score = $away_score; // Already extracted above for overtime detection
        
        // Get team 2 color
        $team2_color_hex = isset($awayTeam['team']['color']) ? $awayTeam['team']['color'] : '000000';
        if ($team2_color_hex == '000000' && isset($awayTeam['team']['alternateColor'])) {
            $team2_color_hex = $awayTeam['team']['alternateColor'];
        }
        $team2_colors = hexToRgb($team2_color_hex);
        
        // Get team 2 record
        $team2_record = '0-0';
        if (isset($awayTeam['records'])) {
            foreach ($awayTeam['records'] as $record) {
                if ($record['name'] == 'overall') {
                    $team2_record = $record['summary'];
                    break;
                }
            }
        }
        
        // Extract odds/gambling info
        $favorite_team = '';
        $favorite_points = '';
        $overUnder = '';
        
        if (isset($competition['odds'][0])) {
            $odds = $competition['odds'][0];
            if (isset($odds['details'])) {
                // Parse details like "SJU -26.5"
                $parts = explode(' ', $odds['details']);
                if (count($parts) == 2) {
                    $favorite_team = $parts[0];
                    $favorite_points = $parts[1];
                }
            }
            if (isset($odds['overUnder'])) {
                $overUnder = $odds['overUnder'];
            }
        }
        
        // Extract broadcast information
        $broadcast = '';
        if (isset($competition['broadcasts']) && is_array($competition['broadcasts'])) {
            $broadcast_parts = array();
            foreach ($competition['broadcasts'] as $bcast) {
                if (isset($bcast['names']) && is_array($bcast['names'])) {
                    foreach ($bcast['names'] as $name) {
                        $broadcast_parts[] = $name;
                    }
                }
            }
            if (!empty($broadcast_parts)) {
                $broadcast = mysqli_real_escape_string($con, implode(', ', $broadcast_parts));
            }
        }
        
        // Default values for football-specific fields
        $who_has_ball = '';
        $short_distance_text = '';
        $pos_text = '';
        $is_red_zone = '';
        $lastPlay = '';
        $score_value = ($game_being_played == 'IN_SESSION' || $game_being_played == 'HALF_TIME') ? '0' : 'NO-SCORE';
        
        // Build SQL query - UPDATED TO INCLUDE BROADCAST FIELD
        $query = "UPDATE sports_list SET 
            Clock = '$clock',
            Period = '$period',
            Game_Played = '$game_being_played',
            Game_Time = '$game_time_formatted',
            Game_Date = '$game_time',
            scoreValue = '$score_value',
            Team_1_Name = '$team1_name',
            Team_2_Name = '$team2_name',
            Team_1_Record = '$team1_record',
            Team_2_Record = '$team2_record',
            Team_1_Score = '$team1_score',
            Team_2_Score = '$team2_score',
            Team_1_Location = '$team1_location',
            Team_2_Location = '$team2_location',
            Team_1_Abbrev = '$team1_abbrev',
            Team_1_Color = '{$team1_colors['rgb']}',
            Team_1_Color_R = '{$team1_colors['r']}',
            Team_1_Color_G = '{$team1_colors['g']}',
            Team_1_Color_B = '{$team1_colors['b']}',
            Team_2_Abbrev = '$team2_abbrev',
            Team_2_Color = '{$team2_colors['rgb']}',
            Team_2_Color_R = '{$team2_colors['r']}',
            Team_2_Color_G = '{$team2_colors['g']}',
            Team_2_Color_B = '{$team2_colors['b']}',
            Who_Has_Ball = '$who_has_ball',
            Distance = '$short_distance_text',
            Possesion = '$pos_text',
            Favorite_Team = '$favorite_team',
            Favorite_Points = '$favorite_points',
            lastPlay = '$lastPlay',
            OverUnder = '$overUnder',
            Red_Zone = '$is_red_zone',
            broadcast = '$broadcast'
            WHERE Game = $gameCounter AND Conference = 'NCAAB'";
        
        $result = mysqli_query($con, $query);
        
        // Store game data for display - display as Away @ Home (matching typical sports format)
        $overtime_indicator = '';
        if ($period > 2) {
            $ot_number = $period - 2;
            $overtime_indicator = $ot_number == 1 ? ' (OT)' : ' (' . $ot_number . 'OT)';
        }
        
        $games[] = array(
            'game_number' => $gameCounter,
            'matchup' => "$team2_location $team2_name @ $team1_location $team1_name", // Away @ Home
            'score' => "$team2_score - $team1_score", // Away - Home
            'status' => $game_being_played,
            'time' => $game_time_formatted,
            'period' => $period,
            'period_display' => $period <= 2 ? "P$period" : ($period == 3 ? "OT" : ($period - 2) . "OT"),
            'clock' => $clock,
            'broadcast' => $broadcast ? $broadcast : 'N/A',
            'updated' => $result ? 'Success' : 'Failed: ' . mysqli_error($con),
            'favorite' => $favorite_team ? "$favorite_team $favorite_points" : 'N/A',
            'over_under' => $overUnder ? $overUnder : 'N/A',
            'date_label' => $dateLabel,
            'overtime' => $overtime_indicator
        );
        
        $gameCounter++;
    }
    
    return array('games' => $games, 'next_game_number' => $gameCounter);
}

// Process TODAY's games (using dynamic date)
$current_date = date("Ymd");
$today_api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/mens-college-basketball/scoreboard?groups=50&limit=200";
$todayResults = processGames($today_api_url, $con, 1, 'Today');

// Determine if we need tomorrow's games
$current_time = date("H:i");
$threshold_time = "11:30";
$allGames = $todayResults['games'];

if ($current_time > $threshold_time) {
    // After 11:30 AM ET, also get tomorrow's games
    $tomorrow_date = date("Ymd", strtotime("+1 day"));
    $tomorrow_api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/mens-college-basketball/scoreboard"
    . "?dates={$tomorrow_date}&groups=50&limit=200";

    $tomorrowResults = processGames($tomorrow_api_url, $con, $todayResults['next_game_number'], 'Tomorrow');
    $allGames = array_merge($allGames, $tomorrowResults['games']);
}

mysqli_close($con);

// Call the NCAAB_Games.php endpoint
$url = 'https://gscroll.us/Scrolling/App_Output/API/Games/NCAAB_Games.php';
$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, false);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
$external_call_success = curl_exec($ch);
$external_call_error = curl_error($ch);
curl_close($ch);

// Generate HTML output
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>NCAAB API Update Results</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: #f5f5f5;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .alert {
            padding: 10px 15px;
            border-radius: 5px;
            margin-bottom: 15px;
        }
        .alert-success {
            background: #c6f6d5;
            color: #22543d;
            border-left: 4px solid #48bb78;
        }
        .alert-error {
            background: #fed7d7;
            color: #742a2a;
            border-left: 4px solid #f56565;
        }
        .alert-info {
            background: #e3f2fd;
            color: #1565c0;
            border-left: 4px solid #2196f3;
        }
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: white;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #333;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        th {
            background: #4a5568;
            color: white;
            padding: 12px;
            text-align: left;
            font-weight: 600;
        }
        td {
            padding: 12px;
            border-bottom: 1px solid #e2e8f0;
        }
        tr:hover {
            background: #f7fafc;
        }
        .date-separator {
            background: #edf2f7;
            font-weight: bold;
            text-align: center;
            color: #4a5568;
        }
        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .status-not-started { background: #e2e8f0; color: #4a5568; }
        .status-in-session { background: #48bb78; color: white; }
        .status-half-time { background: #ed8936; color: white; }
        .status-game-done { background: #9f7aea; color: white; }
        .update-success { color: #48bb78; font-weight: 600; }
        .update-failed { color: #f56565; font-weight: 600; }
        .timestamp {
            text-align: right;
            color: #718096;
            font-size: 14px;
            margin-top: 20px;
        }
        .date-info {
            background: #bee3f8;
            color: #2c5282;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🏀 NCAAB API Update Dashboard</h1>
        <p>Last Updated: <?php echo $time_updated; ?></p>
    </div>
    
    <div class="alert alert-info">
        <strong>Update Triggered:</strong> <?php echo htmlspecialchars($update_check['reason']); ?>
        <?php if ($UPDATE_ANYWAY): ?>
        <br><strong>⚠️ Force Update Mode Active</strong>
        <?php endif; ?>
    </div>
    
    <div class="date-info">
        📅 Processing games for: <strong><?php echo date("F j, Y", strtotime($current_date)); ?></strong>
        <?php if ($current_time > $threshold_time): ?>
            and <strong><?php echo date("F j, Y", strtotime("+1 day")); ?></strong> (after 11:30 AM ET)
        <?php endif; ?>
    </div>
    
    <?php if ($external_call_success !== false): ?>
        <div class="alert alert-success">
            ✅ Successfully triggered NCAAB_Games.php update
        </div>
    <?php else: ?>
        <div class="alert alert-error">
            ❌ Failed to trigger NCAAB_Games.php: <?php echo htmlspecialchars($external_call_error); ?>
        </div>
    <?php endif; ?>
    
    <div class="stats">
        <div class="stat-card">
            <div class="stat-value"><?php echo count($allGames); ?></div>
            <div class="stat-label">Total Games</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo count(array_filter($allGames, function($g) { return strpos($g['updated'], 'Success') !== false; })); ?></div>
            <div class="stat-label">Successful Updates</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo count(array_filter($allGames, function($g) { return $g['status'] == 'IN_SESSION'; })); ?></div>
            <div class="stat-label">Games In Progress</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo count(array_filter($allGames, function($g) { return $g['status'] == 'GAME_DONE'; })); ?></div>
            <div class="stat-label">Completed Games</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo count($todayResults['games']); ?></div>
            <div class="stat-label">Today's Games</div>
        </div>
        <?php if ($current_time > $threshold_time): ?>
        <div class="stat-card">
            <div class="stat-value"><?php echo count($allGames) - count($todayResults['games']); ?></div>
            <div class="stat-label">Tomorrow's Games</div>
        </div>
        <?php endif; ?>
    </div>
    
    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Date</th>
                <th>Matchup</th>
                <th>Score</th>
                <th>Status</th>
                <th>Period/Clock</th>
                <th>Game Time</th>
                <th>Broadcast</th>
                <th>Betting Line</th>
                <th>O/U</th>
                <th>Update Status</th>
            </tr>
        </thead>
        <tbody>
            <?php 
            $lastDateLabel = '';
            foreach ($allGames as $game): 
                if ($game['date_label'] != $lastDateLabel):
                    $lastDateLabel = $game['date_label'];
            ?>
            <tr class="date-separator">
                <td colspan="11"><?php echo $lastDateLabel; ?>'s Games</td>
            </tr>
            <?php endif; ?>
            <tr>
                <td><?php echo $game['game_number']; ?></td>
                <td><?php echo $game['date_label']; ?></td>
                <td><strong><?php echo htmlspecialchars($game['matchup']) . $game['overtime']; ?></strong></td>
                <td><?php echo $game['score']; ?></td>
                <td>
                    <span class="status-badge status-<?php echo strtolower(str_replace('_', '-', $game['status'])); ?>">
                        <?php echo str_replace('_', ' ', $game['status']); ?>
                    </span>
                </td>
                <td><?php echo $game['period_display']; ?> - <?php echo $game['clock']; ?></td>
                <td><?php echo $game['time']; ?></td>
                <td><?php echo htmlspecialchars($game['broadcast']); ?></td>
                <td><?php echo htmlspecialchars($game['favorite']); ?></td>
                <td><?php echo $game['over_under']; ?></td>
                <td class="<?php echo strpos($game['updated'], 'Success') !== false ? 'update-success' : 'update-failed'; ?>">
                    <?php echo $game['updated']; ?>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    
    <div class="timestamp">
        Data Source: ESPN API | Conference: NCAAB | Processing: <?php echo $current_time > $threshold_time ? "Today + Tomorrow" : "Today Only"; ?>
    </div>
</body>
</html>