<?php 
  
require_once '../config.php'; // Include configuration file

error_reporting(E_ALL); // Report all errors and warnings
ini_set('display_errors', 1); // Show errors on screen

/**
 * NCAA Baseball Data Scraper
 * 
 * This script fetches game data from the ESPN API for NCAA Baseball games,
 * parses the data, and updates a database with game information.
 * 
 * Updated: 2025-03-04
 */

// Establish a database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

// Check the connection
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Create an array to hold debug queries
$debug_queries = array();

// Get current time in Eastern Time Zone and format it
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

// Update the API_DB table with the current time
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NCAABase'";
$debug_queries[] = $sql;

// Execute the update query
if (mysqli_query($con, $sql)) {
    if (mysqli_affected_rows($con) > 0) {
        echo "Record updated successfully for API: NCAABase<br>";
    } else {
        echo "No records updated. Please check if the API entry exists.<br>";
    }
} else {
    echo "Error executing query: " . mysqli_error($con) . "<br>";
}

// Track active games
$current_active_games = 0;

/**
 * Helper Functions
 */

// Function to extract team record safely - returns an empty string instead of '0-0'
function extractTeamRecord($competitor) {
    $record = ''; // Default to empty string
    
    // Try to get from records array first
    if (isset($competitor['records']) && is_array($competitor['records']) && !empty($competitor['records'])) {
        foreach ($competitor['records'] as $rec) {
            if (isset($rec['summary']) && !empty($rec['summary'])) {
                $record = $rec['summary'];
                break;
            }
        }
    } 
    // Try alternative sources
    else if (isset($competitor['record']) && !empty($competitor['record'])) {
        $record = $competitor['record'];
    }
    
    return $record;
}

// Function to clean strings by removing special characters
function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

// Function to find the nth occurrence of a substring
function strposX($haystack, $needle, $number = 0) {
    return strpos($haystack, $needle,
        $number > 1 ?
        strposX($haystack, $needle, $number - 1) + strlen($needle) : 0
    );
}

// Function to add leading zeros to RGB values
function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

// Function to get substring between two strings
function getBetween($string, $start = "", $end = "") {
    if (strpos($string, $start)) {
        $startCharCount = strpos($string, $start) + strlen($start);
        $firstSubStr = substr($string, $startCharCount, strlen($string));
        $endCharCount = strpos($firstSubStr, $end);
        if ($endCharCount == 0) {
            $endCharCount = strlen($firstSubStr);
        }
        return substr($firstSubStr, 0, $endCharCount);
    } else {
        return '';
    }
}

// Function to convert date/time between timezones
function dateToTimezone($timeZone = 'UTC', $dateTimeUTC = null, $dateFormat = 'Y-m-d H:i:s') {
    $dateTimeUTC = $dateTimeUTC ? $dateTimeUTC : date("Y-m-d H:i:s");
    $date = new DateTime($dateTimeUTC, new DateTimeZone('UTC'));
    $date->setTimeZone(new DateTimeZone($timeZone));
    return $date->format($dateFormat);
}

// Current date and adjacent dates
$current_date = date("Ymd");
$tomorrow_date = date("Ymd", strtotime("+1 day", strtotime($current_date)));
$yesterdaysDate = date("Ymd", strtotime("-1 day", strtotime($current_date)));
$datesArray = array($current_date, $tomorrow_date, $yesterdaysDate);

// Get today's date in YYYYMMDD format
$today = date('Ymd');

// API endpoint for NCAA Baseball scoreboard with dynamic date
$api_call = "https://site.api.espn.com/apis/site/v2/sports/baseball/college-baseball/scoreboard?dates=$today";


// Make the API call
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);

// Get response from API
$result = curl_exec($curl);
$err = curl_error($curl);
curl_close($curl);

if ($err) {
    echo "cURL Error #:" . $err;
    exit;
}

// Parse the JSON response
$data = json_decode($result, true);

// Check if data is valid
if (empty($data) || !isset($data['events']) || !is_array($data['events'])) {
    echo "Invalid data received from API.";
    exit;
}

// Process each game
$games = $data['events'];
$loop_length = count($games);

for ($i = 1; $i <= $loop_length; $i++) {
    $game_index = $i - 1; // Adjust for zero-based indexing
    if (!isset($games[$game_index])) {
        continue;
    }
    
    $game = $games[$game_index];
    $competition = $game['competitions'][0] ?? [];
    if (empty($competition)) {
        continue;
    }
    
    // Game status information
    $status = $competition['status'] ?? [];
    $status_type = $status['type'] ?? [];
    
    // Default status for internal use (detecting if game is in progress)
    $is_in_progress = false;
    
    // Default status for database
    $game_being_played = 'NOT_STARTED';
    $quarter = (int)($status['period'] ?? 0);
    $clock = "0.0";
    
    if (isset($status_type['name'])) {
        $status_name = trim($status_type['name']);
        switch ($status_name) {
            case 'STATUS_IN_PROGRESS':
                $is_in_progress = true;
                $game_being_played = 'IN_SESSION';
                break;
            case 'STATUS_FINAL':
                $game_being_played = 'GAME_DONE';
                break;
            case 'STATUS_SCHEDULED':
                $game_being_played = 'NOT_STARTED';
                break;
            case 'STATUS_HALFTIME':
                $game_being_played = 'HALF_TIME';
                break;
            case 'STATUS_POSTPONED':
                $game_being_played = 'POSTPONED';
                break;
            case 'STATUS_CANCELED':
                $game_being_played = 'CANCELED';
                break;
        }
    }
    
    if ($game_being_played === 'NOT_STARTED' && isset($status_type['state'])) {
        $state = trim($status_type['state']);
        if ($state === 'in') {
            $is_in_progress = true;
            $game_being_played = 'IN_SESSION';
        } else if ($state === 'post') {
            $game_being_played = 'GAME_DONE';
        }
    }
    
    if ($game_being_played === 'NOT_STARTED' && isset($status_type['description'])) {
        $description = trim($status_type['description']);
        if (strpos($description, 'In Progress') !== false) {
            $is_in_progress = true;
            $game_being_played = 'IN_SESSION';
        }
    }
    
    // Game time
    $game_time = $competition['date'] ?? '';
    $game_time_2 = '';
    if (!empty($game_time)) {
        $game_time_2 = date("g:i a", strtotime($game_time) - 14400); // Convert to Eastern Time
    }
    
    // Initialize variables
    $team1_result = null;
    $team2_result = null;
    $team_name1 = '';
    $team_name2 = '';
    $team1_record = '';  
    $team2_record = '';  
    $team1_score = 0;
    $team2_score = 0;
    $location_team1 = '';
    $location_team2 = '';
    $team_abbrev_1 = '';
    $team_abbrev_2 = '';
    
    // Initialize team colors to white by default
    $team1_color_rgb = '255,255,255';
    $team2_color_rgb = '255,255,255';
    $team1_color_r = addZeros(255);
    $team1_color_g = addZeros(255);
    $team1_color_b = addZeros(255);
    $team2_color_r = addZeros(255);
    $team2_color_g = addZeros(255);
    $team2_color_b = addZeros(255);
    
    $favorite_team = '';
    $favorite_points = '';
    $overUnder = '';
    $short_distance_text = '';
    $pos_text = '';
    $is_red_zone = '';
    $who_has_ball = '';
    $team_id_1 = '';
    $team_id_2 = '';
    $onBases = '0,0,0';
    $outsB = 0;
    $team1HE = '';
    $team2HE = '';
    $lastPlay = '';
    $score_value = 'NO-SCORE';
    
    // Get teams data
    $competitors = $competition['competitors'] ?? [];
    
    foreach ($competitors as $competitor) {
        $homeAway = $competitor['homeAway'] ?? '';
        $team = $competitor['team'] ?? [];
        
        if ($homeAway === 'home') {
            // Team 1 (home team)
            $team_name1 = mysqli_real_escape_string($con, $team['name'] ?? '');
            $location_team1 = mysqli_real_escape_string($con, $team['location'] ?? '');
            $team_abbrev_1 = mysqli_real_escape_string($con, $team['abbreviation'] ?? '');
            $team_id_1 = $team['id'] ?? '';
            $team1_score = isset($competitor['score']) ? (int)$competitor['score'] : 0;
            
            // Extract team record
            $team1_record = mysqli_real_escape_string($con, extractTeamRecord($competitor));
            
            // Team color extraction: if no color exists, default to white.
            $color_hex = $team['color'] ?? '';
            if (!empty($color_hex)) {
                $r = hexdec(substr($color_hex, 0, 2));
                $g = hexdec(substr($color_hex, 2, 2));
                $b = hexdec(substr($color_hex, 4, 2));
                
                // Only force white if the color is exactly black (0,0,0)
                if ($r == 0 && $g == 0 && $b == 0) {
                    $r = 255;
                    $g = 255;
                    $b = 255;
                }
                
                $r = addZeros($r);
                $g = addZeros($g);
                $b = addZeros($b);
                
                $team1_color_r = $r;
                $team1_color_g = $g;
                $team1_color_b = $b;
                $team1_color_rgb = sprintf('%s,%s,%s', $r, $g, $b);
            } else {
                // No color provided; default to white.
                $team1_color_r = addZeros(255);
                $team1_color_g = addZeros(255);
                $team1_color_b = addZeros(255);
                $team1_color_rgb = '255,255,255';
            }
            
            if ($is_in_progress) {
                $hits = isset($competitor['hits']) ? (int)$competitor['hits'] : 0;
                $errors = isset($competitor['errors']) ? (int)$competitor['errors'] : 0;
                $team1HE = $hits . ',' . $errors;
            }
        } else if ($homeAway === 'away') {
            // Team 2 (away team)
            $team_name2 = mysqli_real_escape_string($con, $team['name'] ?? '');
            $location_team2 = mysqli_real_escape_string($con, $team['location'] ?? '');
            $team_abbrev_2 = mysqli_real_escape_string($con, $team['abbreviation'] ?? '');
            $team_id_2 = $team['id'] ?? '';
            $team2_score = isset($competitor['score']) ? (int)$competitor['score'] : 0;
            
            $team2_record = mysqli_real_escape_string($con, extractTeamRecord($competitor));
            
            $color_hex = $team['color'] ?? '';
            if (!empty($color_hex)) {
                $r = hexdec(substr($color_hex, 0, 2));
                $g = hexdec(substr($color_hex, 2, 2));
                $b = hexdec(substr($color_hex, 4, 2));
                
                if ($r == 0 && $g == 0 && $b == 0) {
                    $r = 255;
                    $g = 255;
                    $b = 255;
                }
                
                $r = addZeros($r);
                $g = addZeros($g);
                $b = addZeros($b);
                
                $team2_color_r = $r;
                $team2_color_g = $g;
                $team2_color_b = $b;
                $team2_color_rgb = sprintf('%s,%s,%s', $r, $g, $b);
            } else {
                $team2_color_r = addZeros(255);
                $team2_color_g = addZeros(255);
                $team2_color_b = addZeros(255);
                $team2_color_rgb = '255,255,255';
            }
            
            if ($is_in_progress) {
                $hits = isset($competitor['hits']) ? (int)$competitor['hits'] : 0;
                $errors = isset($competitor['errors']) ? (int)$competitor['errors'] : 0;
                $team2HE = $hits . ',' . $errors;
            }
        }
    }
    
    // Get betting odds if available
    $odds = $competition['odds'] ?? [];
    if (!empty($odds)) {
        $odds_data = $odds[0] ?? [];
        $favorite_team = '';
        $favorite_points = '';
        $overUnder = '';
        
        if (isset($odds_data['awayTeamOdds']['favorite']) && $odds_data['awayTeamOdds']['favorite']) {
            $favorite_team = $team_abbrev_2;
        } else if (isset($odds_data['homeTeamOdds']['favorite']) && $odds_data['homeTeamOdds']['favorite']) {
            $favorite_team = $team_abbrev_1;
        }
        
        if (isset($odds_data['details'])) {
            $details = $odds_data['details'];
            if (strpos($details, $team_abbrev_1) !== false) {
                preg_match('/'.$team_abbrev_1.' (.+)/', $details, $matches);
                $favorite_points = mysqli_real_escape_string($con, $matches[1] ?? '');
            } else if (strpos($details, $team_abbrev_2) !== false) {
                preg_match('/'.$team_abbrev_2.' (.+)/', $details, $matches);
                $favorite_points = mysqli_real_escape_string($con, $matches[1] ?? '');
            }
        }
        
        $overUnder = isset($odds_data['overUnder']) ? (float)$odds_data['overUnder'] : '';
    }
    
    // Get in-game details if game is in progress
    if ($is_in_progress) {
        $current_active_games++;
        $shortDetail = $status_type['shortDetail'] ?? '';
        $topBottom = $shortDetail;
        
        if (preg_match('/(\d+)/', $topBottom, $matches)) {
            $quarter = (int)$matches[1];
        }
        
        if (strpos($topBottom, "Bot") !== false) {
            $who_has_ball = $team_abbrev_1; // Home team batting
        } else {
            $who_has_ball = $team_abbrev_2; // Away team batting
        }
        
        if (strpos($topBottom, "Middle of the") !== false) {
            $who_has_ball = 'mid';
        }
        
        if (strpos($topBottom, "End of the") !== false) {
            $who_has_ball = 'end';
        }
        
        $situation = $competition['situation'] ?? [];
        if (!empty($situation)) {
            $onFirst = isset($situation['onFirst']) && $situation['onFirst'] ? '1' : '0';
            $onSecond = isset($situation['onSecond']) && $situation['onSecond'] ? '1' : '0';
            $onThird = isset($situation['onThird']) && $situation['onThird'] ? '1' : '0';
            $onBases = $onFirst . ',' . $onSecond . ',' . $onThird;
            $outsB = isset($situation['outs']) ? (int)$situation['outs'] : 0;
        }
        
        $lastPlay = mysqli_real_escape_string($con, $competition['lastPlay']['text'] ?? '');
        $scoreValue = isset($competition['lastPlay']['scoreValue']) ? (int)$competition['lastPlay']['scoreValue'] : 0;
        
        if ($scoreValue > 0) {
            $score_value = 'R-' . $scoreValue;
            if (strpos($lastPlay, 'home run') !== false || 
                strpos($lastPlay, 'Home Run') !== false || 
                strpos($lastPlay, 'Home') !== false || 
                strpos($lastPlay, 'home') !== false) {
                $score_value = 'H-' . $scoreValue;
            }
        } else {
            $score_value = 'NO-SCORE';
        }
    }
    
    // Validate team abbreviations and other lengths
    if (empty($team_abbrev_1)) {
        $team_abbrev_1 = "OAK";
    }
    
    if (empty($team_abbrev_2)) {
        $team_abbrev_2 = "OAK";
    }
    
    if (strlen($team_abbrev_2) > 10) {
        $team_abbrev_2 = substr($team_abbrev_2, 0, 3);
    }
    
    if (strlen($team_abbrev_1) > 10) {
        $team_abbrev_1 = substr($team_abbrev_1, 0, 3);
    }
    
    if (strlen($who_has_ball) > 10) {
        $who_has_ball = '';
    }
    
    // Create the update query
    $query = "UPDATE sports_list SET 
        Clock = '$clock', 
        Period = '$quarter', 
        Game_Played = '$game_being_played', 
        Game_Time = '$game_time_2', 
        Game_Date = '$game_time', 
        scoreValue = '$score_value', 
        Team_1_Name = '$team_name1', 
        Team_2_Name = '$team_name2', 
        Team_1_Record = '$team1_record', 
        Team_2_Record = '$team2_record', 
        Team_1_Score = '$team1_score', 
        Team_2_Score = '$team2_score', 
        Team_1_Location = '$location_team1', 
        Team_2_Location = '$location_team2', 
        Team_1_Abbrev = '$team_abbrev_1', 
        Team_1_Color = '$team1_color_rgb', 
        Team_1_Color_R = '$team1_color_r', 
        Team_1_Color_G = '$team1_color_g', 
        Team_1_Color_B = '$team1_color_b', 
        Team_2_Abbrev = '$team_abbrev_2', 
        Team_2_Color = '$team2_color_rgb', 
        Team_2_Color_R = '$team2_color_r', 
        Team_2_Color_G = '$team2_color_g', 
        Team_2_Color_B = '$team2_color_b', 
        Who_Has_Ball = '$who_has_ball', 
        Distance = '$short_distance_text', 
        Possesion = '$pos_text', 
        Favorite_Team = '$favorite_team', 
        Favorite_Points = '$favorite_points', 
        lastPlay = '$lastPlay', 
        OverUnder = '$overUnder', 
        Red_Zone = '$is_red_zone', 
        outsB = '$outsB', 
        onBases = '$onBases', 
        team1HE = '$team1HE', 
        team2HE = '$team2HE' 
    WHERE Game = $i AND Conference = 'NCAABase'";
    
    // Store debug info
    $debug_queries[] = array(
        'game_number' => $i,
        'team1_name' => $team_name1,
        'team1_record' => $team1_record,
        'team2_name' => $team_name2,
        'team2_record' => $team2_record,
        'status' => $game_being_played,
        'team1_color' => $team1_color_rgb,
        'team2_color' => $team2_color_rgb,
        'query' => $query
    );
    
    // Execute the update query
    $query_run = mysqli_query($con, $query);
    if (!$query_run) {
        echo "Error on game $i: " . mysqli_error($con) . "<br>";
    }
}

// Track teams missing record data (optional debug)
$missing_records_teams = [];
foreach ($debug_queries as $debug) {
    if (isset($debug['team1_name']) && empty($debug['team1_record'])) {
        $missing_records_teams[] = $debug['team1_name'];
    }
    if (isset($debug['team2_name']) && empty($debug['team2_record'])) {
        $missing_records_teams[] = $debug['team2_name'];
    }
}

// Close the database connection
mysqli_close($con);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>NCAA Baseball Data Scraper Debug</title>
  <style>
      body {
          font-family: Arial, sans-serif;
          margin: 20px;
          background-color: #f5f5f5;
      }
      
      h1 {
          color: #333;
          text-align: center;
          margin-bottom: 20px;
          padding-bottom: 10px;
          border-bottom: 2px solid #4CAF50;
      }
      
      .debug-table {
          width: 100%;
          border-collapse: collapse;
          margin-bottom: 30px;
          box-shadow: 0 0 20px rgba(0, 0, 0, 0.15);
          background-color: white;
      }
      
      .debug-table th {
          background-color: #4CAF50;
          color: white;
          font-weight: bold;
          padding: 12px 15px;
          text-align: left;
          position: sticky;
          top: 0;
          z-index: 10;
      }
      
      .debug-table td {
          padding: 12px 15px;
          border-bottom: 1px solid #dddddd;
          font-size: 0.9em;
      }
      
      .debug-table tr:nth-child(even) {
          background-color: #f9f9f9;
      }
      
      .debug-table tr:hover {
          background-color: #f1f1f1;
      }
      
      .status {
          display: inline-block;
          padding: 4px 8px;
          border-radius: 4px;
          font-weight: bold;
          text-align: center;
      }
      
      .status-in-session {
          background-color: #FFC107;
          color: #333;
      }
      
      .status-not-started {
          background-color: #2196F3;
          color: white;
      }
      
      .status-game-done {
          background-color: #9E9E9E;
          color: white;
      }
      
      .status-half-time {
          background-color: #673AB7;
          color: white;
      }
      
      .status-canceled {
          background-color: #F44336;
          color: white;
      }
      
      .status-postponed {
          background-color: #FF9800;
          color: white;
      }
      
      .score {
          font-weight: bold;
          font-size: 1.1em;
          text-align: center;
      }
      
      .timestamp {
          text-align: center;
          margin-bottom: 10px;
          color: #666;
          font-style: italic;
      }
      
      .missing-record {
          color: #999;
          font-style: italic;
      }
      
      .api-output {
          background-color: #f8f9fa;
          border: 1px solid #ccc;
          padding: 10px;
          margin-top: 20px;
          font-family: monospace;
          white-space: pre-wrap;
      }
  </style>
</head>
<body>
    <h1>NCAA Baseball Data Scraper Debug</h1>
    <p class="timestamp">Generated on: <?php echo date('Y-m-d H:i:s'); ?></p>
    
    <table class="debug-table">
        <thead>
            <tr>
                <th>#</th>
                <th>Home Team</th>
                <th>Home Record</th>
                <th>Away Team</th>
                <th>Away Record</th>
                <th>Status</th>
                <th>Home Color</th>
                <th>Away Color</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($debug_queries as $debug): ?>
                <?php if (isset($debug['game_number'])): ?>
                <tr>
                    <td><?php echo $debug['game_number']; ?></td>
                    <td><?php echo htmlspecialchars($debug['team1_name']); ?></td>
                    <td><?php 
                        if (!empty($debug['team1_record'])) {
                            echo htmlspecialchars($debug['team1_record']);
                        } else {
                            echo '<span class="missing-record">Not available</span>';
                        }
                    ?></td>
                    <td><?php echo htmlspecialchars($debug['team2_name']); ?></td>
                    <td><?php 
                        if (!empty($debug['team2_record'])) {
                            echo htmlspecialchars($debug['team2_record']);
                        } else {
                            echo '<span class="missing-record">Not available</span>';
                        }
                    ?></td>
                    <td>
                        <?php 
                            $status_class = '';
                            switch($debug['status']) {
                                case 'IN_SESSION':
                                    $status_class = 'status-in-session';
                                    break;
                                case 'NOT_STARTED':
                                    $status_class = 'status-not-started';
                                    break;
                                case 'GAME_DONE':
                                    $status_class = 'status-game-done';
                                    break;
                                case 'HALF_TIME':
                                    $status_class = 'status-half-time';
                                    break;
                                case 'CANCELED':
                                    $status_class = 'status-canceled';
                                    break;
                                case 'POSTPONED':
                                    $status_class = 'status-postponed';
                                    break;
                            }
                        ?>
                        <span class="status <?php echo $status_class; ?>"><?php echo htmlspecialchars($debug['status']); ?></span>
                    </td>
                    <td>
                        <?php 
                            $rgb_vals = explode(',', $debug['team1_color']);
                            echo '<div style="display: flex; align-items: center;">';
                            echo '<div style="background-color: rgb(' . $debug['team1_color'] . '); width: 20px; height: 20px; margin-right: 5px; border: 1px solid #ccc;"></div>';
                            echo '<span>' . $debug['team1_color'] . '</span>';
                            echo '</div>';
                        ?>
                    </td>
                    <td>
                        <?php 
                            $rgb_vals = explode(',', $debug['team2_color']);
                            echo '<div style="display: flex; align-items: center;">';
                            echo '<div style="background-color: rgb(' . $debug['team2_color'] . '); width: 20px; height: 20px; margin-right: 5px; border: 1px solid #ccc;"></div>';
                            echo '<span>' . $debug['team2_color'] . '</span>';
                            echo '</div>';
                        ?>
                    </td>
                </tr>
                <?php else: ?>
                <tr>
                    <td colspan="8"><?php echo htmlspecialchars($debug); ?></td>
                </tr>
                <?php endif; ?>
            <?php endforeach; ?>
        </tbody>
    </table>
    
    <?php if (!empty($missing_records_teams)): ?>
    <div style="margin: 20px 0; padding: 15px; background-color: #f8f9fa; border-left: 4px solid #FFC107; border-radius: 4px;">
        <h3 style="margin-top: 0;">Teams Missing Record Data</h3>
        <p>The following teams did not have record data available in the API:</p>
        <ul>
            <?php foreach($missing_records_teams as $team): ?>
                <li><?php echo htmlspecialchars($team); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>
    
    <?php
    // Additional cURL call to the specified URL at the very end of the script
    $additional_url = SERVER_URL . "Scrolling/App_Output/API/Games/NCAABASE_Games.php";
    $curl2 = curl_init();
    curl_setopt_array($curl2, array(
        CURLOPT_URL => $additional_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
    ));
    $response2 = curl_exec($curl2);
    if (curl_errno($curl2)) {
        $additional_output = "Additional cURL Error: " . curl_error($curl2);
    } else {
        $additional_output = $response2;
    }
    curl_close($curl2);
    ?>
    
    <div class="api-output">
        <h2>Additional API Call Output</h2>
        <pre><?php echo htmlspecialchars($additional_output); ?></pre>
    </div>
    
</body>
</html>
