<?php
require_once '../config.php';

// ====== CONFIG ======
// Define SERVER_URL if not already defined
if (!defined('SERVER_URL')) {
    define('SERVER_URL', 'http://localhost/');
}

// ====== DB CONNECT ======
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// ====== TIMEZONE / TIMESTAMPS ======
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');
$current_timestamp = date('Y-m-d H:i:s');

// ====== SLEEP WINDOW BLOCK (3AM–10AM ET) ======
$current_hour = (int)date('G');
if ($current_hour >= 3 && $current_hour < 10) {
    die("Script is disabled between 3 AM and 10 AM Eastern time. Current time: " . date('g:i A'));
}

// ====== ENSURE TABLE COLUMNS ======
$check_column = "SHOW COLUMNS FROM `sports_list` LIKE 'lastUpdatedTime'";
$column_result = mysqli_query($con, $check_column);
if (mysqli_num_rows($column_result) == 0) {
    $add_column = "ALTER TABLE `sports_list` ADD COLUMN `lastUpdatedTime` DATETIME DEFAULT NULL";
    mysqli_query($con, $add_column);
}

$check_broadcast = "SHOW COLUMNS FROM `sports_list` LIKE 'broadcast'";
$broadcast_result = mysqli_query($con, $check_broadcast);
if (mysqli_num_rows($broadcast_result) == 0) {
    $add_broadcast = "ALTER TABLE `sports_list` ADD COLUMN `broadcast` VARCHAR(50) DEFAULT NULL";
    mysqli_query($con, $add_broadcast);
}

// ====== UPDATE API STAMP ======
$sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'NHL'";
mysqli_query($con, $sql);

// ====== HELPERS ======
function fetchProxyList() {
    $proxy_api_url = "https://proxy.webshare.io/api/v2/proxy/list/download/jtqnktxhwdjdmbvpokcwllcblcalyuiecmiconkc/-/any/username/direct/-/?plan_id=12029912";

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $proxy_api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => false,
    ]);

    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) return [];

    $lines = explode("\n", trim($response));
    $proxies = [];
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '') continue;
        $parts = explode(':', $line);
        if (count($parts) === 4) {
            $proxies[] = [
                'ip' => $parts[0],
                'port' => $parts[1],
                'username' => $parts[2],
                'password' => $parts[3],
                'full' => $line
            ];
        }
    }
    return $proxies;
}

function getRandomProxy($proxies) {
    if (empty($proxies)) return null;
    return $proxies[array_rand($proxies)];
}

function callESPNApiWithProxy($url, $proxies, $maxAttempts = 10) {
    $attempts = 0;
    $success = false;
    $response = null;
    $proxyLog = [];

    while (!$success && $attempts < $maxAttempts && !empty($proxies)) {
        $attempts++;
        $proxy = getRandomProxy($proxies);
        if (!$proxy) {
            $proxyLog[] = [
                'attempt' => $attempts,
                'proxy' => 'No proxy available',
                'status' => 'Failed',
                'error' => 'No proxy available'
            ];
            break;
        }

        $proxyUrl = $proxy['ip'] . ':' . $proxy['port'];
        $proxyAuth = $proxy['username'] . ':' . $proxy['password'];

        $startTime = microtime(true);

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_PROXY => $proxyUrl,
            CURLOPT_PROXYUSERPWD => $proxyAuth,
            CURLOPT_PROXYTYPE => CURLPROXY_HTTP,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $duration = round(microtime(true) - $startTime, 2);

        if ($response && $httpCode == 200) {
            $success = true;
            $proxyLog[] = [
                'attempt' => $attempts,
                'proxy'   => $proxy['ip'] . ':' . $proxy['port'],
                'status'  => 'Success',
                'http_code' => $httpCode,
                'duration'  => $duration . 's',
                'error'   => null
            ];
        } else {
            $proxyLog[] = [
                'attempt' => $attempts,
                'proxy'   => $proxy['ip'] . ':' . $proxy['port'],
                'status'  => 'Failed',
                'http_code' => $httpCode,
                'duration'  => $duration . 's',
                'error'   => $error ?: "HTTP Code: $httpCode"
            ];
            // drop this proxy and try others
            $key = array_search($proxy, $proxies);
            if ($key !== false) {
                unset($proxies[$key]);
                $proxies = array_values($proxies);
            }
        }
    }

    return [
        'success'  => $success,
        'response' => $response,
        'attempts' => $attempts,
        'log'      => $proxyLog
    ];
}

function hexToRgb($hex) {
    $hex = str_replace('#', '', $hex);
    if (strlen($hex) != 6) return "255,255,255";
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    if ($r <= 16 && $g <= 16 && $b <= 16) $r = $g = $b = 255;
    return "$r,$g,$b";
}

function extractGameData($event, $game_num) {
    $gameData = [
        'game_num' => $game_num,
        'event_id' => $event['id'] ?? '',
        'clock' => '0:00',
        'period' => 0,
        'game_status' => 'NOT_STARTED',
        'game_time' => '',
        'game_date' => '',
        'broadcast' => '',
        'score_value' => 'NO-SCORE',
        'team1' => [], // home
        'team2' => [], // away
        'last_play' => '',
        'last_player' => [],
        'who_has_possession' => '',
        'drive_desc' => '',
        'favorite' => [],
        'over_under' => ''
    ];

    $competition = $event['competitions'][0] ?? [];
    $status = $competition['status'] ?? [];
    $gameData['clock'] = $status['displayClock'] ?? '0:00';
    $gameData['period'] = $status['period'] ?? 0;

    $statusType = $status['type']['name'] ?? '';
    switch ($statusType) {
        case 'STATUS_SCHEDULED':
            $gameData['game_status'] = 'NOT_STARTED'; break;
        case 'STATUS_IN_PROGRESS':
            $gameData['game_status'] = 'IN_SESSION'; break;
        case 'STATUS_END_PERIOD':
            $gameData['game_status'] = 'HALF_TIME'; break;
        case 'STATUS_FINAL':
            $gameData['game_status'] = 'GAME_DONE'; break;
    }

    $gameData['game_date'] = $event['date'] ?? '';
    $gameData['game_time'] = date("g:i a", strtotime($gameData['game_date']));

    if (!empty($competition['broadcasts'][0]['names'][0])) {
        $gameData['broadcast'] = $competition['broadcasts'][0]['names'][0];
    }

    $competitors = $competition['competitors'] ?? [];
    foreach ($competitors as $competitor) {
        $teamData = [
            'id' => $competitor['team']['id'] ?? '',
            'name' => $competitor['team']['name'] ?? '',
            'location' => $competitor['team']['location'] ?? '',
            'abbreviation' => $competitor['team']['abbreviation'] ?? '',
            'score' => $competitor['score'] ?? '0',
            'record' => '',
            'color' => '',
            'color_rgb' => '255,255,255'
        ];
        $color   = $competitor['team']['color'] ?? '000000';
        $altColor= $competitor['team']['alternateColor'] ?? '';
        if ($color == '000000' && $altColor) $color = $altColor;
        $teamData['color'] = $color;
        $teamData['color_rgb'] = hexToRgb($color);

        // record
        $records = $competitor['records'] ?? [];
        if ($records) {
            $preferred = ['overall','total','ytd','regularseason','league'];
            foreach ($preferred as $want) {
                foreach ($records as $rec) {
                    $t = strtolower($rec['type'] ?? ($rec['name'] ?? ''));
                    if ($t === $want && !empty($rec['summary'])) {
                        $teamData['record'] = $rec['summary'];
                        break 2;
                    }
                }
            }
            if ($teamData['record'] === '') {
                foreach ($records as $rec) {
                    if (!empty($rec['summary'])) { $teamData['record'] = $rec['summary']; break; }
                }
            }
        }

        if (($competitor['homeAway'] ?? '') === 'home') {
            $gameData['team1'] = $teamData;
        } else {
            $gameData['team2'] = $teamData;
        }
    }

    // Situation (only for in-session/half-time views)
    if (in_array($gameData['game_status'], ['IN_SESSION','HALF_TIME'])) {
        if ($gameData['period'] <= 3)      $gameData['drive_desc'] = "Period " . $gameData['period'];
        elseif ($gameData['period'] == 4)  $gameData['drive_desc'] = "Overtime";
        elseif ($gameData['period'] == 5)  $gameData['drive_desc'] = "Shootout";
    }

    // ======== ODDS ========
    $odds = $competition['odds'][0] ?? [];
    if (!empty($odds)) {
        $gameData['over_under'] = $odds['overUnder'] ?? '';

        $homeOdds = $odds['homeTeamOdds'] ?? [];
        $awayOdds = $odds['awayTeamOdds'] ?? [];

        if (!empty($homeOdds['favorite']) && $homeOdds['favorite']) {
            $gameData['favorite'] = ['team' => $gameData['team1']['abbreviation'], 'spread' => $odds['spread'] ?? ''];
        } elseif (!empty($awayOdds['favorite']) && $awayOdds['favorite']) {
            $gameData['favorite'] = ['team' => $gameData['team2']['abbreviation'], 'spread' => $odds['spread'] ?? ''];
        }

        $spread         = isset($odds['spread']) ? (float)$odds['spread'] : null;
        $homeML         = $homeOdds['moneyLine']  ?? null;
        $awayML         = $awayOdds['moneyLine']  ?? null;
        $homeSpreadOdds = $homeOdds['spreadOdds'] ?? null;
        $awaySpreadOdds = $awayOdds['spreadOdds'] ?? null;

        $gameData['odds'] = [
            'spread'           => $spread,
            'over_under'       => $odds['overUnder'] ?? null,
            'home_moneyline'   => $homeML,
            'away_moneyline'   => $awayML,
            'home_spread_odds' => $homeSpreadOdds,
            'away_spread_odds' => $awaySpreadOdds,
            'provider'         => $odds['provider']['name'] ?? ''
        ];

        $league = 'NHL'; // this script is NHL-only
        $showMoneyline = ($league === 'NHL') && (is_null($spread) || abs($spread) == 1.5);

        if ($showMoneyline) {
            $gameData['odds_display_type'] = 'ML';
            $gameData['odds_display'] = [
                'home' => $homeML,
                'away' => $awayML
            ];
        } else {
            $gameData['odds_display_type'] = 'SPREAD';
            $gameData['odds_display'] = [
                'favorite'         => $gameData['favorite']['team'] ?? '',
                'spread'           => $spread,
                'home_spread_odds' => $homeSpreadOdds,
                'away_spread_odds' => $awaySpreadOdds
            ];
        }
    }
    // ======== END ODDS ========

    return $gameData;
}

function shouldMakeApiCall($con) {
    if (isset($_GET['UPDATE_ANYWAY']) && $_GET['UPDATE_ANYWAY'] == 'true') {
        return ['should_call' => true, 'reason' => 'Manual override (UPDATE_ANYWAY parameter)'];
    }

    $current_time = time();
    $reasons = [];

    $query = "SELECT Game, Game_Played, Game_Date, Clock, Period FROM sports_list WHERE Conference = 'NHL'";
    $result = mysqli_query($con, $query);

    if (!$result || mysqli_num_rows($result) == 0) {
        return ['should_call' => true, 'reason' => 'No games found in database - initial fetch needed'];
    }

    $games = [];
    while ($row = mysqli_fetch_assoc($result)) $games[] = $row;

    $has_active = false; $has_upcoming=false; $has_recent=false;
    $next_game_time = null; $earliest_upcoming = null;

    foreach ($games as $game) {
        $game_status = $game['Game_Played'];
        $game_date = strtotime($game['Game_Date']);

        if ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            $has_active = true;
            $reasons[] = "Game {$game['Game']} is currently active ({$game_status})";
        }

        if ($game_status == 'NOT_STARTED' && $game_date > 0) {
            $until = $game_date - $current_time;
            if ($until <= 600 && $until > -300) {
                $has_upcoming = true;
                $reasons[] = "Game {$game['Game']} starts in " . round($until/60,1) . " minutes";
            }
            if ($until > 0 && ($earliest_upcoming === null || $game_date < $earliest_upcoming)) {
                $earliest_upcoming = $game_date;
                $next_game_time = $game_date;
            }
        }

        if ($game_status == 'GAME_DONE' && $game_date > 0) {
            $time_since_start = $current_time - $game_date;
            if ($time_since_start >= 0 && $time_since_start <= (8 * 3600)) {
                $has_recent = true;
                $reasons[] = "Game {$game['Game']} ended recently (within " . round($time_since_start/3600,1) . " hours)";
            }
        }
    }

    if ($has_active)       return ['should_call'=>true,'reason'=>'Active games in progress','details'=>$reasons];
    if ($has_upcoming)     return ['should_call'=>true,'reason'=>'Games starting soon','details'=>$reasons];
    if ($has_recent)       return ['should_call'=>true,'reason'=>'Recently completed games may have updates','details'=>$reasons];

    if ($next_game_time !== null) {
        $time_until = $next_game_time - $current_time;
        $hours_until = round($time_until/3600,1);
        $mins_until  = round($time_until/60);
        if ($mins_until <= 10) {
            return ['should_call'=>true,'reason'=>"Next game starts in {$mins_until} minutes"];
        } else if ($hours_until < 1) {
            return ['should_call'=>false,'reason'=>"No active games. Next game in {$mins_until} minutes",'next_check'=>date('g:i A',$next_game_time-600)];
        } else {
            return ['should_call'=>false,'reason'=>"No active games. Next game in {$hours_until} hours",'next_check'=>date('g:i A',$next_game_time-600)];
        }
    }

    return ['should_call'=>false,'reason'=>'All games completed or not scheduled - no update needed','next_check'=>'Check schedule for upcoming games'];
}

/* ====== NEW HELPERS (for non-scoring lastPlay refresh) ====== */
function findLatestNonScoringPlay($pbp) {
    $plays = $pbp['plays'] ?? [];
    for ($i = count($plays) - 1; $i >= 0; $i--) {
        $p = $plays[$i];
        if (!empty($p['scoringPlay'])) continue;
        $text = trim($p['text'] ?? '');
        if ($text !== '') {
            return [
                'text' => $text,
                'team_id' => $p['team']['id'] ?? '',
            ];
        }
    }
    return null;
}

function teamIdToAbbrevFromGame($team_id, $gameData) {
    if (!$team_id) return '';
    if (($gameData['team1']['id'] ?? '') == $team_id) return $gameData['team1']['abbreviation'] ?? '';
    if (($gameData['team2']['id'] ?? '') == $team_id) return $gameData['team2']['abbreviation'] ?? '';
    return '';
}
/* ====== END NEW HELPERS ====== */

// ====== MAIN ======
$api_check = shouldMakeApiCall($con);
$should_update = $api_check['should_call'];
$update_reason = $api_check['reason'];
$next_check_time = $api_check['next_check'] ?? null;

$response = null;
$proxyAttempts = 0;
$proxyLog = [];
$api_skipped = false;
$proxies = [];
$games_processed = [];
$skipped_updates = [];
$scoring_plays_detected = [];
$score_snapshot = [];
$debug_log = [];

if ($should_update) {
    // ---- SCORE SNAPSHOT BEFORE API ----
    $debug_log[] = "===== SCORE SNAPSHOT (Before API Call) =====";
    $snapshot_query = "SELECT Game, Team_1_Score, Team_2_Score, Game_Played, scoreValue, lastUpdatedTime
                      FROM sports_list WHERE Conference = 'NHL' ORDER BY Game ASC";
    $snapshot_result = mysqli_query($con, $snapshot_query);
    while ($row = mysqli_fetch_assoc($snapshot_result)) {
        $score_snapshot[$row['Game']] = [
            'home_score' => intval($row['Team_1_Score']),
            'away_score' => intval($row['Team_2_Score']),
            'game_status'=> $row['Game_Played'],
            'score_value'=> $row['scoreValue'],
            'last_updated'=> $row['lastUpdatedTime']
        ];
        $debug_log[] = "Game {$row['Game']}: Home={$row['Team_1_Score']}, Away={$row['Team_2_Score']}, Status={$row['Game_Played']}, ScoreValue={$row['scoreValue']}";
    }
    $debug_log[] = "";

    // ---- FETCH PROXIES + MAIN SCOREBOARD ----
    $proxies = fetchProxyList();
    $debug_log[] = "Fetched " . count($proxies) . " proxies";

    $api_url = "http://site.api.espn.com/apis/site/v2/sports/hockey/nhl/scoreboard";
    $apiResult = callESPNApiWithProxy($api_url, $proxies);

    if (!$apiResult['success']) {
        die("Failed to fetch ESPN API data after {$apiResult['attempts']} attempts");
    }

    $response = $apiResult['response'];
    $proxyAttempts = $apiResult['attempts'];
    $proxyLog = $apiResult['log'];

    $data = json_decode($response, true);
    if (empty($data)) {
        die("Failed to parse API response");
    }

    $events = $data['events'] ?? [];
    $debug_log[] = "Main API returned " . count($events) . " games";
    $debug_log[] = "";

    // ---- PROCESS EACH GAME / WRITE BASE FIELDS ----
    foreach ($events as $index => $event) {
        $game_num = $index + 1;
        $gameData = extractGameData($event, $game_num);

        // 150 pause: if DB has scoreValue=GOAL <150 ago, we partially update only
        $current_db_query = "SELECT scoreValue, lastUpdatedTime FROM sports_list
                             WHERE Game = $game_num AND Conference = 'NHL'";
        $current_db_result = mysqli_query($con, $current_db_query);
        $current_db_data = mysqli_fetch_assoc($current_db_result);

        $skip_update = false;
        $pause_remaining = 0;
        if ($current_db_data && $current_db_data['scoreValue'] === 'GOAL') {
            $lastUpdate = $current_db_data['lastUpdatedTime'];
            if (!empty($lastUpdate)) {
                $timeDiff = time() - strtotime($lastUpdate);
                $pause_remaining = 150 - $timeDiff;
                if ($timeDiff < 150) {
                    $skip_update = true;
                    $skipped_updates[] = $game_num;
                    $debug_log[] = "Game {$game_num}: SKIPPED due to 150 pause ({$pause_remaining}s left)";
                }
            }
        }

        if ($skip_update) {
            // PARTIAL UPDATE: scores/status/clock/period
            $q = "UPDATE sports_list SET
                    Team_1_Score=" . intval($gameData['team1']['score']) . ",
                    Team_2_Score=" . intval($gameData['team2']['score']) . ",
                    Game_Played='" . mysqli_real_escape_string($con, $gameData['game_status']) . "',
                    Clock='" . mysqli_real_escape_string($con, $gameData['clock']) . "',
                    Period=" . intval($gameData['period']) . "
                  WHERE Game=$game_num AND Conference='NHL'";
            mysqli_query($con, $q);

            $gameData['skipped'] = true;
            $gameData['pause_remaining'] = $pause_remaining;
            $debug_log[] = "Game {$game_num}: PAUSED - partial update applied (scores/status only)";
        } else {
            // ======= NHL FAVORITE_POINTS = MONEYLINE (overwrite) =======
            // Compute the value to store in Favorite_Points. For NHL we store the ML of the favorite team.
            $favorite_points_value = $gameData['favorite']['spread'] ?? ''; // default fallback
            $favTeam = $gameData['favorite']['team'] ?? '';
            $homeAbbr = $gameData['team1']['abbreviation'] ?? '';
            $awayAbbr = $gameData['team2']['abbreviation'] ?? '';
            $homeML = $gameData['odds']['home_moneyline'] ?? null;
            $awayML = $gameData['odds']['away_moneyline'] ?? null;

            // NHL-only script: if we have a favorite team and its ML, store that ML instead of spread
            if ($favTeam) {
                if ($favTeam === $homeAbbr && $homeML !== null) {
                    $favorite_points_value = $homeML;
                } elseif ($favTeam === $awayAbbr && $awayML !== null) {
                    $favorite_points_value = $awayML;
                }
            }
            // ==========================================================

            // FULL UPDATE BASE FIELDS (unchanged except Favorite_Points assignment)
            $update_data = [
                'Clock' => mysqli_real_escape_string($con, $gameData['clock']),
                'Period' => intval($gameData['period']),
                'Game_Played' => mysqli_real_escape_string($con, $gameData['game_status']),
                'Game_Time' => mysqli_real_escape_string($con, $gameData['game_time']),
                'Game_Date' => mysqli_real_escape_string($con, $gameData['game_date']),
                'broadcast' => mysqli_real_escape_string($con, $gameData['broadcast']),
                'scoreValue' => mysqli_real_escape_string($con, $gameData['score_value']),
                'Team_1_Name' => mysqli_real_escape_string($con, $gameData['team1']['name']),
                'Team_2_Name' => mysqli_real_escape_string($con, $gameData['team2']['name']),
                'Team_1_Record' => mysqli_real_escape_string($con, $gameData['team1']['record']),
                'Team_2_Record' => mysqli_real_escape_string($con, $gameData['team2']['record']),
                'Team_1_Score' => intval($gameData['team1']['score']),
                'Team_2_Score' => intval($gameData['team2']['score']),
                'Team_1_Location' => mysqli_real_escape_string($con, $gameData['team1']['location']),
                'Team_2_Location' => mysqli_real_escape_string($con, $gameData['team2']['location']),
                'Team_1_Abbrev' => mysqli_real_escape_string($con, $gameData['team1']['abbreviation']),
                'Team_2_Abbrev' => mysqli_real_escape_string($con, $gameData['team2']['abbreviation']),
                'Team_1_Color' => '"' . mysqli_real_escape_string($con, $gameData['team1']['color_rgb']) . '"',
                'Team_2_Color' => '"' . mysqli_real_escape_string($con, $gameData['team2']['color_rgb']) . '"',
                'OverUnder' => mysqli_real_escape_string($con, $gameData['over_under']),
                'driveDesc' => mysqli_real_escape_string($con, $gameData['drive_desc']),
                'Favorite_Team' => mysqli_real_escape_string($con, $gameData['favorite']['team'] ?? ''),
                'lastPlay'  => '"',
                // **** CHANGE IS HERE: write the ML instead of spread for NHL ****
                'Favorite_Points' => mysqli_real_escape_string($con, (string)$favorite_points_value)
                // who_has_possession / lastPlay / lastPlayer* will be updated *only* if delta detects GOAL
            ];

            // Who_Has_Ball: keep prior unless a goal is confirmed below (delta)
            // lastUpdatedTime: only stamp on actual write/goal

            $rgb1 = explode(',', $gameData['team1']['color_rgb']);
            $update_data['Team_1_Color_R'] = $rgb1[0] ?? '255';
            $update_data['Team_1_Color_G'] = $rgb1[1] ?? '255';
            $update_data['Team_1_Color_B'] = $rgb1[2] ?? '255';

            $rgb2 = explode(',', $gameData['team2']['color_rgb']);
            $update_data['Team_2_Color_R'] = $rgb2[0] ?? '255';
            $update_data['Team_2_Color_G'] = $rgb2[1] ?? '255';
            $update_data['Team_2_Color_B'] = $rgb2[2] ?? '255';

            $set = [];
            foreach ($update_data as $field => $value) $set[] = "$field='$value'";

            $q = "UPDATE sports_list SET " . implode(', ', $set) . " WHERE Game=$game_num AND Conference='NHL'";
            mysqli_query($con, $q);

            $debug_log[] = "Game {$game_num}: FULL BASE UPDATE applied (Favorite_Points=" . $favorite_points_value . ")";
            $gameData['skipped'] = false;
            $gameData['pause_remaining'] = 0;
        }

        $games_processed[] = $gameData;
    }

    $debug_log[] = "";
    $debug_log[] = "===== SCORE COMPARISON (Delta-First Goal Detection) =====";

    // ---- DELTA-BASED GOAL DETECTION & WRITE ----
    foreach ($games_processed as &$gameData) {
        $game_num = $gameData['game_num'];
        $snapshot = $score_snapshot[$game_num] ?? null;
        if (!$snapshot) {
            $debug_log[] = "Game {$game_num}: No snapshot found, skipping delta check";
            continue;
        }

        if ($gameData['game_status'] !== 'IN_SESSION') {
            $debug_log[] = "Game {$game_num}: Status={$gameData['game_status']}, skip goal check";
            continue;
        }

        $new_home = intval($gameData['team1']['score']);
        $new_away = intval($gameData['team2']['score']);
        $old_home = intval($snapshot['home_score']);
        $old_away = intval($snapshot['away_score']);

        $home_diff = $new_home - $old_home;
        $away_diff = $new_away - $old_away;

        if ($home_diff <= 0 && $away_diff <= 0) {
            $debug_log[] = "Game {$game_num}: No score increase (ΔH={$home_diff}, ΔA={$away_diff})";
            continue;
        }

        $sides_scored = [];
        if ($home_diff > 0) $sides_scored[] = ['side'=>'home','delta'=>$home_diff];
        if ($away_diff > 0) $sides_scored[] = ['side'=>'away','delta'=>$away_diff];

        foreach ($sides_scored as $entry) {
            $side = $entry['side'];
            $delta = $entry['delta'];

            $scoring_abbrev = ($side === 'home') ? $gameData['team1']['abbreviation'] : $gameData['team2']['abbreviation'];
            $scoring_team_id = ($side === 'home') ? $gameData['team1']['id'] : $gameData['team2']['id'];

            $base_goal_fields = [
                'scoreValue' => 'GOAL',
                'Who_Has_Ball' => mysqli_real_escape_string($con, $scoring_abbrev),
                'lastPlayerTeamID' => mysqli_real_escape_string($con, $scoring_team_id),
                'lastUpdatedTime' => $current_timestamp
            ];

            $safe_goal_text = $scoring_abbrev . " GOAL";
            if ($delta > 1) $safe_goal_text .= " x{$delta}";
            $base_goal_fields['lastPlay'] = mysqli_real_escape_string($con, $safe_goal_text);

            $set_goal = [];
            foreach ($base_goal_fields as $f => $v) $set_goal[] = "$f='$v'";
            $update_goal_base_q = "UPDATE sports_list SET " . implode(', ', $set_goal) . " WHERE Game=$game_num AND Conference='NHL'";
            mysqli_query($con, $update_goal_base_q);

            $debug_log[] = "Game {$game_num}: Δ detected ({$side}, +{$delta}) → who_has_possession={$scoring_abbrev}, teamID={$scoring_team_id}";

            $event_id = $gameData['event_id'];
            $pbp_url = "https://site.api.espn.com/apis/site/v2/sports/hockey/nhl/summary?event=" . $event_id;
            $pbpResult = callESPNApiWithProxy($pbp_url, $proxies);

            if ($pbpResult['success']) {
                $pbp = json_decode($pbpResult['response'], true);
                $plays = $pbp['plays'] ?? [];
                $debug_log[] = "Game {$game_num}: PBP fetch ok (" . count($plays) . " plays)";

                $scoring_play = null;
                for ($i = count($plays)-1; $i >= 0; $i--) {
                    $p = $plays[$i];
                    if (!empty($p['scoringPlay'])) {
                        $pid = $p['team']['id'] ?? null;
                        if ($pid === null || $pid == $scoring_team_id) {
                            $scoring_play = $p;
                            break;
                        }
                    }
                }

                if ($scoring_play) {
                    $goal_text = $scoring_play['text'] ?? $safe_goal_text;

                    $scorer_id = '';
                    $scorer_name = '';
                    $jersey = '';
                    $position = '';

                    $participants = $scoring_play['participants'] ?? [];
                    foreach ($participants as $pt) {
                        if (($pt['type'] ?? '') === 'scorer') {
                            $scorer_id = $pt['athlete']['id'] ?? '';
                            $scorer_name = $pt['athlete']['displayName'] ?? '';
                            break;
                        }
                    }

                    if ($scorer_id && !empty($pbp['boxscore']['players'])) {
                        foreach ($pbp['boxscore']['players'] as $team_entry) {
                            $stats_groups = $team_entry['statistics'] ?? [];
                            foreach ($stats_groups as $sg) {
                                $ath_list = $sg['athletes'] ?? [];
                                foreach ($ath_list as $ath) {
                                    if (($ath['athlete']['id'] ?? '') == $scorer_id) {
                                        $jersey = $ath['athlete']['jersey'] ?? '';
                                        $pos_data = $ath['athlete']['position'] ?? [];
                                        $position = $pos_data['abbreviation'] ?? '';
                                        break 3;
                                    }
                                }
                            }
                        }
                    }

                    $enriched = [
                        'lastPlay'       => mysqli_real_escape_string($con, $goal_text),
                        'lastPlayerID'   => mysqli_real_escape_string($con, $scorer_id),
                        'lastPlayerName' => mysqli_real_escape_string($con, $scorer_name),
                        'jersey'         => mysqli_real_escape_string($con, $jersey),
                        'lastPlayerPos'  => mysqli_real_escape_string($con, $position),
                        'lastPlayerTeamID' => mysqli_real_escape_string($con, $scoring_team_id),
                        'Who_Has_Ball'   => mysqli_real_escape_string($con, $scoring_abbrev),
                        'scoreValue'     => 'GOAL',
                        'lastUpdatedTime'=> $current_timestamp
                    ];
                    $set_en = [];
                    foreach ($enriched as $f => $v) $set_en[] = "$f='$v'";
                    $q_en = "UPDATE sports_list SET " . implode(', ', $set_en) . " WHERE Game=$game_num AND Conference='NHL'";
                    mysqli_query($con, $q_en);

                    $debug_log[] = "Game {$game_num}: Enriched via PBP → player='{$scorer_name}' #{$jersey} {$position}";
                    $gameData['pbp_fetched'] = true;
                    $gameData['last_play'] = $goal_text;
                    $gameData['score_value'] = 'GOAL';
                    $gameData['last_player'] = [
                        'id' => $scorer_id, 'name' => $scorer_name, 'position' => $position,
                        'jersey' => $jersey, 'team_id' => $scoring_team_id
                    ];
                } else {
                    $debug_log[] = "Game {$game_num}: PBP had no matching scoring play; keeping delta-only goal write";
                    $gameData['pbp_fetched'] = false;
                    $gameData['last_play'] = $safe_goal_text;
                    $gameData['score_value'] = 'GOAL';
                    $gameData['last_player'] = ['id'=>'','name'=>'','position'=>'','jersey'=>'','team_id'=>$scoring_team_id];
                }
            } else {
                $debug_log[] = "Game {$game_num}: PBP fetch failed; keeping delta-only goal write";
                $gameData['pbp_fetched'] = false;
                $gameData['last_play'] = $safe_goal_text;
                $gameData['score_value'] = 'GOAL';
                $gameData['last_player'] = ['id'=>'','name'=>'','position'=>'','jersey'=>'','team_id'=>$scoring_team_id];
            }

            $scoring_plays_detected[] = [
                'game_num' => $game_num,
                'home_team' => $gameData['team1']['abbreviation'],
                'away_team' => $gameData['team2']['abbreviation'],
                'scorer_name' => $gameData['last_player']['name'] ?? '',
                'scorer_id' => $gameData['last_player']['id'] ?? '',
                'jersey' => $gameData['last_player']['jersey'] ?? '',
                'position' => $gameData['last_player']['position'] ?? '',
                'team_id' => $scoring_team_id,
                'goal_text' => $gameData['last_play'],
                'time' => $current_timestamp
            ];
        }
    }

    /* ====== NEW: NON-SCORING lastPlay REFRESH ====== */
    foreach ($games_processed as $gameData) {
        $game_num = $gameData['game_num'] ?? null;
        if (!$game_num) continue;

        if (($gameData['game_status'] ?? '') !== 'IN_SESSION') continue;

        $pause_check_q = "SELECT scoreValue, lastUpdatedTime, Team_1_Score, Team_2_Score, lastPlay
                          FROM sports_list
                          WHERE Game={$game_num} AND Conference='NHL' LIMIT 1";
        $pause_res = mysqli_query($con, $pause_check_q);
        $pause_row = $pause_res ? mysqli_fetch_assoc($pause_res) : null;

        $in_goal_pause = false;
        if ($pause_row && ($pause_row['scoreValue'] ?? '') === 'GOAL') {
            $lut = $pause_row['lastUpdatedTime'] ?? null;
            if ($lut && (time() - strtotime($lut)) < 150) {
                $in_goal_pause = true;
            }
        }
        if ($in_goal_pause) {
            $debug_log[] = "Game {$game_num}: Non-scoring refresh skipped (within 150s GOAL pause).";
            continue;
        }

        if (($gameData['score_value'] ?? '') === 'GOAL') {
            $debug_log[] = "Game {$game_num}: Non-scoring refresh skipped (goal already set in this cycle).";
            continue;
        }

        $snapshot = $score_snapshot[$game_num] ?? null;
        if (!$snapshot) continue;

        $cur_home = intval($gameData['team1']['score'] ?? 0);
        $cur_away = intval($gameData['team2']['score'] ?? 0);
        $old_home = intval($snapshot['home_score'] ?? 0);
        $old_away = intval($snapshot['away_score'] ?? 0);

        if ($cur_home > $old_home || $cur_away > $old_away) {
            continue;
        }

        $event_id = $gameData['event_id'] ?? '';
        if (!$event_id) continue;

        $pbp_url = "https://site.api.espn.com/apis/site/v2/sports/hockey/nhl/summary?event=" . $event_id;
        $pbpResult = callESPNApiWithProxy($pbp_url, $proxies);

        if (!$pbpResult['success']) {
            $debug_log[] = "Game {$game_num}: Non-scoring PBP fetch failed; skipping lastPlay refresh.";
            continue;
        }

        $pbp = json_decode($pbpResult['response'], true);
        $latest = findLatestNonScoringPlay($pbp);
        if (!$latest) {
            $debug_log[] = "Game {$game_num}: No non-scoring plays found; skipping lastPlay refresh.";
            continue;
        }

        $new_text = $latest['text'];
        $play_team_id = $latest['team_id'] ?? '';
        $play_team_abbrev = teamIdToAbbrevFromGame($play_team_id, $gameData);

        $old_last_play = $pause_row['lastPlay'] ?? '';
        if ($old_last_play === $new_text) {
            $debug_log[] = "Game {$game_num}: lastPlay unchanged; no DB write.";
            continue;
        }

        $set_fields = "lastPlay='" . mysqli_real_escape_string($con, $new_text) . "'";
        if ($play_team_abbrev !== '') {
            $set_fields .= ", Who_Has_Ball='" . mysqli_real_escape_string($con, $play_team_abbrev) . "'";
        }

        $upd_q = "UPDATE sports_list SET {$set_fields} WHERE Game={$game_num} AND Conference='NHL'";
        mysqli_query($con, $upd_q);

        $debug_log[] = "Game {$game_num}: lastPlay refreshed (non-scoring): " . $new_text;
    }
    /* ====== END NEW NON-SCORING REFRESH ====== */

} else {
    // ====== API SKIPPED: SHOW FROM CACHE ======
    $api_skipped = true;
    $debug_log[] = "API call skipped - using cached data from database";

    $query = "SELECT * FROM sports_list WHERE Conference = 'NHL' ORDER BY Game ASC";
    $result = mysqli_query($con, $query);
    while ($row = mysqli_fetch_assoc($result)) {
        $games_processed[] = [
            'game_num' => $row['Game'],
            'event_id' => '',
            'clock' => $row['Clock'] ?? '0:00',
            'period' => intval($row['Period'] ?? 0),
            'game_status' => $row['Game_Played'] ?? 'NOT_STARTED',
            'game_time' => $row['Game_Time'] ?? '',
            'game_date' => $row['Game_Date'] ?? '',
            'broadcast' => $row['broadcast'] ?? '',
            'score_value' => $row['scoreValue'] ?? 'NO-SCORE',
            'team1' => [
                'name' => $row['Team_1_Name'] ?? '',
                'abbreviation' => $row['Team_1_Abbrev'] ?? '',
                'score' => $row['Team_1_Score'] ?? '0',
                'record' => $row['Team_1_Record'] ?? '',
            ],
            'team2' => [
                'name' => $row['Team_2_Name'] ?? '',
                'abbreviation' => $row['Team_2_Abbrev'] ?? '',
                'score' => $row['Team_2_Score'] ?? '0',
                'record' => $row['Team_2_Record'] ?? '',
            ],
            'last_play' => $row['lastPlay'] ?? '',
            'last_player' => [
                'id' => $row['lastPlayerID'] ?? '',
                'name' => $row['lastPlayerName'] ?? '',
                'position' => $row['lastPlayerPos'] ?? '',
                'jersey' => $row['jersey'] ?? '',
                'team_id' => $row['lastPlayerTeamID'] ?? ''
            ],
            'who_has_possession' => $row['Who_Has_Ball'] ?? '',
            'drive_desc' => $row['driveDesc'] ?? '',
            'favorite' => [
                'team' => $row['Favorite_Team'] ?? '',
                'spread' => $row['Favorite_Points'] ?? '' // in cache this may already be ML (post-change)
            ],
            'over_under' => $row['OverUnder'] ?? '',
            'skipped' => false,
            'pause_remaining' => 0,
            'pbp_fetched' => false,
            'from_cache' => true
        ];
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>NHL Games - Enhanced with Play-by-Play Debug (Delta-First)</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        table { border-collapse: collapse; width: 100%; margin: 20px 0; background: white; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; font-size: 13px; }
        th { background: #333; color: white; font-weight: bold; }
        .status-box { border: 2px solid #333; padding: 15px; margin: 20px 0; background: white; border-radius: 5px; }
        .skipped { background: #fff3cd !important; }
        .goal { background: #ffcccc !important; font-weight: bold; }
        .pbp-fetched { background: #d4edda !important; }
        .debug-box { border: 2px solid #007bff; padding: 15px; margin: 20px 0; background: #e7f3ff; border-radius: 5px; font-family: 'Courier New', monospace; font-size: 12px; max-height: 400px; overflow-y: auto; }
        .debug-box pre { margin: 0; white-space: pre-wrap; }
        .scoring-table { background: #d4edda; }
        .scoring-table th { background: #28a745; }
        h2 { color: #333; border-bottom: 2px solid #333; padding-bottom: 5px; }
        .pause-warning { color: #856404; font-weight: bold; }
    </style>
</head>
<body>
    <h1>NHL Games - Delta-First Goal Detection</h1>
    <p><strong>Last Updated:</strong> <?= htmlspecialchars($time_updated) ?></p>

    <div class="status-box">
        <h3>Update Status</h3>
        <p><strong>API Status:</strong> <?= $should_update ? 'Called' : 'Skipped' ?></p>
        <p><strong>Reason:</strong> <?= htmlspecialchars($update_reason) ?></p>
        <?php if ($next_check_time): ?>
            <p><strong>Next Check:</strong> <?= htmlspecialchars($next_check_time) ?></p>
        <?php endif; ?>
        <?php if ($should_update): ?>
            <p><strong>Proxy Attempts:</strong> <?= intval($proxyAttempts) ?></p>
            <p><strong>Proxies Available:</strong> <?= intval(count($proxies)) ?></p>
        <?php endif; ?>
        <?php if (count($skipped_updates) > 0): ?>
            <p class="pause-warning">⏸️ Games Paused (150-sec rule): <?= htmlspecialchars(implode(', ', $skipped_updates)) ?></p>
        <?php endif; ?>
        <?php if (count($scoring_plays_detected) > 0): ?>
            <p style="color: #28a745; font-weight: bold;">🥅 Goals Detected via Delta: <?= intval(count($scoring_plays_detected)) ?></p>
        <?php endif; ?>
    </div>

    <?php if (count($scoring_plays_detected) > 0): ?>
    <h2>🎯 Scoring Plays Detected This Update</h2>
    <table class="scoring-table">
        <thead>
            <tr>
                <th>Game</th>
                <th>Matchup</th>
                <th>Scorer Name<br><small>(lastPlayerName)</small></th>
                <th>Player ID<br><small>(lastPlayerID)</small></th>
                <th>Jersey #<br><small>(jersey)</small></th>
                <th>Position<br><small>(lastPlayerPos)</small></th>
                <th>Team ID<br><small>(lastPlayerTeamID)</small></th>
                <th>Goal Description<br><small>(lastPlay)</small></th>
                <th>Time Recorded<br><small>(lastUpdatedTime)</small></th>
                <th>DB Status</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($scoring_plays_detected as $sp): ?>
                <tr>
                    <td><strong>#<?= intval($sp['game_num']) ?></strong></td>
                    <td><?= htmlspecialchars($sp['away_team']) ?> @ <?= htmlspecialchars($sp['home_team']) ?></td>
                    <td><strong><?= htmlspecialchars($sp['scorer_name']) ?></strong></td>
                    <td><code><?= htmlspecialchars($sp['scorer_id']) ?></code></td>
                    <td><strong><?= htmlspecialchars($sp['jersey'] ?: '?') ?></strong></td>
                    <td><?= htmlspecialchars($sp['position'] ?: '?') ?></td>
                    <td><code><?= htmlspecialchars($sp['team_id']) ?></code></td>
                    <td style="max-width: 300px; font-size: 11px;"><?= htmlspecialchars($sp['goal_text']) ?></td>
                    <td><?= htmlspecialchars($sp['time']) ?></td>
                    <td><span style="background: #28a745; color: white; padding: 4px 8px; border-radius: 4px; font-weight: bold;">✓ WRITTEN</span></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <div class="status-box" style="background: #d4edda; border-color: #28a745;">
        <h3 style="color: #155724; margin-top: 0;">Database Fields Mapping</h3>
        <p style="color: #155724; margin: 10px 0;">Fields written to <code>sports_list</code> on detected goal:</p>
        <ul style="color: #155724; line-height: 1.8;">
            <li><strong>scoreValue</strong> = 'GOAL' (triggers 150s pause)</li>
            <li><strong>Who_Has_Ball</strong> = scoring team abbreviation (delta derived)</li>
            <li><strong>lastPlayerTeamID</strong> = scoring team’s ESPN ID (delta derived)</li>
            <li><strong>lastPlay</strong> = safe fallback (“{ABBR} GOAL”), overwritten if PBP text found</li>
            <li><strong>lastPlayerName/ID/jersey/lastPlayerPos</strong> = from PBP if available</li>
            <li><strong>lastUpdatedTime</strong> = current timestamp</li>
        </ul>
    </div>
    <?php endif; ?>

    <h2>All Games</h2>
    <table>
        <thead>
            <tr>
                <th>Game</th>
                <th>Status</th>
                <thUpdate>Update</thUpdate>
                <th>Away</th>
                <th>Score</th>
                <th>Home</th>
                <th>Score</th>
                <th>Period</th>
                <th>Clock</th>
                <th>Event</th>
                <th>PBP</th>
                <th>Pause</th>
                <th>Last Play</th>
                <th>Player</th>
                <th>Jersey</th>
                <th>Pos</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($games_processed as $game): ?>
                <tr class="<?= !empty($game['skipped']) ? 'skipped' : '' ?> <?= !empty($game['pbp_fetched']) ? 'pbp-fetched' : '' ?>">
                    <td><?= intval($game['game_num']) ?></td>
                    <td><?= htmlspecialchars($game['game_status']) ?></td>
                    <td><?= isset($game['from_cache']) ? 'CACHED' : (!empty($game['skipped']) ? 'SKIPPED' : 'UPDATED') ?></td>
                    <td><?= htmlspecialchars($game['team2']['abbreviation']) ?></td>
                    <td><?= htmlspecialchars($game['team2']['score']) ?></td>
                    <td><?= htmlspecialchars($game['team1']['abbreviation']) ?></td>
                    <td><?= htmlspecialchars($game['team1']['score']) ?></td>
                    <td><?= intval($game['period']) ?></td>
                    <td><?= htmlspecialchars($game['clock']) ?></td>
                    <td class="<?= ($game['score_value'] ?? '') === 'GOAL' ? 'goal' : '' ?>"><?= htmlspecialchars($game['score_value']) ?></td>
                    <td><?= !empty($game['pbp_fetched']) ? '✓' : '-' ?></td>
                    <td><?= !empty($game['pause_remaining']) && $game['pause_remaining'] > 0 ? intval($game['pause_remaining']).'s' : '-' ?></td>
                    <td style="max-width: 250px; font-size: 11px;"><?= htmlspecialchars($game['last_play'] ?? '') ?></td>
                    <td><?= htmlspecialchars($game['last_player']['name'] ?? '-') ?></td>
                    <td><?= htmlspecialchars($game['last_player']['jersey'] ?? '-') ?></td>
                    <td><?= htmlspecialchars($game['last_player']['position'] ?? '-') ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <h2>🔍 Debug Log</h2>
    <div class="debug-box">
        <pre><?php
        foreach ($debug_log as $log) {
            echo htmlspecialchars($log) . "\n";
        }
        ?></pre>
    </div>
</body>
</html>

<?php
mysqli_close($con);

// ====== TRIGGER RENDERERS ======
// generates the images displayed the glance , think 1.bin, 2.bin
$url = SERVER_URL . "Scrolling/App_Output/API/Games/NHL_Games.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
curl_close($ch);

// generates the images displayed the glance , think 1_ALERT_GOAL_XXX.bin
$url = SERVER_URL . "Scrolling/App_Output/API/Games/Scoring_ALERT.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
curl_close($ch);
?>
