<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../config.php'; // Include the configuration file

// Set timezone
date_default_timezone_set('America/New_York');

/**
 * Modern WNBA ESPN API Scraper
 * Version 2.0 - Complete rewrite using JSON parsing with HTML output
 * 
 * This script fetches WNBA game data from ESPN's API and updates the database
 */

class WNBAScoreboard {
    private $con;
    private $api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/wnba/scoreboard";
    private $headers = [
        'Accept: application/json',
        'Accept-Language: en-US,en;q=0.9',
        'Cache-Control: no-cache',
        'Pragma: no-cache',
        'Referer: https://www.espn.com/',
        'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
    ];
    
    private $html_output = '';
    
    public function __construct($host, $username, $password, $database) {
        $this->con = mysqli_connect($host, $username, $password, $database);
        
        if (!$this->con) {
            die("Connection failed: " . mysqli_connect_error());
        }
        
        // Start HTML output
        $this->initializeHTML();
    }
    
    /**
     * Initialize HTML output
     */
    private function initializeHTML() {
        $this->html_output = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WNBA Scoreboard Update</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            padding: 30px;
        }
        h1 {
            color: #ff6600;
            text-align: center;
            margin-bottom: 10px;
            font-size: 2.5em;
        }
        .timestamp {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 1.1em;
        }
        .games-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
        }
        .games-table th {
            background-color: #ff6600;
            color: white;
            padding: 15px;
            text-align: left;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .games-table td {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }
        .games-table tr:hover {
            background-color: #f9f9f9;
        }
        .game-status {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: 600;
            text-transform: uppercase;
        }
        .status-live {
            background-color: #ff4444;
            color: white;
        }
        .status-halftime {
            background-color: #ff8800;
            color: white;
        }
        .status-final {
            background-color: #4caf50;
            color: white;
        }
        .status-scheduled {
            background-color: #2196f3;
            color: white;
        }
        .team-row {
            display: flex;
            align-items: center;
            margin: 5px 0;
        }
        .team-abbr {
            font-weight: 700;
            width: 50px;
            color: #333;
        }
        .team-name {
            flex: 1;
            color: #555;
        }
        .team-score {
            font-size: 1.3em;
            font-weight: 700;
            width: 40px;
            text-align: right;
        }
        .team-record {
            color: #888;
            font-size: 0.9em;
            margin-left: 10px;
        }
        .betting-info {
            font-size: 0.9em;
            color: #666;
            background-color: #f8f8f8;
            padding: 8px 12px;
            border-radius: 6px;
            margin-top: 5px;
        }
        .last-play {
            font-style: italic;
            color: #666;
            font-size: 0.9em;
            margin-top: 5px;
        }
        .success-message {
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 12px 20px;
            border-radius: 6px;
            margin: 20px 0;
        }
        .error-message {
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
            padding: 12px 20px;
            border-radius: 6px;
            margin: 20px 0;
        }
        .summary {
            background-color: #e3f2fd;
            border: 1px solid #bbdefb;
            color: #1565c0;
            padding: 15px 20px;
            border-radius: 6px;
            text-align: center;
            font-weight: 600;
            margin-top: 20px;
        }
        .debug-query {
            background-color: #f5f5f5;
            border: 1px solid #ddd;
            padding: 15px;
            margin: 10px 0;
            border-radius: 6px;
            font-family: monospace;
            font-size: 0.85em;
            overflow-x: auto;
            display: none;
        }
        .show-debug .debug-query {
            display: block;
        }
        .toggle-debug {
            background-color: #666;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9em;
            margin-bottom: 20px;
        }
        .toggle-debug:hover {
            background-color: #555;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🏀 WNBA Scoreboard Update</h1>
        <div class="timestamp">📅 ' . date('l, F j, Y - g:i A') . ' ET</div>
        <button class="toggle-debug" onclick="document.body.classList.toggle(\'show-debug\')">Toggle SQL Debug</button>
';
    }
    
    /**
     * Fetch data from ESPN API with proper headers
     */
    private function fetchAPIData() {
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => $this->headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ]);
        
        $response = curl_exec($curl);
        $err = curl_error($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        if ($err) {
            throw new Exception("cURL Error: " . $err);
        }
        
        if ($httpCode !== 200) {
            throw new Exception("HTTP Error: " . $httpCode);
        }
        
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("JSON Decode Error: " . json_last_error_msg());
        }
        
        return $data;
    }
    
    /**
     * Parse game status
     */
    private function getGameStatus($game) {
        $status = $game['status'] ?? [];
        $type = $status['type'] ?? [];
        $typeName = $type['name'] ?? '';
        $period = $status['period'] ?? 0;
        $clock = $status['displayClock'] ?? '0:00';
        
        // Check if game is completed
        if ($type['completed'] ?? false) {
            return 'GAME_DONE';
        }
        
        // Check game state
        switch ($typeName) {
            case 'STATUS_SCHEDULED':
                return 'NOT_STARTED';
            case 'STATUS_IN_PROGRESS':
                if ($period == 2 && $clock == '0:00') {
                    return 'HALF_TIME';
                }
                return 'IN_SESSION';
            case 'STATUS_HALFTIME':
                return 'HALF_TIME';
            case 'STATUS_FINAL':
            case 'STATUS_FINAL_OT':
                return 'GAME_DONE';
            default:
                return 'NOT_STARTED';
        }
    }
    
    /**
     * Extract team data
     */
    private function extractTeamData($competitor) {
        $team = $competitor['team'] ?? [];
        
        return [
            'name' => $team['name'] ?? '',
            'location' => $team['location'] ?? '',
            'abbreviation' => $team['abbreviation'] ?? '',
            'displayName' => $team['displayName'] ?? '',
            'color' => $team['color'] ?? '000000',
            'alternateColor' => $team['alternateColor'] ?? '',
            'score' => $competitor['score'] ?? '0',
            'homeAway' => $competitor['homeAway'] ?? '',
            'winner' => $competitor['winner'] ?? false,
            'records' => $this->extractRecords($competitor['records'] ?? []),
            'leaders' => $competitor['leaders'] ?? []
        ];
    }
    
    /**
     * Extract team records
     */
    private function extractRecords($records) {
        foreach ($records as $record) {
            if ($record['name'] === 'overall') {
                return $record['summary'] ?? '0-0';
            }
        }
        return '0-0';
    }
    
    /**
     * Convert hex color to RGB
     */
    private function hexToRgb($hex) {
        $hex = str_replace('#', '', $hex);
        if (strlen($hex) != 6) {
            return ['r' => '000', 'g' => '000', 'b' => '000'];
        }
        
        $r = str_pad(hexdec(substr($hex, 0, 2)), 3, '0', STR_PAD_LEFT);
        $g = str_pad(hexdec(substr($hex, 2, 2)), 3, '0', STR_PAD_LEFT);
        $b = str_pad(hexdec(substr($hex, 4, 2)), 3, '0', STR_PAD_LEFT);
        
        return ['r' => $r, 'g' => $g, 'b' => $b];
    }
    
    /**
     * Extract betting lines
     */
    private function extractBettingData($game) {
        $competitions = $game['competitions'] ?? [];
        if (empty($competitions)) {
            return ['favorite_team' => '', 'favorite_points' => '', 'overUnder' => ''];
        }
        
        $odds = $competitions[0]['odds'] ?? [];
        if (empty($odds)) {
            return ['favorite_team' => '', 'favorite_points' => '', 'overUnder' => ''];
        }
        
        $firstOdd = $odds[0] ?? [];
        $details = $firstOdd['details'] ?? '';
        $overUnder = $firstOdd['overUnder'] ?? '';
        
        // Parse betting line
        $favorite_team = '';
        $favorite_points = '';
        
        if ($details) {
            // Details format: "TEAM -X.X"
            $parts = explode(' ', $details);
            if (count($parts) >= 2) {
                $favorite_team = $parts[0];
                $favorite_points = $parts[1];
            }
        }
        
        return [
            'favorite_team' => $favorite_team,
            'favorite_points' => $favorite_points,
            'overUnder' => $overUnder
        ];
    }
    
    /**
     * Process all games and update database
     */
    public function processGames() {
        try {
            $data = $this->fetchAPIData();
            
            $events = $data['events'] ?? [];
            $gameNumber = 0;
            
            $this->html_output .= '<table class="games-table">
                <thead>
                    <tr>
                        <th>Game</th>
                        <th>Status</th>
                        <th>Teams & Score</th>
                        <th>Game Info</th>
                        <th>Betting Line</th>
                        <th>Update Status</th>
                    </tr>
                </thead>
                <tbody>';
            
            foreach ($events as $event) {
                $gameNumber++;
                $this->processGame($event, $gameNumber);
            }
            
            $this->html_output .= '</tbody></table>';
            
            // Update last update time
            $this->updateLastUpdateTime();
            
            $this->html_output .= '<div class="summary">📊 Successfully processed ' . count($events) . ' games</div>';
            
        } catch (Exception $e) {
            $this->html_output .= '<div class="error-message">❌ Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    /**
     * Process individual game
     */
    private function processGame($game, $gameNumber) {
        $competitions = $game['competitions'] ?? [];
        if (empty($competitions)) {
            return;
        }
        
        $competition = $competitions[0];
        $competitors = $competition['competitors'] ?? [];
        
        if (count($competitors) < 2) {
            return;
        }
        
        // Get game status
        $gameStatus = $this->getGameStatus($competition);
        $status = $competition['status'] ?? [];
        
        // Extract basic game info
        $gameDate = $game['date'] ?? '';
        $gameTime = date("g:i a", strtotime($gameDate));
        $clock = $status['displayClock'] ?? '0:00';
        $period = $status['period'] ?? 0;
        
        // Determine home and away teams
        $homeTeam = null;
        $awayTeam = null;
        
        foreach ($competitors as $competitor) {
            if ($competitor['homeAway'] === 'home') {
                $homeTeam = $this->extractTeamData($competitor);
            } else {
                $awayTeam = $this->extractTeamData($competitor);
            }
        }
        
        if (!$homeTeam || !$awayTeam) {
            $this->html_output .= '<tr><td colspan="6" class="error-message">Could not extract team data for game ' . $gameNumber . '</td></tr>';
            return;
        }
        
        // Get colors
        $team1Color = $this->hexToRgb($awayTeam['color']);
        $team2Color = $this->hexToRgb($homeTeam['color']);
        
        // Get betting data
        $bettingData = $this->extractBettingData($game);
        
        // Get last play (if available)
        $situation = $competition['situation'] ?? [];
        $lastPlay = $situation['lastPlay'] ?? [];
        $lastPlayText = $lastPlay['text'] ?? '';
        
        // Get leaders data
        $leadersText = $this->formatLeaders($awayTeam['leaders'], $homeTeam['leaders']);
        
        // Start HTML row
        $this->html_output .= '<tr>';
        
        // Game number
        $this->html_output .= '<td><strong>Game ' . $gameNumber . '</strong></td>';
        
        // Status
        $statusClass = match($gameStatus) {
            'IN_SESSION' => 'status-live',
            'HALF_TIME' => 'status-halftime',
            'GAME_DONE' => 'status-final',
            'NOT_STARTED' => 'status-scheduled',
            default => ''
        };
        
        $statusText = match($gameStatus) {
            'IN_SESSION' => '🔴 LIVE',
            'HALF_TIME' => '⏸️ HALFTIME',
            'GAME_DONE' => '✅ FINAL',
            'NOT_STARTED' => '🕐 SCHEDULED',
            default => $gameStatus
        };
        
        $this->html_output .= '<td><span class="game-status ' . $statusClass . '">' . $statusText . '</span></td>';
        
        // Teams & Score
        $this->html_output .= '<td>
            <div class="team-row">
                <span class="team-abbr">' . htmlspecialchars($awayTeam['abbreviation']) . '</span>
                <span class="team-name">' . htmlspecialchars($awayTeam['location'] . ' ' . $awayTeam['name']) . '</span>
                <span class="team-score">' . htmlspecialchars($awayTeam['score']) . '</span>
                <span class="team-record">' . htmlspecialchars($awayTeam['records']) . '</span>
            </div>
            <div class="team-row">
                <span class="team-abbr">' . htmlspecialchars($homeTeam['abbreviation']) . '</span>
                <span class="team-name">' . htmlspecialchars($homeTeam['location'] . ' ' . $homeTeam['name']) . '</span>
                <span class="team-score">' . htmlspecialchars($homeTeam['score']) . '</span>
                <span class="team-record">' . htmlspecialchars($homeTeam['records']) . '</span>
            </div>';
        
        if ($lastPlayText) {
            $this->html_output .= '<div class="last-play">📝 ' . htmlspecialchars($lastPlayText) . '</div>';
        }
        
        $this->html_output .= '</td>';
        
        // Game Info
        $this->html_output .= '<td>';
        $this->html_output .= '📅 ' . htmlspecialchars($gameTime) . ' ET<br>';
        
        if ($period > 0) {
            $periodText = match($period) {
                1 => "1st Quarter",
                2 => "2nd Quarter", 
                3 => "3rd Quarter",
                4 => "4th Quarter",
                5 => "Overtime",
                default => "Period {$period}"
            };
            $this->html_output .= '🏀 ' . $periodText . '<br>⏱️ ' . htmlspecialchars($clock);
        }
        
        $this->html_output .= '</td>';
        
        // Betting Line
        $this->html_output .= '<td>';
        if ($bettingData['favorite_team']) {
            $this->html_output .= '<div class="betting-info">';
            $this->html_output .= '💰 Line: ' . htmlspecialchars($bettingData['favorite_team'] . ' ' . $bettingData['favorite_points']) . '<br>';
            $this->html_output .= 'O/U: ' . htmlspecialchars($bettingData['overUnder']);
            $this->html_output .= '</div>';
        } else {
            $this->html_output .= '-';
        }
        $this->html_output .= '</td>';
        
        // Prepare data for database
        $updateData = [
            'Game' => $gameNumber,
            'Clock' => $clock,
            'Period' => $period,
            'Game_Played' => $gameStatus,
            'Game_Time' => $gameTime,
            'Game_Date' => $gameDate,
            'Team_1_Name' => $awayTeam['name'],
            'Team_2_Name' => $homeTeam['name'],
            'Team_1_Record' => $awayTeam['records'],
            'Team_2_Record' => $homeTeam['records'],
            'Team_1_Score' => $awayTeam['score'],
            'Team_2_Score' => $homeTeam['score'],
            'Team_1_Location' => $awayTeam['location'],
            'Team_2_Location' => $homeTeam['location'],
            'Team_1_Abbrev' => $awayTeam['abbreviation'],
            'Team_2_Abbrev' => $homeTeam['abbreviation'],
            'Team_1_Color' => sprintf('%s,%s,%s', $team1Color['r'], $team1Color['g'], $team1Color['b']),
            'Team_1_Color_R' => $team1Color['r'],
            'Team_1_Color_G' => $team1Color['g'],
            'Team_1_Color_B' => $team1Color['b'],
            'Team_2_Color' => sprintf('%s,%s,%s', $team2Color['r'], $team2Color['g'], $team2Color['b']),
            'Team_2_Color_R' => $team2Color['r'],
            'Team_2_Color_G' => $team2Color['g'],
            'Team_2_Color_B' => $team2Color['b'],
            'Favorite_Team' => $bettingData['favorite_team'],
            'Favorite_Points' => $bettingData['favorite_points'],
            'OverUnder' => $bettingData['overUnder'],
            'lastPlay' => $lastPlayText,
            'scoreValue' => 'NO-SCORE', // This field seems deprecated in your schema
            'Who_Has_Ball' => '', // Not available in basketball
            'Distance' => '', // Not available in basketball
            'Possesion' => '', // Not available in basketball
            'Red_Zone' => '' // Not available in basketball
        ];
        
        // Update database and capture result
        $updateResult = $this->updateDatabase($updateData);
        
        // Update Status
        $this->html_output .= '<td>' . $updateResult . '</td>';
        
        $this->html_output .= '</tr>';
    }
    
    /**
     * Format leaders for display
     */
    private function formatLeaders($awayLeaders, $homeLeaders) {
        $leaders = [];
        
        // Process away team leaders (prefix with !)
        foreach ($awayLeaders as $category) {
            if ($category['name'] === 'points' || $category['name'] === 'rebounds') {
                foreach ($category['leaders'] ?? [] as $leader) {
                    $athlete = $leader['athlete'] ?? [];
                    $displayName = $athlete['displayName'] ?? '';
                    $value = $leader['displayValue'] ?? '';
                    
                    if ($displayName) {
                        $lastName = $this->extractLastName($displayName);
                        $statType = ($category['name'] === 'points') ? 'PTS' : 'RBD';
                        $leaders[] = "!{$lastName}: {$value} {$statType}";
                    }
                }
            }
        }
        
        // Process home team leaders (prefix with _)
        foreach ($homeLeaders as $category) {
            if ($category['name'] === 'points' || $category['name'] === 'rebounds') {
                foreach ($category['leaders'] ?? [] as $leader) {
                    $athlete = $leader['athlete'] ?? [];
                    $displayName = $athlete['displayName'] ?? '';
                    $value = $leader['displayValue'] ?? '';
                    
                    if ($displayName) {
                        $lastName = $this->extractLastName($displayName);
                        $statType = ($category['name'] === 'points') ? 'PTS' : 'RBD';
                        $leaders[] = "_{$lastName}: {$value} {$statType}";
                    }
                }
            }
        }
        
        return implode(', ', $leaders);
    }
    
    /**
     * Extract last name from full name
     */
    private function extractLastName($fullName) {
        $parts = explode(' ', $fullName);
        $lastName = end($parts);
        
        // Handle suffixes
        $suffixes = ['JR.', 'III', 'II', 'IV', 'V', 'SR.'];
        if (in_array(strtoupper($lastName), $suffixes) && count($parts) > 2) {
            $lastName = $parts[count($parts) - 2];
        }
        
        return strtoupper($lastName);
    }
    
    /**
     * Update database with game data
     */
    private function updateDatabase($data) {
        // Escape all values
        foreach ($data as $key => $value) {
            $data[$key] = mysqli_real_escape_string($this->con, $value);
        }
        
        $query = "UPDATE sports_list SET 
            Clock = '{$data['Clock']}',
            Period = '{$data['Period']}',
            Game_Played = '{$data['Game_Played']}',
            Game_Time = '{$data['Game_Time']}',
            Game_Date = '{$data['Game_Date']}',
            scoreValue = '{$data['scoreValue']}',
            Team_1_Name = '{$data['Team_1_Name']}',
            Team_2_Name = '{$data['Team_2_Name']}',
            Team_1_Record = '{$data['Team_1_Record']}',
            Team_2_Record = '{$data['Team_2_Record']}',
            Team_1_Score = '{$data['Team_1_Score']}',
            Team_2_Score = '{$data['Team_2_Score']}',
            Team_1_Location = '{$data['Team_1_Location']}',
            Team_2_Location = '{$data['Team_2_Location']}',
            Team_1_Abbrev = '{$data['Team_1_Abbrev']}',
            Team_1_Color = '{$data['Team_1_Color']}',
            Team_1_Color_R = '{$data['Team_1_Color_R']}',
            Team_1_Color_G = '{$data['Team_1_Color_G']}',
            Team_1_Color_B = '{$data['Team_1_Color_B']}',
            Team_2_Abbrev = '{$data['Team_2_Abbrev']}',
            Team_2_Color = '{$data['Team_2_Color']}',
            Team_2_Color_R = '{$data['Team_2_Color_R']}',
            Team_2_Color_G = '{$data['Team_2_Color_G']}',
            Team_2_Color_B = '{$data['Team_2_Color_B']}',
            Who_Has_Ball = '{$data['Who_Has_Ball']}',
            Distance = '{$data['Distance']}',
            Possesion = '{$data['Possesion']}',
            Favorite_Team = '{$data['Favorite_Team']}',
            Favorite_Points = '{$data['Favorite_Points']}',
            lastPlay = '{$data['lastPlay']}',
            OverUnder = '{$data['OverUnder']}',
            Red_Zone = '{$data['Red_Zone']}'
        WHERE Game = {$data['Game']} AND Conference = 'WNBA'";
        
        // Add debug query to HTML
        $this->html_output .= '<div class="debug-query">' . htmlspecialchars($query) . '</div>';
        
        if (!mysqli_query($this->con, $query)) {
            return '<span style="color: red;">❌ ' . htmlspecialchars(mysqli_error($this->con)) . '</span>';
        } else {
            return '<span style="color: green;">✅ Success</span>';
        }
    }
    
    /**
     * Update last update time
     */
    private function updateLastUpdateTime() {
        $timeUpdated = date('m/d/Y | g:iA');
        $query = "UPDATE `API_DB` SET `TIME_UPDATED` = '$timeUpdated' WHERE `API` = 'WNBA'";
        
        if (mysqli_query($this->con, $query)) {
            if (mysqli_affected_rows($this->con) > 0) {
                $this->html_output .= '<div class="success-message">✅ Update time recorded: ' . $timeUpdated . '</div>';
            }
        } else {
            $this->html_output .= '<div class="error-message">❌ Error updating time: ' . htmlspecialchars(mysqli_error($this->con)) . '</div>';
        }
    }
    
    /**
     * Call the output script
     */
    public function callOutputScript() {
        $url = SERVER_URL . "Scrolling/App_Output/API/Games/WNBA_Games.php";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        
        $this->html_output .= '<h2>📤 Output Script Response</h2>';
        
        if (curl_errno($ch)) {
            $this->html_output .= '<div class="error-message">❌ Error calling output script: ' . htmlspecialchars(curl_error($ch)) . '</div>';
        } else {
            $this->html_output .= '<pre style="background-color: #f5f5f5; padding: 15px; border-radius: 6px; overflow-x: auto;">' . htmlspecialchars($response) . '</pre>';
        }
        
        curl_close($ch);
    }
    
    /**
     * Output the HTML
     */
    public function outputHTML() {
        $this->html_output .= '
    </div>
    <script>
        // Auto-refresh every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html>';
        
        echo $this->html_output;
    }
    
    public function __destruct() {
        if ($this->con) {
            mysqli_close($this->con);
        }
    }
}

// Main execution
try {
    $scoreboard = new WNBAScoreboard($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
    $scoreboard->processGames();
    $scoreboard->callOutputScript();
    $scoreboard->outputHTML();
} catch (Exception $e) {
    echo '<div style="background-color: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 20px; margin: 20px; border-radius: 6px;">';
    echo '<h2>Fatal Error</h2>';
    echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
    echo '</div>';
}

?>