<?php
// Enable full error reporting for debugging.
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include configuration and establish a database connection.
require_once '../config.php';
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Define the API endpoint.
$api_call = "https://site.api.espn.com/apis/site/v2/sports/golf/leaderboard";

// ///////////// START API CALL ///////////////////
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);
$result = curl_exec($curl);
$err    = curl_error($curl);
curl_close($curl);

if ($err) {
    die("cURL Error: " . $err);
}

$data = json_decode($result, true);
if (!$data) {
    die("Error decoding JSON data.");
}

// Find the most recent tournament by looking for an event with a competition flagged as recent.
$tournament = null;
$competition = null;
if (isset($data['events']) && is_array($data['events'])) {
    foreach ($data['events'] as $event) {
        if (isset($event['competitions']) && is_array($event['competitions'])) {
            foreach ($event['competitions'] as $comp) {
                if (isset($comp['recent']) && $comp['recent'] === true) {
                    $tournament = $event;
                    $competition = $comp;
                    break 2; // Exit both loops.
                }
            }
        }
    }
}

if (!$tournament || !$competition) {
    die("No recent tournament found.");
}

// Prepare the tournament name for database insertion.
$tournyName = mysqli_real_escape_string($con, $tournament['name']);

// Sort the competitors by sortOrder (ascending).
$competitors = $competition['competitors'];
usort($competitors, function ($a, $b) {
    return $a['sortOrder'] - $b['sortOrder'];
});

// Determine the top 16 competitors. If the 16th competitor’s sortOrder is tied with others, include them.
$topCompetitors = [];
if (count($competitors) >= 16) {
    $thresholdSortOrder = $competitors[15]['sortOrder'];
    foreach ($competitors as $comp) {
        if ($comp['sortOrder'] <= $thresholdSortOrder) {
            $topCompetitors[] = $comp;
        }
    }
} else {
    $topCompetitors = $competitors;
}

// Prepare an array to hold data for the debug table.
$updatedData = [];
$i = 1;
foreach ($topCompetitors as $comp) {

    // Extract teeTime.
    $teeTime = isset($comp['status']['teeTime']) ? $comp['status']['teeTime'] : "";
    $teeTimeEsc = mysqli_real_escape_string($con, $teeTime);

    // Format the athlete's name as FirstInitial.LastName.
    $fullName = isset($comp['athlete']['displayName']) ? $comp['athlete']['displayName'] : "";
    $nameParts = explode(" ", trim($fullName));
    if (count($nameParts) > 1) {
        $firstInitial = strtoupper(substr($nameParts[0], 0, 1));
        $lastName = ucfirst(end($nameParts));
        $formattedName = $firstInitial . "." . $lastName;
    } else {
        $formattedName = $fullName;
    }
    $formattedNameEsc = mysqli_real_escape_string($con, $formattedName);

    // Determine if the golfer is currently playing (if status type state equals "in").
    $golfer_playing = (isset($comp['status']['type']['state']) && $comp['status']['type']['state'] === "in") ? "Yes" : "No";
    $golfer_playingEsc = mysqli_real_escape_string($con, $golfer_playing);

    // --- Get the total score instead of today's score ---
    $totalScore = "";
    if (isset($comp['statistics']) && is_array($comp['statistics'])) {
        foreach ($comp['statistics'] as $stat) {
            if (isset($stat['name']) && $stat['name'] === "scoreToPar") {
                $totalScore = $stat['displayValue'];
                break;
            }
        }
    }
    // Fallback in case the statistic isn't found.
    if ($totalScore === "") {
        $totalScore = isset($comp['score']['displayValue']) ? $comp['score']['displayValue'] : "";
    }
    $totalScoreEsc = mysqli_real_escape_string($con, $totalScore);

    // Extract current position.
    if (isset($comp['status']['position']['displayName'])) {
        $currentPosition = $comp['status']['position']['displayName'];
    } elseif (isset($comp['linescores'][0]['currentPosition'])) {
        $currentPosition = $comp['linescores'][0]['currentPosition'];
    } else {
        $currentPosition = "";
    }
    // Convert to numeric value by removing non-digits. Example: "T2" becomes "2".
    $currentPositionNumeric = preg_replace('/\D/', '', $currentPosition);
    if ($currentPositionNumeric === "") {
        $currentPositionNumeric = 0;
    }
    $currentPositionEsc = mysqli_real_escape_string($con, $currentPositionNumeric);

    // Extract status description.
    $status_name = isset($comp['status']['type']['description']) ? $comp['status']['type']['description'] : "";
    $status_nameEsc = mysqli_real_escape_string($con, $status_name);

    // Extract the 'thru' value.
    $thru = isset($comp['status']['thru']) ? $comp['status']['thru'] : "";
    $thruEsc = mysqli_real_escape_string($con, $thru);

    // Determine the finished status based on API's state.
    // "pre" becomes "scheduled", "in" becomes "golfing", and "post" becomes "finished".
    $finished = "";
    if (isset($comp['status']['type']['state'])) {
        $state = $comp['status']['type']['state'];
        if ($state === "pre") {
            $finished = "scheduled";
        } elseif ($state === "in") {
            $finished = "golfing";
        } elseif ($state === "post") {
            $finished = "finished";
        }
    }
    $finishedEsc = mysqli_real_escape_string($con, $finished);

    // Athlete ID.
    $athleteId = isset($comp['athlete']['id']) ? $comp['athlete']['id'] : "";

    // Build the update query including the new 'thru', 'finished', and updated score columns.
    $query = "UPDATE scoreBoard 
              SET teeTime='$teeTimeEsc', 
                  DN='$formattedNameEsc', 
                  GP='$golfer_playingEsc', 
                  score='$totalScoreEsc', 
                  TN='$tournyName', 
                  POS='$currentPositionEsc', 
                  STATUS='$status_nameEsc',
                  thru='$thruEsc',
                  finished='$finishedEsc'
              WHERE count=$i AND league='golf'";

    // Execute the query.
    $resultUpdate = mysqli_query($con, $query);
    if (!$resultUpdate) {
        $queryMessage = "Error updating record for count $i: " . mysqli_error($con);
    } else {
        $queryMessage = "Record updated for count $i.";
    }

    // Save the details and query result for the debug table.
    $updatedData[] = [
        'count'          => $i,
        'sortOrder'      => isset($comp['sortOrder']) ? $comp['sortOrder'] : "",
        'athleteId'      => $athleteId,
        'golferName'     => $formattedName,
        'score'          => $totalScore,
        'teeTime'        => $teeTime,
        'thru'           => $thru,
        'finished'       => $finished,
        'golfer_playing' => $golfer_playing,
        'POS_original'   => $currentPosition, // For debugging: original value from API.
        'POS_numeric'    => $currentPositionNumeric,
        'queryMessage'   => $queryMessage,
        'TN'             => $tournament['name']   
    ];
    $i++;
}

mysqli_close($con);

// --- Trigger additional API call ---
$update_api_url = SERVER_URL . "Scrolling/App_Output/API/PGA/wrapperPGA.php";
$curl2 = curl_init();
curl_setopt_array($curl2, [
    CURLOPT_URL => $update_api_url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
]);
$updateApiResult = curl_exec($curl2);
$updateApiError = curl_error($curl2);
curl_close($curl2);

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>ScoreBoard Update Debug</title>
    <style>
        body { font-family: Arial, sans-serif; }
        table { border-collapse: collapse; width: 90%; margin: 20px auto; }
        th, td { border: 1px solid #333; padding: 8px; text-align: center; }
        th { background-color: #f0f0f0; }
    </style>
</head>
<body>
    <h1 style="text-align:center;">Tournament: <?php echo htmlspecialchars($tournament['name']); ?></h1>
    <table>
        <thead>
            <tr>
                <th>Count</th>
                <th>Sort Order</th>
                <th>Athlete ID</th>
                <th>Golfer Name</th>
                <th>Score</th>
                <th>TeeTime</th>
                <th>Thru</th>
                <th>Finished</th>
                <th>Currently Golfing?</th>
                <th>POS (API)</th>
                <th>POS (Numeric)</th>
                <th>Query Message</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($updatedData as $row): ?>
            <tr>
                <td><?php echo htmlspecialchars($row['count']); ?></td>
                <td><?php echo htmlspecialchars($row['sortOrder']); ?></td>
                <td><?php echo htmlspecialchars($row['athleteId']); ?></td>
                <td><?php echo htmlspecialchars($row['golferName']); ?></td>
                <td><?php echo htmlspecialchars($row['score']); ?></td>
                <td><?php echo htmlspecialchars($row['teeTime']); ?></td>
                <td><?php echo htmlspecialchars($row['thru']); ?></td>
                <td><?php echo htmlspecialchars($row['finished']); ?></td>
                <td><?php echo htmlspecialchars($row['golfer_playing']); ?></td>
                <td><?php echo htmlspecialchars($row['POS_original']); ?></td>
                <td><?php echo htmlspecialchars($row['POS_numeric']); ?></td>
                <td><?php echo htmlspecialchars($row['queryMessage']); ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</body>
</html>
