<?php
// Robust "last scorer" extractor with fallback for FG/XP to boxscore kicking
$eventId = '401772919';
$url = "https://site.api.espn.com/apis/site/v2/sports/football/nfl/summary?event={$eventId}";

$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 12,
    CURLOPT_USERAGENT => 'Mozilla/5.0',
]);
$raw = curl_exec($ch);
$err = curl_error($ch);
$code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($err || $code !== 200 || !$raw) {
    die("Error fetching summary ($code): " . ($err ?: 'no response'));
}

$data = json_decode($raw, true);
if (!$data) die("Invalid JSON from ESPN.");

// Helper: get scoringPlays from either top-level or gamecast.*
function getScoringPlays(array $d): array {
    if (isset($d['scoringPlays']) && is_array($d['scoringPlays']) && $d['scoringPlays']) return $d['scoringPlays'];
    if (isset($d['gamecast']['scoringPlays']) && is_array($d['gamecast']['scoringPlays']) && $d['gamecast']['scoringPlays']) return $d['gamecast']['scoringPlays'];
    return [];
}

// Helper: find a kicker in boxscore by (teamId, displayName)
function findKickerFromBoxscore(array $d, ?string $teamId, ?string $displayName): ?array {
    if (!isset($d['boxscore']['players']) || !is_array($d['boxscore']['players'])) return null;

    foreach ($d['boxscore']['players'] as $teamBlock) {
        // Filter by team if we have it
        $teamMatches = !$teamId || (isset($teamBlock['team']['id']) && (string)$teamBlock['team']['id'] === (string)$teamId);
        if (!$teamMatches) continue;

        if (!isset($teamBlock['statistics']) || !is_array($teamBlock['statistics'])) continue;

        foreach ($teamBlock['statistics'] as $statGroup) {
            if (($statGroup['name'] ?? '') !== 'kicking') continue;
            if (!isset($statGroup['athletes']) || !is_array($statGroup['athletes'])) continue;

            foreach ($statGroup['athletes'] as $athRow) {
                $ath = $athRow['athlete'] ?? null;
                if (!$ath) continue;
                $name = $ath['displayName'] ?? '';
                if (!$displayName) {
                    // If we couldn't parse a name, just return the first kicker for that team
                    return [
                        'id' => $ath['id'] ?? null,
                        'jersey' => $ath['jersey'] ?? null,
                        'name' => $name,
                    ];
                }
                // Match on full name (case-insensitive). You could also do fuzzy matching if needed.
                if (mb_strtolower($name) === mb_strtolower($displayName)) {
                    return [
                        'id' => $ath['id'] ?? null,
                        'jersey' => $ath['jersey'] ?? null,
                        'name' => $name,
                    ];
                }
                // Loose contains match fallback (handles suffixes like Jr., Sr.)
                if (mb_strtolower($name) === mb_strtolower(trim(preg_replace('/\s+Jr\.?|\s+Sr\.?/i', '', $displayName)))) {
                    return [
                        'id' => $ath['id'] ?? null,
                        'jersey' => $ath['jersey'] ?? null,
                        'name' => $name,
                    ];
                }
            }
        }
    }
    return null;
}

// Helper: for FG/XP text, try to parse "Kicker Name ... Field Goal" or "Extra Point"
function parseKickerNameFromText(?string $text): ?string {
    if (!$text) return null;

    // Examples:
    // "Will Reichard 51 Yd Field Goal"
    // "Will Reichard Made Extra Point"
    // "Will Reichard kicks extra point is good"
    $patterns = [
        '/^([A-Za-z\.\'\- ]+?)\s+\d+\s*(?:Yd|yd|YDS)?\s*Field Goal/i',
        '/^([A-Za-z\.\'\- ]+?)\s+Made\s+Extra\s+Point/i',
        '/^([A-Za-z\.\'\- ]+?)\s+kicks? (?:an )?extra point/i',
        '/^([A-Za-z\.\'\- ]+?)\s+Extra Point/i',
    ];
    foreach ($patterns as $p) {
        if (preg_match($p, $text, $m)) {
            return trim($m[1]);
        }
    }
    return null;
}

$scoringPlays = getScoringPlays($data);
if (!$scoringPlays) die("No scoring plays found for event {$eventId}.");

$last = end($scoringPlays);

// Default values
$teamName   = $last['team']['displayName'] ?? null;
$teamId     = $last['team']['id'] ?? null;
$whenPeriod = $last['period']['number'] ?? null;
$whenClock  = $last['clock']['displayValue'] ?? null;
$typeText   = $last['type']['text'] ?? null;
$desc       = $last['text'] ?? null;

// Primary: if athletes[] is there, use it
$playerId = $last['athletes'][0]['id']      ?? null;
$jersey   = $last['athletes'][0]['jersey']  ?? null;
$fullname = $last['athletes'][0]['fullName']?? null;

// Fallback: FG/XP often lack athletes -> parse name from text & map via boxscore/kicking
$lowerType = mb_strtolower($typeText ?? '');
if ((!$playerId || !$jersey) && (
        strpos($lowerType, 'field goal') !== false ||
        strpos($lowerType, 'extra point') !== false
    )) {
    $parsedName = parseKickerNameFromText($desc);
    $kick = findKickerFromBoxscore($data, $teamId, $parsedName);
    if ($kick) {
        $playerId = $playerId ?: ($kick['id'] ?? null);
        $jersey   = $jersey   ?: ($kick['jersey'] ?? null);
        $fullname = $fullname ?: ($kick['name'] ?? null);
    }
}

// As a last resort (very rare), if still empty you could fetch the CDN play-by-play and match the last scoring play’s players.
// (Left out to keep this single-call, but easy to add if you want.)

echo "Latest scoring play:\n";
echo " Team: " . ($teamName ?? 'N/A') . "\n";
echo " When: Q" . ($whenPeriod ?? '?') . " " . ($whenClock ?? '') . "\n";
echo " Type: " . ($typeText ?? 'N/A') . "\n";
echo " Desc: " . ($desc ?? 'N/A') . "\n";
echo " Player: " . ($fullname ?? 'N/A') .
     " (ID=" . ($playerId ?? 'N/A') . ", Jersey=" . ($jersey ?? 'N/A') . ")\n";
