<?php
/**
 * ESPN Wild Card / Playoff Standings
 * Working version based on actual API structure
 */

// -------------------- CONFIG --------------------
$DEBUG_HTML = true;
$ENABLE_DB = false;
$FIRST_RUN = false;

// Test different endpoints to see what works
$LEAGUES = [
    'MLB' => [
        'url' => 'https://site.api.espn.com/apis/v2/sports/baseball/mlb/standings',
        'label' => 'MLB Standings'
    ],
    'NFL' => [
        'url' => 'https://site.api.espn.com/apis/v2/sports/football/nfl/standings', 
        'label' => 'NFL Standings'
    ],
    'NBA' => [
        'url' => 'https://site.api.espn.com/apis/v2/sports/basketball/nba/standings',
        'label' => 'NBA Standings'
    ],
    'NHL' => [
        'url' => 'https://site.api.espn.com/apis/v2/sports/hockey/nhl/standings',
        'label' => 'NHL Standings'
    ]
];

// -------------------- FUNCTIONS --------------------

function fetchESPNData($url, $league) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HTTPHEADER => [
            'Accept: application/json',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    echo "<!-- $league API Response Code: $httpCode -->\n";
    
    if ($error) {
        return ['error' => "CURL Error: " . $error];
    }
    
    if ($httpCode !== 200) {
        return ['error' => "HTTP $httpCode"];
    }
    
    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['error' => "JSON Error: " . json_last_error_msg()];
    }
    
    return $data;
}

function extractTeamFromEntry($entry) {
    $team = $entry['team'] ?? [];
    $stats = $entry['stats'] ?? [];
    
    // Build stats map by name for easy lookup
    $statsMap = [];
    foreach ($stats as $stat) {
        if (isset($stat['name'])) {
            $statsMap[$stat['name']] = $stat;
        }
    }
    
    // Extract the stats we need
    $data = [
        'team_id' => $team['id'] ?? '',
        'abbreviation' => $team['abbreviation'] ?? '',
        'display_name' => $team['displayName'] ?? '',
        'location' => $team['location'] ?? '',
        'name' => $team['name'] ?? '',
        'logo' => '',
        'wins' => $statsMap['wins']['value'] ?? 0,
        'losses' => $statsMap['losses']['value'] ?? 0,
        'win_pct' => $statsMap['winPercent']['value'] ?? 0,
        'games_back' => isset($statsMap['gamesBehind']) ? $statsMap['gamesBehind']['displayValue'] : '-',
        'points' => $statsMap['points']['value'] ?? null,
        'streak' => $statsMap['streak']['displayValue'] ?? '',
        'playoff_seed' => $statsMap['playoffSeed']['value'] ?? null,
        'clincher' => $statsMap['clincher']['displayValue'] ?? '',
        'ot_losses' => $statsMap['OTLosses']['value'] ?? 0,
        'division_gb' => isset($statsMap['divisionGamesBehind']) ? $statsMap['divisionGamesBehind']['displayValue'] : null
    ];
    
    // Get logo
    if (isset($team['logos']) && is_array($team['logos']) && count($team['logos']) > 0) {
        $data['logo'] = $team['logos'][0]['href'] ?? '';
    }
    
    // For wild card specifically
    $data['wild_card_gb'] = null;
    if (isset($statsMap['wildCardGamesBehind'])) {
        $data['wild_card_gb'] = $statsMap['wildCardGamesBehind']['displayValue'];
    }
    
    return $data;
}

function renderStandingsTable($title, $teams, $league) {
    // Determine which columns to show based on league
    $showPoints = ($league === 'NHL');
    $showOTL = ($league === 'NHL');
    
    $html = '<div class="standings-section">';
    $html .= '<h3>' . htmlspecialchars($title) . '</h3>';
    $html .= '<div class="table-wrapper">';
    $html .= '<table>';
    $html .= '<thead><tr>';
    $html .= '<th class="rank">#</th>';
    $html .= '<th class="team">Team</th>';
    $html .= '<th>W</th>';
    $html .= '<th>L</th>';
    if ($showOTL) $html .= '<th>OTL</th>';
    if ($showPoints) $html .= '<th>PTS</th>';
    $html .= '<th>PCT</th>';
    $html .= '<th>GB</th>';
    $html .= '<th>STRK</th>';
    $html .= '</tr></thead>';
    $html .= '<tbody>';
    
    $rank = 1;
    foreach ($teams as $team) {
        $clincher = $team['clincher'];
        $clincherClass = '';
        if ($clincher === 'x' || $clincher === 'y' || $clincher === 'z') {
            $clincherClass = 'clinched';
        } elseif ($clincher === 'e') {
            $clincherClass = 'eliminated';
        }
        
        $html .= '<tr class="' . $clincherClass . '">';
        $html .= '<td class="rank">' . $rank++ . '</td>';
        $html .= '<td class="team">';
        if ($team['logo']) {
            $html .= '<img src="' . htmlspecialchars($team['logo']) . '" alt="">';
        }
        $html .= '<div class="team-info">';
        $html .= '<span class="team-name">' . htmlspecialchars($team['display_name']) . '</span>';
        $html .= '<span class="team-abbr">(' . htmlspecialchars($team['abbreviation']) . ')</span>';
        if ($clincher) {
            $html .= ' <span class="clincher">' . htmlspecialchars($clincher) . '</span>';
        }
        $html .= '</div>';
        $html .= '</td>';
        $html .= '<td>' . $team['wins'] . '</td>';
        $html .= '<td>' . $team['losses'] . '</td>';
        if ($showOTL) $html .= '<td>' . ($team['ot_losses'] ?? 0) . '</td>';
        if ($showPoints) $html .= '<td class="points">' . ($team['points'] ?? '-') . '</td>';
        $html .= '<td>' . number_format($team['win_pct'], 3) . '</td>';
        $html .= '<td>' . htmlspecialchars($team['games_back'] == '0' ? '-' : $team['games_back']) . '</td>';
        $html .= '<td class="streak">' . htmlspecialchars($team['streak']) . '</td>';
        $html .= '</tr>';
    }
    
    $html .= '</tbody></table>';
    $html .= '</div></div>';
    
    return $html;
}

// -------------------- MAIN --------------------

$allOutput = '';
$apiStatus = [];

foreach ($LEAGUES as $league => $config) {
    $data = fetchESPNData($config['url'], $league);
    
    if (isset($data['error'])) {
        $apiStatus[$league] = 'Error: ' . $data['error'];
        $allOutput .= '<div class="error-section">';
        $allOutput .= '<h2>' . $config['label'] . '</h2>';
        $allOutput .= '<p class="error">API Error: ' . htmlspecialchars($data['error']) . '</p>';
        $allOutput .= '</div>';
        continue;
    }
    
    $apiStatus[$league] = 'Success';
    $allOutput .= '<div class="league-container">';
    $allOutput .= '<h2>' . htmlspecialchars($config['label']) . '</h2>';
    
    $allTeams = [];
    
    // Check if we have children (divisions/conferences)
    if (isset($data['children']) && is_array($data['children'])) {
        foreach ($data['children'] as $group) {
            $groupName = $group['name'] ?? $group['displayName'] ?? 'Group';
            
            // Check for standings at this level
            if (isset($group['standings']['entries'])) {
                $teams = [];
                foreach ($group['standings']['entries'] as $entry) {
                    $teamData = extractTeamFromEntry($entry);
                    if ($teamData['team_id']) {
                        $teams[] = $teamData;
                    }
                }
                
                // Sort by playoff seed if available, otherwise by win percentage
                usort($teams, function($a, $b) {
                    if ($a['playoff_seed'] !== null && $b['playoff_seed'] !== null) {
                        return $a['playoff_seed'] - $b['playoff_seed'];
                    }
                    return ($b['win_pct'] - $a['win_pct']) * 1000;
                });
                
                if (!empty($teams)) {
                    $allOutput .= renderStandingsTable($groupName, $teams, $league);
                    $allTeams = array_merge($allTeams, $teams);
                }
            }
            
            // Check for nested children
            if (isset($group['children']) && is_array($group['children'])) {
                foreach ($group['children'] as $subgroup) {
                    if (isset($subgroup['standings']['entries'])) {
                        $subgroupName = $subgroup['name'] ?? 'Division';
                        $teams = [];
                        foreach ($subgroup['standings']['entries'] as $entry) {
                            $teamData = extractTeamFromEntry($entry);
                            if ($teamData['team_id']) {
                                $teams[] = $teamData;
                            }
                        }
                        
                        usort($teams, function($a, $b) {
                            if ($a['playoff_seed'] !== null && $b['playoff_seed'] !== null) {
                                return $a['playoff_seed'] - $b['playoff_seed'];
                            }
                            return ($b['win_pct'] - $a['win_pct']) * 1000;
                        });
                        
                        if (!empty($teams)) {
                            $allOutput .= renderStandingsTable($groupName . ' - ' . $subgroupName, $teams, $league);
                            $allTeams = array_merge($allTeams, $teams);
                        }
                    }
                }
            }
        }
    }
    
    // If no teams found yet, check root level
    if (empty($allTeams) && isset($data['standings']['entries'])) {
        $teams = [];
        foreach ($data['standings']['entries'] as $entry) {
            $teamData = extractTeamFromEntry($entry);
            if ($teamData['team_id']) {
                $teams[] = $teamData;
            }
        }
        
        if (!empty($teams)) {
            $allOutput .= renderStandingsTable('League Standings', $teams, $league);
        }
    }
    
    if (empty($allTeams)) {
        $allOutput .= '<p class="no-data">No standings data found in API response.</p>';
    }
    
    $allOutput .= '</div>';
}

// -------------------- OUTPUT --------------------

if ($DEBUG_HTML) {
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ESPN Standings</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: -apple-system, system-ui, sans-serif;
            background: #1a1a2e;
            color: #fff;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        h1 {
            text-align: center;
            margin-bottom: 30px;
            font-size: 2.5em;
        }
        
        .api-status {
            background: rgba(255,255,255,0.1);
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 30px;
        }
        
        .api-status h3 {
            margin-bottom: 10px;
        }
        
        .api-status .status-item {
            display: inline-block;
            margin-right: 20px;
            padding: 5px 10px;
            border-radius: 4px;
            background: rgba(255,255,255,0.1);
        }
        
        .league-container {
            margin-bottom: 40px;
        }
        
        .league-container h2 {
            color: #4fc3f7;
            margin-bottom: 20px;
            border-bottom: 2px solid #4fc3f7;
            padding-bottom: 10px;
        }
        
        .standings-section {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            margin-bottom: 20px;
            overflow: hidden;
        }
        
        .standings-section h3 {
            background: rgba(255,255,255,0.1);
            padding: 15px;
            margin: 0;
        }
        
        .table-wrapper {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 10px;
            text-align: left;
        }
        
        th {
            background: rgba(0,0,0,0.3);
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85em;
            position: sticky;
            top: 0;
        }
        
        tbody tr {
            border-top: 1px solid rgba(255,255,255,0.1);
        }
        
        tbody tr:hover {
            background: rgba(255,255,255,0.05);
        }
        
        .rank {
            width: 40px;
            text-align: center;
        }
        
        .team {
            display: flex;
            align-items: center;
            gap: 10px;
            min-width: 200px;
        }
        
        .team img {
            width: 30px;
            height: 30px;
        }
        
        .team-info {
            display: flex;
            flex-direction: column;
        }
        
        .team-name {
            font-weight: 500;
        }
        
        .team-abbr {
            font-size: 0.85em;
            opacity: 0.7;
        }
        
        .clincher {
            display: inline-block;
            background: #4caf50;
            color: #000;
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 0.75em;
            font-weight: bold;
            margin-left: 5px;
        }
        
        .clinched {
            background: rgba(76, 175, 80, 0.1);
        }
        
        .eliminated {
            opacity: 0.5;
        }
        
        .eliminated .clincher {
            background: #f44336;
            color: #fff;
        }
        
        .streak {
            font-weight: 500;
        }
        
        .points {
            font-weight: bold;
            color: #4fc3f7;
        }
        
        .error-section {
            background: rgba(244, 67, 54, 0.1);
            border: 1px solid #f44336;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .error {
            color: #f44336;
        }
        
        .no-data {
            padding: 20px;
            text-align: center;
            opacity: 0.7;
        }
        
        @media (max-width: 768px) {
            body { padding: 10px; }
            th, td { padding: 5px; font-size: 0.85em; }
            .team img { width: 24px; height: 24px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📊 ESPN Standings</h1>
        
        <div class="api-status">
            <h3>API Status:</h3>
            <?php foreach ($apiStatus as $league => $status): ?>
                <span class="status-item"><?php echo "$league: $status"; ?></span>
            <?php endforeach; ?>
        </div>
        
        <?php echo $allOutput; ?>
        
        <div style="text-align: center; margin-top: 40px; opacity: 0.7;">
            Updated: <?php echo date('F j, Y g:i a'); ?>
        </div>
    </div>
</body>
</html>
<?php
} else {
    echo $allOutput;
}
?>