<?php
// Enable error reporting for debugging.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Create a PDO database connection.
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database Connection Error: " . $e->getMessage());
}

// --- Team Lookup Array ---
$teams = array(
    array("id" => "29", "abbreviation" => "ARI", "color" => "aa182c", "alternateColor" => "e3d4ad", "name" => "DIAMONDBACKS"),
    array("id" => "15", "abbreviation" => "ATL", "color" => "0c2340", "alternateColor" => "ba0c2f", "name" => "BRAVES"),
    array("id" => "1",  "abbreviation" => "BAL", "color" => "df4601", "alternateColor" => "000000", "name" => "ORIOLES"),
    array("id" => "2",  "abbreviation" => "BOS", "color" => "0d2b56", "alternateColor" => "bd3039", "name" => "RED SOX"),
    array("id" => "16", "abbreviation" => "CHC", "color" => "0e3386", "alternateColor" => "cc3433", "name" => "CUBS"),
    array("id" => "4",  "abbreviation" => "CHW", "color" => "000000", "alternateColor" => "c4ced4", "name" => "WHITE SOX"),
    array("id" => "17", "abbreviation" => "CIN", "color" => "c6011f", "alternateColor" => "ffffff", "name" => "REDS"),
    array("id" => "5",  "abbreviation" => "CLE", "color" => "002b5c", "alternateColor" => "e31937", "name" => "GUARDIANS"),
    array("id" => "27", "abbreviation" => "COL", "color" => "33006f", "alternateColor" => "c4ced4", "name" => "ROCKIES"),
    array("id" => "6",  "abbreviation" => "DET", "color" => "0a2240", "alternateColor" => "ff4713", "name" => "TIGERS"),
    array("id" => "18", "abbreviation" => "HOU", "color" => "002d62", "alternateColor" => "eb6e1f", "name" => "ASTROS"),
    array("id" => "7",  "abbreviation" => "KC",  "color" => "004687", "alternateColor" => "7ab2dd", "name" => "ROYALS"),
    array("id" => "3",  "abbreviation" => "LAA", "color" => "ba0021", "alternateColor" => "c4ced4", "name" => "ANGELS"),
    array("id" => "19", "abbreviation" => "LAD", "color" => "005a9c", "alternateColor" => "ffffff", "name" => "DODGERS"),
    array("id" => "28", "abbreviation" => "MIA", "color" => "00a3e0", "alternateColor" => "ef3340", "name" => "MARLINS"),
    array("id" => "8",  "abbreviation" => "MIL", "color" => "13294b", "alternateColor" => "ffc62f", "name" => "BREWERS"),
    array("id" => "9",  "abbreviation" => "MIN", "color" => "031f40", "alternateColor" => "e20e32", "name" => "TWINS"),
    array("id" => "21", "abbreviation" => "NYM", "color" => "002d72", "alternateColor" => "ff5910", "name" => "METS"),
    array("id" => "10", "abbreviation" => "NYY", "color" => "132448", "alternateColor" => "c4ced4", "name" => "YANKEES"),
    array("id" => "22", "abbreviation" => "PHI", "color" => "e81828", "alternateColor" => "002d72", "name" => "PHILLIES"),
    array("id" => "23", "abbreviation" => "PIT", "color" => "000000", "alternateColor" => "fdb827", "name" => "PIRATES"),
    array("id" => "25", "abbreviation" => "SD",  "color" => "2f241d", "alternateColor" => "ffc425", "name" => "PADRES"),
    array("id" => "26", "abbreviation" => "SF",  "color" => "000000", "alternateColor" => "fa4616", "name" => "GIANTS"),
    array("id" => "12", "abbreviation" => "SEA", "color" => "005c5c", "alternateColor" => "0c2c56", "name" => "MARINERS"),
    array("id" => "24", "abbreviation" => "STL", "color" => "be0a14", "alternateColor" => "0c2340", "name" => "CARDINALS"),
    array("id" => "30", "abbreviation" => "TB",  "color" => "092c5c", "alternateColor" => "8fbce6", "name" => "RAYS"),
    array("id" => "13", "abbreviation" => "TEX", "color" => "003278", "alternateColor" => "c0111f", "name" => "RANGERS"),
    array("id" => "14", "abbreviation" => "TOR", "color" => "134a8e", "alternateColor" => "1d2d5c", "name" => "BLUE JAYS"),
    array("id" => "20", "abbreviation" => "WSH", "color" => "ab0003", "alternateColor" => "14225a", "name" => "NATIONALS"),
    array("id" => "11", "abbreviation" => "ATH", "color" => "003831", "alternateColor" => "ffffff", "name" => "ATHLETICS")
);

echo "<html><head><title>MLB Team Stats Update</title></head><body>";

// --- Desired Stats Mapping ---
// For each desired stat type, list possible abbreviations and the sort order
$desiredStats = array(
    'battingAverage' => array(
        'displayName' => 'Batting Average',
        'possibleAbbreviations' => array('AVG', 'battingAverage', 'BA', 'BATTING_AVERAGE', 'Batting Avg', 'avg'),
        'sort' => 'desc'
    ),
    'homeRuns' => array(
        'displayName' => 'Home Runs',
        'possibleAbbreviations' => array('HR', 'homeRuns'),
        'sort' => 'desc'
    ),
    'stolenBases' => array(
        'displayName' => 'Stolen Bases',
        'possibleAbbreviations' => array('SB', 'stolenBases'),
        'sort' => 'desc'
    ),
    'strikeouts' => array(
        'displayName' => 'Strikeouts',
        'possibleAbbreviations' => array('SO', 'K', 'strikeouts'),
        'sort' => 'asc'  // Lower is better for strikeouts
    )
);

// Map to database values
$dbTypeNames = array(
    'battingAverage' => 'Batting Average',
    'homeRuns' => 'Home Runs',
    'stolenBases' => 'Stolen Bases',
    'strikeouts' => 'Strikeouts'
);

// Category type abbreviations for the database
$categoryTypeAbbrevs = array(
    'battingAverage' => 'AVG',
    'homeRuns' => 'HR',
    'stolenBases' => 'SB',
    'strikeouts' => 'K'
);

// Initialize leagueStats array
$leagueStats = array();
foreach ($desiredStats as $key => $config) {
    $leagueStats[$key] = array();
}

// Debug: Track all stat names found
$allStatNames = array();

/**
 * Helper function to perform a cURL request and return decoded JSON.
 */
function curlFetch($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return false;
    }
    
    $decoded = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return false;
    }
    
    return $decoded;
}

// Loop through each team and fetch its statistics
foreach ($teams as $team) {
    $teamId = $team['id'];
    $teamAbbr = $team['abbreviation'];
    $teamColor = $team['color'];
    $alternateColor = $team['alternateColor'];
    $teamName = $team['name'];
    
    // Extract Red, Green, Blue components from the main hex color
    $red = hexdec(substr($teamColor, 0, 2));
    $green = hexdec(substr($teamColor, 2, 2));
    $blue = hexdec(substr($teamColor, 4, 2));
    
    // Extract Red, Green, Blue components from the alternate hex color
    $redAlt = hexdec(substr($alternateColor, 0, 2));
    $greenAlt = hexdec(substr($alternateColor, 2, 2));
    $blueAlt = hexdec(substr($alternateColor, 4, 2));
    
    // Build the team's statistics URL
    $statsUrl = "http://sports.core.api.espn.com/v2/sports/baseball/leagues/mlb/seasons/2025/types/2/teams/{$teamId}/statistics/0?lang=en&region=us";
    $statsData = curlFetch($statsUrl);
    if (!$statsData) {
        echo "⚠️ Failed to fetch data for {$teamAbbr}<br>\n";
        continue;
    }
    
    if (!isset($statsData['splits']['categories']) || !is_array($statsData['splits']['categories'])) {
        echo "⚠️ No categories found for {$teamAbbr}<br>\n";
        continue;
    }
    
    // Debug: Only for first team, let's see what stats are available
    if ($teamId == "29") { // ARI
        echo "<h3>Debug: Available stats for {$teamAbbr}:</h3>";
        foreach ($statsData['splits']['categories'] as $categoryIndex => $category) {
            if (isset($category['stats']) && is_array($category['stats'])) {
                echo "<strong>Category {$categoryIndex}:</strong> ";
                foreach ($category['stats'] as $stat) {
                    $statName = isset($stat['name']) ? $stat['name'] : (isset($stat['abbreviation']) ? $stat['abbreviation'] : 'Unknown');
                    echo $statName . " ";
                }
                echo "<br>";
            }
        }
        echo "<br>";
    }
    
    // Loop over each category and its stats
    foreach ($statsData['splits']['categories'] as $category) {
        if (!isset($category['stats']) || !is_array($category['stats'])) {
            continue;
        }
        foreach ($category['stats'] as $stat) {
            $statName = isset($stat['name']) ? $stat['name'] : (isset($stat['abbreviation']) ? $stat['abbreviation'] : null);
            if (!$statName) {
                continue;
            }
            
            // Track all stat names for debugging
            if (!in_array($statName, $allStatNames)) {
                $allStatNames[] = $statName;
            }
            
            // Check if this stat matches one of our desired types
            foreach ($desiredStats as $statKey => $config) {
                if (in_array($statName, $config['possibleAbbreviations'])) {
                    // Get the numeric value
                    $valueNumeric = null;
                    if (isset($stat['value'])) {
                        $valueNumeric = floatval($stat['value']);
                    }
                    if ($valueNumeric === null) {
                        continue;
                    }
                    
                    $displayValue = isset($stat['displayValue']) ? $stat['displayValue'] : "N/A";
                    
                    $leagueStats[$statKey][] = array(
                        'teamAbbr'     => $teamAbbr,
                        'teamColor'    => $teamColor,
                        'teamName'     => $teamName,
                        'value'        => $valueNumeric,
                        'displayValue' => $displayValue,
                        'red'          => $red,
                        'green'        => $green,
                        'blue'         => $blue,
                        'red_alt'      => $redAlt,
                        'green_alt'    => $greenAlt,
                        'blue_alt'     => $blueAlt
                    );
                    
                    // Debug: Show what we found
                    echo "✅ Found {$statName} for {$teamAbbr}: {$displayValue}<br>\n";
                }
            }
        }
    }
}

// Debug: Show all stat names found
echo "<h3>Debug: All stat names found across all teams:</h3>";
sort($allStatNames);
foreach ($allStatNames as $statName) {
    echo $statName . "<br>";
}
echo "<br>";

// Debug: Show what we collected for each stat type
echo "<h3>Debug: Data collected per stat type:</h3>";
foreach ($leagueStats as $statKey => $entries) {
    echo "<strong>{$statKey}:</strong> " . count($entries) . " entries<br>";
}

// Helper function to sort stat entries
function sortEntries(&$entries, $order = 'desc') {
    usort($entries, function($a, $b) use ($order) {
        if ($order === 'desc') {
            return $b['value'] <=> $a['value'];
        } else {
            return $a['value'] <=> $b['value'];
        }
    });
}

// For each desired stat type, sort the entries and select the top 5
$top5Stats = array();
foreach ($leagueStats as $statType => $entries) {
    $order = $desiredStats[$statType]['sort'];
    sortEntries($entries, $order);
    // Slice to top 5; if fewer than 5, you'll see only those available
    $top5Stats[$statType] = array_slice($entries, 0, 5);
}

echo "<h2>Database Update Results:</h2>";

// Update the database rows
foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $categoryAbbrev = isset($categoryTypeAbbrevs[$statType]) ? $categoryTypeAbbrevs[$statType] : $statType;
    $rank = 0;
    foreach ($entries as $entry) {
        $rank++;
        
        // Update using the correct table name and columns - removed category_type
        $updateSQL = "UPDATE team_stats 
                      SET Abbrev = :abbrev, 
                          color = :color, 
                          value = :value, 
                          Red = :red, 
                          Green = :green, 
                          Blue = :blue, 
                          Red_ALT = :red_alt,
                          Green_ALT = :green_alt,
                          Blue_ALT = :blue_alt,
                          current_team = :current_team,
                          Type_Abbrev = :type_abbrev
                      WHERE rank = :rank 
                        AND type = :type 
                        AND league = 'MLB'";
        
        try {
            $stmt = $db->prepare($updateSQL);
            
            $params = array(
                ':abbrev'       => $entry['teamAbbr'],
                ':color'        => $entry['teamColor'],
                ':value'        => $entry['displayValue'],
                ':red'          => $entry['red'],
                ':green'        => $entry['green'],
                ':blue'         => $entry['blue'],
                ':red_alt'      => $entry['red_alt'],
                ':green_alt'    => $entry['green_alt'],
                ':blue_alt'     => $entry['blue_alt'],
                ':current_team' => $entry['teamName'],
                ':type_abbrev'  => $categoryAbbrev,
                ':type'         => $fullType,
                ':rank'         => $rank
            );
            
            $result = $stmt->execute($params);
            
            if (!$result) {
                $errorInfo = $stmt->errorInfo();
                echo "❌ Failed: Rank $rank for $fullType - Error: " . $errorInfo[2] . "<br>\n";
            } else {
                $rowCount = $stmt->rowCount();
                if ($rowCount > 0) {
                    echo "✅ Updated: Rank $rank for $fullType - {$entry['teamAbbr']} ({$entry['displayValue']}) - $rowCount row(s) updated<br>\n";
                } else {
                    echo "⚠️ No rows updated: Rank $rank for $fullType - {$entry['teamAbbr']} (check if row exists)<br>\n";
                }
            }
            
        } catch (PDOException $e) {
            echo "❌ Exception: Rank $rank for $fullType - " . $e->getMessage() . "<br>\n";
        }
    }
}

// Output an HTML table to display the top-5 results
echo "<br><h2>Top 5 Teams per Category:</h2>";
echo "<table border='1' cellpadding='5' cellspacing='0'>";
echo "<tr>
       <th>Rank</th>
       <th>League</th>
       <th>Category</th>
       <th>Team</th>
       <th>Value</th>
       <th>Updated</th>
      </tr>";

foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $rank = 1;
    foreach ($entries as $entry) {
        // Check if this row was updated in the database
        $checkSQL = "SELECT COUNT(*) FROM team_stats 
                     WHERE rank = :rank 
                       AND type = :type 
                       AND league = 'MLB' 
                       AND Abbrev = :abbrev";
        $checkStmt = $db->prepare($checkSQL);
        $checkStmt->execute([
            ':rank' => $rank,
            ':type' => $fullType,
            ':abbrev' => $entry['teamAbbr']
        ]);
        $exists = $checkStmt->fetchColumn() > 0;
        
        $statusIcon = $exists ? '✅' : '❌';
        
        echo "<tr>
                <td>{$rank}</td>
                <td>MLB</td>
                <td>" . htmlspecialchars($fullType) . "</td>
                <td>" . htmlspecialchars($entry['teamAbbr']) . " - " . htmlspecialchars($entry['teamName']) . "</td>
                <td>" . htmlspecialchars($entry['displayValue']) . "</td>
                <td style='text-align: center;'>{$statusIcon}</td>
              </tr>";
        $rank++;
    }
}
echo "</table>";
echo "</body></html>";
?>