<?php
// Enable error reporting for debugging.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Create the database connection.
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database Connection Error: " . $e->getMessage());
}

// NBA Teams lookup array.
$nbaTeams = array(
  array("id" => "1",  "abbreviation" => "ATL",  "color" => "c8102e", "alternateColor" => "fdb927", "name" => "HAWKS"),
    array("id" => "2",  "abbreviation" => "BOS",  "color" => "008348", "alternateColor" => "ffffff", "name" => "CELTICS"),
    array("id" => "17", "abbreviation" => "BKN",  "color" => "ffffff", "alternateColor" => "ffffff", "name" => "NETS"),
    array("id" => "30", "abbreviation" => "CHA",  "color" => "008ca8", "alternateColor" => "1d1160", "name" => "HORNETS"),
    array("id" => "4",  "abbreviation" => "CHI",  "color" => "ce1141", "alternateColor" => "ffffff", "name" => "BULLS"),
    array("id" => "5",  "abbreviation" => "CLE",  "color" => "860038", "alternateColor" => "bc945c", "name" => "CAVS"),
    array("id" => "6",  "abbreviation" => "DAL",  "color" => "0064b1", "alternateColor" => "bbc4ca", "name" => "MAVS"),
    array("id" => "7",  "abbreviation" => "DEN",  "color" => "0e2240", "alternateColor" => "fec524", "name" => "NUGGETS"),
    array("id" => "8",  "abbreviation" => "DET",  "color" => "1d428a", "alternateColor" => "c8102e", "name" => "PISTONS"),
    array("id" => "9",  "abbreviation" => "GS",   "color" => "fdb927", "alternateColor" => "1d428a", "name" => "WARRIORS"),
    array("id" => "10", "abbreviation" => "HOU",  "color" => "ce1141", "alternateColor" => "ffffff", "name" => "ROCKETS"),
    array("id" => "11", "abbreviation" => "IND",  "color" => "002d62", "alternateColor" => "fdbb30", "name" => "PACERS"),
    array("id" => "12", "abbreviation" => "LAC",  "color" => "12173f", "alternateColor" => "c8102e", "name" => "CLIPPERS"),
    array("id" => "13", "abbreviation" => "LAL",  "color" => "552583", "alternateColor" => "fdb927", "name" => "LAKERS"),
    array("id" => "29", "abbreviation" => "MEM",  "color" => "5d76a9", "alternateColor" => "12173f", "name" => "GRIZZLIES"),
    array("id" => "14", "abbreviation" => "MIA",  "color" => "98002e", "alternateColor" => "ffffff", "name" => "HEAT"),
    array("id" => "15", "abbreviation" => "MIL",  "color" => "00471b", "alternateColor" => "eee1c6", "name" => "BUCKS"),
    array("id" => "16", "abbreviation" => "MIN",  "color" => "266092", "alternateColor" => "79bc43", "name" => "WOLVES"),
    array("id" => "3",  "abbreviation" => "NO",   "color" => "0a2240", "alternateColor" => "b4975a", "name" => "PELICANS"),
    array("id" => "18", "abbreviation" => "NY",   "color" => "1d428a", "alternateColor" => "f58426", "name" => "KNICKS"),
    array("id" => "25", "abbreviation" => "OKC",  "color" => "007ac1", "alternateColor" => "ef3b24", "name" => "THUNDER"),
    array("id" => "19", "abbreviation" => "ORL",  "color" => "0077c0", "alternateColor" => "c4ced4", "name" => "MAGIC"),
    array("id" => "20", "abbreviation" => "PHI",  "color" => "1d428a", "alternateColor" => "e01234", "name" => "SIXERS"),
    array("id" => "21", "abbreviation" => "PHX",  "color" => "29127a", "alternateColor" => "e56020", "name" => "SUNS"),
    array("id" => "22", "abbreviation" => "POR",  "color" => "e03a3e", "alternateColor" => "ffffff", "name" => "BLAZERS"),
    array("id" => "23", "abbreviation" => "SAC",  "color" => "5a2d81", "alternateColor" => "6a7a82", "name" => "KINGS"),
    array("id" => "24", "abbreviation" => "SA",   "color" => "c4ced4", "alternateColor" => "ffffff", "name" => "SPURS"),
    array("id" => "28", "abbreviation" => "TOR",  "color" => "d91244", "alternateColor" => "ffffff", "name" => "RAPTORS"),
    array("id" => "26", "abbreviation" => "UTAH", "color" => "fff21f", "alternateColor" => "002b5c", "name" => "JAZZ"),
    array("id" => "27", "abbreviation" => "WSH",  "color" => "e31837", "alternateColor" => "002b5c", "name" => "WIZARDS")
);

// --- Desired NBA Stats Mapping ---
// We want four categories:
// 1. Points (PTS) – descending order.
// 2. Field Goal % (FG%) – descending order.
// 3. Steals (STL) – descending order.
// 4. Rebounds (REB) – descending order.
$desiredStats = array(
    'PTS' => array(
         'possibleAbbreviations' => array('PTS', 'points'),
         'sort' => 'desc'
    ),
    'FG%' => array(
         'possibleAbbreviations' => array('FG%', 'fieldGoalPct'),
         'sort' => 'desc'
    ),
    'STL' => array(
         'possibleAbbreviations' => array('STL', 'steals'),
         'sort' => 'desc'
    ),
    'REB' => array(
         'possibleAbbreviations' => array('REB', 'rebounds'),
         'sort' => 'desc'
    ),
);

// Map keys to full descriptive names.
$fullTypeNames = array(
    'PTS' => "Avg Points",
    'FG%' => "Field Goal %",
    'STL' => "Avg Steals",
    'REB' => "Avg Rebounds"
);

// Initialize an array to collect league-wide stats per category.
$leagueStats = array();
foreach ($desiredStats as $key => $config) {
    $leagueStats[$key] = array();
}

/**
 * Helper function to perform a cURL request and return decoded JSON.
 */
function curlFetch($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
         curl_close($ch);
         return false;
    }
    curl_close($ch);
    return json_decode($response, true);
}

// Loop through each NBA team and fetch its statistics.
foreach ($nbaTeams as $team) {
    $teamId          = $team['id'];
    $teamAbbr        = $team['abbreviation'];
    $teamColor       = $team['color'];
    $alternateColor  = $team['alternateColor'];
    $teamName        = $team['name'];

    // Extract RGB from main color
    $red   = hexdec(substr($teamColor, 0, 2));
    $green = hexdec(substr($teamColor, 2, 2));
    $blue  = hexdec(substr($teamColor, 4, 2));

    // Extract RGB from alternate color
    $redAlt   = hexdec(substr($alternateColor, 0, 2));
    $greenAlt = hexdec(substr($alternateColor, 2, 2));
    $blueAlt  = hexdec(substr($alternateColor, 4, 2));

    // Build the team's statistics URL.
    $statsUrl = "http://sports.core.api.espn.com/v2/sports/basketball/leagues/nba/seasons/2026/types/2/teams/{$teamId}/statistics/0?lang=en&region=us";
    $statsData = curlFetch($statsUrl);
    if (!$statsData) continue;

    if (!isset($statsData['splits']['categories']) || !is_array($statsData['splits']['categories'])) continue;

    // Loop over each category and its stats.
    foreach ($statsData['splits']['categories'] as $category) {
        if (!isset($category['stats']) || !is_array($category['stats'])) continue;

        foreach ($category['stats'] as $stat) {
            if (!isset($stat['abbreviation'])) continue;

            $abbr = $stat['abbreviation'];
            foreach ($desiredStats as $statKey => $config) {
                if (in_array($abbr, $config['possibleAbbreviations'])) {
                    $valueNumeric = null;
                    if (isset($stat['perGameValue'])) {
                        $valueNumeric = floatval($stat['perGameValue']);
                    } elseif (isset($stat['value'])) {
                        $valueNumeric = floatval($stat['value']);
                    }
                    if ($valueNumeric === null) continue;

                    $displayValue = isset($stat['perGameDisplayValue'])
                                    ? $stat['perGameDisplayValue']
                                    : (isset($stat['displayValue']) ? $stat['displayValue'] : "N/A");

                    $leagueStats[$statKey][] = array(
                        'teamAbbr'     => $teamAbbr,
                        'teamColor'    => $teamColor,
                        'teamName'     => $teamName,
                        'value'        => $valueNumeric,
                        'displayValue' => $displayValue,
                        'red'          => $red,
                        'green'        => $green,
                        'blue'         => $blue,
                        'red_alt'      => $redAlt,
                        'green_alt'    => $greenAlt,
                        'blue_alt'     => $blueAlt
                    );
                }
            }
        }
    }
}

// Helper function to sort entries.
function sortEntries(&$entries, $order = 'desc') {
    usort($entries, function($a, $b) use ($order) {
        return $order === 'desc' ? $b['value'] <=> $a['value'] : $a['value'] <=> $b['value'];
    });
}

// For each desired stat type, sort the entries and then select the top 5 unique teams.
$top5Stats = array();
foreach ($leagueStats as $statType => $entries) {
    $order = $desiredStats[$statType]['sort'];
    sortEntries($entries, $order);
    $unique = array();
    foreach ($entries as $entry) {
        if (!isset($unique[$entry['teamAbbr']])) {
            $unique[$entry['teamAbbr']] = $entry;
        }
        if (count($unique) >= 5) break;
    }
    $top5Stats[$statType] = array_values($unique);
}

// Update the pre‑populated NBA rows in the database using the composite key (league, type, rank).
foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    $rank = 0;
    foreach ($entries as $entry) {
        $rank++;
        $sql = "UPDATE team_stats
                SET Abbrev = :abbrev,
                    color = :color,
                    value = :value,
                    current_team = :current_team,
                    Red = :red,
                    Green = :green,
                    Blue = :blue,
                    Red_ALT = :red_alt,
                    Green_ALT = :green_alt,
                    Blue_ALT = :blue_alt
                WHERE league = 'NBA'
                  AND type = :type
                  AND rank = :rank";
        $stmt = $db->prepare($sql);
        $stmt->execute(array(
            ':abbrev'       => $entry['teamAbbr'],
            ':color'        => $entry['teamColor'],
            ':value'        => $entry['displayValue'],
            ':current_team' => $entry['teamName'],
            ':red'          => $entry['red'],
            ':green'        => $entry['green'],
            ':blue'         => $entry['blue'],
            ':red_alt'      => $entry['red_alt'],
            ':green_alt'    => $entry['green_alt'],
            ':blue_alt'     => $entry['blue_alt'],
            ':type'         => $fullType,
            ':rank'         => $rank
        ));
    }
}

// Output the updated NBA top-5 results in an HTML table.
echo "<html><head><title>NBA League Leaders - Top 5 per Category (Database Updated)</title></head><body>";
echo "<h1>NBA League Leaders (Top 5 per Category)</h1>";
echo "<table border='1' cellpadding='5' cellspacing='0'>";
echo "<tr>
        <th>Rank</th>
        <th>League</th>
        <th>Type</th>
        <th>Abbrev</th>
        <th>Hex Color</th>
        <th>Team Name</th>
        <th>Red</th>
        <th>Green</th>
        <th>Blue</th>
        <th>Red_ALT</th>
        <th>Green_ALT</th>
        <th>Blue_ALT</th>
        <th>Value</th>
      </tr>";

foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    $rank = 1;
    foreach ($entries as $entry) {
        echo "<tr>
                <td>{$rank}</td>
                <td>NBA</td>
                <td>" . htmlspecialchars($fullType) . "</td>
                <td>" . htmlspecialchars($entry['teamAbbr']) . "</td>
                <td>" . htmlspecialchars($entry['teamColor']) . "</td>
                <td>" . htmlspecialchars($entry['teamName']) . "</td>
                <td>" . htmlspecialchars($entry['red']) . "</td>
                <td>" . htmlspecialchars($entry['green']) . "</td>
                <td>" . htmlspecialchars($entry['blue']) . "</td>
                <td>" . htmlspecialchars($entry['red_alt']) . "</td>
                <td>" . htmlspecialchars($entry['green_alt']) . "</td>
                <td>" . htmlspecialchars($entry['blue_alt']) . "</td>
                <td>" . htmlspecialchars($entry['displayValue']) . "</td>
              </tr>";
        $rank++;
    }
}
echo "</table>";
echo "</body></html>";
?>
