<?php
// Enable error reporting for debugging.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h1>NCAAF Stats Update Script - Debug Mode</h1>";
echo "<pre>"; // Use preformatted text for better debug output

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Create the database connection.
echo "=== DATABASE CONNECTION ===\n";
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "✓ Database connection successful\n\n";
} catch (PDOException $e) {
    die("✗ Database Connection Error: " . $e->getMessage());
}

// First, let's check what's currently in the database
echo "=== CHECKING EXISTING DATABASE RECORDS ===\n";
$checkSql = "SELECT league, type, rank, Abbrev, value FROM team_stats 
             WHERE league = 'NCAAF' 
             ORDER BY type, rank LIMIT 10";
$checkStmt = $db->prepare($checkSql);
$checkStmt->execute();
$existingRows = $checkStmt->fetchAll(PDO::FETCH_ASSOC);
echo "Found " . count($existingRows) . " existing NCAAF rows in database:\n";
foreach ($existingRows as $row) {
    echo "  - Type: {$row['type']}, Rank: {$row['rank']}, Team: {$row['Abbrev']}, Value: {$row['value']}\n";
}
echo "\n";

$NCAAFteams = array(
    // SEC teams
    array("id" => "2", "abbreviation" => "AUB", "color" => "002b5c", "alternateColor" => "f26522", "name" => "TIGERS"),
    array("id" => "8", "abbreviation" => "ARK", "color" => "a41f35", "alternateColor" => "ffffff", "name" => "RAZORBACKS"),
    array("id" => "57", "abbreviation" => "FLA", "color" => "0021a5", "alternateColor" => "fa4616", "name" => "GATORS"),
    array("id" => "61", "abbreviation" => "UGA", "color" => "ba0c2f", "alternateColor" => "ffffff", "name" => "BULLDOGS"),
    array("id" => "96", "abbreviation" => "UK", "color" => "0033a0", "alternateColor" => "ffffff", "name" => "WILDCATS"),
    array("id" => "99", "abbreviation" => "LSU", "color" => "461d7c", "alternateColor" => "fdd023", "name" => "TIGERS"),
    array("id" => "142", "abbreviation" => "MIZ", "color" => "f1b82d", "alternateColor" => "000000", "name" => "TIGERS"),
    array("id" => "145", "abbreviation" => "MISS", "color" => "13294b", "alternateColor" => "c8102e", "name" => "REBELS"),
    array("id" => "201", "abbreviation" => "OU", "color" => "a32036", "alternateColor" => "ffffff", "name" => "SOONERS"),
    array("id" => "238", "abbreviation" => "VAN", "color" => "000000", "alternateColor" => "231f20", "name" => "COMMODORES"),
    array("id" => "245", "abbreviation" => "TA&M", "color" => "500000", "alternateColor" => "ffffff", "name" => "AGGIES"),
    array("id" => "251", "abbreviation" => "TEX", "color" => "c15d26", "alternateColor" => "ffffff", "name" => "LONGHORNS"),
    array("id" => "333", "abbreviation" => "ALA", "color" => "9e1632", "alternateColor" => "ffffff", "name" => "CRIMSON TI"),
    array("id" => "344", "abbreviation" => "MSST", "color" => "5d1725", "alternateColor" => "c1c6c8", "name" => "BULLDOGS"),
    array("id" => "2579", "abbreviation" => "SC", "color" => "73000a", "alternateColor" => "ffffff", "name" => "GAMECOCKS"),
    array("id" => "2633", "abbreviation" => "TENN", "color" => "ff8200", "alternateColor" => "58595b", "name" => "VOLUNTEERS"),

    // ACC teams
    array("id" => "24", "abbreviation" => "STAN", "color" => "8c1515", "alternateColor" => "ffffff", "name" => "CARDINAL"),
    array("id" => "25", "abbreviation" => "CAL", "color" => "031522", "alternateColor" => "ffc423", "name" => "GOLDEN BEA"),
    array("id" => "52", "abbreviation" => "FSU", "color" => "782f40", "alternateColor" => "ceb888", "name" => "SEMINOLES"),
    array("id" => "59", "abbreviation" => "GT", "color" => "003057", "alternateColor" => "b3a369", "name" => "YELLOW JAC"),
    array("id" => "97", "abbreviation" => "LOU", "color" => "c9001f", "alternateColor" => "000000", "name" => "CARDINALS"),
    array("id" => "103", "abbreviation" => "BC", "color" => "8c2232", "alternateColor" => "dbcca6", "name" => "EAGLES"),
    array("id" => "150", "abbreviation" => "DUKE", "color" => "013088", "alternateColor" => "ffffff", "name" => "BLUE DEVIL"),
    array("id" => "152", "abbreviation" => "NCSU", "color" => "cc0000", "alternateColor" => "ffffff", "name" => "WOLFPACK"),
    array("id" => "153", "abbreviation" => "UNC", "color" => "7bafd4", "alternateColor" => "13294b", "name" => "TAR HEELS"),
    array("id" => "154", "abbreviation" => "WAKE", "color" => "000000", "alternateColor" => "ceb888", "name" => "DEMON DEAC"),
    array("id" => "183", "abbreviation" => "SYR", "color" => "ff6500", "alternateColor" => "000e54", "name" => "ORANGE"),
    array("id" => "221", "abbreviation" => "PITT", "color" => "003263", "alternateColor" => "231f20", "name" => "PANTHERS"),
    array("id" => "228", "abbreviation" => "CLEM", "color" => "f56600", "alternateColor" => "522d80", "name" => "TIGERS"),
    array("id" => "258", "abbreviation" => "UVA", "color" => "232d4b", "alternateColor" => "f84c1e", "name" => "CAVALIERS"),
    array("id" => "259", "abbreviation" => "VT", "color" => "630031", "alternateColor" => "cf4520", "name" => "HOKIES"),
    array("id" => "2390", "abbreviation" => "MIA", "color" => "005030", "alternateColor" => "f47321", "name" => "HURRICANES"),
    array("id" => "2567", "abbreviation" => "SMU", "color" => "354ca1", "alternateColor" => "cc0035", "name" => "MUSTANGS"),

    // Big 12 teams
    array("id" => "9", "abbreviation" => "ASU", "color" => "8e0c3a", "alternateColor" => "ffc72c", "name" => "SUN DEVILS"),
    array("id" => "12", "abbreviation" => "ARIZ", "color" => "0c234b", "alternateColor" => "ab0520", "name" => "WILDCATS"),
    array("id" => "38", "abbreviation" => "COLO", "color" => "000000", "alternateColor" => "cfb87c", "name" => "BUFFALOES"),
    array("id" => "66", "abbreviation" => "ISU", "color" => "822433", "alternateColor" => "fdca2f", "name" => "CYCLONES"),
    array("id" => "197", "abbreviation" => "OKST", "color" => "000000", "alternateColor" => "dddddd", "name" => "COWBOYS"),
    array("id" => "239", "abbreviation" => "BAY", "color" => "154734", "alternateColor" => "ffb81c", "name" => "BEARS"),
    array("id" => "248", "abbreviation" => "HOU", "color" => "c92a39", "alternateColor" => "ffffff", "name" => "COUGARS"),
    array("id" => "252", "abbreviation" => "BYU", "color" => "003da5", "alternateColor" => "ffffff", "name" => "COUGARS"),
    array("id" => "254", "abbreviation" => "UTAH", "color" => "ea002a", "alternateColor" => "ffffff", "name" => "UTES"),
    array("id" => "277", "abbreviation" => "WVU", "color" => "002855", "alternateColor" => "eaaa00", "name" => "MOUNTAINEE"),
    array("id" => "2116", "abbreviation" => "UCF", "color" => "000000", "alternateColor" => "b4a269", "name" => "KNIGHTS"),
    array("id" => "2132", "abbreviation" => "CIN", "color" => "000000", "alternateColor" => "717073", "name" => "BEARCATS"),
    array("id" => "2305", "abbreviation" => "KU", "color" => "0051ba", "alternateColor" => "e8000d", "name" => "JAYHAWKS"),
    array("id" => "2306", "abbreviation" => "KSU", "color" => "3c0969", "alternateColor" => "e2e3e4", "name" => "WILDCATS"),
    array("id" => "2628", "abbreviation" => "TCU", "color" => "4d1979", "alternateColor" => "f1f2f3", "name" => "H-FROGS"),
    array("id" => "2641", "abbreviation" => "TTU", "color" => "000000", "alternateColor" => "da291c", "name" => "RED RAIDER"),

    // Big 10 teams
    array("id" => "26", "abbreviation" => "UCLA", "color" => "2774ae", "alternateColor" => "f2a900", "name" => "BRUINS"),
    array("id" => "30", "abbreviation" => "USC", "color" => "9e2237", "alternateColor" => "ffcc00", "name" => "TROJANS"),
    array("id" => "77", "abbreviation" => "NU", "color" => "582c83", "alternateColor" => "ffffff", "name" => "WILDCATS"),
    array("id" => "84", "abbreviation" => "IU", "color" => "990000", "alternateColor" => "edebeb", "name" => "HOOSIERS"),
    array("id" => "120", "abbreviation" => "MD", "color" => "D5002B", "alternateColor" => "ffcd00", "name" => "TERRAPINS"),
    array("id" => "127", "abbreviation" => "MSU", "color" => "18453b", "alternateColor" => "ffffff", "name" => "SPARTANS"),
    array("id" => "130", "abbreviation" => "MICH", "color" => "00274c", "alternateColor" => "ffcb05", "name" => "WOLVERINES"),
    array("id" => "135", "abbreviation" => "MINN", "color" => "5e0a2f", "alternateColor" => "fab41c", "name" => "GOLDEN GOP"),
    array("id" => "158", "abbreviation" => "NEB", "color" => "d00000", "alternateColor" => "ffffff", "name" => "CORNHUSKER"),
    array("id" => "164", "abbreviation" => "RUTG", "color" => "d21034", "alternateColor" => "ffffff", "name" => "SCARLET KN"),
    array("id" => "194", "abbreviation" => "OSU", "color" => "ce1141", "alternateColor" => "505056", "name" => "BUCKEYES"),
    array("id" => "213", "abbreviation" => "PSU", "color" => "00265D", "alternateColor" => "002e5c", "name" => "NITTANY LI"),
    array("id" => "264", "abbreviation" => "WASH", "color" => "33006f", "alternateColor" => "e8d3a2", "name" => "HUSKIES"),
    array("id" => "275", "abbreviation" => "WIS", "color" => "c4012f", "alternateColor" => "ffffff", "name" => "BADGERS"),
    array("id" => "356", "abbreviation" => "ILL", "color" => "ff5f05", "alternateColor" => "13294b", "name" => "FIGHTING I"),
    array("id" => "2294", "abbreviation" => "IOWA", "color" => "000000", "alternateColor" => "fcd116", "name" => "HAWKEYES"),
    array("id" => "2483", "abbreviation" => "ORE", "color" => "007030", "alternateColor" => "fee11a", "name" => "DUCKS"),
    array("id" => "2509", "abbreviation" => "PUR", "color" => "000000", "alternateColor" => "cfb991", "name" => "BOILERMAKE"),

    // MW teams
    array("id" => "21", "abbreviation" => "SDSU", "color" => "c41230", "alternateColor" => "000000", "name" => "AZTECS"),
    array("id" => "23", "abbreviation" => "SJSU", "color" => "005893", "alternateColor" => "fdba31", "name" => "SPARTANS"),
    array("id" => "36", "abbreviation" => "CSU", "color" => "1e4d2b", "alternateColor" => "c8c372", "name" => "RAMS"),
    array("id" => "62", "abbreviation" => "HAW", "color" => "003420", "alternateColor" => "ffffff", "name" => "RAINBOW WA"),
    array("id" => "68", "abbreviation" => "BOIS", "color" => "0033a0", "alternateColor" => "fa4616", "name" => "BRONCOS"),
    array("id" => "167", "abbreviation" => "UNM", "color" => "000000", "alternateColor" => "231f20", "name" => "LOBOS"),
    array("id" => "278", "abbreviation" => "FRES", "color" => "c41230", "alternateColor" => "13284c", "name" => "BULLDOGS"),
    array("id" => "328", "abbreviation" => "USU", "color" => "00263a", "alternateColor" => "949ca1", "name" => "AGGIES"),
    array("id" => "2005", "abbreviation" => "AFA", "color" => "004a7b", "alternateColor" => "ffffff", "name" => "FALCONS"),
    array("id" => "2426", "abbreviation" => "NAVY", "color" => "00225b", "alternateColor" => "b5a67c", "name" => "MIDSHIPMEN"),
    array("id" => "2439", "abbreviation" => "UNLV", "color" => "b10202", "alternateColor" => "ffffff", "name" => "REBELS"),
    array("id" => "2440", "abbreviation" => "NEV", "color" => "002d62", "alternateColor" => "ffffff", "name" => "WOLF PACK"),
    array("id" => "2751", "abbreviation" => "WYO", "color" => "492f24", "alternateColor" => "ffc425", "name" => "COWBOYS"),

    // Sunbelt teams  
    array("id" => "6", "abbreviation" => "USA", "color" => "003E7E", "alternateColor" => "", "name" => "JAGUARS"),
    array("id" => "256", "abbreviation" => "JMU", "color" => "450084", "alternateColor" => "b5a068", "name" => "DUKES"),
    array("id" => "276", "abbreviation" => "MRSH", "color" => "00ae42", "alternateColor" => "be854c", "name" => "THUNDERING"),
    array("id" => "290", "abbreviation" => "GASO", "color" => "003775", "alternateColor" => "f0f0f0", "name" => "EAGLES"),
    array("id" => "295", "abbreviation" => "ODU", "color" => "00507d", "alternateColor" => "a1d2f1", "name" => "MONARCHS"),
    array("id" => "309", "abbreviation" => "UL", "color" => "ce181e", "alternateColor" => "ffffff", "name" => "RAGIN' CAJ"),
    array("id" => "324", "abbreviation" => "CCU", "color" => "007073", "alternateColor" => "876447", "name" => "CHANTICLEE"),
    array("id" => "326", "abbreviation" => "TXST", "color" => "4e1719", "alternateColor" => "b4975a", "name" => "BOBCATS"),
    array("id" => "2026", "abbreviation" => "APP", "color" => "000000", "alternateColor" => "ffcd00", "name" => "MOUNTAINEE"),
    array("id" => "2032", "abbreviation" => "ARST", "color" => "e81018", "alternateColor" => "000000", "name" => "RED WOLVES"),
    array("id" => "2247", "abbreviation" => "GAST", "color" => "1e539a", "alternateColor" => "ebebeb", "name" => "PANTHERS"),
    array("id" => "2433", "abbreviation" => "ULM", "color" => "231F20", "alternateColor" => "b18445", "name" => "WARHAWKS"),
    array("id" => "2572", "abbreviation" => "USM", "color" => "FFAA3C", "alternateColor" => "ffc423", "name" => "GOLDEN EAG"),
    array("id" => "2653", "abbreviation" => "TROY", "color" => "AE0210", "alternateColor" => "88898c", "name" => "TROJANS"),

    // AAC teams
    array("id" => "5", "abbreviation" => "UAB", "color" => "003b28", "alternateColor" => "ffc845", "name" => "BLAZERS"),
    array("id" => "58", "abbreviation" => "USF", "color" => "004A36", "alternateColor" => "231f20", "name" => "BULLS"),
    array("id" => "151", "abbreviation" => "ECU", "color" => "4b1869", "alternateColor" => "f0907b", "name" => "PIRATES"),
    array("id" => "202", "abbreviation" => "TLSA", "color" => "003595", "alternateColor" => "d0b787", "name" => "GOLDEN HUR"),
    array("id" => "218", "abbreviation" => "TEM", "color" => "A80532", "alternateColor" => "a7a9ac", "name" => "OWLS"),
    array("id" => "235", "abbreviation" => "MEM", "color" => "004991", "alternateColor" => "8e908f", "name" => "TIGERS"),
    array("id" => "242", "abbreviation" => "RICE", "color" => "d1d5d8", "alternateColor" => "003d7d", "name" => "OWLS"),
    array("id" => "249", "abbreviation" => "UNT", "color" => "ffffff", "alternateColor" => "000000", "name" => "MEAN GREEN"),
    array("id" => "349", "abbreviation" => "ARMY", "color" => "ce9c00", "alternateColor" => "231f20", "name" => "BLACK KNIG"),
    array("id" => "2226", "abbreviation" => "FAU", "color" => "00447c", "alternateColor" => "d31245", "name" => "OWLS"),
    array("id" => "2429", "abbreviation" => "CLT", "color" => "ffffff", "alternateColor" => "cfab7a", "name" => "49ERS"),
    array("id" => "2636", "abbreviation" => "UTSA", "color" => "002A5C", "alternateColor" => "f47321", "name" => "ROADRUNNER"),
    array("id" => "2655", "abbreviation" => "TULN", "color" => "006547", "alternateColor" => "468ac9", "name" => "GREEN WAVE"),

    // Additional teams
    array("id" => "231", "abbreviation" => "FUR", "color" => "582c83", "alternateColor" => "808080", "name" => "PALADINS"),
    array("id" => "236", "abbreviation" => "UTC", "color" => "00386b", "alternateColor" => "eeb111", "name" => "MOCS"),
    array("id" => "2193", "abbreviation" => "ETSU", "color" => "002d61", "alternateColor" => "ffc72c", "name" => "BUCCANEERS"),
    array("id" => "2382", "abbreviation" => "MER", "color" => "ff7f29", "alternateColor" => "000000", "name" => "BEARS"),
    array("id" => "2535", "abbreviation" => "SAM", "color" => "005485", "alternateColor" => "bb133e", "name" => "BULLDOGS"),
    array("id" => "2643", "abbreviation" => "CIT", "color" => "002b5c", "alternateColor" => "ffffff", "name" => "BULLDOGS"),
    array("id" => "2678", "abbreviation" => "VMI", "color" => "ae122a", "alternateColor" => "ffcd00", "name" => "KEYDETS"),
    array("id" => "2717", "abbreviation" => "WCU", "color" => "492F91", "alternateColor" => "d4af37", "name" => "CATAMOUNTS"),
    array("id" => "2747", "abbreviation" => "WOF", "color" => "533B23", "alternateColor" => "000000", "name" => "TERRIERS"),
    array("id" => "87", "abbreviation" => "ND", "color" => "0C2340", "alternateColor" => "ae9142", "name" => "NOTRE DAME")
);

echo "=== TEAM CONFIGURATION ===\n";
echo "Total teams to process: " . count($NCAAFteams) . "\n\n";

// Desired NCAAF Stats Mapping
$desiredStats = array(
    'netYardsPerGame' => array(
         'possibleAbbreviations' => array('NTYDS/G'),
         'sort' => 'desc'
    ),
    'netPassingYardsPerGame' => array(
         'possibleAbbreviations' => array('NYDS/G'),
         'sort' => 'desc'
    ),
    'rushingYardsPerGame' => array(
         'possibleAbbreviations' => array('YDS/G'),
         'sort' => 'desc'
    ),
    'defensivePoints' => array(
         'possibleAbbreviations' => array('DP'),
         'sort' => 'asc'
    ),
    'totalTouchdowns' => array(
         'possibleAbbreviations' => array('TD'),
         'sort' => 'desc'
    ),
    'sacks' => array(
         'possibleAbbreviations' => array('SACK'),
         'sort' => 'desc'
    ),
);

// Map keys to full descriptive names.
$fullTypeNames = array(
    'netYardsPerGame'      => "Net Yards Per Game",
    'netPassingYardsPerGame' => "Net Passing Yards Per Game",
    'rushingYardsPerGame'  => "Rushing Yards Per Game",
    'defensivePoints'      => "Defensive Points",
    'totalTouchdowns'      => "Total Touchdowns",
    'sacks'                => "Total Sacks"
);

echo "=== STATS TO TRACK ===\n";
foreach ($fullTypeNames as $key => $name) {
    echo "  - $name (key: $key)\n";
}
echo "\n";

// Initialize an array to collect league-wide stats per category.
$leagueStats = array();
foreach ($desiredStats as $key => $config) {
    $leagueStats[$key] = array();
}

/**
 * Helper function to perform a cURL request and return decoded JSON.
 */
function curlFetch($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        echo "    ✗ cURL Error: " . curl_error($ch) . "\n";
        curl_close($ch);
        return false;
    }
    
    if ($httpCode !== 200) {
        echo "    ✗ HTTP Error: Code $httpCode\n";
        curl_close($ch);
        return false;
    }
    
    curl_close($ch);
    $decoded = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        echo "    ✗ JSON Decode Error: " . json_last_error_msg() . "\n";
        return false;
    }
    
    return $decoded;
}

echo "=== FETCHING TEAM STATISTICS ===\n";
$teamsProcessed = 0;
$teamsFailed = 0;

// Loop through each NCAAF team and fetch its statistics.
foreach ($NCAAFteams as $index => $team) {
    $teamId          = $team['id'];
    $teamAbbr        = $team['abbreviation'];
    $teamColor       = $team['color'];
    $alternateColor  = $team['alternateColor'];
    $teamName        = $team['name'];
    
    if ($index < 5 || $index % 10 == 0) { // Show first 5 and every 10th team
        echo "Processing team #{$index}: $teamAbbr ($teamName) ID: $teamId\n";
    }
    
    // Extract RGB from main color
    $red   = hexdec(substr($teamColor, 0, 2));
    $green = hexdec(substr($teamColor, 2, 2));
    $blue  = hexdec(substr($teamColor, 4, 2));

    // Extract RGB from alternate color
    $redAlt   = hexdec(substr($alternateColor, 0, 2));
    $greenAlt = hexdec(substr($alternateColor, 2, 2));
    $blueAlt  = hexdec(substr($alternateColor, 4, 2));

    // Build the team's statistics URL for NCAAF.
    $statsUrl = "http://sports.core.api.espn.com/v2/sports/football/leagues/college-football/seasons/2025/types/3/teams/{$teamId}/statistics/0?lang=en&region=us";
    
    $statsData = curlFetch($statsUrl);
    if (!$statsData) {
        echo "  ✗ Failed to fetch data for $teamAbbr\n";
        $teamsFailed++;
        continue;
    }

    if (!isset($statsData['splits']['categories']) || !is_array($statsData['splits']['categories'])) {
        echo "  ✗ No categories found for $teamAbbr\n";
        $teamsFailed++;
        continue;
    }

    $statsFoundForTeam = 0;
    
    // Loop over each category and its stats.
    foreach ($statsData['splits']['categories'] as $category) {
        if (!isset($category['stats']) || !is_array($category['stats'])) continue;
        
        $categoryName = isset($category['name']) ? $category['name'] : 'Unknown';

        foreach ($category['stats'] as $stat) {
            if (!isset($stat['abbreviation'])) continue;

            $abbr = $stat['abbreviation'];
            $statName = isset($stat['name']) ? $stat['name'] : '';
            $displayName = isset($stat['displayName']) ? $stat['displayName'] : '';
            
            // Enhanced debug output for first few teams
            if ($index < 3 && $statName == 'totalTouchdowns') {
                echo "    Team $teamAbbr - Found totalTouchdowns in $categoryName category:\n";
                echo "      abbreviation: '$abbr'\n";
                echo "      value: " . (isset($stat['value']) ? $stat['value'] : 'N/A') . "\n";
            }
            
            // CRITICAL: Check exact stat name and category to avoid duplicates
            $matchFound = false;
            $statKey = '';
            
            // Rushing Yards Per Game - MUST be the specific rushingYardsPerGame, not generic yardsPerGame
            if ($categoryName == 'rushing' && $statName == 'rushingYardsPerGame' && $abbr == 'YDS/G') {
                $matchFound = true;
                $statKey = 'rushingYardsPerGame';
            }
            
            // Net Passing Yards Per Game
            elseif ($categoryName == 'passing' && $statName == 'netPassingYardsPerGame' && $abbr == 'NYDS/G') {
                $matchFound = true;
                $statKey = 'netPassingYardsPerGame';
            }
            
            // Net Yards Per Game (Total)
            elseif ($categoryName == 'passing' && $statName == 'netYardsPerGame' && $abbr == 'NTYDS/G') {
                $matchFound = true;
                $statKey = 'netYardsPerGame';
            }
            
            // Defensive Points
            elseif ($categoryName == 'scoring' && $statName == 'defensivePoints' && $abbr == 'DP') {
                $matchFound = true;
                $statKey = 'defensivePoints';
            }
            
            // Total Touchdowns - CRITICAL: Only from scoring category!
            elseif ($categoryName == 'scoring' && $statName == 'totalTouchdowns' && $abbr == 'TD') {
                $matchFound = true;
                $statKey = 'totalTouchdowns';
                if ($index < 3) {
                    echo "      ✓ MATCHED as totalTouchdowns from SCORING category (value: {$stat['value']})\n";
                }
            }
            // Skip totalTouchdowns from other categories
            elseif ($statName == 'totalTouchdowns' && $categoryName !== 'scoring' && $index < 3) {
                echo "      ✗ SKIPPED totalTouchdowns from $categoryName category (value: {$stat['value']})\n";
            }
            
            // Sacks (from defensive category)
            elseif ($categoryName == 'defensive' && $statName == 'sacks' && $abbr == 'SACK') {
                $matchFound = true;
                $statKey = 'sacks';
            }
            
            if ($matchFound && $statKey) {
                $valueNumeric = null;
                // Always use the 'value' field directly
                if (isset($stat['value'])) {
                    $valueNumeric = floatval($stat['value']);
                }
                
                if ($valueNumeric === null) continue;

                $displayValue = isset($stat['displayValue']) ? $stat['displayValue'] : "N/A";

                $leagueStats[$statKey][] = array(
                    'teamAbbr'     => $teamAbbr,
                    'teamColor'    => $teamColor,
                    'teamName'     => $teamName,
                    'value'        => $valueNumeric,
                    'displayValue' => $displayValue,
                    'red'          => $red,
                    'green'        => $green,
                    'blue'         => $blue,
                    'red_alt'      => $redAlt,
                    'green_alt'    => $greenAlt,
                    'blue_alt'     => $blueAlt
                );
                $statsFoundForTeam++;
            }
        }
    }
    
    if ($statsFoundForTeam > 0) {
        $teamsProcessed++;
    }
}

echo "\nTeam Processing Summary:\n";
echo "  ✓ Successfully processed: $teamsProcessed teams\n";
echo "  ✗ Failed: $teamsFailed teams\n\n";

echo "=== STATS COLLECTED PER CATEGORY ===\n";
foreach ($leagueStats as $statType => $entries) {
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    echo "  $fullType: " . count($entries) . " entries\n";
}
echo "\n";

// Helper function to sort stat entries.
function sortEntries(&$entries, $order = 'desc') {
    usort($entries, function($a, $b) use ($order) {
        return $order === 'desc' ? $b['value'] <=> $a['value'] : $a['value'] <=> $b['value'];
    });
}

echo "=== SORTING AND SELECTING TOP 5 ===\n";
// For each desired stat type, sort the entries and then select the top 5 unique teams.
$top5Stats = array();
foreach ($leagueStats as $statType => $entries) {
    $order = $desiredStats[$statType]['sort'];
    sortEntries($entries, $order);
    $unique = array();
    foreach ($entries as $entry) {
        if (!isset($unique[$entry['teamAbbr']])) {
            $unique[$entry['teamAbbr']] = $entry;
        }
        if (count($unique) >= 5) break;
    }
    $top5Stats[$statType] = array_values($unique);
    
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    echo "  $fullType: Top 5 selected\n";
    foreach ($unique as $teamAbbr => $data) {
        echo "    - $teamAbbr: {$data['displayValue']}\n";
    }
}
echo "\n";

echo "=== DATABASE UPDATE OPERATIONS ===\n";
$totalUpdates = 0;
$successfulUpdates = 0;
$failedUpdates = 0;

// Update the pre-populated NCAAF rows in the database using the composite key (league, type, rank).
foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    echo "\nUpdating: $fullType\n";
    
    $rank = 0;
    foreach ($entries as $entry) {
        $rank++;
        $totalUpdates++;
        
        // First check if the row exists
        $checkSql = "SELECT COUNT(*) as count FROM team_stats 
                     WHERE league = 'NCAAF' AND type = :type AND rank = :rank";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute(array(':type' => $fullType, ':rank' => $rank));
        $rowExists = $checkStmt->fetch(PDO::FETCH_ASSOC)['count'] > 0;
        
        if (!$rowExists) {
            echo "  ✗ Row doesn't exist: Type='$fullType', Rank=$rank - SKIPPING\n";
            $failedUpdates++;
            continue;
        }
        
        $sql = "UPDATE team_stats
                SET Abbrev = :abbrev,
                    color = :color,
                    value = :value,
                    current_team = :current_team,
                    Red = :red,
                    Green = :green,
                    Blue = :blue,
                    Red_ALT = :red_alt,
                    Green_ALT = :green_alt,
                    Blue_ALT = :blue_alt
                WHERE league = 'NCAAF'
                  AND type = :type
                  AND rank = :rank";
        
        try {
            $stmt = $db->prepare($sql);
            $result = $stmt->execute(array(
                ':abbrev'       => $entry['teamAbbr'],
                ':color'        => $entry['teamColor'],
                ':value'        => $entry['displayValue'],
                ':current_team' => $entry['teamName'],
                ':red'          => $entry['red'],
                ':green'        => $entry['green'],
                ':blue'         => $entry['blue'],
                ':red_alt'      => $entry['red_alt'],
                ':green_alt'    => $entry['green_alt'],
                ':blue_alt'     => $entry['blue_alt'],
                ':type'         => $fullType,
                ':rank'         => $rank
            ));
            
            $rowsAffected = $stmt->rowCount();
            
            if ($rowsAffected > 0) {
                echo "  ✓ Rank $rank: Updated to {$entry['teamAbbr']} = {$entry['displayValue']} (Rows affected: $rowsAffected)\n";
                $successfulUpdates++;
            } else {
                echo "  ⚠ Rank $rank: No rows affected for {$entry['teamAbbr']}\n";
                $failedUpdates++;
            }
        } catch (PDOException $e) {
            echo "  ✗ Rank $rank: Database error - " . $e->getMessage() . "\n";
            $failedUpdates++;
        }
    }
}

echo "\n=== FINAL SUMMARY ===\n";
echo "Total update attempts: $totalUpdates\n";
echo "✓ Successful updates: $successfulUpdates\n";
echo "✗ Failed updates: $failedUpdates\n\n";

// Verify the updates by checking the database again
echo "=== VERIFYING UPDATES ===\n";
$verifySql = "SELECT type, rank, Abbrev, value FROM team_stats 
              WHERE league = 'NCAAF' 
              AND type IN ('Net Yards Per Game', 'Net Passing Yards Per Game', 
                          'Rushing Yards Per Game', 'Defensive Points', 
                          'Total Touchdowns', 'Total Sacks')
              ORDER BY type, rank LIMIT 30";
$verifyStmt = $db->prepare($verifySql);
$verifyStmt->execute();
$updatedRows = $verifyStmt->fetchAll(PDO::FETCH_ASSOC);

echo "Current database state (first 30 rows):\n";
$lastType = '';
foreach ($updatedRows as $row) {
    if ($row['type'] !== $lastType) {
        echo "\n{$row['type']}:\n";
        $lastType = $row['type'];
    }
    echo "  Rank {$row['rank']}: {$row['Abbrev']} = {$row['value']}\n";
}

echo "</pre>"; // End preformatted text

// Output the updated NCAAF top-5 results in an HTML table.
echo "<h2>NCAAF League Leaders (Top 5 per Category)</h2>";
echo "<table border='1' cellpadding='5' cellspacing='0'>";
echo "<tr>
        <th>Rank</th>
        <th>League</th>
        <th>Type</th>
        <th>Abbrev</th>
        <th>Hex Color</th>
        <th>Team Name</th>
        <th>Red</th>
        <th>Green</th>
        <th>Blue</th>
        <th>Red_ALT</th>
        <th>Green_ALT</th>
        <th>Blue_ALT</th>
        <th>Value</th>
      </tr>";

foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($fullTypeNames[$statType]) ? $fullTypeNames[$statType] : $statType;
    $rank = 1;
    foreach ($entries as $entry) {
        echo "<tr>
                <td>{$rank}</td>
                <td>NCAAF</td>
                <td>" . htmlspecialchars($fullType) . "</td>
                <td>" . htmlspecialchars($entry['teamAbbr']) . "</td>
                <td style='background-color:#{$entry['teamColor']}; color: white;'>" . htmlspecialchars($entry['teamColor']) . "</td>
                <td>" . htmlspecialchars($entry['teamName']) . "</td>
                <td>" . htmlspecialchars($entry['red']) . "</td>
                <td>" . htmlspecialchars($entry['green']) . "</td>
                <td>" . htmlspecialchars($entry['blue']) . "</td>
                <td>" . htmlspecialchars($entry['red_alt']) . "</td>
                <td>" . htmlspecialchars($entry['green_alt']) . "</td>
                <td>" . htmlspecialchars($entry['blue_alt']) . "</td>
                <td>" . htmlspecialchars($entry['displayValue']) . "</td>
              </tr>";
        $rank++;
    }
}
echo "</table>";
echo "</body></html>";
?>