<?php
/**
 * NCAAF Team Leaders Update Script
 * Updates statistical leaders for all NCAAF teams across all conferences
 * Includes progress tracking and API rate limiting
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
set_time_limit(600); // 10 minutes max execution time

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Headshot directory for NCAAF
$headshot_dir = '/home/gscroll1/getStockPanel/Logos/HEADSHOTS/NCAAF/';

// Create directory if it doesn't exist
if (!file_exists($headshot_dir)) {
    mkdir($headshot_dir, 0755, true);
}

// Conference mapping (excluding TOP-25)
$conferences = [
    'SEC' => 8,
    'B10' => 5,
    'P12' => 9,
    'ACC' => 1,
    'B12' => 4,
    'MWC' => 17,
    'MAC' => 15,
    'CUSA' => 12,
    'IVY' => 22,
    'SBC' => 37,
    'AAC' => 151
];

// Full conference names for display
$conferenceFullNames = [
    'SEC' => 'SEC',
    'B10' => 'Big Ten',
    'P12' => 'Pac-12',
    'ACC' => 'ACC',
    'B12' => 'Big 12',
    'MWC' => 'Mountain West',
    'MAC' => 'MAC',
    'CUSA' => 'Conference USA',
    'IVY' => 'Ivy League',
    'SBC' => 'Sun Belt',
    'AAC' => 'American Athletic'
];

// Function to download headshot
function downloadHeadshot($url, $playerId, $headshot_dir) {
    if (empty($url) || empty($playerId)) {
        return false;
    }
    
    $extension = (strpos($url, '.jpg') !== false) ? 'jpg' : 'png';
    $filename = $headshot_dir . $playerId . '.' . $extension;
    
    if (file_exists($filename)) {
        return true;
    }
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $imageData = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200 && !empty($imageData)) {
        file_put_contents($filename, $imageData);
        return true;
    }
    
    return false;
}

// Function to format player stats string
function formatPlayerStats($athlete, $stats, $position, $teamAbbrev, $categoryName) {
    $playerId = $athlete['id'] ?? '';
    $fullName = $athlete['displayName'] ?? '';
    $jersey = $athlete['jersey'] ?? '';
    
    // Format stats based on category
    $statLine = '';
    switch($categoryName) {
        case 'passing':
            $statLine = $stats[0] . ', ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            if (!empty($stats[4]) && $stats[4] > 0) {
                $statLine .= ', ' . $stats[4] . ' INT';
            }
            break;
            
        case 'rushing':
            $statLine = $stats[0] . ' CAR, ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            break;
            
        case 'receiving':
            $statLine = $stats[0] . ' REC, ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            break;
    }
    
    return "DN: $fullName | STATS: $statLine | ID: $playerId | JERSEY: $jersey | POS: $position | TEAM: $teamAbbrev";
}

// Function to format leader from the leaders section
function formatLeaderStats($leaderData, $position, $teamAbbrev) {
    $athlete = $leaderData['athlete'] ?? [];
    $playerId = $athlete['id'] ?? '';
    $fullName = $athlete['displayName'] ?? '';
    $jersey = $athlete['jersey'] ?? '';
    $displayValue = $leaderData['displayValue'] ?? '';
    
    // Download headshot if available
    global $headshot_dir;
    if (isset($athlete['headshot']['href'])) {
        downloadHeadshot($athlete['headshot']['href'], $playerId, $headshot_dir);
    }
    
    return "DN: $fullName | STATS: $displayValue | ID: $playerId | JERSEY: $jersey | POS: $position | TEAM: $teamAbbrev";
}

// Function to process season leaders
function processSeasonLeaders($data, $con, $debug = false) {
    $updateCount = 0;
    
    if (!isset($data['leaders']) || !is_array($data['leaders'])) {
        return 0;
    }
    
    foreach ($data['leaders'] as $teamLeaders) {
        $team = $teamLeaders['team'] ?? [];
        $teamAbbrev = $team['abbreviation'] ?? '';
        
        if (empty($teamAbbrev)) continue;
        
        $leaders = [
            'qb' => '',
            'rb' => '',
            'wr' => ''
        ];
        
        if (isset($teamLeaders['leaders']) && is_array($teamLeaders['leaders'])) {
            foreach ($teamLeaders['leaders'] as $category) {
                $categoryName = $category['name'] ?? '';
                
                if (!empty($category['leaders']) && isset($category['leaders'][0])) {
                    $leader = $category['leaders'][0];
                    
                    switch($categoryName) {
                        case 'passingYards':
                            $leaders['qb'] = formatLeaderStats($leader, 'QB', $teamAbbrev);
                            break;
                        case 'rushingYards':
                            $leaders['rb'] = formatLeaderStats($leader, 'RB', $teamAbbrev);
                            break;
                        case 'receivingYards':
                            $leaders['wr'] = formatLeaderStats($leader, 'WR', $teamAbbrev);
                            break;
                    }
                }
            }
        }
        
        // Update database
        $query = "UPDATE team_leaders SET ";
        $updates = [];
        
        if (!empty($leaders['qb'])) {
            $updates[] = "qb_leader = '" . mysqli_real_escape_string($con, $leaders['qb']) . "'";
        }
        if (!empty($leaders['rb'])) {
            $updates[] = "rb_leader = '" . mysqli_real_escape_string($con, $leaders['rb']) . "'";
        }
        if (!empty($leaders['wr'])) {
            $updates[] = "wr_leader = '" . mysqli_real_escape_string($con, $leaders['wr']) . "'";
        }
        
        if (!empty($updates)) {
            $query .= implode(', ', $updates);
            $query .= " WHERE team_abbrev = '" . mysqli_real_escape_string($con, $teamAbbrev) . "' AND league = 'NCAAF'";
            
            if (mysqli_query($con, $query)) {
                $updateCount++;
            }
        }
    }
    
    return $updateCount;
}

// Function to process a single game
function processGame($gameId, $con, $debug = false) {
    global $headshot_dir;
    
    $summaryUrl = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/summary?event=" . $gameId;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $summaryUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $result = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);
    
    if ($httpCode != 200 || empty($result)) {
        return 0;
    }
    
    $data = json_decode($result, true);
    $updateCount = 0;
    
    // Try boxscore.players first, then fall back to leaders
    if (isset($data['boxscore']['players']) && !empty($data['boxscore']['players'])) {
        // Process game-specific stats
        foreach ($data['boxscore']['players'] as $teamData) {
            $teamInfo = $teamData['team'] ?? [];
            $teamAbbrev = $teamInfo['abbreviation'] ?? '';
            
            if (empty($teamAbbrev)) continue;
            
            $leaders = [
                'qb' => '',
                'rb' => '',
                'wr' => ''
            ];
            
            if (isset($teamData['statistics'])) {
                foreach ($teamData['statistics'] as $category) {
                    $categoryName = $category['name'] ?? '';
                    
                    if (isset($category['athletes']) && count($category['athletes']) > 0) {
                        $leader = $category['athletes'][0];
                        $athlete = $leader['athlete'] ?? [];
                        $stats = $leader['stats'] ?? [];
                        
                        if (!empty($athlete) && !empty($stats)) {
                            if (isset($athlete['headshot']['href'])) {
                                downloadHeadshot($athlete['headshot']['href'], $athlete['id'], $headshot_dir);
                            }
                            
                            switch($categoryName) {
                                case 'passing':
                                    $leaders['qb'] = formatPlayerStats($athlete, $stats, 'QB', $teamAbbrev, $categoryName);
                                    break;
                                case 'rushing':
                                    $leaders['rb'] = formatPlayerStats($athlete, $stats, 'RB', $teamAbbrev, $categoryName);
                                    break;
                                case 'receiving':
                                    $leaders['wr'] = formatPlayerStats($athlete, $stats, 'WR', $teamAbbrev, $categoryName);
                                    break;
                            }
                        }
                    }
                }
            }
            
            // Update database
            $query = "UPDATE team_leaders SET ";
            $updates = [];
            
            if (!empty($leaders['qb'])) {
                $updates[] = "qb_leader = '" . mysqli_real_escape_string($con, $leaders['qb']) . "'";
            }
            if (!empty($leaders['rb'])) {
                $updates[] = "rb_leader = '" . mysqli_real_escape_string($con, $leaders['rb']) . "'";
            }
            if (!empty($leaders['wr'])) {
                $updates[] = "wr_leader = '" . mysqli_real_escape_string($con, $leaders['wr']) . "'";
            }
            
            if (!empty($updates)) {
                $query .= implode(', ', $updates);
                $query .= " WHERE team_abbrev = '" . mysqli_real_escape_string($con, $teamAbbrev) . "' AND league = 'NCAAF'";
                
                if (mysqli_query($con, $query)) {
                    $updateCount++;
                }
            }
        }
    } else {
        // Fall back to season leaders
        $updateCount = processSeasonLeaders($data, $con, $debug);
    }
    
    return $updateCount;
}

// MAIN EXECUTION
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Set timezone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

$debug = isset($_GET['debug']);

// Start HTML output
echo "<!DOCTYPE html>
<html>
<head>
    <title>NCAAF Team Leaders Update</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1400px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        h1 { color: #333; border-bottom: 3px solid #cc0000; padding-bottom: 10px; }
        .progress-container {
            width: 100%;
            background-color: #f0f0f0;
            border-radius: 10px;
            overflow: hidden;
            margin: 20px 0;
            height: 30px;
        }
        .progress-bar {
            height: 100%;
            background: linear-gradient(90deg, #4CAF50, #45a049);
            text-align: center;
            line-height: 30px;
            color: white;
            transition: width 0.3s ease;
        }
        .conference-section {
            margin: 20px 0;
            padding: 15px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        .conference-header {
            font-size: 18px;
            font-weight: bold;
            color: #cc0000;
            margin-bottom: 10px;
        }
        .game-info {
            padding: 5px 10px;
            margin: 5px 0;
            background: #f9f9f9;
            border-left: 3px solid #4CAF50;
            font-size: 12px;
        }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th { background: #cc0000; color: white; padding: 10px; text-align: left; font-size: 12px; }
        td { padding: 8px; border-bottom: 1px solid #ddd; font-size: 11px; }
        tr:hover { background: #f9f9f9; }
        .success { color: green; }
        .error { color: red; }
        .info { color: #0066cc; }
        .stats-summary {
            margin-top: 20px;
            padding: 15px;
            background: #e8f4f8;
            border-radius: 5px;
        }
        .update-time { text-align: center; color: #666; margin: 10px 0; }
    </style>
    <script>
        function updateProgress(current, total) {
            const percentage = Math.round((current / total) * 100);
            document.getElementById('progressBar').style.width = percentage + '%';
            document.getElementById('progressBar').innerHTML = percentage + '%';
            document.getElementById('progressText').innerHTML = 'Processing: ' + current + ' / ' + total + ' items';
        }
    </script>
</head>
<body>
<div class='container'>";

echo "<h1>🏈 NCAAF Team Leaders Update</h1>";
echo "<div class='update-time'>Started: $time_updated Eastern</div>";

// Progress bar
echo "<div class='progress-container'>";
echo "<div id='progressBar' class='progress-bar' style='width: 0%;'>0%</div>";
echo "</div>";
echo "<div id='progressText' style='text-align: center; margin: 10px 0;'>Initializing...</div>";

// Flush output to show progress
ob_flush();
flush();

// Statistics tracking
$totalGames = 0;
$totalUpdates = 0;
$apiCalls = 0;
$conferenceStats = [];

// First, count total games across all conferences
echo "<h3>Scanning Conferences...</h3>";
$allGames = [];

foreach ($conferences as $confName => $confId) {
    $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard?groups=" . $confId;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
    ]);
    
    $result = curl_exec($curl);
    curl_close($curl);
    $apiCalls++;
    
    $data = json_decode($result, true);
    
    if (isset($data['events'])) {
        foreach ($data['events'] as $event) {
            $gameId = $event['id'] ?? '';
            if (!empty($gameId)) {
                $allGames[] = [
                    'id' => $gameId,
                    'name' => $event['name'] ?? '',
                    'conference' => $confName,
                    'status' => $event['status']['type']['state'] ?? 'unknown'
                ];
            }
        }
    }
    
    // Small delay between conference API calls
    usleep(250000); // 0.25 second
}

$totalGames = count($allGames);
echo "<p class='info'>Found $totalGames total games across all conferences</p>";

// Process each game
echo "<h3>Processing Games...</h3>";
$processedCount = 0;

foreach ($allGames as $index => $game) {
    $processedCount++;
    
    // Update progress
    echo "<script>updateProgress($processedCount, $totalGames);</script>";
    
    if ($debug) {
        echo "<div class='game-info'>";
        echo "[$processedCount/$totalGames] {$game['conference']}: {$game['name']} (Status: {$game['status']})";
        echo "</div>";
    }
    
    // Process the game
    $updates = processGame($game['id'], $con, $debug);
    $apiCalls++;
    
    if ($updates > 0) {
        $totalUpdates += $updates;
        if (!isset($conferenceStats[$game['conference']])) {
            $conferenceStats[$game['conference']] = 0;
        }
        $conferenceStats[$game['conference']] += $updates;
    }
    
    // Flush output periodically
    if ($processedCount % 10 == 0) {
        ob_flush();
        flush();
    }
    
    // Rate limiting - delay between API calls
    // Adjust delay based on progress to avoid timeouts
    if ($processedCount < 50) {
        usleep(500000); // 0.5 second for first 50
    } else {
        usleep(250000); // 0.25 second for rest
    }
}

// Final progress update
echo "<script>updateProgress($totalGames, $totalGames);</script>";

// Display summary
echo "<div class='stats-summary'>";
echo "<h2>Update Summary</h2>";
echo "<p><strong>Total API Calls:</strong> $apiCalls</p>";
echo "<p><strong>Total Games Processed:</strong> $totalGames</p>";
echo "<p><strong>Total Team Records Updated:</strong> $totalUpdates</p>";

if (!empty($conferenceStats)) {
    echo "<h3>Updates by Conference:</h3>";
    echo "<ul>";
    foreach ($conferenceStats as $conf => $count) {
        $fullName = $conferenceFullNames[$conf] ?? $conf;
        echo "<li>$fullName: $count team updates</li>";
    }
    echo "</ul>";
}

echo "<p class='info'>Completed at: " . date('m/d/Y | g:iA') . " Eastern</p>";
echo "</div>";

// Display current leaders table
echo "<h2>Current NCAAF Team Leaders</h2>";

// Add filter by conference
$selectedConf = $_GET['conf'] ?? 'all';
echo "<form method='get' style='margin-bottom: 20px;'>";
echo "<label>Filter by Conference: </label>";
echo "<select name='conf' onchange='this.form.submit()'>";
echo "<option value='all'" . ($selectedConf == 'all' ? ' selected' : '') . ">All Conferences</option>";
foreach ($conferences as $confName => $confId) {
    $fullName = $conferenceFullNames[$confName] ?? $confName;
    echo "<option value='$confName'" . ($selectedConf == $confName ? ' selected' : '') . ">$fullName</option>";
}
echo "</select>";
if ($debug) echo "<input type='hidden' name='debug' value='1'>";
echo "</form>";

// Query based on filter
$whereClause = "WHERE league = 'NCAAF'";
if ($selectedConf != 'all') {
    $whereClause .= " AND conference = '" . mysqli_real_escape_string($con, $selectedConf) . "'";
}

$query = "SELECT * FROM team_leaders $whereClause ORDER BY conference, team_abbrev";
$result = mysqli_query($con, $query);

if ($result && mysqli_num_rows($result) > 0) {
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>Conference</th>";
    echo "<th>Team</th>";
    echo "<th>QB Leader</th>";
    echo "<th>RB Leader</th>";
    echo "<th>WR Leader</th>";
    echo "<th>Updated</th>";
    echo "</tr></thead><tbody>";
    
    while ($row = mysqli_fetch_assoc($result)) {
        // Parse leader strings to display names only
        $qbName = $rbName = $wrName = '-';
        
        if (!empty($row['qb_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['qb_leader'], $matches);
            $qbName = $matches[1] ?? '-';
        }
        if (!empty($row['rb_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['rb_leader'], $matches);
            $rbName = $matches[1] ?? '-';
        }
        if (!empty($row['wr_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['wr_leader'], $matches);
            $wrName = $matches[1] ?? '-';
        }
        
        echo "<tr>";
        echo "<td>{$row['conference']}</td>";
        echo "<td><strong>{$row['team_abbrev']}</strong> - {$row['team_name']}</td>";
        echo "<td title='" . htmlspecialchars($row['qb_leader'] ?? '') . "'>$qbName</td>";
        echo "<td title='" . htmlspecialchars($row['rb_leader'] ?? '') . "'>$rbName</td>";
        echo "<td title='" . htmlspecialchars($row['wr_leader'] ?? '') . "'>$wrName</td>";
        echo "<td>" . date('g:i a', strtotime($row['last_updated'])) . "</td>";
        echo "</tr>";
    }
    
    echo "</tbody></table>";
} else {
    echo "<p>No NCAAF teams found in database</p>";
}

echo "<h3>Options</h3>";
echo "<ul>";
echo "<li><a href='?debug=1'>Run with debug output</a></li>";
echo "<li><a href='?conf=SEC'>View specific conference (e.g., SEC)</a></li>";
echo "<li>Note: Processing all conferences may take several minutes due to API rate limiting</li>";
echo "</ul>";

echo "</div></body></html>";

mysqli_close($con);
?>