<?php
/**
 * NCAAF Team Population Script
 * Populates the existing team_leaders table with all NCAAF teams
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Conference mapping (excluding TOP-25)
$conferences = [
    'SEC' => 8,
    'BIG-10' => 5,
    'PAC-12' => 9,
    'ACC' => 1,
    'BIG-12' => 4,
    'MW' => 17,
    'MAC' => 15,
    'CUSA' => 12,
    'IVY' => 22,
    'Sun-Belt' => 37,
    'AAC' => 151
];

// Connect to database
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

echo "<!DOCTYPE html>
<html>
<head>
    <title>NCAAF Team Population</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        h1 { color: #333; border-bottom: 3px solid #cc0000; padding-bottom: 10px; }
        .conference-box { 
            margin: 20px 0; 
            padding: 15px; 
            border: 1px solid #ddd; 
            border-radius: 5px;
            background: #f9f9f9;
        }
        .conference-title { 
            font-size: 18px; 
            font-weight: bold; 
            color: #0066cc; 
            margin-bottom: 10px;
        }
        .team-list { 
            display: grid; 
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); 
            gap: 10px; 
            margin-top: 10px;
        }
        .team-item { 
            padding: 8px; 
            background: white; 
            border: 1px solid #e0e0e0; 
            border-radius: 3px;
            font-size: 12px;
        }
        .success { color: green; }
        .error { color: red; }
        .info { color: #0066cc; }
        .stats-summary {
            margin-top: 20px;
            padding: 15px;
            background: #e8f4f8;
            border-radius: 5px;
        }
    </style>
</head>
<body>
<div class='container'>";

echo "<h1>🏈 NCAAF Team Database Population</h1>";

// First, clear any existing NCAAF teams (optional - comment out if you want to keep existing)
$clearQuery = "DELETE FROM team_leaders WHERE league = 'NCAAF'";
if (mysqli_query($con, $clearQuery)) {
    echo "<p class='info'>Cleared existing NCAAF teams from database</p>";
}

$totalTeamsAdded = 0;
$conferenceTeamCounts = [];

// Process each conference
foreach ($conferences as $confName => $confId) {
    echo "<div class='conference-box'>";
    echo "<div class='conference-title'>$confName (Conference ID: $confId)</div>";
    
    // Fetch teams for this conference
    $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard?groups=" . $confId;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
    ]);
    
    $result = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);
    
    if ($httpCode != 200) {
        echo "<p class='error'>Failed to fetch data for $confName (HTTP $httpCode)</p>";
        continue;
    }
    
    $data = json_decode($result, true);
    
    // Extract unique teams from events
    $teams = [];
    
    if (isset($data['events']) && is_array($data['events'])) {
        foreach ($data['events'] as $event) {
            if (isset($event['competitions'][0]['competitors'])) {
                foreach ($event['competitions'][0]['competitors'] as $competitor) {
                    $team = $competitor['team'] ?? [];
                    $teamId = $team['id'] ?? '';
                    
                    if (!empty($teamId) && !isset($teams[$teamId])) {
                        $teams[$teamId] = [
                            'id' => $teamId,
                            'abbrev' => $team['abbreviation'] ?? '',
                            'name' => $team['name'] ?? '',
                            'location' => $team['location'] ?? '',
                            'display_name' => $team['displayName'] ?? ''
                        ];
                    }
                }
            }
        }
    }
    
    // If no events, try to get teams from the groups section
    if (empty($teams) && isset($data['groups'])) {
        // Alternative method to get teams if no games are scheduled
        echo "<p class='info'>No games found, checking groups data...</p>";
    }
    
    $conferenceTeamCount = 0;
    
    if (!empty($teams)) {
        echo "<div class='team-list'>";
        
        foreach ($teams as $teamId => $teamData) {
            // Insert into database
            $query = "INSERT INTO team_leaders 
                     (team_abbrev, team_name, conference, league) 
                     VALUES (?, ?, ?, 'NCAAF')
                     ON DUPLICATE KEY UPDATE 
                     team_name = VALUES(team_name),
                     conference = VALUES(conference)";
            
            $stmt = mysqli_prepare($con, $query);
            mysqli_stmt_bind_param($stmt, 'sss', 
                $teamData['abbrev'],
                $teamData['display_name'],
                $confName
            );
            
            if (mysqli_stmt_execute($stmt)) {
                echo "<div class='team-item'>";
                echo "<strong>{$teamData['abbrev']}</strong> - {$teamData['display_name']}";
                echo "</div>";
                $conferenceTeamCount++;
                $totalTeamsAdded++;
            } else {
                echo "<div class='team-item error'>Failed: {$teamData['abbrev']}</div>";
            }
            
            mysqli_stmt_close($stmt);
        }
        
        echo "</div>";
    } else {
        echo "<p class='error'>No teams found for this conference</p>";
    }
    
    $conferenceTeamCounts[$confName] = $conferenceTeamCount;
    echo "<p class='success'>Added $conferenceTeamCount teams from $confName</p>";
    echo "</div>";
}

// Display summary
echo "<div class='stats-summary'>";
echo "<h2>Population Summary</h2>";
echo "<p><strong>Total teams added:</strong> $totalTeamsAdded</p>";
echo "<h3>Teams by Conference:</h3>";
echo "<ul>";
foreach ($conferenceTeamCounts as $conf => $count) {
    echo "<li>$conf: $count teams</li>";
}
echo "</ul>";

// Show database totals
$countQuery = "SELECT 
                league, 
                COUNT(*) as total,
                COUNT(DISTINCT conference) as conf_count 
               FROM team_leaders 
               GROUP BY league";
$result = mysqli_query($con, $countQuery);

echo "<h3>Database Totals:</h3>";
echo "<table style='width: 100%; border-collapse: collapse;'>";
echo "<tr style='background: #f0f0f0;'><th style='padding: 8px; text-align: left;'>League</th><th>Total Teams</th><th>Conferences</th></tr>";
while ($row = mysqli_fetch_assoc($result)) {
    echo "<tr>";
    echo "<td style='padding: 8px; border-top: 1px solid #ddd;'>{$row['league']}</td>";
    echo "<td style='padding: 8px; border-top: 1px solid #ddd;'>{$row['total']}</td>";
    echo "<td style='padding: 8px; border-top: 1px solid #ddd;'>{$row['conf_count']}</td>";
    echo "</tr>";
}
echo "</table>";

echo "</div>";

// Display current NCAAF teams
echo "<h2>Current NCAAF Teams in Database</h2>";
$query = "SELECT team_abbrev, team_name, conference 
          FROM team_leaders 
          WHERE league = 'NCAAF' 
          ORDER BY conference, team_abbrev";
$result = mysqli_query($con, $query);

if ($result && mysqli_num_rows($result) > 0) {
    $currentConf = '';
    echo "<div style='columns: 2; column-gap: 30px;'>";
    
    while ($row = mysqli_fetch_assoc($result)) {
        if ($currentConf != $row['conference']) {
            if ($currentConf != '') echo "</ul>";
            $currentConf = $row['conference'];
            echo "<h4 style='margin-top: 20px; color: #cc0000; break-inside: avoid;'>$currentConf</h4>";
            echo "<ul style='break-inside: avoid;'>";
        }
        echo "<li><strong>{$row['team_abbrev']}</strong> - {$row['team_name']}</li>";
    }
    
    echo "</ul></div>";
}

echo "</div></body></html>";

mysqli_close($con);
?>