<?php
// Enable error reporting for debugging.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Create a PDO database connection.
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database Connection Error: " . $e->getMessage());
}

// --- Team Lookup Array ---
$teams = array(
    array("id" => "22", "abbreviation" => "ARI", "color" => "a40227", "alternateColor" => "ffffff", "name" => "CARDINALS"),
    array("id" => "1",  "abbreviation" => "ATL", "color" => "a71930", "alternateColor" => "ffffff", "name" => "FALCONS"),
    array("id" => "33", "abbreviation" => "BAL", "color" => "29126f", "alternateColor" => "ffffff", "name" => "RAVENS"),
    array("id" => "2",  "abbreviation" => "BUF", "color" => "00338d", "alternateColor" => "d50a0a", "name" => "BILLS"),
    array("id" => "29", "abbreviation" => "CAR", "color" => "0085ca", "alternateColor" => "ffffff", "name" => "PANTHERS"),
    array("id" => "3",  "abbreviation" => "CHI", "color" => "0b1c3a", "alternateColor" => "e64100", "name" => "BEARS"),
    array("id" => "4",  "abbreviation" => "CIN", "color" => "fb4f14", "alternateColor" => "ffffff", "name" => "BENGALS"),
    array("id" => "5",  "abbreviation" => "CLE", "color" => "472a08", "alternateColor" => "ff3c00", "name" => "BROWNS"),
    array("id" => "6",  "abbreviation" => "DAL", "color" => "002a5c", "alternateColor" => "b0b7bc", "name" => "COWBOYS"),
    array("id" => "7",  "abbreviation" => "DEN", "color" => "0a2343", "alternateColor" => "fc4c02", "name" => "BRONCOS"),
    array("id" => "8",  "abbreviation" => "DET", "color" => "0076b6", "alternateColor" => "bbbbbb", "name" => "LIONS"),
    array("id" => "9",  "abbreviation" => "GB",  "color" => "204e32", "alternateColor" => "ffb612", "name" => "PACKERS"),
    array("id" => "34", "abbreviation" => "HOU", "color" => "00143f", "alternateColor" => "c41230", "name" => "TEXANS"),
    array("id" => "11", "abbreviation" => "IND", "color" => "003b75", "alternateColor" => "ffffff", "name" => "COLTS"),
    array("id" => "30", "abbreviation" => "JAX", "color" => "007487", "alternateColor" => "d7a22a", "name" => "JAGUARS"),
    array("id" => "12", "abbreviation" => "KC",  "color" => "e31837", "alternateColor" => "ffb612", "name" => "CHIEFS"),
    array("id" => "13", "abbreviation" => "LV",  "color" => "000000", "alternateColor" => "a5acaf", "name" => "RAIDERS"),
    array("id" => "24", "abbreviation" => "LAC", "color" => "0080c6", "alternateColor" => "ffc20e", "name" => "CHARGERS"),
    array("id" => "14", "abbreviation" => "LAR", "color" => "003594", "alternateColor" => "ffd100", "name" => "RAMS"),
    array("id" => "15", "abbreviation" => "MIA", "color" => "008e97", "alternateColor" => "fc4c02", "name" => "DOLPHINS"),
    array("id" => "16", "abbreviation" => "MIN", "color" => "4f2683", "alternateColor" => "ffc62f", "name" => "VIKINGS"),
    array("id" => "17", "abbreviation" => "NE",  "color" => "002a5c", "alternateColor" => "c60c30", "name" => "PATRIOTS"),
    array("id" => "18", "abbreviation" => "NO",  "color" => "d3bc8d", "alternateColor" => "ffffff", "name" => "SAINTS"),
    array("id" => "19", "abbreviation" => "NYG", "color" => "003c7f", "alternateColor" => "c9243f", "name" => "GIANTS"),
    array("id" => "20", "abbreviation" => "NYJ", "color" => "115740", "alternateColor" => "ffffff", "name" => "JETS"),
    array("id" => "21", "abbreviation" => "PHI", "color" => "06424d", "alternateColor" => "a5acaf", "name" => "EAGLES"),
    array("id" => "23", "abbreviation" => "PIT", "color" => "000000", "alternateColor" => "ffb612", "name" => "STEELERS"),
    array("id" => "25", "abbreviation" => "SF",  "color" => "aa0000", "alternateColor" => "b3995d", "name" => "49ERS"),
    array("id" => "26", "abbreviation" => "SEA", "color" => "002a5c", "alternateColor" => "69be28", "name" => "SEAHAWKS"),
    array("id" => "27", "abbreviation" => "TB",  "color" => "bd1c36", "alternateColor" => "d9d9d9", "name" => "BUCCS"),
    array("id" => "10", "abbreviation" => "TEN", "color" => "4b92db", "alternateColor" => "c8102e", "name" => "TITANS"),
    array("id" => "28", "abbreviation" => "WSH", "color" => "5a1414", "alternateColor" => "ffb612", "name" => "CMDRS")
);

echo "<html><head><title>NFL Team Stats Update</title></head><body>";

// --- Desired Stats Mapping ---
// FIXED: Added 'category' parameter to ensure we get the right stat from the right category
$desiredStats = array(
    'pointsAllowed' => array(
        'displayName' => 'Points Allowed',
        'sort' => 'asc',  // Lower is better for defense
        'possibleNames' => array('pointsAllowed', 'defensePoints', 'defensivePoints'),
        'category' => 'defensive'  // Only get from defensive category
    ),
    'netYardsPerGame' => array(
        'displayName' => 'Net Yards Per Game',
        'sort' => 'desc',
        'possibleNames' => array('netYardsPerGame'),
        'category' => null  // Any category
    ),
    'netPassingYardsPerGame' => array(
        'displayName' => 'Net Passing Yards Per Game',
        'sort' => 'desc',
        'possibleNames' => array('netPassingYardsPerGame'),
        'category' => 'passing'  // Only from passing category
    ),
    'rushingYardsPerGame' => array(
        'displayName' => 'Rushing Yards Per Game',
        'sort' => 'desc',
        'possibleNames' => array('rushingYardsPerGame'),
        'category' => 'rushing'  // Only from rushing category
    ),
    'sacks' => array(
        'displayName' => 'Total Sacks',
        'sort' => 'desc',
        'possibleNames' => array('sacks', 'totalSacks'),
        'category' => 'defensive'  // CRITICAL FIX: Only get defensive sacks made, not sacks allowed from passing category
    ),
    'totalTouchdowns' => array(
        'displayName' => 'Total Touchdowns',
        'sort' => 'desc',
        'possibleNames' => array('totalTouchdowns'),
        'category' => null  // Any category
    )
);

// Map to database values
$dbTypeNames = array(
    'pointsAllowed' => 'Points Allowed',
    'netYardsPerGame' => 'Net Yards Per Game',
    'netPassingYardsPerGame' => 'Net Passing Yards Per Game',
    'rushingYardsPerGame' => 'Rushing Yards Per Game',
    'sacks' => 'Total Sacks',
    'totalTouchdowns' => 'Total Touchdowns'
);

// Category type abbreviations for the database
$categoryTypeAbbrevs = array(
    'pointsAllowed' => 'PTS',
    'netYardsPerGame' => 'YDS',
    'netPassingYardsPerGame' => 'YDS',
    'rushingYardsPerGame' => 'YDS',
    'sacks' => 'SACK',
    'totalTouchdowns' => 'TDS'
);

// Initialize leagueStats array
$leagueStats = array();
foreach ($desiredStats as $key => $config) {
    $leagueStats[$key] = array();
}

/**
 * Helper function to perform a cURL request and return decoded JSON.
 */
function curlFetch($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return false;
    }
    
    $decoded = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return false;
    }
    
    return $decoded;
}

// Debug: Track all stat names found
$allStatNames = array();

// Loop through each team and fetch its statistics
foreach ($teams as $team) {
    $teamId = $team['id'];
    $teamAbbr = $team['abbreviation'];
    $teamColor = $team['color'];
    $alternateColor = $team['alternateColor'];
    $teamName = $team['name'];
    
    // Extract Red, Green, Blue components from the main hex color
    $red = hexdec(substr($teamColor, 0, 2));
    $green = hexdec(substr($teamColor, 2, 2));
    $blue = hexdec(substr($teamColor, 4, 2));
    
    // Extract Red, Green, Blue components from the alternate hex color
    $redAlt = hexdec(substr($alternateColor, 0, 2));
    $greenAlt = hexdec(substr($alternateColor, 2, 2));
    $blueAlt = hexdec(substr($alternateColor, 4, 2));
    
    // Build the team's statistics URL
    $statsUrl = "http://sports.core.api.espn.com/v2/sports/football/leagues/nfl/seasons/2025/types/2/teams/{$teamId}/statistics/0?lang=en&region=us";
    $statsData = curlFetch($statsUrl);
    if (!$statsData) {
        echo "⚠️ Failed to fetch data for {$teamAbbr}<br>\n";
        continue;
    }
    
    if (!isset($statsData['splits']['categories']) || !is_array($statsData['splits']['categories'])) {
        echo "⚠️ No categories found for {$teamAbbr}<br>\n";
        continue;
    }
    
    // Debug: Only for first team, let's see what stats are available
    if ($teamId == "22") { // ARI
        echo "<h3>Debug: Available stats for {$teamAbbr}:</h3>";
        foreach ($statsData['splits']['categories'] as $categoryIndex => $category) {
            $categoryName = isset($category['name']) ? $category['name'] : 'Unknown';
            if (isset($category['stats']) && is_array($category['stats'])) {
                echo "<strong>Category: {$categoryName}</strong> - ";
                foreach ($category['stats'] as $stat) {
                    $statName = isset($stat['name']) ? $stat['name'] : 'Unknown';
                    $statValue = isset($stat['value']) ? $stat['value'] : 'N/A';
                    if ($statName === 'sacks') {
                        echo "<span style='color: red; font-weight: bold;'>{$statName} ({$statValue})</span> ";
                    } else {
                        echo $statName . " ";
                    }
                }
                echo "<br>";
            }
        }
        echo "<br>";
    }
    
    // Loop over each category and its stats
    foreach ($statsData['splits']['categories'] as $category) {
        if (!isset($category['stats']) || !is_array($category['stats'])) {
            continue;
        }
        
        // Get the category name for filtering
        $categoryName = isset($category['name']) ? $category['name'] : '';
        
        foreach ($category['stats'] as $stat) {
            if (!isset($stat['name'])) {
                continue;
            }
            $statName = $stat['name'];
            
            // Track all stat names for debugging
            if (!in_array($statName, $allStatNames)) {
                $allStatNames[] = $statName;
            }
            
            // Check if this stat matches one of our desired types
            foreach ($desiredStats as $statKey => $config) {
                if (in_array($statName, $config['possibleNames'])) {
                    // CRITICAL FIX: Check if category filtering is required
                    if (isset($config['category']) && $config['category'] !== null) {
                        if ($categoryName !== $config['category']) {
                            // Skip this stat if it's not in the required category
                            continue;
                        }
                    }
                    
                    // Check if we already have this team for this stat (prevent duplicates)
                    $alreadyExists = false;
                    foreach ($leagueStats[$statKey] as $existingEntry) {
                        if ($existingEntry['teamAbbr'] === $teamAbbr) {
                            $alreadyExists = true;
                            break;
                        }
                    }
                    
                    if ($alreadyExists) {
                        continue; // Skip duplicate
                    }
                    
                    // Get the numeric value
                    $valueNumeric = null;
                    if (isset($stat['value'])) {
                        $valueNumeric = floatval($stat['value']);
                    }
                    if ($valueNumeric === null) {
                        continue;
                    }
                    
                    $displayValue = isset($stat['displayValue']) ? $stat['displayValue'] : "N/A";
                    
                    // Special formatting for sacks - show as integer
                    if ($statKey === 'sacks') {
                        $displayValue = number_format($valueNumeric, 0);
                    }
                    
                    $leagueStats[$statKey][] = array(
                        'teamAbbr'     => $teamAbbr,
                        'teamColor'    => $teamColor,
                        'teamName'     => $teamName,
                        'value'        => $valueNumeric,
                        'displayValue' => $displayValue,
                        'red'          => $red,
                        'green'        => $green,
                        'blue'         => $blue,
                        'red_alt'      => $redAlt,
                        'green_alt'    => $greenAlt,
                        'blue_alt'     => $blueAlt
                    );
                    
                    // Debug: Show what we found
                    if ($statKey === 'sacks') {
                        echo "✅ <strong>Found {$statName} in '{$categoryName}' category for {$teamAbbr}: {$displayValue}</strong><br>\n";
                    } else {
                        echo "✅ Found {$statName} for {$teamAbbr}: {$displayValue}<br>\n";
                    }
                }
            }
        }
    }
}

// Debug: Show all stat names found
echo "<h3>Debug: All stat names found across all teams:</h3>";
sort($allStatNames);
foreach ($allStatNames as $statName) {
    echo $statName . " ";
}
echo "<br><br>";

// Debug: Show what we collected for each stat type
echo "<h3>Debug: Data collected per stat type:</h3>";
foreach ($leagueStats as $statKey => $entries) {
    echo "<strong>{$statKey}:</strong> " . count($entries) . " entries<br>";
    if ($statKey === 'sacks' && count($entries) > 0) {
        echo "<ul>";
        foreach (array_slice($entries, 0, 5) as $entry) {
            echo "<li>{$entry['teamAbbr']}: {$entry['displayValue']} sacks</li>";
        }
        echo "</ul>";
    }
}
echo "<br>";

// Helper function to sort stat entries
function sortEntries(&$entries, $order = 'desc') {
    usort($entries, function($a, $b) use ($order) {
        if ($order === 'desc') {
            return $b['value'] <=> $a['value'];
        } else {
            return $a['value'] <=> $b['value'];
        }
    });
}

// For each desired stat type, sort the entries and select the top 5
$top5Stats = array();
foreach ($leagueStats as $statType => $entries) {
    $order = $desiredStats[$statType]['sort'];
    sortEntries($entries, $order);
    // Slice to top 5; if fewer than 5, you'll see only those available
    $top5Stats[$statType] = array_slice($entries, 0, 5);
}

echo "<h2>Database Update Results:</h2>";

// Update the database rows - removed category_type
foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $categoryAbbrev = isset($categoryTypeAbbrevs[$statType]) ? $categoryTypeAbbrevs[$statType] : $statType;
    $rank = 0;
    foreach ($entries as $entry) {
        $rank++;
        
        // Update using the correct table name and columns - removed category_type
        $updateSQL = "UPDATE team_stats 
                      SET Abbrev = :abbrev, 
                          color = :color, 
                          value = :value, 
                          Red = :red, 
                          Green = :green, 
                          Blue = :blue, 
                          Red_ALT = :red_alt,
                          Green_ALT = :green_alt,
                          Blue_ALT = :blue_alt,
                          current_team = :current_team,
                          Type_Abbrev = :type_abbrev
                      WHERE rank = :rank 
                        AND type = :type 
                        AND league = 'NFL'";
        
        try {
            $stmt = $db->prepare($updateSQL);
            
            $params = array(
                ':abbrev'       => $entry['teamAbbr'],
                ':color'        => $entry['teamColor'],
                ':value'        => $entry['displayValue'],
                ':red'          => $entry['red'],
                ':green'        => $entry['green'],
                ':blue'         => $entry['blue'],
                ':red_alt'      => $entry['red_alt'],
                ':green_alt'    => $entry['green_alt'],
                ':blue_alt'     => $entry['blue_alt'],
                ':current_team' => $entry['teamName'],
                ':type_abbrev'  => $categoryAbbrev,
                ':type'         => $fullType,
                ':rank'         => $rank
            );
            
            $result = $stmt->execute($params);
            
            if (!$result) {
                $errorInfo = $stmt->errorInfo();
                echo "❌ Failed: Rank $rank for $fullType - Error: " . $errorInfo[2] . "<br>\n";
            } else {
                $rowCount = $stmt->rowCount();
                if ($rowCount > 0) {
                    echo "✅ Updated: Rank $rank for $fullType - {$entry['teamAbbr']} ({$entry['displayValue']}) - $rowCount row(s) updated<br>\n";
                } else {
                    echo "⚠️ No rows updated: Rank $rank for $fullType - {$entry['teamAbbr']} (check if row exists)<br>\n";
                }
            }
            
        } catch (PDOException $e) {
            echo "❌ Exception: Rank $rank for $fullType - " . $e->getMessage() . "<br>\n";
        }
    }
}

// Output an HTML table to display the top-5 results
echo "<br><h2>Top 5 Teams per Category:</h2>";
echo "<table border='1' cellpadding='5' cellspacing='0'>";
echo "<tr>
       <th>Rank</th>
       <th>League</th>
       <th>Category</th>
       <th>Team</th>
       <th>Value</th>
       <th>Updated</th>
      </tr>";

foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $rank = 1;
    foreach ($entries as $entry) {
        // Check if this row was updated in the database
        $checkSQL = "SELECT COUNT(*) FROM team_stats 
                     WHERE rank = :rank 
                       AND type = :type 
                       AND league = 'NFL' 
                       AND Abbrev = :abbrev";
        $checkStmt = $db->prepare($checkSQL);
        $checkStmt->execute([
            ':rank' => $rank,
            ':type' => $fullType,
            ':abbrev' => $entry['teamAbbr']
        ]);
        $exists = $checkStmt->fetchColumn() > 0;
        
        $statusIcon = $exists ? '✅' : '❌';
        
        echo "<tr>
                <td>{$rank}</td>
                <td>NFL</td>
                <td>" . htmlspecialchars($fullType) . "</td>
                <td>" . htmlspecialchars($entry['teamAbbr']) . " - " . htmlspecialchars($entry['teamName']) . "</td>
                <td>" . htmlspecialchars($entry['displayValue']) . "</td>
                <td style='text-align: center;'>{$statusIcon}</td>
              </tr>";
        $rank++;
    }
}
echo "</table>";
echo "</body></html>";
?>