<?php
/**
 * NFL Team Leaders Daily Updater
 * Updates current statistical leaders for all NFL teams
 * Uses game stats when available, falls back to season leaders
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Headshot directory
$headshot_dir = '/home/gscroll1/getStockPanel/Logos/HEADSHOTS/NFL/';

// Create directory if it doesn't exist
if (!file_exists($headshot_dir)) {
    mkdir($headshot_dir, 0755, true);
}

// Function to download headshot
function downloadHeadshot($url, $playerId, $headshot_dir) {
    if (empty($url) || empty($playerId)) {
        return false;
    }
    
    $extension = (strpos($url, '.jpg') !== false) ? 'jpg' : 'png';
    $filename = $headshot_dir . $playerId . '.' . $extension;
    
    if (file_exists($filename)) {
        return true;
    }
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $imageData = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200 && !empty($imageData)) {
        file_put_contents($filename, $imageData);
        return true;
    }
    
    return false;
}

// Function to format player stats string
function formatPlayerStats($athlete, $stats, $position, $teamAbbrev, $categoryName) {
    $playerId = $athlete['id'] ?? '';
    $fullName = $athlete['displayName'] ?? '';
    $jersey = $athlete['jersey'] ?? '';
    
    // Format stats based on category
    $statLine = '';
    switch($categoryName) {
        case 'passing':
            // C/ATT, YDS, TD, INT
            $statLine = $stats[0] . ', ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            if (!empty($stats[4]) && $stats[4] > 0) {
                $statLine .= ', ' . $stats[4] . ' INT';
            }
            break;
            
        case 'rushing':
            // CAR, YDS, TD
            $statLine = $stats[0] . ' CAR, ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            break;
            
        case 'receiving':
            // REC, YDS, TD
            $statLine = $stats[0] . ' REC, ' . $stats[1] . ' YDS';
            if (!empty($stats[3]) && $stats[3] > 0) {
                $statLine .= ', ' . $stats[3] . ' TD';
            }
            break;
    }
    
    return "DN: $fullName | STATS: $statLine | ID: $playerId | JERSEY: $jersey | POS: $position | TEAM: $teamAbbrev";
}

// Function to format leader from the leaders section
function formatLeaderStats($leaderData, $position, $teamAbbrev) {
    $athlete = $leaderData['athlete'] ?? [];
    $playerId = $athlete['id'] ?? '';
    $fullName = $athlete['displayName'] ?? '';
    $jersey = $athlete['jersey'] ?? '';
    $displayValue = $leaderData['displayValue'] ?? '';
    
    // Download headshot if available
    global $headshot_dir;
    if (isset($athlete['headshot']['href'])) {
        downloadHeadshot($athlete['headshot']['href'], $playerId, $headshot_dir);
    }
    
    return "DN: $fullName | STATS: $displayValue | ID: $playerId | JERSEY: $jersey | POS: $position | TEAM: $teamAbbrev";
}

// Function to process season leaders from summary API
function processSeasonLeaders($data, $con, $debug = false) {
    $updateCount = 0;
    
    // Check if leaders section exists
    if (!isset($data['leaders']) || !is_array($data['leaders'])) {
        if ($debug) echo "No leaders section found<br>";
        return 0;
    }
    
    // Process each team's leaders
    foreach ($data['leaders'] as $teamLeaders) {
        $team = $teamLeaders['team'] ?? [];
        $teamAbbrev = $team['abbreviation'] ?? '';
        
        if (empty($teamAbbrev)) continue;
        
        $leaders = [
            'qb' => '',
            'rb' => '',
            'wr' => ''
        ];
        
        // Process each stat category
        if (isset($teamLeaders['leaders']) && is_array($teamLeaders['leaders'])) {
            foreach ($teamLeaders['leaders'] as $category) {
                $categoryName = $category['name'] ?? '';
                
                if (!empty($category['leaders']) && isset($category['leaders'][0])) {
                    $leader = $category['leaders'][0];
                    
                    switch($categoryName) {
                        case 'passingYards':
                            $leaders['qb'] = formatLeaderStats($leader, 'QB', $teamAbbrev);
                            break;
                        case 'rushingYards':
                            $leaders['rb'] = formatLeaderStats($leader, 'RB', $teamAbbrev);
                            break;
                        case 'receivingYards':
                            $leaders['wr'] = formatLeaderStats($leader, 'WR', $teamAbbrev);
                            break;
                    }
                }
            }
        }
        
        // Update database for this team
        $query = "UPDATE team_leaders SET ";
        $updates = [];
        
        if (!empty($leaders['qb'])) {
            $updates[] = "qb_leader = '" . mysqli_real_escape_string($con, $leaders['qb']) . "'";
        }
        if (!empty($leaders['rb'])) {
            $updates[] = "rb_leader = '" . mysqli_real_escape_string($con, $leaders['rb']) . "'";
        }
        if (!empty($leaders['wr'])) {
            $updates[] = "wr_leader = '" . mysqli_real_escape_string($con, $leaders['wr']) . "'";
        }
        
        if (!empty($updates)) {
            $query .= implode(', ', $updates);
            $query .= " WHERE team_abbrev = '" . mysqli_real_escape_string($con, $teamAbbrev) . "'";
            
            if (mysqli_query($con, $query)) {
                $updateCount++;
                if ($debug) {
                    echo "✓ Updated {$teamAbbrev} (Season Leaders): ";
                    if (!empty($leaders['qb'])) echo "QB ✓ ";
                    if (!empty($leaders['rb'])) echo "RB ✓ ";
                    if (!empty($leaders['wr'])) echo "WR ✓ ";
                    echo "<br>";
                }
            } else {
                if ($debug) echo "✗ Failed to update {$teamAbbrev}: " . mysqli_error($con) . "<br>";
            }
        }
    }
    
    return $updateCount;
}

// Main function to process a single game
function processGame($gameId, $con, $debug = false) {
    global $headshot_dir;
    
    $summaryUrl = "http://site.api.espn.com/apis/site/v2/sports/football/nfl/summary?event=" . $gameId;
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $summaryUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $result = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);
    
    if ($httpCode != 200 || empty($result)) {
        if ($debug) echo "Failed to fetch game $gameId (HTTP $httpCode)<br>";
        return 0;
    }
    
    $data = json_decode($result, true);
    $updateCount = 0;
    
    // First try to use boxscore.players (game-specific stats)
    if (isset($data['boxscore']['players']) && !empty($data['boxscore']['players'])) {
        if ($debug) echo "Using game-specific stats from boxscore.players<br>";
        
        // Process each team in the game
        foreach ($data['boxscore']['players'] as $teamData) {
            $teamInfo = $teamData['team'] ?? [];
            $teamAbbrev = $teamInfo['abbreviation'] ?? '';
            
            if (empty($teamAbbrev)) continue;
            
            $leaders = [
                'qb' => '',
                'rb' => '',
                'wr' => ''
            ];
            
            // Process statistics
            if (isset($teamData['statistics'])) {
                foreach ($teamData['statistics'] as $category) {
                    $categoryName = $category['name'] ?? '';
                    
                    // Get the leading player in each category
                    if (isset($category['athletes']) && count($category['athletes']) > 0) {
                        $leader = $category['athletes'][0];
                        $athlete = $leader['athlete'] ?? [];
                        $stats = $leader['stats'] ?? [];
                        
                        if (!empty($athlete) && !empty($stats)) {
                            // Download headshot
                            if (isset($athlete['headshot']['href'])) {
                                downloadHeadshot($athlete['headshot']['href'], $athlete['id'], $headshot_dir);
                            }
                            
                            switch($categoryName) {
                                case 'passing':
                                    $leaders['qb'] = formatPlayerStats($athlete, $stats, 'QB', $teamAbbrev, $categoryName);
                                    break;
                                case 'rushing':
                                    $leaders['rb'] = formatPlayerStats($athlete, $stats, 'RB', $teamAbbrev, $categoryName);
                                    break;
                                case 'receiving':
                                    $leaders['wr'] = formatPlayerStats($athlete, $stats, 'WR', $teamAbbrev, $categoryName);
                                    break;
                            }
                        }
                    }
                }
            }
            
            // Update database for this team
            $query = "UPDATE team_leaders SET ";
            $updates = [];
            
            if (!empty($leaders['qb'])) {
                $updates[] = "qb_leader = '" . mysqli_real_escape_string($con, $leaders['qb']) . "'";
            }
            if (!empty($leaders['rb'])) {
                $updates[] = "rb_leader = '" . mysqli_real_escape_string($con, $leaders['rb']) . "'";
            }
            if (!empty($leaders['wr'])) {
                $updates[] = "wr_leader = '" . mysqli_real_escape_string($con, $leaders['wr']) . "'";
            }
            
            if (!empty($updates)) {
                $query .= implode(', ', $updates);
                $query .= " WHERE team_abbrev = '" . mysqli_real_escape_string($con, $teamAbbrev) . "'";
                
                if (mysqli_query($con, $query)) {
                    $updateCount++;
                    if ($debug) {
                        echo "✓ Updated {$teamAbbrev} (Game Stats): ";
                        if (!empty($leaders['qb'])) echo "QB ✓ ";
                        if (!empty($leaders['rb'])) echo "RB ✓ ";
                        if (!empty($leaders['wr'])) echo "WR ✓ ";
                        echo "<br>";
                    }
                }
            }
        }
    } else {
        // Fallback to season leaders
        if ($debug) echo "No boxscore.players found, using season leaders<br>";
        $updateCount = processSeasonLeaders($data, $con, $debug);
    }
    
    return $updateCount;
}

// MAIN EXECUTION
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

$totalStart = microtime(true);  // Track total execution time

// Set timezone
date_default_timezone_set('America/New_York');
$time_updated = date('m/d/Y | g:iA');

$debug = isset($_GET['debug']);

echo "<!DOCTYPE html>
<html>
<head>
    <title>NFL Team Leaders Update</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1400px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        h1 { color: #333; border-bottom: 3px solid #0066cc; padding-bottom: 10px; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th { background: #0066cc; color: white; padding: 10px; text-align: left; font-size: 12px; }
        td { padding: 8px; border-bottom: 1px solid #ddd; font-size: 11px; }
        tr:hover { background: #f9f9f9; }
        .afc { background: #ff4444; color: white; padding: 2px 6px; border-radius: 3px; }
        .nfc { background: #4444ff; color: white; padding: 2px 6px; border-radius: 3px; }
        .success { color: green; }
        .error { color: red; }
        .info { color: #0066cc; }
        .warning { color: orange; }
        .update-time { text-align: center; color: #666; margin: 10px 0; }
        .game-status { 
            padding: 2px 6px; 
            border-radius: 3px; 
            font-size: 10px; 
            font-weight: bold; 
        }
        .status-pre { background: #ffc107; color: #333; }
        .status-in { background: #28a745; color: white; }
        .status-post { background: #6c757d; color: white; }
    </style>
</head>
<body>
<div class='container'>";

echo "<h1>NFL Team Leaders Update</h1>";
echo "<div class='update-time'>Last Updated: $time_updated Eastern</div>";

// Fetch current games from scoreboard
$api_call = "http://site.api.espn.com/apis/site/v2/sports/football/nfl/scoreboard";
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
]);

$result = curl_exec($curl);
curl_close($curl);

$data = json_decode($result, true);

if (isset($data['events'])) {
    echo "<h2>Processing " . count($data['events']) . " Games</h2>";
    
    $totalUpdates = 0;
    $gamesByStatus = ['pre' => 0, 'in' => 0, 'post' => 0];
    
    foreach ($data['events'] as $event) {
        $gameId = $event['id'] ?? '';
        $gameName = $event['name'] ?? '';
        $gameStatus = $event['status']['type']['state'] ?? 'unknown';
        
        if (isset($gamesByStatus[$gameStatus])) {
            $gamesByStatus[$gameStatus]++;
        }
        
        if ($debug) {
            $statusClass = 'status-' . $gameStatus;
            echo "<h3>$gameName <span class='game-status $statusClass'>$gameStatus</span></h3>";
            echo "<p class='info'>Game ID: $gameId</p>";
        }
        
        $updates = processGame($gameId, $con, $debug);
        $totalUpdates += $updates;
    }
    
    echo "<h3>Summary</h3>";
    echo "<p class='success'>✓ Updated $totalUpdates team records</p>";
    echo "<p>Games by status: ";
    echo "<span class='game-status status-pre'>Not Started: {$gamesByStatus['pre']}</span> ";
    echo "<span class='game-status status-in'>In Progress: {$gamesByStatus['in']}</span> ";
    echo "<span class='game-status status-post'>Completed: {$gamesByStatus['post']}</span>";
    echo "</p>";
    echo "<p class='info'>Note: Using game-specific stats when available, otherwise using season leaders</p>";
}

// Display current leaders table
echo "<h2>Current Team Leaders</h2>";

$query = "SELECT * FROM team_leaders ORDER BY conference, team_name";
$result = mysqli_query($con, $query);

if ($result && mysqli_num_rows($result) > 0) {
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>Team</th>";
    echo "<th>Conf</th>";
    echo "<th>QB Leader</th>";
    echo "<th>RB Leader</th>";
    echo "<th>WR Leader</th>";
    echo "<th>Updated</th>";
    echo "</tr></thead><tbody>";
    
    while ($row = mysqli_fetch_assoc($result)) {
        $confClass = strtolower($row['conference']);
        
        // Parse leader strings to display names only in table
        $qbName = '';
        $rbName = '';
        $wrName = '';
        
        if (!empty($row['qb_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['qb_leader'], $matches);
            $qbName = $matches[1] ?? '';
        }
        if (!empty($row['rb_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['rb_leader'], $matches);
            $rbName = $matches[1] ?? '';
        }
        if (!empty($row['wr_leader'])) {
            preg_match('/DN: ([^|]+)/', $row['wr_leader'], $matches);
            $wrName = $matches[1] ?? '';
        }
        
        echo "<tr>";
        echo "<td><strong>{$row['team_abbrev']}</strong> - {$row['team_name']}</td>";
        echo "<td><span class='{$confClass}'>{$row['conference']}</span></td>";
        echo "<td title='" . htmlspecialchars($row['qb_leader']) . "'>$qbName</td>";
        echo "<td title='" . htmlspecialchars($row['rb_leader']) . "'>$rbName</td>";
        echo "<td title='" . htmlspecialchars($row['wr_leader']) . "'>$wrName</td>";
        echo "<td>" . date('g:i a', strtotime($row['last_updated'])) . "</td>";
        echo "</tr>";
    }
    
    echo "</tbody></table>";
}

echo "<h3>Options</h3>";
echo "<ul>";
echo "<li><a href='?debug=1'>Run with debug output</a></li>";
echo "<li>Game stats are used when available (completed or in-progress games)</li>";
echo "<li>Season leaders are used for upcoming games</li>";
echo "<li>Run daily to keep stats current</li>";
echo "</ul>";


// ==================== GENERATE TEAM LEADER IMAGES ====================
echo "<h2>Generating Team Leader Images</h2>";

// Call the TeamTopPlayers wrapper to generate all team leader images
$wrapperUrl = SERVER_URL . "Scrolling/App_Output/API/Stats/TeamTopPlayers.php?league=NFL";

echo "<p class='info'>Calling wrapper to generate bin files: <a href='$wrapperUrl' target='_blank'>$wrapperUrl</a></p>";

// Call the wrapper to generate all team leader images
$wrapperStart = microtime(true);

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $wrapperUrl,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 300, // 5 minutes timeout for processing all teams
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_SSL_VERIFYPEER => false,
]);

$wrapperResult = curl_exec($curl);
$wrapperHttpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
$wrapperError = curl_error($curl);
curl_close($curl);

$wrapperElapsed = microtime(true) - $wrapperStart;

if ($wrapperHttpCode == 200 && !empty($wrapperResult) && empty($wrapperError)) {
    echo "<p class='success'>✓ Successfully generated team leader images</p>";
    echo "<p>Wrapper execution time: " . round($wrapperElapsed, 2) . " seconds</p>";
    
    if ($debug) {
        echo "<details>";
        echo "<summary>Wrapper Response (click to expand)</summary>";
        echo "<div style='max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; margin: 10px 0;'>";
        echo $wrapperResult;
        echo "</div>";
        echo "</details>";
    }
} else {
    echo "<p class='error'>✗ Failed to generate team leader images</p>";
    echo "<p class='error'>HTTP Code: $wrapperHttpCode</p>";
    if (!empty($wrapperError)) {
        echo "<p class='error'>Error: " . htmlspecialchars($wrapperError) . "</p>";
    }
    
    if ($debug && !empty($wrapperResult)) {
        echo "<p>Response: " . htmlspecialchars(substr($wrapperResult, 0, 500)) . "...</p>";
    }
}

echo "<hr>";
echo "<h3>Complete Update Summary</h3>";
echo "<p class='success'>✓ Database updated with current team leaders</p>";
echo "<p class='success'>✓ Team leader images generated</p>";
echo "<p class='info'>Total process time: " . round(microtime(true) - $totalStart, 2) . " seconds</p>";

echo "</div></body></html>";

mysqli_close($con);
?>