<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Start HTML output with styling
echo "<!DOCTYPE html>
<html>
<head>
    <title>NHL Player Stats Update Log</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; background-color: #f5f5f5; }
        .section { background: white; padding: 15px; margin: 10px 0; border-radius: 5px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #28a745; font-weight: bold; }
        .error { color: #dc3545; font-weight: bold; }
        .info { color: #007bff; }
        .warning { color: #ffc107; }
        .player-update { margin-left: 20px; padding: 5px; background: #f8f9fa; margin-bottom: 5px; border-left: 3px solid #007bff; }
        .category-header { font-size: 18px; font-weight: bold; color: #333; margin-top: 20px; }
        .summary { background: #e7f3ff; padding: 15px; border-radius: 5px; margin-top: 20px; }
        h1 { color: #333; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        th, td { padding: 8px; text-align: left; border: 1px solid #ddd; }
        th { background-color: #007bff; color: white; }
    </style>
</head>
<body>
<h1>NHL Player Stats Database Update Process</h1>
";

echo "<div class='section'>";
echo "<h2>Process Started at: " . date('Y-m-d H:i:s') . "</h2>";
echo "</div>";

$teams = array(
    array("id" => "25", "abbreviation" => "ANA",   "color" => "fc4c02", "alternateColor" => "ffffff", "name" => "DUCKS"),
    array("id" => "1",  "abbreviation" => "BOS",   "color" => "fdbb30", "alternateColor" => "000000", "name" => "BRUINS"),
    array("id" => "2",  "abbreviation" => "BUF",   "color" => "00468b", "alternateColor" => "fdb71a", "name" => "SABRES"),
    array("id" => "3",  "abbreviation" => "CGY",   "color" => "dd1a32", "alternateColor" => "ffffff", "name" => "FLAMES"),
    array("id" => "7",  "abbreviation" => "CAR",   "color" => "e30426", "alternateColor" => "ffffff", "name" => "HURRICANES"),
    array("id" => "4",  "abbreviation" => "CHI",   "color" => "e31937", "alternateColor" => "ffffff", "name" => "B-HAWK"),
    array("id" => "17", "abbreviation" => "COL",   "color" => "860038", "alternateColor" => "005ea3", "name" => "AVALANCHE"),
    array("id" => "29", "abbreviation" => "CBJ",   "color" => "002d62", "alternateColor" => "c8102e", "name" => "B-JACKET"),
    array("id" => "9",  "abbreviation" => "DAL",   "color" => "20864c", "alternateColor" => "ffffff", "name" => "STARS"),
    array("id" => "5",  "abbreviation" => "DET",   "color" => "e30526", "alternateColor" => "ffffff", "name" => "RED-WINGS"),
    array("id" => "6",  "abbreviation" => "EDM",   "color" => "00205b", "alternateColor" => "ff4c00", "name" => "OILERS"),
    array("id" => "26", "abbreviation" => "FLA",   "color" => "e51937", "alternateColor" => "003e7e", "name" => "PANTHERS"),
    array("id" => "8",  "abbreviation" => "LA",    "color" => "a2aaad", "alternateColor" => "ffffff", "name" => "KINGS"),
    array("id" => "30", "abbreviation" => "MIN",   "color" => "124734", "alternateColor" => "a6192e", "name" => "WILD"),
    array("id" => "10", "abbreviation" => "MTL",   "color" => "c41230", "alternateColor" => "013a81", "name" => "CANADIENS"),
    array("id" => "27", "abbreviation" => "NSH",   "color" => "fdba31", "alternateColor" => "002b5c", "name" => "PREDATORS"),
    array("id" => "11", "abbreviation" => "NJ",    "color" => "e30b2b", "alternateColor" => "ffffff", "name" => "DEVILS"),
    array("id" => "12", "abbreviation" => "NYI",   "color" => "00529b", "alternateColor" => "f47d31", "name" => "ISLANDERS"),
    array("id" => "13", "abbreviation" => "NYR",   "color" => "0056ae", "alternateColor" => "e51937", "name" => "RANGERS"),
    array("id" => "14", "abbreviation" => "OTT",   "color" => "dd1a32", "alternateColor" => "b79257", "name" => "SENATORS"),
    array("id" => "15", "abbreviation" => "PHI",   "color" => "fe5823", "alternateColor" => "ffffff", "name" => "FLYERS"),
    array("id" => "16", "abbreviation" => "PIT",   "color" => "fdb71a", "alternateColor" => "ffffff", "name" => "PENGUINS"),
    array("id" => "18", "abbreviation" => "SJ",    "color" => "00788a", "alternateColor" => "ffffff", "name" => "SHARKS"),
    array("id" => "124292", "abbreviation" => "SEA", "color" => "001628", "alternateColor" => "99d9d9", "name" => "KRAKEN"),
    array("id" => "19", "abbreviation" => "STL",   "color" => "00468b", "alternateColor" => "fdb71a", "name" => "BLUES"),
    array("id" => "20", "abbreviation" => "TB",    "color" => "003e7e", "alternateColor" => "ffffff", "name" => "LIGHTNING"),
    array("id" => "21", "abbreviation" => "TOR",   "color" => "003e7e", "alternateColor" => "ffffff", "name" => "MAPLE LEAFS"),
    array("id" => "129764", "abbreviation" => "UTAH", "color" => "231f20", "alternateColor" => "ffffff", "name" => "UTAH"),
    array("id" => "22", "abbreviation" => "VAN",   "color" => "003e7e", "alternateColor" => "008752", "name" => "CANUCKS"),
    array("id" => "37", "abbreviation" => "VGK",   "color" => "344043", "alternateColor" => "b4975a", "name" => "G-KNIGHT"),
    array("id" => "23", "abbreviation" => "WSH",   "color" => "d71830", "alternateColor" => "0b1f41", "name" => "CAPITALS"),
    array("id" => "28", "abbreviation" => "WPG",   "color" => "002d62", "alternateColor" => "a8a9ad", "name" => "JETS")
);

function curlFetch($url) {
    echo "<span class='info'>Fetching: " . htmlspecialchars($url) . "</span><br>";
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        echo "<span class='error'>cURL error when fetching " . htmlspecialchars($url) . ": " . curl_error($ch) . "</span><br>";
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $data = json_decode($response, true);
    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        echo "<span class='warning'>JSON decode error: " . json_last_error_msg() . "</span><br>";
        return false;
    }
    echo "<span class='success'>✓ Successfully fetched data</span><br>";
    return $data;
}

// Fetch leaders data
echo "<div class='section'>";
echo "<h3>Step 1: Fetching NHL Leaders Data</h3>";
$leadersUrl = "https://sports.core.api.espn.com/v2/sports/hockey/leagues/nhl/seasons/2026/types/2/leaders?lang=en&region=us";
$leadersData = curlFetch($leadersUrl);
if (!$leadersData) {
    die("<span class='error'>Error fetching leaders data. Process terminated.</span></div></body></html>");
}
echo "<span class='success'>Leaders data fetched successfully!</span>";
echo "</div>";

// Database connection
echo "<div class='section'>";
echo "<h3>Step 2: Connecting to Database</h3>";
require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "<span class='success'>✓ Database connection established successfully!</span><br>";
    echo "Database: <strong>$database_gscroll</strong> on host: <strong>$host_gscroll</strong>";
} catch (PDOException $e) {
    die("<span class='error'>Database Connection Error: " . $e->getMessage() . "</span></div></body></html>");
}
echo "</div>";

// Prepare update statement
$updateSQL = "UPDATE player_stats
              SET Player = :player,
                  Stat_Value = :stat_value,
                  Team = :team,
                  Red = :red,
                  Green = :green,
                  Blue = :blue,
                  Red_ALT = :red_alt,
                  Green_ALT = :green_alt,
                  Blue_ALT = :blue_alt,
                  ID = :id
              WHERE Rank = :rank AND Category = :category AND league = 'NHL'";
$stmt = $db->prepare($updateSQL);

// Initialize tracking variables
$playerIds = [];
$totalUpdates = 0;
$updateErrors = 0;
$categoriesProcessed = 0;
$updateDetails = [];

// Process categories and update database
echo "<div class='section'>";
echo "<h3>Step 3: Processing Categories and Updating Database</h3>";

if (isset($leadersData['categories'])) {
    $totalCategories = count($leadersData['categories']);
    echo "<p>Found <strong>$totalCategories</strong> categories to process</p>";

    foreach ($leadersData['categories'] as $categoryIndex => $category) {
        $categoriesProcessed++;
        $categoryName = isset($category['displayName']) ? $category['displayName'] : 'N/A';
        $categoryName = str_replace('/', '-', $categoryName);

        echo "<div class='category-header'>Category $categoriesProcessed/$totalCategories: $categoryName</div>";

        if (isset($category['leaders']) && is_array($category['leaders'])) {
            $rank = 1;
            $categoryUpdates = 0;

            foreach ($category['leaders'] as $leader) {
                if ($rank > 5) break;

                echo "<div class='player-update'>";
                echo "<strong>Rank $rank:</strong> ";

                $displayValue = $leader['displayValue'] ?? 'N/A';

                // Fetch athlete data
                $athleteUrl = $leader['athlete']['$ref'] ?? '';
                if ($athleteUrl) {
                    $athleteData = curlFetch($athleteUrl);
                    $playerName = $athleteData['fullName'] ?? 'N/A';
                } else {
                    $playerName = 'N/A';
                    echo "<span class='warning'>No athlete URL available</span><br>";
                }

                // Extract player ID
                $playerId = 'N/A';
                if (preg_match('/athletes\/(\d+)/', $leader['athlete']['$ref'], $matches)) {
                    $playerId = $matches[1];
                    $playerIds[] = $playerId;
                }

                // Fetch team data
                $teamUrl = $leader['team']['$ref'] ?? '';
                if ($teamUrl) {
                    $teamData = curlFetch($teamUrl);
                    $teamAbbr = $teamData['abbreviation'] ?? 'N/A';
                } else {
                    $teamAbbr = 'N/A';
                    echo "<span class='warning'>No team URL available</span><br>";
                }

                // Get team colors
                $red = $green = $blue = $redAlt = $greenAlt = $blueAlt = 0;
                $teamFound = false;
                foreach ($teams as $t) {
                    if ($t['abbreviation'] === $teamAbbr) {
                        $teamFound = true;
                        $main = $t['color'];
                        $alt = $t['alternateColor'];
                        $red = hexdec(substr($main, 0, 2));
                        $green = hexdec(substr($main, 2, 2));
                        $blue = hexdec(substr($main, 4, 2));
                        $redAlt = hexdec(substr($alt, 0, 2));
                        $greenAlt = hexdec(substr($alt, 2, 2));
                        $blueAlt = hexdec(substr($alt, 4, 2));
                        break;
                    }
                }

                if (!$teamFound && $teamAbbr !== 'N/A') {
                    echo "<span class='warning'>Team colors not found for: $teamAbbr</span><br>";
                }

                // Execute database update
                try {
                    $stmt->execute([
                        ':player'     => $playerName,
                        ':stat_value' => $displayValue,
                        ':team'       => $teamAbbr,
                        ':red'        => $red,
                        ':green'      => $green,
                        ':blue'       => $blue,
                        ':red_alt'    => $redAlt,
                        ':green_alt'  => $greenAlt,
                        ':blue_alt'   => $blueAlt,
                        ':id'         => $playerId,
                        ':rank'       => $rank,
                        ':category'   => $categoryName,
                    ]);

                    $rowsAffected = $stmt->rowCount();
                    $totalUpdates++;
                    $categoryUpdates++;

                    echo "<span class='success'>✓ Updated:</span> ";
                    echo "<strong>$playerName</strong> ($teamAbbr) - ";
                    echo "Value: <strong>$displayValue</strong> - ";
                    echo "Player ID: $playerId - ";
                    echo "Colors: RGB($red,$green,$blue) / Alt RGB($redAlt,$greenAlt,$blueAlt) - ";
                    echo "Rows affected: <strong>$rowsAffected</strong>";

                    // Store update details
                    $updateDetails[] = [
                        'category' => $categoryName,
                        'rank' => $rank,
                        'player' => $playerName,
                        'team' => $teamAbbr,
                        'value' => $displayValue,
                        'rows_affected' => $rowsAffected
                    ];

                } catch (PDOException $e) {
                    $updateErrors++;
                    echo "<span class='error'>✗ Update failed for $playerName: " . $e->getMessage() . "</span>";
                }

                echo "</div>";
                $rank++;
            }

            echo "<p class='info'>Category complete: $categoryUpdates players updated</p>";
        } else {
            echo "<span class='warning'>No leaders found for this category</span><br>";
        }
    }
} else {
    echo "<span class='error'>No categories found in the data!</span>";
}
echo "</div>";

// Headshot Processing
echo "<div class='section'>";
echo "<h3>Step 4: Processing Player Headshots</h3>";

$downloadedCount = 0;
$skippedCount = 0;
$errorCount = 0;
$processedCount = 0;

$totalHeadshots = count($playerIds);
echo "<p>Processing headshots for <strong>$totalHeadshots</strong> unique players</p>";

foreach ($playerIds as $index => $playerId) {
    $playerNumber = $index + 1;
    echo "<div style='margin-left: 20px; padding: 3px;'>";
    echo "[$playerNumber/$totalHeadshots] Player ID: <strong>$playerId</strong> - ";

    // Build the save path using the player ID
    $savePath = SERVER_USERNAME . "getStockPanel/Logos/HEADSHOTS/NHL/" . $playerId . ".png";
    // Build the headshot image URL dynamically
    $imageUrl = "https://a.espncdn.com/combiner/i?img=/i/headshots/nhl/players/full/" . $playerId . ".png&w=30&h=30";

    if (!file_exists($savePath)) {
        echo "<span class='info'>Downloading headshot...</span> ";
        $ch = curl_init($imageUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $data = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($data !== false && $httpCode == 200) {
            file_put_contents($savePath, $data);
            $downloadedCount++;
            echo "<span class='success'>✓ Downloaded</span> ";
        } else {
            echo "<span class='error'>✗ Download failed (HTTP $httpCode)</span> ";
            $errorCount++;
            echo "</div>";
            continue;
        }
    } else {
        $skippedCount++;
        echo "<span class='info'>⊘ Already exists</span> ";
    }

    // Remove near-white background from the PNG
    echo "<span class='info'>Processing image...</span> ";
    $img = @imagecreatefrompng($savePath);
    if ($img !== false) {
        imagealphablending($img, false);
        imagesavealpha($img, true);
        $width = imagesx($img);
        $height = imagesy($img);
        $pixelsModified = 0;

        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $argb = imagecolorat($img, $x, $y);
                $r = ($argb >> 16) & 0xFF;
                $g = ($argb >> 8) & 0xFF;
                $b = $argb & 0xFF;
                if ($r > 220 && $g > 220 && $b > 220) {
                    $transparentColor = imagecolorallocatealpha($img, 255, 255, 255, 127);
                    imagesetpixel($img, $x, $y, $transparentColor);
                    $pixelsModified++;
                }
            }
        }
        imagepng($img, $savePath);
        imagedestroy($img);
        $processedCount++;
        echo "<span class='success'>✓ Processed ($pixelsModified pixels made transparent)</span>";
    } else {
        echo "<span class='error'>✗ Could not open PNG for processing</span>";
        $errorCount++;
    }
    echo "</div>";
}

echo "<div class='summary' style='margin-top: 10px;'>";
echo "<strong>Headshot Summary:</strong><br>";
echo "• Downloaded: <strong>$downloadedCount</strong> new images<br>";
echo "• Skipped: <strong>$skippedCount</strong> existing images<br>";
echo "• Processed: <strong>$processedCount</strong> images for transparency<br>";
echo "• Errors: <strong>$errorCount</strong> failed operations";
echo "</div>";
echo "</div>";

// Final Summary
echo "<div class='section summary'>";
echo "<h2>Process Summary</h2>";
echo "<table>";
echo "<tr><th>Metric</th><th>Value</th></tr>";
echo "<tr><td>Process Completed At</td><td>" . date('Y-m-d H:i:s') . "</td></tr>";
echo "<tr><td>Categories Processed</td><td><strong>$categoriesProcessed</strong></td></tr>";
echo "<tr><td>Database Updates</td><td><strong>$totalUpdates</strong></td></tr>";
echo "<tr><td>Update Errors</td><td><strong>$updateErrors</strong></td></tr>";
echo "<tr><td>Unique Players</td><td><strong>$totalHeadshots</strong></td></tr>";
echo "<tr><td>New Headshots Downloaded</td><td><strong>$downloadedCount</strong></td></tr>";
echo "<tr><td>Existing Headshots</td><td><strong>$skippedCount</strong></td></tr>";
echo "<tr><td>Image Processing Errors</td><td><strong>$errorCount</strong></td></tr>";
echo "</table>";

// Show update details summary
if (count($updateDetails) > 0) {
    echo "<h3>Database Update Details</h3>";
    echo "<table>";
    echo "<tr><th>Category</th><th>Rank</th><th>Player</th><th>Team</th><th>Value</th><th>Rows Affected</th></tr>";
    foreach ($updateDetails as $detail) {
        $rowClass = $detail['rows_affected'] > 0 ? 'success' : 'warning';
        echo "<tr>";
        echo "<td>{$detail['category']}</td>";
        echo "<td>{$detail['rank']}</td>";
        echo "<td>{$detail['player']}</td>";
        echo "<td>{$detail['team']}</td>";
        echo "<td>{$detail['value']}</td>";
        echo "<td class='$rowClass'>{$detail['rows_affected']}</td>";
        echo "</tr>";
    }
    echo "</table>";
}

echo "</div>";

echo "</body></html>";
?>