<?php
// Enable error reporting for debugging.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

// Create a PDO database connection.
try {
    $db = new PDO("mysql:host=$host_gscroll;dbname=$database_gscroll", $username_gscroll, $password_gscroll);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database Connection Error: " . $e->getMessage());
}

// --- Team Lookup Array ---
$teams = array(
    array("id" => "25", "abbreviation" => "ANA",   "color" => "fc4c02", "alternateColor" => "ffffff", "name" => "DUCKS"),
    array("id" => "1",  "abbreviation" => "BOS",   "color" => "fdbb30", "alternateColor" => "000000", "name" => "BRUINS"),
    array("id" => "2",  "abbreviation" => "BUF",   "color" => "00468b", "alternateColor" => "fdb71a", "name" => "SABRES"),
    array("id" => "3",  "abbreviation" => "CGY",   "color" => "dd1a32", "alternateColor" => "ffffff", "name" => "FLAMES"),
    array("id" => "7",  "abbreviation" => "CAR",   "color" => "e30426", "alternateColor" => "ffffff", "name" => "HURRICANES"),
    array("id" => "4",  "abbreviation" => "CHI",   "color" => "e31937", "alternateColor" => "ffffff", "name" => "B-HAWK"),
    array("id" => "17", "abbreviation" => "COL",   "color" => "860038", "alternateColor" => "005ea3", "name" => "AVALANCHE"),
    array("id" => "29", "abbreviation" => "CBJ",   "color" => "002d62", "alternateColor" => "c8102e", "name" => "B-JACKET"),
    array("id" => "9",  "abbreviation" => "DAL",   "color" => "20864c", "alternateColor" => "ffffff", "name" => "STARS"),
    array("id" => "5",  "abbreviation" => "DET",   "color" => "e30526", "alternateColor" => "ffffff", "name" => "RED-WINGS"),
    array("id" => "6",  "abbreviation" => "EDM",   "color" => "00205b", "alternateColor" => "ff4c00", "name" => "OILERS"),
    array("id" => "26", "abbreviation" => "FLA",   "color" => "e51937", "alternateColor" => "003e7e", "name" => "PANTHERS"),
    array("id" => "8",  "abbreviation" => "LA",    "color" => "a2aaad", "alternateColor" => "ffffff", "name" => "KINGS"),
    array("id" => "30", "abbreviation" => "MIN",   "color" => "124734", "alternateColor" => "a6192e", "name" => "WILD"),
    array("id" => "10", "abbreviation" => "MTL",   "color" => "c41230", "alternateColor" => "013a81", "name" => "CANADIENS"),
    array("id" => "27", "abbreviation" => "NSH",   "color" => "fdba31", "alternateColor" => "002b5c", "name" => "PREDATORS"),
    array("id" => "11", "abbreviation" => "NJ",    "color" => "e30b2b", "alternateColor" => "ffffff", "name" => "DEVILS"),
    array("id" => "12", "abbreviation" => "NYI",   "color" => "00529b", "alternateColor" => "f47d31", "name" => "ISLANDERS"),
    array("id" => "13", "abbreviation" => "NYR",   "color" => "0056ae", "alternateColor" => "e51937", "name" => "RANGERS"),
    array("id" => "14", "abbreviation" => "OTT",   "color" => "dd1a32", "alternateColor" => "b79257", "name" => "SENATORS"),
    array("id" => "15", "abbreviation" => "PHI",   "color" => "fe5823", "alternateColor" => "ffffff", "name" => "FLYERS"),
    array("id" => "16", "abbreviation" => "PIT",   "color" => "fdb71a", "alternateColor" => "ffffff", "name" => "PENGUINS"),
    array("id" => "18", "abbreviation" => "SJ",    "color" => "00788a", "alternateColor" => "ffffff", "name" => "SHARKS"),
    array("id" => "124292", "abbreviation" => "SEA", "color" => "001628", "alternateColor" => "99d9d9", "name" => "KRAKEN"),
    array("id" => "19", "abbreviation" => "STL",   "color" => "00468b", "alternateColor" => "fdb71a", "name" => "BLUES"),
    array("id" => "20", "abbreviation" => "TB",    "color" => "003e7e", "alternateColor" => "ffffff", "name" => "LIGHTNING"),
    array("id" => "21", "abbreviation" => "TOR",   "color" => "003e7e", "alternateColor" => "ffffff", "name" => "MAPLE LEAFS"),
    array("id" => "129764", "abbreviation" => "UTAH", "color" => "231f20", "alternateColor" => "ffffff", "name" => "UTAH"), // Placeholder if used
    array("id" => "22", "abbreviation" => "VAN",   "color" => "003e7e", "alternateColor" => "008752", "name" => "CANUCKS"),
    array("id" => "37", "abbreviation" => "VGK",   "color" => "344043", "alternateColor" => "b4975a", "name" => "G-KNIGHT"),
    array("id" => "23", "abbreviation" => "WSH",   "color" => "d71830", "alternateColor" => "0b1f41", "name" => "CAPITALS"),
    array("id" => "28", "abbreviation" => "WPG",   "color" => "002d62", "alternateColor" => "a8a9ad", "name" => "JETS")
);

echo "<html><head><title>NHL Team Stats Update</title></head><body>";

// --- Desired Stats Mapping ---
// For each desired stat type, list possible abbreviations and the sort order.
$desiredStats = array(
    'GF/G' => array(
        'possibleAbbreviations' => array('GF/G', 'avgGoalsFor'),
        'sort' => 'desc'
    ),
    'GA/G' => array(
        'possibleAbbreviations' => array('GA/G', 'GAA', 'avgGoalsAgainst'),
        'sort' => 'asc'
    ),
    'PP%'  => array(
        'possibleAbbreviations' => array('PP%', 'powerPlayPct', 'PCT'),
        'sort' => 'desc'
    ),
);

// Map abbreviated keys to full type names and database values.
$fullTypeNames = array(
    'GF/G' => "Goals Per Game",
    'GA/G' => "Goals Against Per Game",
    'PP%'  => "Power Play %",
);

// Shorter names for the database 'type' column
$dbTypeNames = array(
    'GF/G' => "Goals Per Game",
    'GA/G' => "Goals Against Per Game", 
    'PP%'  => "Power Play %",
);

// Category type abbreviations for the database
$categoryTypeAbbrevs = array(
    'GF/G' => "GF",
    'GA/G' => "GAA",
    'PP%'  => "PP%",
);

// Initialize leagueStats array.
$leagueStats = array();
foreach ($desiredStats as $key => $config) {
    $leagueStats[$key] = array();
}

/**
 * Helper function to perform a cURL request and return decoded JSON.
 */
function curlFetch($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        echo "cURL error: " . curl_error($ch) . "<br>\n";
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return false;
    }
    
    $decoded = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return false;
    }
    
    return $decoded;
}

// Debug: Track all stat names found
$allStatNames = array();

// Loop through each team and fetch its statistics.
foreach ($teams as $team) {
    $teamId          = $team['id'];
    $teamAbbr        = $team['abbreviation'];
    $teamColor       = $team['color'];
    $alternateColor  = $team['alternateColor'];
    $teamName        = $team['name'];
    
    // Extract Red, Green, Blue components from the main hex color.
    $red   = hexdec(substr($teamColor, 0, 2));
    $green = hexdec(substr($teamColor, 2, 2));
    $blue  = hexdec(substr($teamColor, 4, 2));
    
    // Extract Red, Green, Blue components from the alternate hex color.
    $redAlt   = hexdec(substr($alternateColor, 0, 2));
    $greenAlt = hexdec(substr($alternateColor, 2, 2));
    $blueAlt  = hexdec(substr($alternateColor, 4, 2));
    
    // Build the team's statistics URL.
    $statsUrl = "http://sports.core.api.espn.com/v2/sports/hockey/leagues/nhl/seasons/2026/types/2/teams/{$teamId}/statistics/0?lang=en&region=us";
    $statsData = curlFetch($statsUrl);
    if (!$statsData) {
        echo "⚠️ Failed to fetch data for {$teamAbbr}<br>\n";
        continue;
    }
    
    if (!isset($statsData['splits']['categories']) || !is_array($statsData['splits']['categories'])) {
        echo "⚠️ No categories found for {$teamAbbr}<br>\n";
        continue;
    }
    
    // Debug: Only for first team, let's see what stats are available
    if ($teamId == "25") { // ANA
        echo "<h3>Debug: Available stats for {$teamAbbr}:</h3>";
        foreach ($statsData['splits']['categories'] as $categoryIndex => $category) {
            if (isset($category['stats']) && is_array($category['stats'])) {
                echo "<strong>Category {$categoryIndex}:</strong> ";
                foreach ($category['stats'] as $stat) {
                    $statName = isset($stat['abbreviation']) ? $stat['abbreviation'] : 'Unknown';
                    echo $statName . " ";
                }
                echo "<br>";
            }
        }
        echo "<br>";
    }
    
    // Loop over each category and its stats.
    foreach ($statsData['splits']['categories'] as $category) {
        if (!isset($category['stats']) || !is_array($category['stats'])) {
            continue;
        }
        foreach ($category['stats'] as $stat) {
            if (!isset($stat['abbreviation'])) {
                continue;
            }
            $abbr = $stat['abbreviation'];
            
            // Track all stat names for debugging
            if (!in_array($abbr, $allStatNames)) {
                $allStatNames[] = $abbr;
            }
            
            // Check if this stat matches one of our desired types.
            foreach ($desiredStats as $statKey => $config) {
                if (in_array($abbr, $config['possibleAbbreviations'])) {
                    // Prefer perGameValue if available; otherwise, use value.
                    $valueNumeric = null;
                    if (isset($stat['perGameValue'])) {
                        $valueNumeric = floatval($stat['perGameValue']);
                    } elseif (isset($stat['value'])) {
                        $valueNumeric = floatval($stat['value']);
                    }
                    if ($valueNumeric === null) {
                        continue;
                    }
                    
                    $displayValue = isset($stat['perGameDisplayValue']) 
                                    ? $stat['perGameDisplayValue'] 
                                    : (isset($stat['displayValue']) ? $stat['displayValue'] : "N/A");
                    
                    // Debug: Show what we found for PP%
                    if ($statKey == 'PP%') {
                        echo "🔍 PP% Debug for {$teamAbbr}: abbr={$abbr}, value={$valueNumeric}, displayValue={$displayValue}<br>\n";
                        if (isset($stat['value'])) echo "   Raw value: {$stat['value']}<br>\n";
                        if (isset($stat['perGameValue'])) echo "   Per game value: {$stat['perGameValue']}<br>\n";
                        if (isset($stat['displayValue'])) echo "   Display value: {$stat['displayValue']}<br>\n";
                        if (isset($stat['perGameDisplayValue'])) echo "   Per game display: {$stat['perGameDisplayValue']}<br>\n";
                    }
                    
                    $leagueStats[$statKey][] = array(
                        'teamAbbr'     => $teamAbbr,
                        'teamColor'    => $teamColor,
                        'teamName'     => $teamName,
                        'value'        => $valueNumeric,
                        'displayValue' => $displayValue,
                        'red'          => $red,
                        'green'        => $green,
                        'blue'         => $blue,
                        'red_alt'      => $redAlt,
                        'green_alt'    => $greenAlt,
                        'blue_alt'     => $blueAlt
                    );
                    
                    // Debug: Show what we found
                    echo "✅ Found {$abbr} for {$teamAbbr}: {$displayValue}<br>\n";
                }
            }
        }
    }
}

// Helper function to sort stat entries.
function sortEntries(&$entries, $order = 'desc') {
    usort($entries, function($a, $b) use ($order) {
        if ($order === 'desc') {
            return $b['value'] <=> $a['value'];
        } else {
            return $a['value'] <=> $b['value'];
        }
    });
}

// For each desired stat type, sort the entries and select the top 5.
$top5Stats = array();
foreach ($leagueStats as $statType => $entries) {
    $order = $desiredStats[$statType]['sort'];
    sortEntries($entries, $order);
    // Slice to top 5; if fewer than 5, you'll see only those available.
    $top5Stats[$statType] = array_slice($entries, 0, 5);
}

echo "<h2>Database Update Results:</h2>";

// Update the pre-inserted rows using correct table and column names - removed category_type
foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $categoryAbbrev = isset($categoryTypeAbbrevs[$statType]) ? $categoryTypeAbbrevs[$statType] : $statType;
    $rank = 0;
    foreach ($entries as $entry) {
        $rank++;
        
        // Update using the correct table name and columns - removed category_type
        $updateSQL = "UPDATE team_stats 
                      SET Abbrev = :abbrev, 
                          color = :color, 
                          value = :value, 
                          Red = :red, 
                          Green = :green, 
                          Blue = :blue, 
                          Red_ALT = :red_alt,
                          Green_ALT = :green_alt,
                          Blue_ALT = :blue_alt,
                          current_team = :current_team,
                          Type_Abbrev = :type_abbrev
                      WHERE rank = :rank 
                        AND type = :type 
                        AND league = 'NHL'";
        
        try {
            $stmt = $db->prepare($updateSQL);
            
            $params = array(
                ':abbrev'       => $entry['teamAbbr'],
                ':color'        => $entry['teamColor'],
                ':value'        => $entry['displayValue'],
                ':red'          => $entry['red'],
                ':green'        => $entry['green'],
                ':blue'         => $entry['blue'],
                ':red_alt'      => $entry['red_alt'],
                ':green_alt'    => $entry['green_alt'],
                ':blue_alt'     => $entry['blue_alt'],
                ':current_team' => $entry['teamName'],
                ':type_abbrev'  => $categoryAbbrev,
                ':type'         => $fullType,
                ':rank'         => $rank
            );
            
            $result = $stmt->execute($params);
            
            if (!$result) {
                $errorInfo = $stmt->errorInfo();
                echo "❌ Failed: Rank $rank for $fullType - Error: " . $errorInfo[2] . "<br>\n";
            } else {
                $rowCount = $stmt->rowCount();
                if ($rowCount > 0) {
                    echo "✅ Updated: Rank $rank for $fullType - {$entry['teamAbbr']} ({$entry['displayValue']}) - $rowCount row(s) updated<br>\n";
                } else {
                    echo "⚠️ No rows updated: Rank $rank for $fullType - {$entry['teamAbbr']} (check if row exists)<br>\n";
                }
            }
            
        } catch (PDOException $e) {
            echo "❌ Exception: Rank $rank for $fullType - " . $e->getMessage() . "<br>\n";
        }
    }
}

// Optionally, output an HTML table to display the updated top-5 results.
echo "<br><h2>Top 5 Teams per Category:</h2>";
echo "<table border='1' cellpadding='5' cellspacing='0'>";
echo "<tr>
       <th>Rank</th>
       <th>League</th>
       <th>Category</th>
       <th>Team</th>
       <th>Value</th>
       <th>Updated</th>
      </tr>";

foreach ($top5Stats as $statType => $entries) {
    $fullType = isset($dbTypeNames[$statType]) ? $dbTypeNames[$statType] : $statType;
    $rank = 1;
    foreach ($entries as $entry) {
        // Check if this row was updated in the database
        $checkSQL = "SELECT COUNT(*) FROM team_stats 
                     WHERE rank = :rank 
                       AND type = :type 
                       AND league = 'NHL' 
                       AND Abbrev = :abbrev";
        $checkStmt = $db->prepare($checkSQL);
        $checkStmt->execute([
            ':rank' => $rank,
            ':type' => $fullType,
            ':abbrev' => $entry['teamAbbr']
        ]);
        $exists = $checkStmt->fetchColumn() > 0;
        
        $statusIcon = $exists ? '✅' : '❌';
        
        echo "<tr>
                <td>{$rank}</td>
                <td>NHL</td>
                <td>" . htmlspecialchars($fullType) . "</td>
                <td>" . htmlspecialchars($entry['teamAbbr']) . " - " . htmlspecialchars($entry['teamName']) . "</td>
                <td>" . htmlspecialchars($entry['displayValue']) . "</td>
                <td style='text-align: center;'>{$statusIcon}</td>
              </tr>";
        $rank++;
    }
}
echo "</table>";
echo "</body></html>";
?>