<?php
//************************************************//
// Intelligent CFB Score Updater - Cron Version with Broadcast
// Runs every 2 minutes, only updates when games are active
// Updated as of 08/23/2025 - ENHANCED VERSION WITH FORCE ALL OPTION + BROADCAST
// - Fixed UTC handling for West Coast games
// - Added 2-minute score update protection
// - Added score alert validation
// - Added smart field preservation for scoring plays
// - Added FORCE ALL option to update all conferences
// - Added broadcast/channel information parsing
//************************************************//

// Check for force update parameter (can be passed via GET, POST, or command line)
$forceUpdateAll = false;

// Check GET parameter - both 'input' and 'update' for compatibility
if ((isset($_GET['input']) && strtoupper($_GET['input']) === 'ALL') ||
    (isset($_GET['update']) && strtoupper($_GET['update']) === 'ALL')) {
    $forceUpdateAll = true;
}

// Check POST parameter - both 'input' and 'update' for compatibility
if ((isset($_POST['input']) && strtoupper($_POST['input']) === 'ALL') ||
    (isset($_POST['update']) && strtoupper($_POST['update']) === 'ALL')) {
    $forceUpdateAll = true;
}

// Check command line argument (for cron usage)
if (php_sapi_name() === 'cli' && $argc > 1) {
    if (strtoupper($argv[1]) === 'ALL') {
        $forceUpdateAll = true;
    }
}

// Conference mappings
$conferences = [
    'TOP-25' => null,
    'SEC' => 8,
    'BIG-10' => 5,
    'PAC-12' => 9,
    'ACC' => 1,
    'BIG-12' => 4,
    'MW' => 17,
    'MAC' => 15,
    'CUSA' => 12,
    'IVY' => 22,
    'Sun-Belt' => 37,
    'AAC' => 151
];

require_once '../config.php';
session_start();

// Start output buffering for HTML report
ob_start();

// HTML header
echo '<!DOCTYPE html>
<html>
<head>
    <title>CFB Score Update Report - Enhanced with Broadcast</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .header { background: #2c3e50; color: white; padding: 20px; border-radius: 5px; }
        .conference-section { background: white; margin: 15px 0; padding: 15px; border-radius: 5px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        .active { border-left: 5px solid #27ae60; }
        .inactive { border-left: 5px solid #95a5a6; }
        .skipped { border-left: 5px solid #e74c3c; }
        .forced { border-left: 5px solid #f39c12; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px; margin: 10px 0; }
        .stat-box { background: #ecf0f1; padding: 10px; border-radius: 3px; }
        .timestamp { color: #7f8c8d; font-size: 0.9em; }
        .summary { background: #3498db; color: white; padding: 15px; border-radius: 5px; margin: 20px 0; }
        h2 { margin: 0 0 10px 0; }
        .game-status { display: inline-block; padding: 2px 8px; border-radius: 3px; font-size: 0.85em; }
        .status-active { background: #27ae60; color: white; }
        .status-done { background: #95a5a6; color: white; }
        .status-upcoming { background: #3498db; color: white; }
        .status-skipped { background: #f39c12; color: white; }
        table { border-collapse: collapse; width: 100%; margin: 10px 0; }
        th, td { padding: 5px; text-align: left; border: 1px solid #ddd; font-size: 11px; }
        th { background: #34495e; color: white; }
        .timing-restriction { background: #fff3cd; border: 1px solid #ffc107; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .score-alert { background: #ff5252; color: white; font-weight: bold; padding: 2px 5px; }
        .force-mode-notice { background: #f39c12; color: white; padding: 15px; margin: 10px 0; border-radius: 5px; font-weight: bold; }
        .broadcast-cell { background: #e3f2fd; font-weight: bold; }
    </style>
</head>
<body>';

date_default_timezone_set('America/New_York');
$start_time = microtime(true);
$current_time = date('m/d/Y | g:i:s A');
$current_timestamp = date('Y-m-d H:i:s'); // For database storage

echo "<div class='header'>";
echo "<h1>CFB Score Update Report - Enhanced with Broadcast</h1>";
echo "<div class='timestamp'>Generated: $current_time</div>";
echo "<div class='timestamp'>Features: UTC Fix | 2-Min Score Protection | Alert Validation | Smart Field Preservation | Broadcast Info</div>";
echo "</div>";

// Display force mode notice if active
if ($forceUpdateAll) {
    echo "<div class='force-mode-notice'>";
    echo "⚠️ FORCE UPDATE MODE ACTIVE - All conferences will be updated regardless of game status";
    echo "</div>";
}

// Track statistics
$total_api_calls = 0;
$total_games_updated = 0;
$total_games_skipped_timing = 0;
$conferences_with_active_games = [];
$conferences_skipped = [];
$conferences_forced = [];
$errors = [];
$score_alerts_fixed = 0;

// Database connection
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Check if new columns exist, if not, add them
$columns_to_check = [
    'lastUpdatedTime' => 'DATETIME DEFAULT NULL',
    'lastPlay' => 'VARCHAR(255) DEFAULT NULL',
    'lastPlayerID' => 'VARCHAR(50) DEFAULT NULL',
    'lastPlayerName' => 'VARCHAR(100) DEFAULT NULL',
    'lastPlayerPos' => 'VARCHAR(10) DEFAULT NULL',
    'lastPlayerTeamID' => 'VARCHAR(50) DEFAULT NULL',
    'jersey' => 'VARCHAR(10) DEFAULT NULL',
    'driveDesc' => 'VARCHAR(255) DEFAULT NULL',
    'broadcast' => 'VARCHAR(50) DEFAULT NULL'
];

foreach ($columns_to_check as $column => $definition) {
    $check_column = "SHOW COLUMNS FROM `sports_list` LIKE '$column'";
    $column_result = mysqli_query($con, $check_column);
    if (mysqli_num_rows($column_result) == 0) {
        $add_column = "ALTER TABLE `sports_list` ADD COLUMN `$column` $definition";
        mysqli_query($con, $add_column);
        echo "<div class='stat-box'>Added column: $column</div>";
    }
}

//////////////////// HELPER FUNCTIONS ////////////

function checkActiveGames($con, $conference) {
    $query = "SELECT COUNT(*) as active_count 
              FROM sports_list 
              WHERE Conference = '$conference' 
              AND (Game_Played = 'IN_SESSION' OR Game_Played = 'HALF_TIME')";
    
    $result = mysqli_query($con, $query);
    if ($result) {
        $row = mysqli_fetch_assoc($result);
        return $row['active_count'] > 0;
    }
    return false;
}

function getUpcomingGames($con, $conference, $minutes_ahead = 5) {  // Changed to minutes, default 5
    // Use gmdate for proper UTC timestamps
    $current_time = gmdate('Y-m-d\TH:i:s\Z');
    $future_time = gmdate('Y-m-d\TH:i:s\Z', time() + ($minutes_ahead * 60));  // Changed to minutes
    
    $query = "SELECT COUNT(*) as upcoming_count 
              FROM sports_list 
              WHERE Conference = '$conference' 
              AND Game_Date BETWEEN '$current_time' AND '$future_time'
              AND Game_Played = 'NOT_STARTED'";
    
    $result = mysqli_query($con, $query);
    if ($result) {
        $row = mysqli_fetch_assoc($result);
        return $row['upcoming_count'] > 0;
    }
    return false;
}

// NEW FUNCTION: Check if update should be skipped due to timing restriction
function shouldSkipUpdate($game_num, $conference, $scoreValue, $con) {
    // Only check timing for scores (scoreValue not empty and not NO-SCORE)
    if (empty($scoreValue) || $scoreValue == 'NO-SCORE' || $scoreValue == '') {
        return false;
    }
    
    // Query the last update time for this game
    $check_query = "SELECT lastUpdatedTime, scoreValue FROM sports_list 
                   WHERE Game = $game_num AND Conference = '$conference'";
    $result = mysqli_query($con, $check_query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $lastUpdate = $row['lastUpdatedTime'];
        $lastScoreValue = $row['scoreValue'];
        
        // If there's a previous update time and the last scoreValue was also a score
        if (!empty($lastUpdate) && !empty($lastScoreValue) && 
            $lastScoreValue != 'NO-SCORE' && $lastScoreValue != '') {
            
            // Calculate time difference in seconds
            $lastUpdateTime = strtotime($lastUpdate);
            $currentTime = time();
            $timeDifference = $currentTime - $lastUpdateTime;
            
            // If less than 2 minutes (120 seconds), skip update
            if ($timeDifference < 120) {
                return ['skip' => true, 'time_remaining' => 120 - $timeDifference];
            }
        }
    }
    
    return ['skip' => false, 'time_remaining' => 0];
}

// NEW FUNCTION: Validate and potentially override score alerts
function validateScoreAlert($scoreVal, $lastPlayText, $scoringTeamAbbrev) {
    global $score_alerts_fixed;
    
    // Convert to lowercase for comparison
    $lastPlayLower = strtolower($lastPlayText);
    
    // Check for touchdown (scoreVal = 6)
    if ($scoreVal == 6) {
        // Check if lastPlay contains touchdown-related keywords
        if (strpos($lastPlayLower, 'touchdown') === false && 
            strpos($lastPlayLower, 'td') === false &&
            strpos($lastPlayLower, 'score') === false) {
            $score_alerts_fixed++;
            return strtoupper($scoringTeamAbbrev) . " TOUCHDOWN!";
        }
    }
    
    // Check for field goal (scoreVal = 3)
    elseif ($scoreVal == 3) {
        // Check if lastPlay contains field goal-related keywords
        if (strpos($lastPlayLower, 'field goal') === false && 
            strpos($lastPlayLower, 'fg') === false &&
            strpos($lastPlayLower, 'kick') === false) {
            $score_alerts_fixed++;
            return strtoupper($scoringTeamAbbrev) . " FIELD GOAL!";
        }
    }
    
    // Check for extra point (scoreVal = 1)
    elseif ($scoreVal == 1) {
        // Check if lastPlay contains extra point-related keywords
        if (strpos($lastPlayLower, 'extra point') === false && 
            strpos($lastPlayLower, 'pat') === false &&
            strpos($lastPlayLower, 'xp') === false &&
            strpos($lastPlayLower, 'kick') === false) {
            $score_alerts_fixed++;
            return strtoupper($scoringTeamAbbrev) . " EXTRA POINT!";
        }
    }
    
    // Return original text if validation passes
    return $lastPlayText;
}

function getGameStatus($status_data) {
    $type_id = $status_data['type']['id'] ?? '';
    $completed = $status_data['type']['completed'] ?? false;
    $state = $status_data['type']['state'] ?? '';
    
    // Use the explicit status from ESPN
    if ($type_id == '3' || $completed === true || $state == 'post') {
        return 'GAME_DONE';
    } elseif ($type_id == '2' || $state == 'in') {
        // Check for halftime
        $period = $status_data['period'] ?? 0;
        $clock = $status_data['displayClock'] ?? '';
        if ($period == 2 && $clock == '0:00') {
            return 'HALF_TIME';
        }
        return 'IN_SESSION';
    } elseif ($type_id == '1' || $state == 'pre') {
        return 'NOT_STARTED';
    }
    
    // Fallback to basic check
    return 'NOT_STARTED';
}

function clean($string) {
    $string = str_replace(' ', '-', $string);
    return preg_replace('/[^A-Za-z0-9\-]/', '', $string);
}

function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

function parseTeamColor($team) {
    $color_hex = $team['color'] ?? '000000';
    
    // If color is black, try alternate color
    if ($color_hex == '000000') {
        $color_hex = $team['alternateColor'] ?? '000000';
    }
    
    $color_hex = clean($color_hex);
    
    // Convert hex to RGB
    $split = str_split($color_hex, 2);
    $r = hexdec($split[0] ?? '00');
    $g = hexdec($split[1] ?? '00');
    $b = hexdec($split[2] ?? '00');
    
    // Add leading zeros
    $r = addZeros($r);
    $g = addZeros($g);
    $b = addZeros($b);
    
    // Check if color is too dark
    if ($r < 30 && $g < 30 && $b < 30) {
        $r = 255;
        $g = 255;
        $b = 255;
    }
    
    return [
        'rgb' => sprintf('"%s,%s,%s"', $r, $g, $b),
        'r' => $r,
        'g' => $g,
        'b' => $b
    ];
}

// Main processing loop
$games_with_timing_restrictions = [];

foreach ($conferences as $conference_name => $conference_id) {
    echo "<div class='conference-section ";
    
    // Check if there are active games or games starting soon
    $has_active_games = checkActiveGames($con, $conference_name);
    $has_upcoming_games = getUpcomingGames($con, $conference_name);
    
    // Decide whether to update
    $should_update = $has_active_games || $has_upcoming_games || $forceUpdateAll;
    
    if (!$should_update) {
        echo "inactive'>";
        echo "<h2>$conference_name - SKIPPED</h2>";
        echo "<div class='timestamp'>No active or upcoming games (checked database)</div>";
        $conferences_skipped[] = $conference_name;
        echo "</div>";
        continue;
    }
    
    // Determine the CSS class and status text
    if ($forceUpdateAll && !$has_active_games && !$has_upcoming_games) {
        echo "forced'>";
        echo "<h2>$conference_name - FORCE UPDATING</h2>";
        $conferences_forced[] = $conference_name;
    } else {
        echo "active'>";
        echo "<h2>$conference_name - UPDATING</h2>";
    }
    
    // Build API URL
    if ($conference_name == "TOP-25") {
        $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard";
    } else {
        $api_url = "http://site.api.espn.com/apis/site/v2/sports/football/college-football/scoreboard?groups=" . $conference_id;
    }
    
    echo "<div class='stats'>";
    echo "<div class='stat-box'>API URL: " . htmlspecialchars($api_url) . "</div>";
    if ($forceUpdateAll && !$has_active_games && !$has_upcoming_games) {
        echo "<div class='stat-box' style='background: #f39c12; color: white;'>Forced Update (No Active Games)</div>";
    }
    
    // Make API call
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
    ]);
    
    $result = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if ($err) {
        echo "<div class='stat-box' style='background: #e74c3c; color: white;'>Error: $err</div>";
        $errors[] = "$conference_name: $err";
        echo "</div></div>";
        continue;
    }
    
    $total_api_calls++;
    
    // Parse JSON
    $data = json_decode($result, true);
    if (!$data || !isset($data['events'])) {
        echo "<div class='stat-box' style='background: #e74c3c; color: white;'>Invalid JSON response</div>";
        $errors[] = "$conference_name: Invalid JSON";
        echo "</div></div>";
        continue;
    }
    
    $events = $data['events'];
    $games_count = count($events);
    $games_in_progress = 0;
    $games_completed = 0;
    $games_upcoming = 0;
    $games_skipped_in_conference = 0;
    
    // Small table for games in this conference
    echo "</div>"; // Close stats div
    echo "<details><summary>Game Details (click to expand)</summary>";
    echo "<table>";
    echo "<tr><th>Game</th><th>Status</th><th>Score</th><th>Clock</th><th>Broadcast</th><th>Update</th><th>Last Play</th></tr>";
    
    // Process each game
    for ($i = 0; $i < $games_count; $i++) {
        $event = $events[$i];
        $competition = $event['competitions'][0] ?? [];
        
        // Get game time and status
        $game_date = $event['date'] ?? '';
        $status = $competition['status'] ?? [];
        $clock = $status['displayClock'] ?? '0:00';
        $period = $status['period'] ?? 0;
        $game_status = getGameStatus($status);
        
        // Convert to Eastern time for display
        $game_time_et = date("g:i a", strtotime($game_date));
        
        // Extract broadcast information
        $broadcast = '';
        if (isset($competition['broadcasts']) && is_array($competition['broadcasts']) && !empty($competition['broadcasts'])) {
            $broadcast_data = $competition['broadcasts'][0] ?? [];
            if (isset($broadcast_data['names']) && is_array($broadcast_data['names']) && !empty($broadcast_data['names'])) {
                $broadcast = $broadcast_data['names'][0] ?? '';
            }
        }
        
        // Get competitors (teams)
        $competitors = $competition['competitors'] ?? [];
        $team1 = null;
        $team2 = null;
        
        foreach ($competitors as $competitor) {
            if ($competitor['homeAway'] == 'home') {
                $team1 = $competitor;
            } else {
                $team2 = $competitor;
            }
        }
        
        if (!$team1 || !$team2) {
            continue; // Skip if teams not found
        }
        
        // Extract team data
        $team1_data = $team1['team'] ?? [];
        $team2_data = $team2['team'] ?? [];
        
        $team1_name = $team1_data['name'] ?? '';
        $team2_name = $team2_data['name'] ?? '';
        $team1_location = $team1_data['location'] ?? '';
        $team2_location = $team2_data['location'] ?? '';
        $team1_abbrev = $team1_data['abbreviation'] ?? '';
        $team2_abbrev = $team2_data['abbreviation'] ?? '';
        $team1_id = $team1_data['id'] ?? '';
        $team2_id = $team2_data['id'] ?? '';
        
        // Fix Texas A&M abbreviation
        if ($team1_abbrev == "TA&M") $team1_abbrev = "TAMU";
        if ($team2_abbrev == "TA&M") $team2_abbrev = "TAMU";
        
        // Get records
        $team1_records = $team1['records'] ?? [];
        $team2_records = $team2['records'] ?? [];
        $team1_record = '0-0';
        $team2_record = '0-0';
        
        foreach ($team1_records as $record) {
            if ($record['type'] == 'total') {
                $team1_record = $record['summary'] ?? '0-0';
                break;
            }
        }
        
        foreach ($team2_records as $record) {
            if ($record['type'] == 'total') {
                $team2_record = $record['summary'] ?? '0-0';
                break;
            }
        }
        
        // Get scores
        $team1_score = ($game_status != 'NOT_STARTED') ? ($team1['score'] ?? '0') : '0';
        $team2_score = ($game_status != 'NOT_STARTED') ? ($team2['score'] ?? '0') : '0';
        
        // Get colors
        $team1_colors = parseTeamColor($team1_data);
        $team2_colors = parseTeamColor($team2_data);
        
        // Get odds (betting lines)
        $odds = $competition['odds'][0] ?? [];
        $favorite_team = '';
        $favorite_points = '';
        $overUnder = '';
        
        if (!empty($odds)) {
            $overUnder = $odds['overUnder'] ?? '';
            $details = $odds['details'] ?? '';
            
            if (strpos($details, $team1_abbrev) !== false) {
                $favorite_team = $team1_abbrev;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            } elseif (strpos($details, $team2_abbrev) !== false) {
                $favorite_team = $team2_abbrev;
                preg_match('/([+-]?\d+\.?\d*)/', $details, $matches);
                $favorite_points = $matches[1] ?? '';
            }
        }
        
        // Get situation data (in-game info)
        $situation = $competition['situation'] ?? [];
        $who_has_ball = '';
        $short_distance_text = '';
        $pos_text = '';
        $is_red_zone = '';
        $score_value = '';
        
        // Initialize last player variables
        $lastPlayerID = '';
        $lastPlayerName = '';
        $lastPlayerPos = '';
        $lastPlayerTeamID = '';
        $lastPlayerJersey = '';
        $driveDescription = '';
        $last_play_text = '';
        
        // Track scoreVal for timing check
        $scoreVal = 0;
        
        if ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            $lastPlay = $situation['lastPlay'] ?? [];
            $scoreVal = $lastPlay['scoreValue'] ?? 0;
            
            // Determine score type
            if ($scoreVal == 0) {
                $score_value = "NO-SCORE";
            } elseif ($scoreVal == 1) {
                $score_value = "EXTRA-POINT";
            } elseif ($scoreVal == 2) {
                $score_value = "TWO-PT";
            } elseif ($scoreVal == 3) {
                $score_value = "FIELD-GOAL-GOOD";
            } elseif ($scoreVal == 6) {
                $score_value = "TOUCHDOWN";
            } else {
                $score_value = "SCORE-" . $scoreVal;
            }
            
            // Extract last play text
            if (isset($lastPlay['text'])) {
                $last_play_text = $lastPlay['text'];
                // Sanitize the text
                $last_play_text = strip_tags($last_play_text);
                $last_play_text = trim($last_play_text);
                // Limit length if needed
                if (strlen($last_play_text) > 255) {
                    $last_play_text = substr($last_play_text, 0, 252) . '...';
                }
            }
            
            // SCORE ALERT VALIDATION
            if (($scoreVal == 3 || $scoreVal == 6 || $scoreVal == 1)) {
                // Get the team that has possession (likely the scoring team)
                $scoring_team_abbrev = '';
                $possession_team_id = $lastPlay['team']['id'] ?? '';
                if ($possession_team_id == $team1_id) {
                    $scoring_team_abbrev = $team1_abbrev;
                } elseif ($possession_team_id == $team2_id) {
                    $scoring_team_abbrev = $team2_abbrev;
                }
                
                // If we have a scoring team, validate the play text
                if (!empty($scoring_team_abbrev)) {
                    $last_play_text = validateScoreAlert($scoreVal, $last_play_text, $scoring_team_abbrev);
                }
            }
            
            // Extract drive description
            if (isset($lastPlay['drive']['description'])) {
                $driveDescription = $lastPlay['drive']['description'];
                // Sanitize the drive description
                $driveDescription = strip_tags($driveDescription);
                $driveDescription = trim($driveDescription);
                // Limit length if needed
                if (strlen($driveDescription) > 100) {
                    $driveDescription = substr($driveDescription, 0, 97) . '...';
                }
            }
            
            // Extract player information from athletesInvolved
            if (isset($lastPlay['athletesInvolved']) && is_array($lastPlay['athletesInvolved']) && count($lastPlay['athletesInvolved']) > 0) {
                $firstPlayer = $lastPlay['athletesInvolved'][0];
                
                // Extract player details
                $lastPlayerID = $firstPlayer['id'] ?? '';
                $lastPlayerName = $firstPlayer['fullName'] ?? $firstPlayer['displayName'] ?? '';
                $lastPlayerPos = $firstPlayer['position'] ?? '';
                $lastPlayerJersey = $firstPlayer['jersey'] ?? '';
                
                // Extract player team ID
                if (isset($firstPlayer['team']['id'])) {
                    $lastPlayerTeamID = $firstPlayer['team']['id'];
                }
            }
            
            // Get possession
            $possession_team_id = $lastPlay['team']['id'] ?? '';
            if ($possession_team_id == $team1_id) {
                $who_has_ball = $team1_abbrev;
            } elseif ($possession_team_id == $team2_id) {
                $who_has_ball = $team2_abbrev;
            }
            
            // Get down and distance
            $short_distance_text = $situation['shortDownDistanceText'] ?? '';
            $downDistanceText = $situation['downDistanceText'] ?? '';
            
            if ($downDistanceText) {
                if (strpos($downDistanceText, ' at ') !== false) {
                    $pos_text = substr($downDistanceText, strpos($downDistanceText, ' at ') + 4);
                    $pos_text = str_replace(' ', '-', trim($pos_text));
                }
            }
            
            // Red zone
            $is_red_zone = ($situation['isRedZone'] ?? false) ? 'true' : 'false';
        }
        
        // Get player stats
        $leaders = $competition['leaders'] ?? [];
        
        $rusher1 = '';
        $rusher2 = '';
        $QB1 = '';
        $QB2 = '';
        $Rec1 = '';
        $Rec2 = '';
        
        if ($game_status == 'NOT_STARTED') {
            // For games not started, prefix with team indicators
            $rusher1 = '_TBD';
            $rusher2 = '!TBD';
            $QB1 = '_TBD';
            $QB2 = '!TBD';
            $Rec1 = '_TBD';
            $Rec2 = '!TBD';
        } else {
            // Extract actual stats
            foreach ($leaders as $leader) {
                if ($leader['name'] == 'rushingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $rusher1 = $signal . $fullName . ': ' . $displayValue;
                }
                
                if ($leader['name'] == 'passingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $QB1 = $signal . $fullName . ': ' . $displayValue;
                }
                
                if ($leader['name'] == 'receivingYards' && !empty($leader['leaders'][0])) {
                    $player = $leader['leaders'][0];
                    $team_id = $player['team']['id'] ?? '';
                    $signal = ($team_id == $team1_id) ? '_' : '!';
                    $fullName = $player['athlete']['fullName'] ?? '';
                    $displayValue = $player['displayValue'] ?? '';
                    $Rec1 = $signal . $fullName . ': ' . $displayValue;
                }
            }
            
            // Set second values same as first for in-game
            $rusher2 = $rusher1;
            $QB2 = $QB1;
            $Rec2 = $Rec1;
        }
        
        // Clean special characters for database
        $team1_location = str_replace("'", "", $team1_location);
        $team2_location = str_replace("'", "", $team2_location);
        $team1_name = str_replace("'", "", $team1_name);
        $team2_name = str_replace("'", "", $team2_name);
        $pos_text = str_replace("&", "A", $pos_text);
        
        // CHECK IF UPDATE SHOULD BE SKIPPED DUE TO TIMING RESTRICTION
        $game_num = $i + 1;
        
        // Disable timing restriction if in force mode
        $skip_update = false;
        $time_remaining = 0;
        
        if (!$forceUpdateAll) {
            $skip_info = shouldSkipUpdate($game_num, $conference_name, $score_value, $con);
            $skip_update = $skip_info['skip'];
            $time_remaining = $skip_info['time_remaining'];
        }
        
        if ($skip_update) {
            $games_skipped_in_conference++;
            $total_games_skipped_timing++;
            $games_with_timing_restrictions[] = [
                'conference' => $conference_name,
                'game_num' => $game_num,
                'teams' => "{$team2_abbrev} @ {$team1_abbrev}",
                'time_remaining' => $time_remaining,
                'score_value' => $score_value
            ];
        }
        
        // Display in summary table
        $status_class = '';
        $update_status = 'UPDATED';
        
        if ($forceUpdateAll && $skip_info['skip'] ?? false) {
            $update_status = "FORCED";
            $status_class = 'status-skipped';
        } elseif ($skip_update) {
            $update_status = "HOLD ({$time_remaining}s)";
            $status_class = 'status-skipped';
        } elseif ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME') {
            $status_class = 'status-active';
            $games_in_progress++;
        } elseif ($game_status == 'GAME_DONE') {
            $status_class = 'status-done';
            $games_completed++;
        } else {
            $status_class = 'status-upcoming';
            $games_upcoming++;
        }
        
        // Check if this was a score alert override
        $play_display = substr($last_play_text, 0, 50);
        if (strlen($last_play_text) > 50) $play_display .= '...';
        
        if (strpos($last_play_text, 'TOUCHDOWN!') !== false || 
            strpos($last_play_text, 'FIELD GOAL!') !== false || 
            strpos($last_play_text, 'EXTRA POINT!') !== false) {
            $play_display = "<span class='score-alert'>$play_display</span>";
        }
        
        echo "<tr>";
        echo "<td>{$team2_abbrev} @ {$team1_abbrev}</td>";
        echo "<td><span class='game-status $status_class'>$game_status</span></td>";
        echo "<td>{$team1_score} - {$team2_score}</td>";
        echo "<td>$clock</td>";
        echo "<td class='broadcast-cell'>" . ($broadcast ?: '-') . "</td>";
        echo "<td>$update_status</td>";
        echo "<td>$play_display</td>";
        echo "</tr>";
        
        // Check if game is older than 5 days (skip old games unless forced)
        if (!$forceUpdateAll) {
            $gameDateTime = strtotime($game_date);
            $fiveDaysAgo = strtotime('-5 days');
            
            if ($gameDateTime < $fiveDaysAgo) {
                continue; // Skip database update for old games
            }
        }
        
        // SKIP DATABASE UPDATE IF WITHIN TIMING RESTRICTION (unless forced)
        if ($skip_update && !$forceUpdateAll) {
            continue;
        }
        
        // Update database with smart field preservation
        // Only update scoring-related fields when there's an actual score
        if ($scoreVal > 0 && ($game_status == 'IN_SESSION' || $game_status == 'HALF_TIME')) {
            // There's a score - update ALL fields including lastPlay and player info
            $query = "
                UPDATE sports_list
                SET
                    Clock = '" . addslashes($clock) . "',
                    Period = '" . addslashes($period) . "',
                    Game_Played = '" . addslashes($game_status) . "',
                    Game_Time = '" . addslashes($game_time_et) . "',
                    Game_Date = '" . addslashes($game_date) . "',
                    broadcast = '" . addslashes($broadcast) . "',
                    scoreValue = '" . addslashes($score_value) . "',
                    lastPlay = '" . addslashes($last_play_text) . "',
                    lastPlayerID = '" . addslashes($lastPlayerID) . "',
                    lastPlayerName = '" . addslashes($lastPlayerName) . "',
                    lastPlayerPos = '" . addslashes($lastPlayerPos) . "',
                    lastPlayerTeamID = '" . addslashes($lastPlayerTeamID) . "',
                    jersey = '" . addslashes($lastPlayerJersey) . "',
                    driveDesc = '" . addslashes($driveDescription) . "',
                    lastUpdatedTime = '" . addslashes($current_timestamp) . "',
                    
                    Team_1_Name = '" . addslashes($team1_name) . "',
                    Team_1_Record = '" . addslashes($team1_record) . "',
                    Team_1_Score = '" . addslashes($team1_score) . "',
                    Team_1_Location = '" . addslashes($team1_location) . "',
                    Team_1_Abbrev = '" . addslashes($team1_abbrev) . "',
                    Team_1_Color = '" . addslashes($team1_colors['rgb']) . "',
                    Team_1_Color_R = '" . addslashes($team1_colors['r']) . "',
                    Team_1_Color_G = '" . addslashes($team1_colors['g']) . "',
                    Team_1_Color_B = '" . addslashes($team1_colors['b']) . "',
                    
                    Team_2_Name = '" . addslashes($team2_name) . "',
                    Team_2_Record = '" . addslashes($team2_record) . "',
                    Team_2_Score = '" . addslashes($team2_score) . "',
                    Team_2_Location = '" . addslashes($team2_location) . "',
                    Team_2_Abbrev = '" . addslashes($team2_abbrev) . "',
                    Team_2_Color = '" . addslashes($team2_colors['rgb']) . "',
                    Team_2_Color_R = '" . addslashes($team2_colors['r']) . "',
                    Team_2_Color_G = '" . addslashes($team2_colors['g']) . "',
                    Team_2_Color_B = '" . addslashes($team2_colors['b']) . "',
                    
                    Who_Has_Ball = '" . addslashes($who_has_ball) . "',
                    Distance = '" . addslashes($short_distance_text) . "',
                    Possesion = '" . addslashes($pos_text) . "',
                    Favorite_Team = '" . addslashes($favorite_team) . "',
                    Favorite_Points = '" . addslashes($favorite_points) . "',
                    OverUnder = '" . addslashes($overUnder) . "',
                    Red_Zone = '" . addslashes($is_red_zone) . "',
                    Ru1 = '" . addslashes($rusher1) . "',
                    Ru2 = '" . addslashes($rusher2) . "',
                    QB1 = '" . addslashes($QB1) . "',
                    QB2 = '" . addslashes($QB2) . "',
                    Rec1 = '" . addslashes($Rec1) . "',
                    Rec2 = '" . addslashes($Rec2) . "'
                WHERE
                    Game = $game_num
                    AND Conference = '$conference_name'
            ";
        } else {
            // No score (scoreVal = 0) - update everything EXCEPT lastPlay and player fields
            // This preserves the scoring information from the previous score
            $query = "
                UPDATE sports_list
                SET
                    Clock = '" . addslashes($clock) . "',
                    Period = '" . addslashes($period) . "',
                    Game_Played = '" . addslashes($game_status) . "',
                    Game_Time = '" . addslashes($game_time_et) . "',
                    Game_Date = '" . addslashes($game_date) . "',
                    broadcast = '" . addslashes($broadcast) . "',
                    scoreValue = '" . addslashes($score_value) . "',
                    
                    Team_1_Name = '" . addslashes($team1_name) . "',
                    Team_1_Record = '" . addslashes($team1_record) . "',
                    Team_1_Score = '" . addslashes($team1_score) . "',
                    Team_1_Location = '" . addslashes($team1_location) . "',
                    Team_1_Abbrev = '" . addslashes($team1_abbrev) . "',
                    Team_1_Color = '" . addslashes($team1_colors['rgb']) . "',
                    Team_1_Color_R = '" . addslashes($team1_colors['r']) . "',
                    Team_1_Color_G = '" . addslashes($team1_colors['g']) . "',
                    Team_1_Color_B = '" . addslashes($team1_colors['b']) . "',
                    
                    Team_2_Name = '" . addslashes($team2_name) . "',
                    Team_2_Record = '" . addslashes($team2_record) . "',
                    Team_2_Score = '" . addslashes($team2_score) . "',
                    Team_2_Location = '" . addslashes($team2_location) . "',
                    Team_2_Abbrev = '" . addslashes($team2_abbrev) . "',
                    Team_2_Color = '" . addslashes($team2_colors['rgb']) . "',
                    Team_2_Color_R = '" . addslashes($team2_colors['r']) . "',
                    Team_2_Color_G = '" . addslashes($team2_colors['g']) . "',
                    Team_2_Color_B = '" . addslashes($team2_colors['b']) . "',
                    
                    Who_Has_Ball = '" . addslashes($who_has_ball) . "',
                    Distance = '" . addslashes($short_distance_text) . "',
                    Possesion = '" . addslashes($pos_text) . "',
                    Favorite_Team = '" . addslashes($favorite_team) . "',
                    Favorite_Points = '" . addslashes($favorite_points) . "',
                    OverUnder = '" . addslashes($overUnder) . "',
                    Red_Zone = '" . addslashes($is_red_zone) . "',
                    Ru1 = '" . addslashes($rusher1) . "',
                    Ru2 = '" . addslashes($rusher2) . "',
                    QB1 = '" . addslashes($QB1) . "',
                    QB2 = '" . addslashes($QB2) . "',
                    Rec1 = '" . addslashes($Rec1) . "',
                    Rec2 = '" . addslashes($Rec2) . "'
                WHERE
                    Game = $game_num
                    AND Conference = '$conference_name'
            ";
            // Note: We're NOT updating: lastPlay, lastPlayerID, lastPlayerName, lastPlayerPos, lastPlayerTeamID, jersey, driveDesc
        }
        
        mysqli_query($con, $query);
        $total_games_updated++;
    }
    
    echo "</table>";
    echo "</details>";
    
    // Update API timestamp
    $time_updated = date('m/d/Y | g:iA');
    $sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'CFB-$conference_name'";
    mysqli_query($con, $sql);
    
    // Display stats for this conference
    echo "<div class='stats'>";
    echo "<div class='stat-box'>Total Games: $games_count</div>";
    echo "<div class='stat-box'><span class='game-status status-active'>In Progress: $games_in_progress</span></div>";
    echo "<div class='stat-box'><span class='game-status status-done'>Completed: $games_completed</span></div>";
    echo "<div class='stat-box'><span class='game-status status-upcoming'>Upcoming: $games_upcoming</span></div>";
    if ($games_skipped_in_conference > 0 && !$forceUpdateAll) {
        echo "<div class='stat-box'><span class='game-status status-skipped'>Timing Hold: $games_skipped_in_conference</span></div>";
    }
    echo "</div>";
    
    if ($games_in_progress > 0) {
        $conferences_with_active_games[] = "$conference_name ($games_in_progress active)";
    }
    
    echo "</div>";
}

// Display timing restrictions if any (not shown in force mode)
if (count($games_with_timing_restrictions) > 0 && !$forceUpdateAll) {
    echo "<div class='conference-section skipped'>";
    echo "<h2>⏱️ Timing Restrictions Active</h2>";
    echo "<div class='timing-restriction'>";
    echo "<p><strong>2-Minute Score Protection:</strong> Updates paused after scoring to preserve play details</p>";
    echo "<table>";
    echo "<tr><th>Conference</th><th>Game</th><th>Teams</th><th>Score Type</th><th>Hold Time Remaining</th></tr>";
    foreach ($games_with_timing_restrictions as $restricted) {
        echo "<tr>";
        echo "<td>{$restricted['conference']}</td>";
        echo "<td>#{$restricted['game_num']}</td>";
        echo "<td>{$restricted['teams']}</td>";
        echo "<td>{$restricted['score_value']}</td>";
        echo "<td><strong>{$restricted['time_remaining']}s</strong></td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    echo "</div>";
}

// Summary section
$execution_time = round(microtime(true) - $start_time, 2);

echo "<div class='summary'>";
echo "<h2>Execution Summary</h2>";
echo "<div class='stats'>";
echo "<div class='stat-box'>Execution Time: {$execution_time}s</div>";
echo "<div class='stat-box'>API Calls Made: $total_api_calls</div>";
echo "<div class='stat-box'>Games Updated: $total_games_updated</div>";
if (!$forceUpdateAll) {
    echo "<div class='stat-box'>Games on Hold: $total_games_skipped_timing</div>";
}
echo "<div class='stat-box'>Score Alerts Fixed: $score_alerts_fixed</div>";
echo "<div class='stat-box'>Conferences Skipped: " . count($conferences_skipped) . "</div>";
if (count($conferences_forced) > 0) {
    echo "<div class='stat-box' style='background: #f39c12; color: white;'>Conferences Forced: " . count($conferences_forced) . "</div>";
}
echo "</div>";

echo "<h3>Enhanced Features Active:</h3>";
echo "<ul>";
echo "<li>✅ UTC Time Fix - West Coast games update past midnight ET</li>";
if (!$forceUpdateAll) {
    echo "<li>✅ 2-Minute Score Protection - Preserves scoring plays</li>";
} else {
    echo "<li>⚠️ 2-Minute Score Protection - DISABLED (Force Mode)</li>";
}
echo "<li>✅ Score Alert Validation - Fixes incorrect ESPN descriptions</li>";
echo "<li>✅ Smart Field Preservation - Keeps scoring data visible</li>";
echo "<li>✅ Enhanced Play Tracking - Player details and drive info</li>";
echo "<li>✅ Broadcast Information - Channel/network data included</li>";
if ($forceUpdateAll) {
    echo "<li>⚠️ FORCE UPDATE MODE - All conferences updated regardless of game status</li>";
}
echo "</ul>";

if (!empty($conferences_with_active_games)) {
    echo "<p><strong>Active Conferences:</strong> " . implode(", ", $conferences_with_active_games) . "</p>";
}

if (!empty($conferences_forced)) {
    echo "<p style='color: #f39c12;'><strong>Force Updated (No Active Games):</strong> " . implode(", ", $conferences_forced) . "</p>";
}

if (!empty($conferences_skipped)) {
    echo "<p><strong>Skipped (No Active Games):</strong> " . implode(", ", $conferences_skipped) . "</p>";
}

if (!empty($errors)) {
    echo "<p style='color: #e74c3c;'><strong>Errors:</strong><br>" . implode("<br>", $errors) . "</p>";
}

// Add usage instructions
echo "<div style='margin-top: 20px; padding: 15px; background: #ecf0f1; border-radius: 5px;'>";
echo "<h3>Usage Instructions:</h3>";
echo "<ul>";
echo "<li><strong>Normal Mode:</strong> Run without parameters - only updates conferences with active/upcoming games</li>";
echo "<li><strong>Force Update All:</strong> Use one of these methods:";
echo "<ul>";
echo "<li>Web URL: <code>yoursite.com/cfb_updater.php?input=ALL</code> or <code>?update=ALL</code></li>";
echo "<li>Command Line: <code>php cfb_updater.php ALL</code></li>";
echo "<li>POST Request: Send <code>input=ALL</code> or <code>update=ALL</code> in POST body</li>";
echo "</ul>";
echo "</li>";
echo "<li><strong>Note:</strong> Force mode bypasses timing restrictions and old game filters</li>";
echo "<li><strong>Broadcast Info:</strong> Game broadcast/channel information is now included for all games</li>";
echo "</ul>";
echo "</div>";

echo "</div>";

// Call the output API once at the end
// Trigger if: games were updated OR force mode is active OR API calls were made
if ($total_games_updated > 0 || $forceUpdateAll || $total_api_calls > 0) {
    $url = SERVER_URL . "Scrolling/App_Output/API/Games/NCAAF_Games.php";
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        echo "<div class='conference-section skipped'>";
        echo "<h3>Output API Error</h3>";
        echo "<p>Error: " . curl_error($ch) . "</p>";
        echo "</div>";
    } else {
        echo "<div class='conference-section inactive'>";
        echo "<h3>Output API Called Successfully</h3>";
        echo "<p>NCAAF_Games.php has been updated";
        if ($total_games_updated == 0 && $forceUpdateAll) {
            echo " (Force mode - no games needed updating)";
        } elseif ($total_games_updated == 0 && $total_api_calls > 0) {
            echo " (API data refreshed with broadcast info)";
        }
        echo "</p>";
        echo "</div>";
    }
    
    curl_close($ch);
} else {
    // Only show this message if NOT in force mode
    if (!$forceUpdateAll) {
        echo "<div class='conference-section inactive'>";
        echo "<h3>Output API Not Called</h3>";
        echo "<p>No updates were needed</p>";
        echo "</div>";
    }
}

echo "</body></html>";

// Close database connection
mysqli_close($con);
session_write_close();

// Output the HTML
$html_output = ob_get_clean();
echo $html_output;

?>