<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);
require_once '../../config.php';

session_start();
$conn = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

// EPL standings API URL
$url = 'https://site.api.espn.com/apis/v2/sports/soccer/eng.1/standings';

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($ch, CURLOPT_HTTPHEADER, array('Accept: application/json'));
$response = curl_exec($ch);
curl_close($ch);

$data = json_decode($response, true);
if ($data === null) {
    echo "Failed to decode JSON.\n";
    echo "Response:\n";
    echo $response;
    exit;
}

// Helper functions
function get_stat_value($stats, $name) {
    foreach ($stats as $stat) {
        if ($stat['name'] === $name) {
            return $stat['value'];
        }
    }
    return null;
}

function addZeros($rgb) {
    return str_pad($rgb, 3, '0', STR_PAD_LEFT);
}

function normalize_team_name($name) {
    return strtoupper(preg_replace('/[^A-Za-z0-9 ]/', '', iconv('UTF-8', 'ASCII//TRANSLIT', $name)));
}

// Debug log function
function debug_log($message) {
    echo "<div style='background: #f8f8f8; padding: 5px; margin: 5px 0; border: 1px solid #ddd;'>";
    echo "<strong>DEBUG:</strong> " . htmlspecialchars($message) . "</div>";
}

// Team name corrections mapping - Updated for 2025-26 season
$teamNameCorrections = array(
    'MANCHESTER CITY' => 'MAN CITY',
    'MANCHESTER UNITED' => 'MAN UNITED',
    'TOTTENHAM HOTSPUR' => 'TOTTENHAM',
    'WEST HAM UNITED' => 'WEST HAM',
    'WOLVERHAMPTON WANDERERS' => 'WOLVES',
    'WOLVERHAMPTON' => 'WOLVES',
    'NOTTINGHAM FOREST' => 'NOTTM FOREST',
    'AFC BOURNEMOUTH' => 'BOURNEMOUTH',
    'BOURNEMOUTH' => 'BOURNEMOUTH',
    'BRIGHTON HOVE ALBION' => 'BRIGHTON',  // This is the key fix
    'BRIGHTON  HOVE ALBION' => 'BRIGHTON',  // Alternative with double space
    'BRIGHTON & HOVE ALBION' => 'BRIGHTON', // In case & isn't removed
    'CRYSTAL PALACE' => 'CRYSTAL PAL',
    'NEWCASTLE UNITED' => 'NEWCASTLE',
    'LEEDS UNITED' => 'LEEDS UNITED',  // Keep full name
    'BURNLEY' => 'BURNLEY',
    'SUNDERLAND' => 'SUNDERLAND'
);

// EPL team colors and abbreviations - Updated for 2025-26 season (20 teams)
$teamColors = array(
    "LIVERPOOL"      => array(200, 16, 46),    // Red
    "ARSENAL"        => array(239, 1, 7),      // Red
    "NOTTM FOREST"   => array(229, 39, 39),    // Red
    "MAN CITY"       => array(108, 171, 221),  // Sky Blue
    "CHELSEA"        => array(3, 70, 148),     // Blue
    "NEWCASTLE"      => array(0, 0, 0),        // Black and White
    "BOURNEMOUTH"    => array(218, 41, 28),    // Red and Black
    "BRIGHTON"       => array(0, 87, 184),     // Blue
    "FULHAM"         => array(255, 255, 255),  // White
    "ASTON VILLA"    => array(149, 191, 229),  // Claret and Blue
    "BRENTFORD"      => array(227, 6, 19),     // Red and White
    "CRYSTAL PAL"    => array(26, 13, 171),    // Red and Blue
    "TOTTENHAM"      => array(19, 34, 87),     // Navy Blue
    "MAN UNITED"     => array(218, 41, 28),    // Red
    "WEST HAM"       => array(122, 38, 58),    // Claret and Blue
    "EVERTON"        => array(39, 68, 136),    // Blue
    "WOLVES"         => array(253, 185, 19),   // Gold
    "BURNLEY"        => array(108, 35, 69),    // Claret and Blue
    "LEEDS UNITED"   => array(255, 205, 0),    // White and Yellow
    "SUNDERLAND"     => array(211, 8, 23)      // Red and White
);

$teamAbbreviations = array(
    "LIVERPOOL"      => "LIV",
    "ARSENAL"        => "ARS",
    "NOTTM FOREST"   => "NFO",
    "MAN CITY"       => "MNC",
    "CHELSEA"        => "CHE",
    "NEWCASTLE"      => "NEW",
    "BOURNEMOUTH"    => "BOU",
    "BRIGHTON"       => "BHA",
    "FULHAM"         => "FUL",
    "ASTON VILLA"    => "AVL",
    "BRENTFORD"      => "BRE",
    "CRYSTAL PAL"    => "CRY",
    "TOTTENHAM"      => "TOT",
    "MAN UNITED"     => "MAN",
    "WEST HAM"       => "WHU",
    "EVERTON"        => "EVE",
    "WOLVES"         => "WOL",
    "BURNLEY"        => "BUR",
    "LEEDS UNITED"   => "LEE",
    "SUNDERLAND"     => "SUN"
);

// Build lookup arrays by normalized full name
$team_colors = array();
$abbrByFullName = array();
foreach ($teamColors as $fullName => $colors) {
    $normalizedFullName = normalize_team_name($fullName);
    $team_colors[$normalizedFullName] = $colors;
    $abbrByFullName[$normalizedFullName] = $teamAbbreviations[$fullName];
}

// Process API data
$apiTeams = array();
$unmatchedTeams = array();

// DEBUG: Print out all API entries before processing
echo "<h3>API Raw Data for Teams:</h3>";
echo "<table border='1'>";
echo "<tr><th>API Team</th><th>Normalized</th><th>After Correction</th><th>In team_colors</th></tr>";

// EPL has only one table (no conferences)
if (isset($data['children']) && !empty($data['children'][0]['standings']['entries'])) {
    foreach ($data['children'][0]['standings']['entries'] as $entry) {
        $full_team_name = $entry['team']['displayName'];
        $normalizedFullName = normalize_team_name($full_team_name);
        $correctedName = $normalizedFullName;
        
        // Special case for Brighton - force correction
        if (strpos($normalizedFullName, 'BRIGHTON') !== false) {
            $correctedName = 'BRIGHTON';
            debug_log("Forced Brighton correction: $normalizedFullName -> $correctedName");
        }
        // Apply correction if API name differs from our standard 
        else if (isset($teamNameCorrections[$normalizedFullName])) {
            $correctedName = $teamNameCorrections[$normalizedFullName];
        }

        // Check if in team_colors lookup
        $inTeamColors = array_key_exists($correctedName, $team_colors) ? "Yes" : "No";
        
        // Debug table row
        echo "<tr>";
        echo "<td>" . htmlspecialchars($full_team_name) . "</td>";
        echo "<td>" . htmlspecialchars($normalizedFullName) . "</td>";
        echo "<td>" . htmlspecialchars($correctedName) . "</td>";
        echo "<td>" . $inTeamColors . "</td>";
        echo "</tr>";
        
        $wins = get_stat_value($entry['stats'], 'wins');
        $losses = get_stat_value($entry['stats'], 'losses');
        $ties = get_stat_value($entry['stats'], 'ties'); // Draws in football/soccer
        
        // Format for EPL: W-D-L (Wins-Draws-Losses)
        $record = "$wins-$ties-$losses";
        $ranking = get_stat_value($entry['stats'], 'rank');
        
        // Use the corrected name for team lookup
        if (array_key_exists($correctedName, $team_colors)) {
            $apiTeams[$correctedName] = array(
                'wins' => $wins,
                'losses' => $losses,
                'ties' => $ties,
                'record' => $record,
                'ranking' => $ranking,
                'abbrev' => $entry['team']['abbreviation'],
                'gamesPlayed' => get_stat_value($entry['stats'], 'gamesPlayed'),
                'goalsFor' => get_stat_value($entry['stats'], 'pointsFor'),
                'goalsAgainst' => get_stat_value($entry['stats'], 'pointsAgainst'),
                'goalDiff' => get_stat_value($entry['stats'], 'pointDifferential'),
                'points' => get_stat_value($entry['stats'], 'points')
            );
        } else {
            $unmatchedTeams[] = "Rank $ranking: $full_team_name (Normalized: $normalizedFullName, Corrected: $correctedName)";
        }
    }
}
echo "</table>";

// Debug unmatched teams
if (!empty($unmatchedTeams)) {
    echo "<h3>Unmatched Teams from API:</h3><ul>";
    foreach ($unmatchedTeams as $unmatched) {
        echo "<li>$unmatched</li>";
    }
    echo "</ul>";
} else {
    echo "<h3>All teams successfully matched!</h3>";
}

// Debug: Show what values are stored in the apiTeams array
echo "<h3>Teams Successfully Matched from API:</h3>";
echo "<table border='1'>";
echo "<tr><th>Normalized Name</th><th>Record</th><th>Rank</th></tr>";
foreach ($apiTeams as $teamName => $teamData) {
    echo "<tr>";
    echo "<td>" . htmlspecialchars($teamName) . "</td>";
    echo "<td>" . htmlspecialchars($teamData['record']) . "</td>";
    echo "<td>" . htmlspecialchars($teamData['ranking']) . "</td>";
    echo "</tr>";
}
echo "</table>";

// --- Create a mapping of normalized team names to unique_count values ---
// Updated for 2025-26 season teams
$uniqueCountsMapping = array(
    normalize_team_name("LIVERPOOL")      => array(6000),
    normalize_team_name("ARSENAL")        => array(6001),
    normalize_team_name("NOTTM FOREST")   => array(6002),
    normalize_team_name("MAN CITY")       => array(6003),
    normalize_team_name("CHELSEA")        => array(6004),
    normalize_team_name("NEWCASTLE")      => array(6005),
    normalize_team_name("BOURNEMOUTH")    => array(6006),
    normalize_team_name("BRIGHTON")       => array(6007),
    normalize_team_name("FULHAM")         => array(6008),
    normalize_team_name("ASTON VILLA")    => array(6009),
    normalize_team_name("BRENTFORD")      => array(6010),
    normalize_team_name("CRYSTAL PAL")    => array(6011),
    normalize_team_name("TOTTENHAM")      => array(6012),
    normalize_team_name("MAN UNITED")     => array(6013),
    normalize_team_name("WEST HAM")       => array(6014),
    normalize_team_name("EVERTON")        => array(6015),
    normalize_team_name("WOLVES")         => array(6016),

    // match the rows you actually have in leader_board:
    normalize_team_name("BURNLEY")        => array(6020),
    normalize_team_name("LEEDS UNITED")   => array(6021),
    normalize_team_name("SUNDERLAND")     => array(6022)
);


// --- Now update the database using unique_count as the identifier ---
echo "<h3>Database Updates:</h3>";
echo "<table border='1'>";
echo "<tr><th>Team Name</th><th>Record (W-D-L)</th><th>Rank</th><th>Red</th><th>Green</th><th>Blue</th><th>SQL Query</th><th>Status</th></tr>";

foreach ($uniqueCountsMapping as $normalizedFullName => $uniqueCountsArray) {
    // Apply same corrections as we did for API data
    $correctedName = $normalizedFullName;
    if (isset($teamNameCorrections[$normalizedFullName])) {
        $correctedName = $teamNameCorrections[$normalizedFullName];
    }
    // Force Brighton correction
    if (strpos($normalizedFullName, 'BRIGHTON') !== false) {
        $correctedName = 'BRIGHTON';
    }
    
    // Get team colors and abbreviation from our lookup arrays using corrected name
    if (!isset($team_colors[$correctedName])) {
        // If the team is not in our color mapping, skip it.
        debug_log("Team not found in color mapping: $correctedName (normalized: $normalizedFullName)");
        continue;
    }
    $colors = $team_colors[$correctedName];
    $red = $colors[0];
    $green = $colors[1];
    $blue = $colors[2];
    $abbr = $abbrByFullName[$correctedName] ?? '';

    // If API data is available, use it; otherwise use default values.
    if (isset($apiTeams[$correctedName])) {
        $record = $apiTeams[$correctedName]['record'];
        $ranking = $apiTeams[$correctedName]['ranking'];
        $gamesPlayed = $apiTeams[$correctedName]['gamesPlayed'];
        $goalsFor = $apiTeams[$correctedName]['goalsFor'];
        $goalsAgainst = $apiTeams[$correctedName]['goalsAgainst'];
        $goalDiff = $apiTeams[$correctedName]['goalDiff'];
        $points = $apiTeams[$correctedName]['points'];
    } else {
        debug_log("Team not found in API data: $correctedName (normalized: $normalizedFullName)");
        $record = "0-0-0";
        $ranking = 0;
        $gamesPlayed = 0;
        $goalsFor = 0;
        $goalsAgainst = 0;
        $goalDiff = 0;
        $points = 0;
    }
    
    // Execute an update for each unique_count
    foreach ($uniqueCountsArray as $uniqueCount) {
        // Use the corrected name for the database
        $query = "UPDATE leader_board SET 
            current_team = '$correctedName', 
            winLossRecord = '$record', 
            Abbrev = '$abbr',
            Red = $red, 
            Green = $green, 
            Blue = $blue,
            team_rank = $ranking
            WHERE unique_count = $uniqueCount;";
            
        $status = mysqli_query($conn, $query) ? "Query executed successfully." : "Error: " . mysqli_error($conn);
        
        echo "<tr>";
        echo "<td>$correctedName</td>";
        echo "<td>$record</td>";
        echo "<td>$ranking</td>";

        echo "<td>" . addZeros($red) . "</td>";
        echo "<td>" . addZeros($green) . "</td>";
        echo "<td>" . addZeros($blue) . "</td>";
        echo "<td>" . htmlspecialchars($query) . "</td>";
        echo "<td>$status (unique_count: $uniqueCount)</td>";
        echo "</tr>";
    }
}
echo "</table>";

mysqli_close($conn);
?>