<?php
// ============================================
// NFL Standings -> leader_board updater
// - Includes ties in overall only if > 0
// - Echoes an HTML data table of updates
// ============================================

// Constants and Configurations
define('NFL_STANDINGS_URL', 'https://site.api.espn.com/apis/v2/sports/football/nfl/standings');
error_reporting(0);

// Corrected Team Information Mapping with abbreviations, colors, and conferences (DB 'division' column)
$TEAM_INFO = array(
    'ARI' => array('name' => 'Cardinals',   'colors' => array(155,16,30),  'division' => 'NFC'),
    'ATL' => array('name' => 'Falcons',     'colors' => array(255,255,255),'division' => 'NFC'),
    'BAL' => array('name' => 'Ravens',      'colors' => array(26,25,95),   'division' => 'AFC'),
    'BUF' => array('name' => 'Bills',       'colors' => array(0,51,141),   'division' => 'AFC'),
    'CAR' => array('name' => 'Panthers',    'colors' => array(0,133,202),  'division' => 'NFC'),
    'CHI' => array('name' => 'Bears',       'colors' => array(200,56,3),   'division' => 'NFC'),
    'CIN' => array('name' => 'Bengals',     'colors' => array(251,79,20),  'division' => 'AFC'),
    'CLE' => array('name' => 'Browns',      'colors' => array(49,29,0),    'division' => 'AFC'),
    'DAL' => array('name' => 'Cowboys',     'colors' => array(127,150,149),'division' => 'NFC'),
    'DEN' => array('name' => 'Broncos',     'colors' => array(0,35,77),    'division' => 'AFC'),
    'DET' => array('name' => 'Lions',       'colors' => array(0,118,182),  'division' => 'NFC'),
    'GB'  => array('name' => 'Packers',     'colors' => array(24,48,40),   'division' => 'NFC'),
    'HOU' => array('name' => 'Texans',      'colors' => array(3,32,47),    'division' => 'AFC'),
    'IND' => array('name' => 'Colts',       'colors' => array(0,44,95),    'division' => 'AFC'),
    'JAX' => array('name' => 'Jaguars',     'colors' => array(0,103,120),  'division' => 'AFC'),
    'KC'  => array('name' => 'Chiefs',      'colors' => array(227,24,55),  'division' => 'AFC'),
    'LV'  => array('name' => 'Raiders',     'colors' => array(165,172,175),'division' => 'AFC'),
    'LAC' => array('name' => 'Chargers',    'colors' => array(0,128,198),  'division' => 'AFC'),
    'LAR' => array('name' => 'Rams',        'colors' => array(0,53,148),   'division' => 'NFC'),
    'MIA' => array('name' => 'Dolphins',    'colors' => array(0,142,151),  'division' => 'AFC'),
    'MIN' => array('name' => 'Vikings',     'colors' => array(79,38,131),  'division' => 'NFC'),
    'NE'  => array('name' => 'Patriots',    'colors' => array(0,34,68),    'division' => 'AFC'),
    'NO'  => array('name' => 'Saints',      'colors' => array(211,188,141),'division' => 'NFC'),
    'NYG' => array('name' => 'Giants',      'colors' => array(1,35,82),    'division' => 'NFC'),
    'NYJ' => array('name' => 'Jets',        'colors' => array(18,87,64),   'division' => 'AFC'),
    'PHI' => array('name' => 'Eagles',      'colors' => array(0,76,84),    'division' => 'NFC'),
    'PIT' => array('name' => 'Steelers',    'colors' => array(165,172,175),'division' => 'AFC'),
    'SEA' => array('name' => 'Seahawks',    'colors' => array(0,34,68),    'division' => 'NFC'),
    'SF'  => array('name' => '49ers',       'colors' => array(170,0,0),    'division' => 'NFC'),
    'TB'  => array('name' => 'Buccaneers',  'colors' => array(213,10,10),  'division' => 'NFC'),
    'TEN' => array('name' => 'Titans',      'colors' => array(75,146,219), 'division' => 'AFC'),
    'WSH' => array('name' => 'Commanders',  'colors' => array(90,20,20),   'division' => 'NFC')
);

// ------------------ Helpers ------------------
function get_nfl_standings() {
    $response = file_get_contents(NFL_STANDINGS_URL);
    if ($response === FALSE) {
        echo "Failed to retrieve data.\n";
        return array();
    }
    return json_decode($response, true);
}

function safe_int($v, $default = 0) {
    return is_numeric($v) ? intval($v) : $default;
}

// ------------------ DB Connect ------------------
require_once '../../config.php'; // Adjust the path if needed
session_start();
$conn = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

// ------------------ Fetch Data ------------------
$data = get_nfl_standings();
if (empty($data)) {
    echo "No data retrieved.\n";
    exit;
}

// ------------------ Prepare Statement ------------------
/*
Table: leader_board
Columns updated:
- current_team (TEAM NAME ONLY, custom abbreviations for some)
- winLossRecord (e.g., "5-2" or "4-1-1" if ties > 0)
- Abbrev (e.g., "GB")
- Red, Green, Blue (RGB ints)
Filtered by:
- team_rank (from playoffSeed)
- division  (from TEAM_INFO; "AFC"/"NFC")
- league    = 'NFL'
*/
$stmt = $conn->prepare("UPDATE leader_board SET 
    current_team = ?, 
    winLossRecord = ?, 
    Abbrev = ?,
    Red = ?, 
    Green = ?, 
    Blue = ? 
    WHERE team_rank = ? 
    AND division = ? 
    AND league = 'NFL'");

// For HTML echo table at the end
$rows = [];

// ------------------ Process + Update ------------------
foreach ($data['children'] as $conference) {
    if (!isset($conference['standings']['entries'])) { continue; }

    foreach ($conference['standings']['entries'] as $team_entry) {
        if (!isset($team_entry['team']['abbreviation'])) { continue; }

        $team_id = strtoupper($team_entry['team']['abbreviation']);

        if (!isset($TEAM_INFO[$team_id])) {
            // Team not in mapping → skip
            continue;
        }

        // Base totals
        $wins   = 0;
        $losses = 0;
        $ties   = 0;
        $ranking = 0; // playoffSeed
        $overall_summary = '';

        // Pull from stats array
        if (isset($team_entry['stats']) && is_array($team_entry['stats'])) {
            foreach ($team_entry['stats'] as $stat) {
                $name = isset($stat['name']) ? $stat['name'] : (isset($stat['type']) ? $stat['type'] : '');
                switch ($name) {
                    case 'wins':
                        $wins = safe_int($stat['value']);
                        break;
                    case 'losses':
                        $losses = safe_int($stat['value']);
                        break;
                    case 'ties':
                        $ties = safe_int($stat['value']);
                        break;
                    case 'playoffseed':
                    case 'playoffSeed':
                        $ranking = safe_int($stat['value']);
                        break;
                }
            }
        }

        // Some ESPN payloads also include an "overall/total" chunk with a summary like "5-2" or "4-1-1".
        // We don't strictly need it, but capture if present (useful for echo table).
        if (isset($team_entry['stats']) && is_array($team_entry['stats'])) {
            foreach ($team_entry['stats'] as $stat) {
                if ((isset($stat['name']) && $stat['name'] === 'overall') ||
                    (isset($stat['type']) && $stat['type'] === 'total')) {
                    if (isset($stat['summary'])) {
                        $overall_summary = $stat['summary'];
                    } elseif (isset($stat['displayValue'])) {
                        $overall_summary = $stat['displayValue'];
                    }
                }
            }
        }

        // Build the record string: include "-T" only when ties > 0
        $winLossRecord = ($ties > 0)
            ? "{$wins}-{$losses}-{$ties}"
            : "{$wins}-{$losses}";

        // Map color + division (DB column)
        $colors   = $TEAM_INFO[$team_id]['colors'];
        if (empty($colors) || count($colors) !== 3) { continue; }

        $team_name = strtoupper($TEAM_INFO[$team_id]['name']);
        if ($team_name === "COMMANDERS")  { $team_name = "COMMDRS"; }
        if ($team_name === "BUCCANEERS")  { $team_name = "BUCS"; }

        $abbrev   = $team_id;
        $red      = safe_int($colors[0]);
        $green    = safe_int($colors[1]);
        $blue     = safe_int($colors[2]);
        $team_rank = safe_int($ranking);
        $division = $TEAM_INFO[$team_id]['division']; // "AFC"/"NFC"

        // Bind + execute
        $stmt->bind_param(
            'sssiiiss',
            $team_name,
            $winLossRecord,
            $abbrev,
            $red,
            $green,
            $blue,
            $team_rank,
            $division
        );
        $ok = $stmt->execute();

        // Collect for table output
        $rows[] = array(
            'rank'            => $team_rank,
            'league_division' => $division,
            'team'            => $team_name,
            'abbrev'          => $abbrev,
            'wins'            => $wins,
            'losses'          => $losses,
            'ties'            => $ties,
            'record_written'  => $winLossRecord,     // what we actually wrote to DB
            'overall_summary' => $overall_summary,   // what ESPN provided (if any)
            'red'             => $red,
            'green'           => $green,
            'blue'            => $blue,
            'db_status'       => $ok ? 'UPDATED' : ('ERROR: '.$stmt->error),
        );
    }
}

// Close DB
$stmt->close();
mysqli_close($conn);

// ------------------ Echo HTML Data Table ------------------
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8" />
<title>NFL Leader Board Updates</title>
<style>
    body { font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif; background:#0b0f14; color:#e6edf3; margin:20px; }
    h1 { font-size: 20px; margin: 0 0 12px 0; }
    table { border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #233041; padding: 8px 10px; text-align: left; font-size: 14px; }
    th { background: #15202b; position: sticky; top: 0; }
    tr:nth-child(even) td { background: #0f1520; }
    .chip { display:inline-block; padding:2px 8px; border-radius:999px; font-size:12px; border:1px solid #2a3b51; }
    .ok { background:#0d2f1b; color:#9fe29f; border-color:#245d3c; }
    .err{ background:#3a1313; color:#ffb3b3; border-color:#6b1e1e; }
    .swatch { width: 36px; height: 18px; border-radius:3px; display:inline-block; border:1px solid #111; vertical-align:middle; margin-right:6px; }
    .muted { color:#9fb0c2; font-size:12px; }
</style>
</head>
<body>
<h1>NFL leader_board updates (ties included only when present)</h1>
<table>
    <thead>
        <tr>
            <th>Rank</th>
            <th>AFC/NFC</th>
            <th>Team</th>
            <th>Abbrev</th>
            <th>Wins</th>
            <th>Losses</th>
            <th>Ties</th>
            <th>Record (written)</th>
            <th>ESPN overall</th>
            <th>Color</th>
            <th>Status</th>
        </tr>
    </thead>
    <tbody>
    <?php foreach ($rows as $r): 
        $rgbText = $r['red'].','.$r['green'].','.$r['blue'];
        $bg = "rgb({$r['red']},{$r['green']},{$r['blue']})";
        $chipClass = (strpos($r['db_status'], 'ERROR') === false) ? 'ok' : 'err';
    ?>
        <tr>
            <td><?= htmlspecialchars($r['rank']) ?></td>
            <td><?= htmlspecialchars($r['league_division']) ?></td>
            <td><?= htmlspecialchars($r['team']) ?></td>
            <td><?= htmlspecialchars($r['abbrev']) ?></td>
            <td><?= htmlspecialchars($r['wins']) ?></td>
            <td><?= htmlspecialchars($r['losses']) ?></td>
            <td><?= htmlspecialchars($r['ties']) ?></td>
            <td><strong><?= htmlspecialchars($r['record_written']) ?></strong></td>
            <td class="muted"><?= htmlspecialchars($r['overall_summary']) ?></td>
            <td>
                <span class="swatch" style="background: <?= $bg ?>;"></span>
                <span class="muted"><?= htmlspecialchars($rgbText) ?></span>
            </td>
            <td><span class="chip <?= $chipClass ?>"><?= htmlspecialchars($r['db_status']) ?></span></td>
        </tr>
    <?php endforeach; ?>
    </tbody>
</table>
</body>
</html>
