<?php
// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../../config.php';
session_start();

// Create mysqli connection
$conn = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

echo "✓ Database connected\n";
mysqli_set_charset($conn, "utf8mb4");

// Fetch data from ESPN API
$api_url = "https://site.web.api.espn.com/apis/v2/sports/football/college-football/standings?season=2025&seasontype=2&type=0&level=3";

echo "→ Fetching data from ESPN API...\n";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $api_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_HTTPHEADER, array(
    'Accept: application/json',
    'User-Agent: Mozilla/5.0'
));

$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($http_code !== 200 || !$response) {
    die("Failed to fetch data. HTTP Code: " . $http_code);
}

echo "✓ API data fetched (HTTP $http_code)\n";

$data = json_decode($response, true);
if (!$data) {
    die("Failed to parse JSON");
}

echo "✓ JSON parsed successfully\n\n";

// Helper function to safely get value from array
function safe_value($value, $default = '') {
    return isset($value) ? $value : $default;
}

// Helper function to find stat by type
function get_stat($stats, $type, $field = 'value', $default = '0') {
    foreach ($stats as $stat) {
        if (isset($stat['type']) && $stat['type'] === $type) {
            return isset($stat[$field]) ? $stat[$field] : $default;
        }
    }
    return $default;
}

// Process data
$teams_processed = 0;
$conferences_processed = 0;
$errors = [];

// Get conferences
$conferences = isset($data['children']) ? $data['children'] : [$data];
echo "Found " . count($conferences) . " conferences\n\n";

foreach ($conferences as $conference) {
    $conf_name = safe_value($conference['name'], 'Unknown');
    $conf_id = safe_value($conference['id'], '');
    
    echo "═══ Conference: $conf_name ═══\n";
    $conferences_processed++;
    
    if (!isset($conference['standings']['entries'])) {
        echo "  No teams found\n\n";
        continue;
    }
    
    $entries = $conference['standings']['entries'];
    echo "  Teams: " . count($entries) . "\n";
    
    foreach ($entries as $entry) {
        if (!isset($entry['team'])) continue;
        
        $team = $entry['team'];
        $stats = isset($entry['stats']) ? $entry['stats'] : [];
        
        // Get team data
        $team_id = safe_value($team['id']);
        $team_uid = safe_value($team['uid']);
        $full_name = safe_value($team['displayName'], 'Unknown Team');
        $short_name = safe_value($team['shortDisplayName'], $full_name);
        $location = safe_value($team['location']);
        $name = safe_value($team['name']);
        $abbr = safe_value($team['abbreviation']);
        
        // Get logos
        $logo_url = '';
        $logo_dark = '';
        if (isset($team['logos'])) {
            foreach ($team['logos'] as $logo) {
                if (isset($logo['href'])) {
                    if (in_array('default', $logo['rel'] ?? [])) {
                        $logo_url = $logo['href'];
                    }
                    if (in_array('dark', $logo['rel'] ?? [])) {
                        $logo_dark = $logo['href'];
                    }
                }
            }
        }
        
        // Get stats - all as strings for flexibility
        $overall_record = get_stat($stats, 'total', 'summary', '0-0');
        preg_match('/(\d+)-(\d+)/', $overall_record, $overall_parts);
        $overall_wins = isset($overall_parts[1]) ? $overall_parts[1] : '0';
        $overall_losses = isset($overall_parts[2]) ? $overall_parts[2] : '0';
        
        $conf_record = get_stat($stats, 'vsconf', 'summary', '0-0');
        preg_match('/(\d+)-(\d+)/', $conf_record, $conf_parts);
        $conf_wins = isset($conf_parts[1]) ? $conf_parts[1] : '0';
        $conf_losses = isset($conf_parts[2]) ? $conf_parts[2] : '0';
        
        $league_pct = get_stat($stats, 'leaguewinpercent', 'value', '0.000');
        $playoff_seed = get_stat($stats, 'playoffseed', 'value', '0');
        $points_for = get_stat($stats, 'pointsfor', 'value', '0');
        $points_against = get_stat($stats, 'pointsagainst', 'value', '0');
        $point_diff = get_stat($stats, 'pointdifferential', 'value', '0');
        $streak = get_stat($stats, 'streak', 'displayValue', '-');
        $home_record = get_stat($stats, 'homerecord', 'summary', '0-0');
        $away_record = get_stat($stats, 'awayrecord', 'summary', '0-0');
        
        // Escape all values for SQL
        $team_id = mysqli_real_escape_string($conn, $team_id);
        $team_uid = mysqli_real_escape_string($conn, $team_uid);
        $full_name = mysqli_real_escape_string($conn, $full_name);
        $short_name = mysqli_real_escape_string($conn, $short_name);
        $location = mysqli_real_escape_string($conn, $location);
        $name = mysqli_real_escape_string($conn, $name);
        $abbr = mysqli_real_escape_string($conn, $abbr);
        $conf_name = mysqli_real_escape_string($conn, $conf_name);
        $conf_id = mysqli_real_escape_string($conn, $conf_id);
        $overall_wins = mysqli_real_escape_string($conn, $overall_wins);
        $overall_losses = mysqli_real_escape_string($conn, $overall_losses);
        $overall_record = mysqli_real_escape_string($conn, $overall_record);
        $conf_wins = mysqli_real_escape_string($conn, $conf_wins);
        $conf_losses = mysqli_real_escape_string($conn, $conf_losses);
        $conf_record = mysqli_real_escape_string($conn, $conf_record);
        $league_pct = mysqli_real_escape_string($conn, $league_pct);
        $playoff_seed = mysqli_real_escape_string($conn, $playoff_seed);
        $points_for = mysqli_real_escape_string($conn, $points_for);
        $points_against = mysqli_real_escape_string($conn, $points_against);
        $point_diff = mysqli_real_escape_string($conn, $point_diff);
        $streak = mysqli_real_escape_string($conn, $streak);
        $home_record = mysqli_real_escape_string($conn, $home_record);
        $away_record = mysqli_real_escape_string($conn, $away_record);
        $logo_url = mysqli_real_escape_string($conn, $logo_url);
        $logo_dark = mysqli_real_escape_string($conn, $logo_dark);
        
        // Build and execute query
        $sql = "INSERT INTO cfb_standings (
            team_id, team_uid, full_display_name, short_display_name,
            location, name, abbreviation, conference_name, conference_id,
            overall_wins, overall_losses, overall_record,
            conference_wins, conference_losses, conference_record,
            league_win_percent, playoff_seed,
            points_for, points_against, point_differential,
            current_streak, home_record, away_record,
            logo_url, logo_dark_url
        ) VALUES (
            '$team_id', '$team_uid', '$full_name', '$short_name',
            '$location', '$name', '$abbr', '$conf_name', '$conf_id',
            '$overall_wins', '$overall_losses', '$overall_record',
            '$conf_wins', '$conf_losses', '$conf_record',
            '$league_pct', '$playoff_seed',
            '$points_for', '$points_against', '$point_diff',
            '$streak', '$home_record', '$away_record',
            '$logo_url', '$logo_dark'
        ) ON DUPLICATE KEY UPDATE
            team_uid = VALUES(team_uid),
            full_display_name = VALUES(full_display_name),
            short_display_name = VALUES(short_display_name),
            location = VALUES(location),
            name = VALUES(name),
            abbreviation = VALUES(abbreviation),
            conference_name = VALUES(conference_name),
            conference_id = VALUES(conference_id),
            overall_wins = VALUES(overall_wins),
            overall_losses = VALUES(overall_losses),
            overall_record = VALUES(overall_record),
            conference_wins = VALUES(conference_wins),
            conference_losses = VALUES(conference_losses),
            conference_record = VALUES(conference_record),
            league_win_percent = VALUES(league_win_percent),
            playoff_seed = VALUES(playoff_seed),
            points_for = VALUES(points_for),
            points_against = VALUES(points_against),
            point_differential = VALUES(point_differential),
            current_streak = VALUES(current_streak),
            home_record = VALUES(home_record),
            away_record = VALUES(away_record),
            logo_url = VALUES(logo_url),
            logo_dark_url = VALUES(logo_dark_url),
            last_updated = NOW()";
        
        if (mysqli_query($conn, $sql)) {
            $affected = mysqli_affected_rows($conn);
            if ($affected > 0) {
                echo "  ✓ $full_name ($abbr) - " . ($affected == 1 ? "inserted" : "updated") . "\n";
            } else {
                echo "  ○ $full_name ($abbr) - no changes\n";
            }
            $teams_processed++;
        } else {
            echo "  ✗ $full_name - ERROR: " . mysqli_error($conn) . "\n";
            $errors[] = "$full_name: " . mysqli_error($conn);
        }
    }
    echo "\n";
}

mysqli_close($conn);

echo "════════════════════════════════════════\n";
echo "IMPORT COMPLETE!\n";
echo "Conferences: $conferences_processed\n";
echo "Teams: $teams_processed\n";
if (count($errors) > 0) {
    echo "Errors: " . count($errors) . "\n";
    foreach ($errors as $error) {
        echo "  - $error\n";
    }
}
echo "════════════════════════════════════════\n";
?>