<?php
require_once '../../config.php';
session_start();

// Create mysqli connection
$conn = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);

if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

mysqli_set_charset($conn, "utf8mb4");

// Helper function to safely get value
function safe_value($value, $default = '') {
    return isset($value) ? $value : $default;
}

// Helper function to find stat by type
function get_stat($stats, $type, $field = 'value', $default = '0') {
    foreach ($stats as $stat) {
        if (isset($stat['type']) && $stat['type'] === $type) {
            return isset($stat[$field]) ? $stat[$field] : $default;
        }
    }
    return $default;
}

// Update function
function updateStandings($conn) {
    $api_url = "https://site.web.api.espn.com/apis/v2/sports/football/college-football/standings?season=2025&seasontype=2&type=0&level=3";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Accept: application/json',
        'User-Agent: Mozilla/5.0'
    ));
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200 || !$response) {
        return ['success' => false, 'message' => "Failed to fetch data. HTTP Code: $http_code"];
    }
    
    $data = json_decode($response, true);
    if (!$data) {
        return ['success' => false, 'message' => "Failed to parse JSON"];
    }
    
    $teams_updated = 0;
    $conferences = isset($data['children']) ? $data['children'] : [$data];
    
    foreach ($conferences as $conference) {
        $conf_name = safe_value($conference['name'], 'Unknown');
        $conf_id = safe_value($conference['id'], '');
        
        if (!isset($conference['standings']['entries'])) continue;
        
        foreach ($conference['standings']['entries'] as $entry) {
            if (!isset($entry['team'])) continue;
            
            $team = $entry['team'];
            $stats = isset($entry['stats']) ? $entry['stats'] : [];
            
            // Get all team data
            $team_id = mysqli_real_escape_string($conn, safe_value($team['id']));
            $team_uid = mysqli_real_escape_string($conn, safe_value($team['uid']));
            $full_name = mysqli_real_escape_string($conn, safe_value($team['displayName'], 'Unknown Team'));
            $short_name = mysqli_real_escape_string($conn, safe_value($team['shortDisplayName'], $full_name));
            $location = mysqli_real_escape_string($conn, safe_value($team['location']));
            $name = mysqli_real_escape_string($conn, safe_value($team['name']));
            $abbr = mysqli_real_escape_string($conn, safe_value($team['abbreviation']));
            
            // Get logos
            $logo_url = '';
            $logo_dark = '';
            if (isset($team['logos'])) {
                foreach ($team['logos'] as $logo) {
                    if (isset($logo['href'])) {
                        if (in_array('default', $logo['rel'] ?? [])) {
                            $logo_url = $logo['href'];
                        }
                        if (in_array('dark', $logo['rel'] ?? [])) {
                            $logo_dark = $logo['href'];
                        }
                    }
                }
            }
            
            // Get all stats
            $overall_record = get_stat($stats, 'total', 'summary', '0-0');
            preg_match('/(\d+)-(\d+)/', $overall_record, $overall_parts);
            $overall_wins = isset($overall_parts[1]) ? $overall_parts[1] : '0';
            $overall_losses = isset($overall_parts[2]) ? $overall_parts[2] : '0';
            
            $conf_record = get_stat($stats, 'vsconf', 'summary', '0-0');
            preg_match('/(\d+)-(\d+)/', $conf_record, $conf_parts);
            $conf_wins = isset($conf_parts[1]) ? $conf_parts[1] : '0';
            $conf_losses = isset($conf_parts[2]) ? $conf_parts[2] : '0';
            
            $league_pct = mysqli_real_escape_string($conn, get_stat($stats, 'leaguewinpercent', 'value', '0.000'));
            $playoff_seed = mysqli_real_escape_string($conn, get_stat($stats, 'playoffseed', 'value', '0'));
            $points_for = mysqli_real_escape_string($conn, get_stat($stats, 'pointsfor', 'value', '0'));
            $points_against = mysqli_real_escape_string($conn, get_stat($stats, 'pointsagainst', 'value', '0'));
            $point_diff = mysqli_real_escape_string($conn, get_stat($stats, 'pointdifferential', 'value', '0'));
            $streak = mysqli_real_escape_string($conn, get_stat($stats, 'streak', 'displayValue', '-'));
            $home_record = mysqli_real_escape_string($conn, get_stat($stats, 'homerecord', 'summary', '0-0'));
            $away_record = mysqli_real_escape_string($conn, get_stat($stats, 'awayrecord', 'summary', '0-0'));
            
            $conf_name = mysqli_real_escape_string($conn, $conf_name);
            $conf_id = mysqli_real_escape_string($conn, $conf_id);
            $logo_url = mysqli_real_escape_string($conn, $logo_url);
            $logo_dark = mysqli_real_escape_string($conn, $logo_dark);
            
            $sql = "INSERT INTO cfb_standings (
                team_id, team_uid, full_display_name, short_display_name,
                location, name, abbreviation, conference_name, conference_id,
                overall_wins, overall_losses, overall_record,
                conference_wins, conference_losses, conference_record,
                league_win_percent, playoff_seed,
                points_for, points_against, point_differential,
                current_streak, home_record, away_record,
                logo_url, logo_dark_url
            ) VALUES (
                '$team_id', '$team_uid', '$full_name', '$short_name',
                '$location', '$name', '$abbr', '$conf_name', '$conf_id',
                '$overall_wins', '$overall_losses', '$overall_record',
                '$conf_wins', '$conf_losses', '$conf_record',
                '$league_pct', '$playoff_seed',
                '$points_for', '$points_against', '$point_diff',
                '$streak', '$home_record', '$away_record',
                '$logo_url', '$logo_dark'
            ) ON DUPLICATE KEY UPDATE
                team_uid = VALUES(team_uid),
                full_display_name = VALUES(full_display_name),
                short_display_name = VALUES(short_display_name),
                location = VALUES(location),
                name = VALUES(name),
                abbreviation = VALUES(abbreviation),
                conference_name = VALUES(conference_name),
                conference_id = VALUES(conference_id),
                overall_wins = VALUES(overall_wins),
                overall_losses = VALUES(overall_losses),
                overall_record = VALUES(overall_record),
                conference_wins = VALUES(conference_wins),
                conference_losses = VALUES(conference_losses),
                conference_record = VALUES(conference_record),
                league_win_percent = VALUES(league_win_percent),
                playoff_seed = VALUES(playoff_seed),
                points_for = VALUES(points_for),
                points_against = VALUES(points_against),
                point_differential = VALUES(point_differential),
                current_streak = VALUES(current_streak),
                home_record = VALUES(home_record),
                away_record = VALUES(away_record),
                logo_url = VALUES(logo_url),
                logo_dark_url = VALUES(logo_dark_url),
                last_updated = NOW()";
            
            if (mysqli_query($conn, $sql)) {
                $teams_updated++;
            }
        }
    }
    
    return ['success' => true, 'teams_updated' => $teams_updated];
}

// Run update if requested
$update_result = null;
if (isset($_GET['update']) && $_GET['update'] == '1') {
    $update_result = updateStandings($conn);
}

// Fetch current data from database
$query = "SELECT * FROM cfb_standings ORDER BY conference_name, CAST(playoff_seed AS UNSIGNED), CAST(overall_wins AS UNSIGNED) DESC";
$result = mysqli_query($conn, $query);
$standings = [];
while ($row = mysqli_fetch_assoc($result)) {
    $standings[] = $row;
}

// Group by conference
$conferences = [];
foreach ($standings as $team) {
    $conf = $team['conference_name'];
    if (!isset($conferences[$conf])) {
        $conferences[$conf] = [];
    }
    $conferences[$conf][] = $team;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CFB Standings Debug Table</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
        }
        
        .header h1 {
            color: #2d3748;
            font-size: 2.5em;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header .subtitle {
            color: #718096;
            font-size: 1.1em;
        }
        
        .controls {
            display: flex;
            gap: 15px;
            margin-top: 20px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        }
        
        .update-status {
            padding: 15px 20px;
            border-radius: 10px;
            margin-top: 20px;
            font-weight: 500;
        }
        
        .update-success {
            background: #c6f6d5;
            color: #22543d;
            border: 1px solid #9ae6b4;
        }
        
        .update-error {
            background: #fed7d7;
            color: #742a2a;
            border: 1px solid #fc8181;
        }
        
        .conference-section {
            background: rgba(255, 255, 255, 0.98);
            border-radius: 20px;
            margin-bottom: 30px;
            overflow: hidden;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
        }
        
        .conference-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 30px;
            font-size: 1.4em;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .table-wrapper {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            background: #f7fafc;
            padding: 15px 10px;
            text-align: left;
            font-weight: 700;
            color: #4a5568;
            font-size: 0.85em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid #e2e8f0;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        td {
            padding: 12px 10px;
            border-bottom: 1px solid #e2e8f0;
            color: #2d3748;
            font-size: 0.95em;
        }
        
        tr:hover {
            background: #f7fafc;
            transition: background 0.2s;
        }
        
        .team-cell {
            display: flex;
            align-items: center;
            gap: 12px;
            font-weight: 600;
        }
        
        .team-logo {
            width: 35px;
            height: 35px;
            object-fit: contain;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.1));
        }
        
        .record-win {
            color: #38a169;
            font-weight: 700;
        }
        
        .record-loss {
            color: #e53e3e;
            font-weight: 700;
        }
        
        .playoff-seed {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 4px 10px;
            border-radius: 20px;
            font-weight: 700;
            font-size: 0.85em;
            min-width: 30px;
            text-align: center;
        }
        
        .seed-0 {
            background: #cbd5e0;
            color: #4a5568;
        }
        
        .win-pct {
            font-weight: 700;
            font-size: 1.05em;
        }
        
        .pct-high { color: #38a169; }
        .pct-med { color: #ed8936; }
        .pct-low { color: #e53e3e; }
        
        .streak {
            padding: 4px 8px;
            border-radius: 6px;
            font-weight: 600;
            font-size: 0.85em;
        }
        
        .streak-win {
            background: #c6f6d5;
            color: #22543d;
        }
        
        .streak-loss {
            background: #fed7d7;
            color: #742a2a;
        }
        
        .stats-cell {
            font-family: 'Monaco', 'Courier New', monospace;
            font-size: 0.9em;
        }
        
        .diff-positive {
            color: #38a169;
            font-weight: 700;
        }
        
        .diff-negative {
            color: #e53e3e;
            font-weight: 700;
        }
        
        @media (max-width: 768px) {
            .header h1 {
                font-size: 1.8em;
            }
            
            th, td {
                padding: 8px 5px;
                font-size: 0.85em;
            }
            
            .team-logo {
                width: 25px;
                height: 25px;
            }
        }
        
        .last-updated {
            text-align: center;
            color: rgba(255, 255, 255, 0.9);
            margin-top: 30px;
            font-size: 0.9em;
        }
        
        .emoji {
            font-size: 1.2em;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>
                <span class="emoji">🏈</span>
                CFB Standings Debug Table
            </h1>
            <div class="subtitle">
                <?php echo count($standings); ?> teams across <?php echo count($conferences); ?> conferences
            </div>
            
            <div class="controls">
                <a href="?update=1" class="btn">
                    🔄 Update Standings
                </a>
                <button class="btn btn-secondary" onclick="location.reload()">
                    🔃 Refresh Page
                </button>
            </div>
            
            <?php if ($update_result): ?>
                <?php if ($update_result['success']): ?>
                    <div class="update-status update-success">
                        ✅ Successfully updated <?php echo $update_result['teams_updated']; ?> teams
                    </div>
                <?php else: ?>
                    <div class="update-status update-error">
                        ❌ Update failed: <?php echo $update_result['message']; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <?php foreach ($conferences as $conf_name => $teams): ?>
        <div class="conference-section">
            <div class="conference-header">
                <?php echo htmlspecialchars($conf_name); ?>
                <span style="float: right; font-size: 0.8em; font-weight: 400;">
                    <?php echo count($teams); ?> teams
                </span>
            </div>
            <div class="table-wrapper">
                <table>
                    <thead>
                        <tr>
                            <th>Seed</th>
                            <th>Team</th>
                            <th>Overall</th>
                            <th>Conference</th>
                            <th>Win %</th>
                            <th>PF</th>
                            <th>PA</th>
                            <th>Diff</th>
                            <th>Home</th>
                            <th>Away</th>
                            <th>Streak</th>
                            <th>Updated</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($teams as $team): ?>
                        <tr>
                            <td>
                                <?php 
                                $seed = $team['playoff_seed'];
                                $seed_class = $seed == '0' ? 'seed-0' : '';
                                ?>
                                <span class="playoff-seed <?php echo $seed_class; ?>">
                                    <?php echo $seed == '0' ? '-' : $seed; ?>
                                </span>
                            </td>
                            <td>
                                <div class="team-cell">
                                    <?php if ($team['logo_url']): ?>
                                        <img src="<?php echo htmlspecialchars($team['logo_url']); ?>" 
                                             alt="<?php echo htmlspecialchars($team['abbreviation']); ?>" 
                                             class="team-logo">
                                    <?php endif; ?>
                                    <div>
                                        <div><?php echo htmlspecialchars($team['full_display_name']); ?></div>
                                        <div style="color: #718096; font-size: 0.85em; font-weight: 400;">
                                            <?php echo htmlspecialchars($team['abbreviation']); ?>
                                        </div>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="record-win"><?php echo $team['overall_wins']; ?></span>-<span class="record-loss"><?php echo $team['overall_losses']; ?></span>
                            </td>
                            <td>
                                <span class="record-win"><?php echo $team['conference_wins']; ?></span>-<span class="record-loss"><?php echo $team['conference_losses']; ?></span>
                            </td>
                            <td>
                                <?php 
                                $pct = floatval($team['league_win_percent']);
                                $pct_class = $pct >= 0.600 ? 'pct-high' : ($pct >= 0.400 ? 'pct-med' : 'pct-low');
                                ?>
                                <span class="win-pct <?php echo $pct_class; ?>">
                                    <?php echo number_format($pct, 3); ?>
                                </span>
                            </td>
                            <td class="stats-cell"><?php echo $team['points_for']; ?></td>
                            <td class="stats-cell"><?php echo $team['points_against']; ?></td>
                            <td>
                                <?php 
                                $diff = intval($team['point_differential']);
                                $diff_class = $diff > 0 ? 'diff-positive' : ($diff < 0 ? 'diff-negative' : '');
                                ?>
                                <span class="<?php echo $diff_class; ?>">
                                    <?php echo $diff > 0 ? '+' . $diff : $diff; ?>
                                </span>
                            </td>
                            <td><?php echo $team['home_record']; ?></td>
                            <td><?php echo $team['away_record']; ?></td>
                            <td>
                                <?php 
                                $streak = $team['current_streak'];
                                $streak_class = '';
                                if (strpos($streak, 'W') !== false) {
                                    $streak_class = 'streak-win';
                                } elseif (strpos($streak, 'L') !== false) {
                                    $streak_class = 'streak-loss';
                                }
                                ?>
                                <span class="streak <?php echo $streak_class; ?>">
                                    <?php echo htmlspecialchars($streak); ?>
                                </span>
                            </td>
                            <td style="color: #718096; font-size: 0.85em;">
                                <?php echo date('m/d H:i', strtotime($team['last_updated'])); ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endforeach; ?>
        
        <div class="last-updated">
            Last database update: <?php 
            $last_update_query = "SELECT MAX(last_updated) as last FROM cfb_standings";
            $last_result = mysqli_query($conn, $last_update_query);
            $last_row = mysqli_fetch_assoc($last_result);
            echo $last_row['last'] ? date('F j, Y g:i A', strtotime($last_row['last'])) : 'Never';
            ?>
        </div>
    </div>
</body>
</html>
<?php
mysqli_close($conn);
?>