<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../../../config.php';
date_default_timezone_set('America/New_York');

/**
 * WNBA Scoreboard Scraper (procedural)
 * Gate:
 *  - UPDATE_ANYWAY=true => force update
 *  - else update only if:
 *      * any game is live, OR
 *      * now ∈ [earliestTip - 10m, latestTip + 5h] AND not all games are final
 *  - if all games are GAME_DONE => skip (unless forced)
 * Updated: 2025-10-23
 */

$UPDATE_ANYWAY = false;
if (isset($_GET['UPDATE_ANYWAY'])) {
    $UPDATE_ANYWAY = strtolower(trim($_GET['UPDATE_ANYWAY'])) === 'true' || $_GET['UPDATE_ANYWAY'] === '1';
} elseif (isset($_POST['UPDATE_ANYWAY'])) {
    $UPDATE_ANYWAY = strtolower(trim($_POST['UPDATE_ANYWAY'])) === 'true' || $_POST['UPDATE_ANYWAY'] === '1';
}

$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

function pad3($n){ return str_pad((string)$n, 3, '0', STR_PAD_LEFT); }

function hexToRgbPadded($hex) {
    $hex = ltrim((string)$hex, '#');
    if (strlen($hex) !== 6) {
        return ['r'=>'255','g'=>'255','b'=>'255','rgb'=>'255,255,255'];
    }
    $r = hexdec(substr($hex,0,2));
    $g = hexdec(substr($hex,2,2));
    $b = hexdec(substr($hex,4,2));
    if ($r === 0 && $g === 0 && $b === 0) { $r = 255; $g = 255; $b = 255; } // avoid pure black
    $rp = pad3($r); $gp = pad3($g); $bp = pad3($b);
    return ['r'=>$rp,'g'=>$gp,'b'=>$bp,'rgb'=>"$rp,$gp,$bp"];
}

// Avoid returning "0-0" when record missing
function extractTeamRecordSoft($competitor) {
    if (isset($competitor['records']) && is_array($competitor['records'])) {
        foreach ($competitor['records'] as $rec) {
            if (!empty($rec['summary'])) return $rec['summary'];
        }
    }
    return '';
}

function curlJson($url, $headers = []) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
    ]);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($err) throw new Exception("cURL Error: $err");
    if ($code !== 200) throw new Exception("HTTP Error: $code");
    $data = json_decode($res, true);
    if (json_last_error() !== JSON_ERROR_NONE) throw new Exception("JSON Decode Error: " . json_last_error_msg());
    return $data;
}

// -------- Fetch ESPN --------
$todayYmd = date('Ymd');
$api_url = "https://site.api.espn.com/apis/site/v2/sports/basketball/wnba/scoreboard?";
$headers = [
    'Accept: application/json',
    'Accept-Language: en-US,en;q=0.9',
    'Cache-Control: no-cache',
    'Pragma: no-cache',
    'Referer: https://www.espn.com/',
    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64)',
];

$debug_queries = [];
$rows_debug = [];
$errors_debug = [];

try {
    $payload = curlJson($api_url, $headers);
} catch (Exception $e) {
    mysqli_close($con);
    die("<div style='background:#fde2e1;border:1px solid #f8b4ad;color:#7f1d1d;padding:12px;border-radius:8px;margin:20px;'>❌ ".$e->getMessage()."</div>");
}

$events = $payload['events'] ?? [];
$loop_length = count($events);

// -------- Determine gate window --------
$now = time();
$any_live = false;
$all_final = true;
$earliestTip = null;
$latestTip = null;

function classify_status($comp) {
    $status = $comp['status'] ?? [];
    $type = $status['type'] ?? [];
    $name = $type['name'] ?? '';
    $period = (int)($status['period'] ?? 0);
    $clock = (string)($status['displayClock'] ?? '0:00');
    $desc = trim((string)($type['description'] ?? ''));

    // Completed final?
    if (!empty($type['completed'])) return 'GAME_DONE';

    switch ($name) {
        case 'STATUS_FINAL':
        case 'STATUS_FINAL_OT':
            return 'GAME_DONE';
        case 'STATUS_IN_PROGRESS':
            if ($period === 2 && $clock === '0:00') return 'HALF_TIME';
            return 'IN_SESSION';
        case 'STATUS_HALFTIME':
            return 'HALF_TIME';
        case 'STATUS_SCHEDULED':
        default:
            // Sometimes state/description says "in"/"post"
            $state = strtolower((string)($type['state'] ?? ''));
            if ($state === 'in') return 'IN_SESSION';
            if ($state === 'post') return 'GAME_DONE';
            if (stripos($desc, 'halftime') !== false) return 'HALF_TIME';
            return 'NOT_STARTED';
    }
}

for ($i=0; $i<$loop_length; $i++) {
    $event = $events[$i];
    $comp = $event['competitions'][0] ?? null;
    if (!$comp) continue;

    $statusText = classify_status($comp);
    if ($statusText !== 'GAME_DONE') $all_final = false;
    if ($statusText === 'IN_SESSION' || $statusText === 'HALF_TIME') $any_live = true;

    // tip (UTC ISO8601)
    $tipIso = $event['date'] ?? ($comp['date'] ?? null);
    if ($tipIso) {
        $tipTs = strtotime($tipIso); // epoch (server tz, but strtotime treats ISO as UTC -> epoch is okay)
        if ($tipTs) {
            if ($earliestTip === null || $tipTs < $earliestTip) $earliestTip = $tipTs;
            if ($latestTip === null || $tipTs > $latestTip) $latestTip = $tipTs;
        }
    }
}

// No games — bail gracefully (and do not update)
if ($loop_length === 0) {
    mysqli_close($con);
    echo "<!doctype html><html><head><meta charset='utf-8'><title>WNBA Scoreboard Update</title></head><body style='font-family:Arial,sans-serif;padding:20px;'>
        <h1>WNBA Scoreboard Update</h1>
        <p>No games returned for today ($todayYmd). Skipping update.</p>
    </body></html>";
    exit;
}

// Compute window
$windowStart = $earliestTip ? $earliestTip - (10 * 60) : null;
$windowEnd   = $latestTip   ? $latestTip   + (5 * 3600) : null;

$reasons = [];
$should_update = false;

if ($UPDATE_ANYWAY) {
    $should_update = true;
    $reasons[] = "UPDATE_ANYWAY=true";
} else {
    if ($any_live) {
        $should_update = true;
        $reasons[] = "At least one game is live";
    } else {
        if ($windowStart !== null && $windowEnd !== null) {
            if ($now >= $windowStart && $now <= $windowEnd) {
                if ($all_final) {
                    $should_update = false;
                    $reasons[] = "Inside time window but all games are final (skip)";
                } else {
                    $should_update = true;
                    $reasons[] = "Within allowed time window";
                }
            } else {
                $reasons[] = "Outside allowed time window";
            }
        } else {
            $reasons[] = "No valid tip times; cannot open window (skip)";
        }
    }
}

// HTML header
echo '<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>WNBA Scoreboard Update</title>
<style>
 body{font-family:Arial,sans-serif;background:#f5f5f5;margin:0;padding:20px;color:#333}
 .container{max-width:1200px;margin:0 auto;background:#fff;border-radius:12px;box-shadow:0 2px 8px rgba(0,0,0,0.1);padding:24px}
 h1{color:#ff6600;margin:0 0 6px 0}
 .timestamp{color:#666;margin-bottom:16px}
 table{width:100%;border-collapse:collapse;margin-top:10px}
 th{background:#ff6600;color:#fff;text-align:left;padding:10px;font-size:12px;text-transform:uppercase;letter-spacing:.03em}
 td{padding:10px;border-bottom:1px solid #eee;vertical-align:top}
 .status{display:inline-block;padding:4px 10px;border-radius:18px;font-weight:700;font-size:12px;color:#fff}
 .live{background:#e53935}.half{background:#fb8c00}.final{background:#43a047}.sched{background:#1e88e5}
 .pill{display:inline-block;background:#eef2ff;color:#3730a3;padding:3px 8px;border-radius:999px;font-size:12px;margin-right:8px}
 .ok{color:#155724;background:#d4edda;border:1px solid #c3e6cb;padding:10px;border-radius:8px;margin:10px 0}
 .err{color:#7f1d1d;background:#fde2e1;border:1px solid #f8b4ad;padding:10px;border-radius:8px;margin:10px 0}
 .muted{color:#666}
 .debug{font-family:monospace;font-size:12px;background:#f7f7f9;border:1px solid #e1e1e8;padding:10px;border-radius:6px;overflow-x:auto}
 .rgb{display:flex;align-items:center;gap:8px}
 .swatch{width:16px;height:16px;border:1px solid #ccc;border-radius:4px}
</style>
</head>
<body>
<div class="container">
<h1>🏀 WNBA Scoreboard Update</h1>
<div class="timestamp">Generated: '.date('l, F j, Y - g:i A').' ET</div>';

echo '<div>';
echo '<span class="pill">Live: '.($any_live?'Yes':'No').'</span>';
echo '<span class="pill">All Final: '.($all_final?'Yes':'No').'</span>';
if ($earliestTip) echo '<span class="pill">First Tip: '.date('g:i A', $earliestTip).' ET</span>';
if ($latestTip)   echo '<span class="pill">Last Tip: '.date('g:i A', $latestTip).' ET</span>';
if ($windowStart && $windowEnd) echo '<span class="pill">Window: '.date('g:i A',$windowStart).' - '.date('g:i A',$windowEnd).'</span>';
echo '<span class="pill">UPDATE_ANYWAY: '.($UPDATE_ANYWAY?'true':'false').'</span>';
echo '</div>';

echo '<div class="'.($should_update?'ok':'err').'">'.($should_update?'Proceeding with update':'Skipping update').': '.htmlspecialchars(implode(' | ', $reasons)).'</div>';

// If we are NOT updating, show a quick table of the slate and exit
if (!$should_update) {
    echo '<table>
        <thead><tr>
            <th>#</th><th>Status</th><th>Away</th><th>Score</th><th>Home</th><th>Score</th><th>Tip/Clock</th>
        </tr></thead><tbody>';
    $gnum=0;
    foreach ($events as $event) {
        $gnum++;
        $comp = $event['competitions'][0] ?? [];
        $status = $comp['status'] ?? [];
        $type = $status['type'] ?? [];
        $period = (int)($status['period'] ?? 0);
        $clock = $status['displayClock'] ?? '0:00';
        $statusText = classify_status($comp);
        $cls = ($statusText==='IN_SESSION'?'live':($statusText==='HALF_TIME'?'half':($statusText==='GAME_DONE'?'final':'sched')));
        $tipIso = $event['date'] ?? ($comp['date'] ?? '');
        $tipStr = $tipIso ? date('g:i a', strtotime($tipIso)) : '-';

        $away = ['abbr'=>'','name'=>'','score'=>0];
        $home = ['abbr'=>'','name'=>'','score'=>0];

        foreach (($comp['competitors'] ?? []) as $c) {
            $t = $c['team'] ?? [];
            $row = [
                'abbr' => $t['abbreviation'] ?? '',
                'name' => trim(($t['location'] ?? '').' '.($t['name'] ?? '')),
                'score'=> $c['score'] ?? '0'
            ];
            if (($c['homeAway'] ?? '') === 'home') $home = $row; else $away = $row;
        }

        echo '<tr>';
        echo '<td>'.$gnum.'</td>';
        echo '<td><span class="status '.$cls.'">'.$statusText.'</span></td>';
        echo '<td>'.htmlspecialchars($away['name']).' ('.htmlspecialchars($away['abbr']).')</td>';
        echo '<td>'.htmlspecialchars($away['score']).'</td>';
        echo '<td>'.htmlspecialchars($home['name']).' ('.htmlspecialchars($home['abbr']).')</td>';
        echo '<td>'.htmlspecialchars($home['score']).'</td>';
        echo '<td>'.($statusText==='IN_SESSION' || $statusText==='HALF_TIME' ? 'P'.$period.' • '.$clock : $tipStr.' ET').'</td>';
        echo '</tr>';
    }
    echo '</tbody></table></div></body></html>';
    mysqli_close($con);
    exit;
}

// -------- Proceed with UPDATE --------
$updated_ok = 0;

echo '<table>
    <thead><tr>
        <th>#</th>
        <th>Status</th>
        <th>Away (Abbr / Rec / RGB)</th>
        <th>Score</th>
        <th>Home (Abbr / Rec / RGB)</th>
        <th>Score</th>
        <th>Game Info</th>
        <th>Betting</th>
        <th>DB</th>
    </tr></thead><tbody>';

for ($i=1; $i <= $loop_length; $i++) {
    $idx = $i-1;
    $event = $events[$idx] ?? null;
    if (!$event) continue;

    $comp = $event['competitions'][0] ?? null;
    if (!$comp) continue;

    $status = $comp['status'] ?? [];
    $type   = $status['type'] ?? [];
    $period = (int)($status['period'] ?? 0);
    $clock  = $status['displayClock'] ?? '0:00';
    $statusText = classify_status($comp);

    $tipIso = $event['date'] ?? ($comp['date'] ?? '');
    $game_date_iso = $tipIso;
    $game_time_et  = $tipIso ? date('g:i a', strtotime($tipIso)) : '';

    // Teams (Team_1 = Away, Team_2 = Home to match your prior NBA style)
    $away = ['team'=>[],'score'=>'0','homeAway'=>'away'];
    $home = ['team'=>[],'score'=>'0','homeAway'=>'home'];

    foreach (($comp['competitors'] ?? []) as $c) {
        $ha = $c['homeAway'] ?? '';
        if ($ha === 'home') $home = $c; else $away = $c;
    }

    $awayTeam = $away['team'] ?? [];
    $homeTeam = $home['team'] ?? [];

    $away_abbr = $awayTeam['abbreviation'] ?? '';
    $home_abbr = $homeTeam['abbreviation'] ?? '';
    $away_loc  = $awayTeam['location'] ?? '';
    $home_loc  = $homeTeam['location'] ?? '';
    $away_name = $awayTeam['name'] ?? '';
    $home_name = $homeTeam['name'] ?? '';
    $away_score = isset($away['score']) ? (string)$away['score'] : '0';
    $home_score = isset($home['score']) ? (string)$home['score'] : '0';

    $away_rec = extractTeamRecordSoft($away);
    $home_rec = extractTeamRecordSoft($home);

    $away_color = hexToRgbPadded($awayTeam['color'] ?? '');
    $home_color = hexToRgbPadded($homeTeam['color'] ?? '');

    // Betting
    $favorite_team = '';
    $favorite_points = '';
    $overUnder = '';
    if (!empty($comp['odds'][0])) {
        $odd = $comp['odds'][0];
        $overUnder = isset($odd['overUnder']) ? (string)$odd['overUnder'] : '';
        $details = $odd['details'] ?? ''; // e.g., "SEA -3.5"
        if ($details) {
            $parts = explode(' ', $details);
            if (count($parts) >= 2) {
                $favorite_team = $parts[0];
                $favorite_points = $parts[1];
            }
        }
    }

    // lastPlay (rare for basketball in this endpoint, but attempt)
    $lastPlayText = '';
    if (!empty($comp['situation']['lastPlay']['text'])) {
        $lastPlayText = $comp['situation']['lastPlay']['text'];
    } elseif (!empty($comp['situation']['lastPlay']['shortText'])) {
        $lastPlayText = $comp['situation']['lastPlay']['shortText'];
    }

    // Build UPDATE
    $Clock = mysqli_real_escape_string($con, $clock);
    $Period = (int)$period;
    $Game_Played = mysqli_real_escape_string($con, $statusText);
    $Game_Time = mysqli_real_escape_string($con, $game_time_et);
    $Game_Date = mysqli_real_escape_string($con, $game_date_iso);
    $scoreValue = 'NO-SCORE';

    $Team_1_Name = mysqli_real_escape_string($con, $away_name);
    $Team_2_Name = mysqli_real_escape_string($con, $home_name);
    $Team_1_Record = mysqli_real_escape_string($con, $away_rec);
    $Team_2_Record = mysqli_real_escape_string($con, $home_rec);
    $Team_1_Score = (int)$away_score;
    $Team_2_Score = (int)$home_score;
    $Team_1_Location = mysqli_real_escape_string($con, $away_loc);
    $Team_2_Location = mysqli_real_escape_string($con, $home_loc);
    $Team_1_Abbrev = mysqli_real_escape_string($con, $away_abbr);
    $Team_2_Abbrev = mysqli_real_escape_string($con, $home_abbr);

    $Team_1_Color = mysqli_real_escape_string($con, $away_color['rgb']);
    $Team_1_Color_R = mysqli_real_escape_string($con, $away_color['r']);
    $Team_1_Color_G = mysqli_real_escape_string($con, $away_color['g']);
    $Team_1_Color_B = mysqli_real_escape_string($con, $away_color['b']);
    $Team_2_Color = mysqli_real_escape_string($con, $home_color['rgb']);
    $Team_2_Color_R = mysqli_real_escape_string($con, $home_color['r']);
    $Team_2_Color_G = mysqli_real_escape_string($con, $home_color['g']);
    $Team_2_Color_B = mysqli_real_escape_string($con, $home_color['b']);

    $Favorite_Team = mysqli_real_escape_string($con, $favorite_team);
    $Favorite_Points = mysqli_real_escape_string($con, $favorite_points);
    $OverUnder = mysqli_real_escape_string($con, $overUnder);
    $lastPlay = mysqli_real_escape_string($con, $lastPlayText);

    $Who_Has_Ball = ''; // not used for basketball
    $Distance = '';
    $Possesion = '';
    $Red_Zone = '';

    $query = "UPDATE sports_list SET 
        Clock = '$Clock',
        Period = '$Period',
        Game_Played = '$Game_Played',
        Game_Time = '$Game_Time',
        Game_Date = '$Game_Date',
        scoreValue = '$scoreValue',
        Team_1_Name = '$Team_1_Name',
        Team_2_Name = '$Team_2_Name',
        Team_1_Record = '$Team_1_Record',
        Team_2_Record = '$Team_2_Record',
        Team_1_Score = '$Team_1_Score',
        Team_2_Score = '$Team_2_Score',
        Team_1_Location = '$Team_1_Location',
        Team_2_Location = '$Team_2_Location',
        Team_1_Abbrev = '$Team_1_Abbrev',
        Team_1_Color = '$Team_1_Color',
        Team_1_Color_R = '$Team_1_Color_R',
        Team_1_Color_G = '$Team_1_Color_G',
        Team_1_Color_B = '$Team_1_Color_B',
        Team_2_Abbrev = '$Team_2_Abbrev',
        Team_2_Color = '$Team_2_Color',
        Team_2_Color_R = '$Team_2_Color_R',
        Team_2_Color_G = '$Team_2_Color_G',
        Team_2_Color_B = '$Team_2_Color_B',
        Who_Has_Ball = '$Who_Has_Ball',
        Distance = '$Distance',
        Possesion = '$Possesion',
        Favorite_Team = '$Favorite_Team',
        Favorite_Points = '$Favorite_Points',
        lastPlay = '$lastPlay',
        OverUnder = '$OverUnder',
        Red_Zone = '$Red_Zone'
    WHERE Game = $i AND Conference = 'WNBA'";

    $debug_queries[] = $query;

    // Output row visual
    $cls = ($statusText==='IN_SESSION'?'live':($statusText==='HALF_TIME'?'half':($statusText==='GAME_DONE'?'final':'sched')));
    echo '<tr>';
    echo '<td>'.$i.'</td>';
    echo '<td><span class="status '.$cls.'">'.$statusText.'</span></td>';
    echo '<td>'.htmlspecialchars($away_loc.' '.$away_name).' ('.htmlspecialchars($away_abbr).')'
        .'<div class="muted">'.($away_rec!=='' ? htmlspecialchars($away_rec) : '<em>No record</em>').'</div>'
        .'<div class="rgb"><span class="swatch" style="background:rgb('.$away_color['rgb'].')"></span><span class="muted">'.$away_color['rgb'].'</span></div>'
        .'</td>';
    echo '<td>'.htmlspecialchars($away_score).'</td>';
    echo '<td>'.htmlspecialchars($home_loc.' '.$home_name).' ('.htmlspecialchars($home_abbr).')'
        .'<div class="muted">'.($home_rec!=='' ? htmlspecialchars($home_rec) : '<em>No record</em>').'</div>'
        .'<div class="rgb"><span class="swatch" style="background:rgb('.$home_color['rgb'].')"></span><span class="muted">'.$home_color['rgb'].'</span></div>'
        .'</td>';
    echo '<td>'.htmlspecialchars($home_score).'</td>';
    echo '<td>';
    if ($statusText==='IN_SESSION' || $statusText==='HALF_TIME') {
        echo 'P'.$period.' • '.$clock;
    } else {
        echo ($game_time_et ?: '-') . ' ET';
    }
    if ($lastPlayText) echo '<div class="muted" style="margin-top:6px;"><em>📝 '.htmlspecialchars($lastPlayText).'</em></div>';
    echo '</td>';

    echo '<td>';
    if ($favorite_team !== '' || $overUnder !== '') {
        echo '<div class="muted">Line: '.htmlspecialchars(trim($favorite_team.' '.$favorite_points)).'</div>';
        if ($overUnder !== '') echo '<div class="muted">O/U: '.htmlspecialchars($overUnder).'</div>';
    } else {
        echo '<span class="muted">-</span>';
    }
    echo '</td>';

    $ok = mysqli_query($con, $query);
    if ($ok) {
        $updated_ok++;
        echo '<td class="muted">✅</td>';
    } else {
        $err = mysqli_error($con);
        $errors_debug[] = "Game $i: $err";
        echo '<td class="muted" style="color:#b00020;">❌ '.htmlspecialchars($err).'</td>';
    }
    echo '</tr>';
}

echo '</tbody></table>';

// Update TIME_UPDATED only if we updated rows
if ($updated_ok > 0) {
    $time_updated = date('m/d/Y | g:iA');
    $sql = "UPDATE `API_DB` SET `TIME_UPDATED` = '$time_updated' WHERE `API` = 'WNBA'";
    if (mysqli_query($con, $sql)) {
        echo '<div class="ok">Updated API_DB.TIME_UPDATED = '.$time_updated.'</div>';
    } else {
        echo '<div class="err">Error updating API_DB: '.htmlspecialchars(mysqli_error($con)).'</div>';
    }

    // Call output script
    $additional_url = SERVER_URL . "Scrolling/App_Output/API/Games/WNBA_Games.php";
    $ch2 = curl_init();
    curl_setopt_array($ch2, [
        CURLOPT_URL => $additional_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 20,
    ]);
    $resp2 = curl_exec($ch2);
    if (curl_errno($ch2)) {
        echo '<div class="err">Output Script Error: '.htmlspecialchars(curl_error($ch2)).'</div>';
    } else {
        echo '<div class="ok">Output Script Response:</div>';
        echo '<div class="debug">'.htmlspecialchars($resp2).'</div>';
    }
    curl_close($ch2);
} else {
    echo '<div class="err">No rows updated.</div>';
}

// Show debug queries compactly
echo '<h3 style="margin-top:20px;">SQL Debug ('.count($debug_queries).')</h3>';
echo '<div class="debug">'.htmlspecialchars(implode(";\n\n", $debug_queries))."</div>";

if (!empty($errors_debug)) {
    echo '<h3>Errors</h3><div class="debug">'.htmlspecialchars(implode("\n", $errors_debug)).'</div>';
}

echo '</div></body></html>';

mysqli_close($con);
