<?php
/**
 * NCAA Baseball Updater (ESPN)
 * - Gate: Live, ≤10m to first pitch, within 5h after last first pitch
 * - Skip if ALL games are GAME_DONE (unless UPDATE_ANYWAY=true)
 * - Writes to sports_list (Conference='NCAABase') using slot = sequential game #
 * - Renders a neat HTML table showing what was written
 *
 * Updated: 2025-10-23
 */

require_once '../../../config.php'; // Include configuration (DB creds, SERVER_URL, etc.)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ------------------------- Setup -------------------------
date_default_timezone_set('America/New_York');
$time_updated       = date('m/d/Y | g:iA');
$current_timestamp  = time();
$UPDATE_ANYWAY      = isset($_GET['UPDATE_ANYWAY']) && strtolower($_GET['UPDATE_ANYWAY']) === 'true';

// DB connect
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) die("Connection failed: " . mysqli_connect_error());

// Bump API_DB time marker (NCAABase)
mysqli_query($con, "UPDATE `API_DB` SET `TIME_UPDATED` = '" . mysqli_real_escape_string($con, $time_updated) . "' WHERE `API` = 'NCAABase'");

// ------------------------- Helpers -------------------------
function esc($con, $s) { return mysqli_real_escape_string($con, (string)$s); }
function addZeros($v){ return str_pad((string)$v, 3, '0', STR_PAD_LEFT); }

// hex like "1a2b3c" -> "r,g,b" with leading zeros as strings; force white if pure 0,0,0
function colorHexToRgbTriplet($hex){
    $hex = preg_replace('/[^0-9a-fA-F]/','', $hex ?? '');
    if (strlen($hex) !== 6) return ['255','255','255'];
    $r = hexdec(substr($hex,0,2));
    $g = hexdec(substr($hex,2,2));
    $b = hexdec(substr($hex,4,2));
    if ($r==0 && $g==0 && $b==0) { $r=$g=$b=255; }
    return [addZeros($r), addZeros($g), addZeros($b)];
}

// Extract a readable record string or '' if absent
function extractTeamRecordFromCompetitor(array $competitor): string {
    if (!empty($competitor['records']) && is_array($competitor['records'])) {
        foreach ($competitor['records'] as $rec) {
            if (!empty($rec['summary'])) return $rec['summary'];
        }
    }
    if (!empty($competitor['record'])) return $competitor['record'];
    return '';
}

// Soft map ESPN status into our statuses
function mapStatus(array $statusType): string {
    $name  = $statusType['name']        ?? '';
    $state = $statusType['state']       ?? '';
    $desc  = $statusType['description'] ?? '';

    if ($state === 'in' || $name === 'STATUS_IN_PROGRESS' || stripos($desc,'In Progress') !== false) return 'IN_SESSION';
    if ($state === 'post' || $name === 'STATUS_FINAL') return 'GAME_DONE';
    if ($name === 'STATUS_HALFTIME') return 'HALF_TIME';
    if ($name === 'STATUS_POSTPONED') return 'POSTPONED';
    if ($name === 'STATUS_CANCELED') return 'CANCELED';
    return 'NOT_STARTED';
}

// ------------------------- Fetch ESPN -------------------------
$today = date('Ymd'); // you can change this to a specific YYYYMMDD if needed
$api_call = "https://site.api.espn.com/apis/site/v2/sports/baseball/college-baseball/scoreboard";

$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => $api_call,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => "GET",
]);
$result = curl_exec($curl);
$err    = curl_error($curl);
curl_close($curl);
if ($err) die("cURL Error: " . htmlspecialchars($err));

$data = json_decode($result, true);
$events = (is_array($data) && !empty($data['events']) && is_array($data['events'])) ? $data['events'] : [];

// ------------------------- Gate: decide if we should update -------------------------
$any_starting_soon = false;  // within 10 minutes
$any_live          = false;  // in progress
$all_done          = !empty($events); // assume all done, disprove
$first_pitch_ts    = null;
$last_scheduled_ts = null;

foreach ($events as $event) {
    $comp    = $event['competitions'][0] ?? [];
    $status  = $comp['status'] ?? [];
    $stype   = $status['type'] ?? [];
    $state   = $stype['state'] ?? ''; // 'pre' | 'in' | 'post'
    $dateISO = $event['date'] ?? '';
    $ts      = $dateISO ? strtotime($dateISO) : null;

    if ($ts !== null) {
        if ($first_pitch_ts === null || $ts < $first_pitch_ts) $first_pitch_ts = $ts;
        if ($last_scheduled_ts === null || $ts > $last_scheduled_ts) $last_scheduled_ts = $ts;
    }

    if ($state !== 'post') $all_done = false;

    if ($state === 'in') {
        $any_live = true;
    } elseif ($state === 'pre' && $ts !== null) {
        $delta = $ts - $current_timestamp;
        if ($delta <= 600 && $delta >= 0) $any_starting_soon = true; // within 10 minutes
    }
}

$reasons = [];
$should_update = false;

if ($UPDATE_ANYWAY) {
    $should_update = true;
    $reasons[] = "Forced (UPDATE_ANYWAY=true)";
} else {
    if ($any_live) {
        $should_update = true;
        $reasons[] = "Live";
    } elseif ($any_starting_soon) {
        $should_update = true;
        $reasons[] = "Starting within 10 minutes";
    } elseif ($all_done) {
        // explicitly skip when all games are final
        $should_update = false;
        $reasons[] = "All games done — skipping";
    } else {
        $WINDOW_AFTER_LAST_FIRST_PITCH = 5 * 3600; // 5 hours
        if ($last_scheduled_ts !== null && $current_timestamp <= ($last_scheduled_ts + $WINDOW_AFTER_LAST_FIRST_PITCH)) {
            $should_update = true;
            $reasons[] = "Within 5h of last first pitch";
        } else {
            $should_update = false;
            $reasons[] = "No triggers (not live, not starting soon, outside 5h window)";
        }
    }
}

// If we shouldn't update, render a light HTML notice and exit
if (!$should_update) {
    ?>
    <!doctype html>
    <html>
    <head>
        <meta charset="utf-8" />
        <title>NCAA Baseball — No Update</title>
        <style>
            body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;background:#f6f7fb;margin:0;padding:40px}
            .card{max-width:720px;margin:0 auto;background:#fff;border-radius:16px;padding:28px;box-shadow:0 10px 30px rgba(0,0,0,.08)}
            h1{margin:0 0 8px 0;font-size:22px}
            .pill{display:inline-block;font-size:12px;padding:4px 10px;border-radius:999px;background:#eee;color:#333;margin-right:8px}
            ul{margin:12px 0 0 18px;color:#555}
            .muted{color:#6b7280;font-size:13px;margin-top:10px}
        </style>
    </head>
    <body>
        <div class="card">
            <h1>⚾ NCAA Baseball — No Database Update</h1>
            <div>
                <?php foreach ($reasons as $r): ?>
                    <span class="pill"><?php echo htmlspecialchars($r); ?></span>
                <?php endforeach; ?>
            </div>
            <p class="muted">Last check: <?php echo htmlspecialchars($time_updated); ?> Eastern</p>
        </div>
    </body>
    </html>
    <?php
    mysqli_close($con);
    exit;
}

// ------------------------- Process + Update DB -------------------------
$written_rows = []; // for table rendering
$slot = 0;

foreach ($events as $event) {
    $slot++; // Game slot (1..N)
    $comp          = $event['competitions'][0] ?? [];
    $status        = $comp['status'] ?? [];
    $statusType    = $status['type'] ?? [];
    $mappedStatus  = mapStatus($statusType);
    $period        = (int)($status['period'] ?? 0); // inning number if available
    $displayClock  = (string)($status['displayClock'] ?? '0:00');
    $isoDate       = $event['date'] ?? '';
    $game_time_e   = $isoDate ? date("g:i a", strtotime($isoDate)) : '';
    $competitors   = $comp['competitors'] ?? [];

    // Defaults
    $home = [
        'id'=>'','name'=>'','location'=>'','abbrev'=>'','record'=>'','score'=>0,
        'rgb'=>['255','255','255']
    ];
    $away = [
        'id'=>'','name'=>'','location'=>'','abbrev'=>'','record'=>'','score'=>0,
        'rgb'=>['255','255','255']
    ];

    foreach ($competitors as $c) {
        $team = $c['team'] ?? [];
        $isHome = ($c['homeAway'] ?? '') === 'home';
        $t = [
            'id'       => $team['id'] ?? '',
            'name'     => $team['name'] ?? '',
            'location' => $team['location'] ?? '',
            'abbrev'   => $team['abbreviation'] ?? '',
            'record'   => extractTeamRecordFromCompetitor($c),
            'score'    => isset($c['score']) ? (int)$c['score'] : 0,
            'rgb'      => colorHexToRgbTriplet($team['color'] ?? '')
        ];
        if ($isHome) $home = $t; else $away = $t;
    }

    // Odds (favorite / O/U)
    $favorite_team = '';
    $favorite_points = '';
    $overUnder = '';
    if (!empty($comp['odds'][0])) {
        $od = $comp['odds'][0];
        if (!empty($od['awayTeamOdds']['favorite'])) $favorite_team = $away['abbrev'];
        if (!empty($od['homeTeamOdds']['favorite'])) $favorite_team = $home['abbrev'];
        if (!empty($od['details'])) {
            $d = $od['details'];
            if (strpos($d, $home['abbrev']) !== false) {
                if (preg_match('/'.$home['abbrev'].'\s+([\-+]?\d+(\.\d+)?)/', $d, $m)) $favorite_points = $m[1];
            } elseif (strpos($d, $away['abbrev']) !== false) {
                if (preg_match('/'.$away['abbrev'].'\s+([\-+]?\d+(\.\d+)?)/', $d, $m)) $favorite_points = $m[1];
            }
        }
        if (isset($od['overUnder'])) $overUnder = (float)$od['overUnder'];
    }

    // Situation (bases/outs, who's batting)
    $who_has_ball = '';
    $outsB = 0;
    $onBases = '0,0,0';
    $lastPlay = '';
    $score_value = 'NO-SCORE';

    $shortDetail = (string)($statusType['shortDetail'] ?? '');
    // crude parse to decide who bats
    if (stripos($shortDetail, 'Top') !== false)      $who_has_ball = $away['abbrev'];
    elseif (stripos($shortDetail, 'Bot') !== false)  $who_has_ball = $home['abbrev'];
    elseif (stripos($shortDetail, 'Middle') !== false) $who_has_ball = 'mid';
    elseif (stripos($shortDetail, 'End') !== false)    $who_has_ball = 'end';

    $situation = $comp['situation'] ?? [];
    if (!empty($situation)) {
        $onFirst  = !empty($situation['onFirst'])  ? '1':'0';
        $onSecond = !empty($situation['onSecond']) ? '1':'0';
        $onThird  = !empty($situation['onThird'])  ? '1':'0';
        $onBases = $onFirst . ',' . $onSecond . ',' . $onThird;
        $outsB   = isset($situation['outs']) ? (int)$situation['outs'] : 0;

        if (!empty($situation['lastPlay']['text'])) {
            $lastPlay = $situation['lastPlay']['text'];
        }
        $sv = isset($situation['lastPlay']['scoreValue']) ? (int)$situation['lastPlay']['scoreValue'] : 0;
        if ($sv > 0) {
            $score_value = 'R-' . $sv;
            if (stripos($lastPlay, 'home run') !== false) $score_value = 'H-' . $sv;
        }
    }

    // Team “H,E” if available in competitor (non-standard; keep blank if absent)
    $team1HE = ''; $team2HE = '';
    if (!empty($competitors)) {
        foreach ($competitors as $c) {
            $hits   = isset($c['hits']) ? (int)$c['hits'] : null;
            $errors = isset($c['errors']) ? (int)$c['errors'] : null;
            if ($hits !== null && $errors !== null) {
                if (($c['homeAway'] ?? '') === 'home') $team1HE = $hits . ',' . $errors;
                if (($c['homeAway'] ?? '') === 'away') $team2HE = $hits . ',' . $errors;
            }
        }
    }

    // Keep abbreviations tidy
    if (strlen($home['abbrev']) > 10) $home['abbrev'] = substr($home['abbrev'], 0, 3);
    if (strlen($away['abbrev']) > 10) $away['abbrev'] = substr($away['abbrev'], 0, 3);

    // Build UPDATE for sports_list row by slot
    $rgb1 = implode(',', $home['rgb']);
    $rgb2 = implode(',', $away['rgb']);

    $q = "
        UPDATE sports_list SET
            Clock = '".esc($con, $displayClock)."',
            Period = '".intval($period)."',
            Game_Played = '".esc($con, $mappedStatus)."',
            Game_Time = '".esc($con, $game_time_e)."',
            Game_Date = '".esc($con, $isoDate)."',
            scoreValue = '".esc($con, $score_value)."',

            Team_1_Name = '".esc($con, $home['name'])."',
            Team_1_Location = '".esc($con, $home['location'])."',
            Team_1_Abbrev = '".esc($con, $home['abbrev'])."',
            Team_1_Record = '".esc($con, $home['record'])."',
            Team_1_Score = '".intval($home['score'])."',
            Team_1_Color = '".esc($con, $rgb1)."',
            Team_1_Color_R = '".esc($con, $home['rgb'][0])."',
            Team_1_Color_G = '".esc($con, $home['rgb'][1])."',
            Team_1_Color_B = '".esc($con, $home['rgb'][2])."',

            Team_2_Name = '".esc($con, $away['name'])."',
            Team_2_Location = '".esc($con, $away['location'])."',
            Team_2_Abbrev = '".esc($con, $away['abbrev'])."',
            Team_2_Record = '".esc($con, $away['record'])."',
            Team_2_Score = '".intval($away['score'])."',
            Team_2_Color = '".esc($con, $rgb2)."',
            Team_2_Color_R = '".esc($con, $away['rgb'][0])."',
            Team_2_Color_G = '".esc($con, $away['rgb'][1])."',
            Team_2_Color_B = '".esc($con, $away['rgb'][2])."',

            Who_Has_Ball = '".esc($con, $who_has_ball)."',
            Distance = '',
            Possesion = '',
            Favorite_Team = '".esc($con, $favorite_team)."',
            Favorite_Points = '".esc($con, $favorite_points)."',
            lastPlay = '".esc($con, $lastPlay)."',
            OverUnder = '".esc($con, $overUnder)."',
            Red_Zone = '',
            outsB = '".intval($outsB)."',
            onBases = '".esc($con, $onBases)."',
            team1HE = '".esc($con, $team1HE)."',
            team2HE = '".esc($con, $team2HE)."',
            lastUpdatedTime = '".date('Y-m-d H:i:s')."'
        WHERE Game = ".intval($slot)." AND Conference = 'NCAABase'
    ";

    $ok = mysqli_query($con, $q);
    $written_rows[] = [
        'slot' => $slot,
        'ok'   => (bool)$ok,
        'err'  => $ok ? '' : mysqli_error($con),

        'status' => $mappedStatus,
        'clock'  => $displayClock,
        'period' => $period,
        'time'   => $game_time_e,
        'date'   => $isoDate,

        'away_abbrev' => $away['abbrev'],
        'away_name'   => $away['name'],
        'away_rec'    => $away['record'],
        'away_score'  => $away['score'],
        'away_rgb'    => $rgb2,

        'home_abbrev' => $home['abbrev'],
        'home_name'   => $home['name'],
        'home_rec'    => $home['record'],
        'home_score'  => $home['score'],
        'home_rgb'    => $rgb1,

        'fav_team'  => $favorite_team,
        'fav_pts'   => $favorite_points,
        'ou'        => $overUnder,

        'who'       => $who_has_ball,
        'lastPlay'  => $lastPlay,
        'score_val' => $score_value,
        'outs'      => $outsB,
        'bases'     => $onBases,
        'h1e'       => $team1HE,
        'h2e'       => $team2HE,
    ];
}

// Optional: hit your downstream generator (like your old script did)
if (defined('SERVER_URL')) {
    $additional_url = SERVER_URL . "Scrolling/App_Output/API/Games/NCAABASE_Games.php";
    $curl2 = curl_init();
    curl_setopt_array($curl2, [
        CURLOPT_URL => $additional_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 15,
    ]);
    $downstream_response = curl_exec($curl2);
    if (curl_errno($curl2)) {
        $downstream_response = "cURL Error: " . curl_error($curl2);
    }
    curl_close($curl2);
} else {
    $downstream_response = "SERVER_URL not defined.";
}

mysqli_close($con);

// ------------------------- Render Neat HTML Table -------------------------
?>
<!doctype html>
<html>
<head>
<meta charset="utf-8" />
<title>NCAA Baseball — Database Update</title>
<style>
    :root{
        --bg:#f6f7fb;--card:#fff;--text:#111827;--muted:#6b7280;
        --ok:#16a34a;--bad:#dc2626;--chip:#eef2ff;--blue:#2563eb;
        --head:#111827;
    }
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;background:var(--bg);margin:0;padding:32px}
    .wrap{max-width:1200px;margin:0 auto}
    h1{margin:0 0 4px 0;font-size:24px;color:var(--head)}
    .muted{color:var(--muted);font-size:13px}
    .card{background:var(--card);border-radius:16px;box-shadow:0 10px 30px rgba(0,0,0,.07);padding:18px;margin-top:18px}
    table{width:100%;border-collapse:collapse;font-size:13px}
    th,td{padding:10px 8px;border-bottom:1px solid #eee;vertical-align:middle}
    th{position:sticky;top:0;background:#fafafa;text-align:left;font-weight:700;font-size:12px;text-transform:uppercase;letter-spacing:.04em}
    tr:hover{background:#f9fbff}
    .chip{display:inline-block;padding:3px 8px;border-radius:999px;background:var(--chip);font-size:12px}
    .ok{color:var(--ok)} .bad{color:var(--bad)}
    .status{font-weight:700}
    .rgbdot{width:18px;height:18px;border-radius:4px;border:1px solid #ddd;display:inline-block;margin-right:6px;vertical-align:middle}
    .pill{display:inline-block;margin-right:8px;background:#eef2ff;color:#3730a3;font-size:12px;padding:4px 8px;border-radius:999px}
    .footer{margin-top:14px;color:var(--muted);font-size:12px}
    .flex{display:flex;gap:8px;align-items:center}
</style>
</head>
<body>
<div class="wrap">
    <h1>⚾ NCAA Baseball — Database Update</h1>
    <div class="muted">
        Last Updated: <?php echo htmlspecialchars($time_updated); ?> Eastern
        <?php foreach ($reasons as $r): ?>
            <span class="pill"><?php echo htmlspecialchars($r); ?></span>
        <?php endforeach; ?>
        <?php if ($UPDATE_ANYWAY): ?>
            <span class="pill">Forced</span>
        <?php endif; ?>
    </div>

    <div class="card">
        <table>
            <thead>
            <tr>
                <th>#</th>
                <th>OK</th>
                <th>Status</th>
                <th>Time</th>
                <th>Inning</th>
                <th>Away</th>
                <th>Score</th>
                <th>Away Rec</th>
                <th>Home</th>
                <th>Score</th>
                <th>Home Rec</th>
                <th>Fav</th>
                <th>Line</th>
                <th>O/U</th>
                <th>Bat</th>
                <th>Outs</th>
                <th>Bases</th>
                <th>Last Play</th>
                <th>Score Val</th>
                <th>Home H,E</th>
                <th>Away H,E</th>
                <th>Home RGB</th>
                <th>Away RGB</th>
            </tr>
            </thead>
            <tbody>
            <?php foreach ($written_rows as $row): ?>
                <tr>
                    <td><?php echo (int)$row['slot']; ?></td>
                    <td class="<?php echo $row['ok'] ? 'ok' : 'bad'; ?>">
                        <?php echo $row['ok'] ? '✔' : '✖'; ?>
                        <?php if(!$row['ok']) echo '<div class="muted">'.htmlspecialchars($row['err']).'</div>'; ?>
                    </td>
                    <td class="status"><?php echo htmlspecialchars($row['status']); ?></td>
                    <td><?php echo htmlspecialchars($row['time']); ?></td>
                    <td><?php echo (int)$row['period']; ?></td>

                    <td>
                        <strong><?php echo htmlspecialchars($row['away_abbrev']); ?></strong>
                        <div class="muted"><?php echo htmlspecialchars($row['away_name']); ?></div>
                    </td>
                    <td><strong><?php echo (int)$row['away_score']; ?></strong></td>
                    <td><?php echo htmlspecialchars($row['away_rec']); ?></td>

                    <td>
                        <strong><?php echo htmlspecialchars($row['home_abbrev']); ?></strong>
                        <div class="muted"><?php echo htmlspecialchars($row['home_name']); ?></div>
                    </td>
                    <td><strong><?php echo (int)$row['home_score']; ?></strong></td>
                    <td><?php echo htmlspecialchars($row['home_rec']); ?></td>

                    <td><?php echo htmlspecialchars($row['fav_team'] ?: '-'); ?></td>
                    <td><?php echo htmlspecialchars($row['fav_pts'] ?: '-'); ?></td>
                    <td><?php echo htmlspecialchars($row['ou'] !== '' ? $row['ou'] : '-'); ?></td>
                    <td><?php echo htmlspecialchars($row['who'] ?: '-'); ?></td>
                    <td><?php echo (int)$row['outs']; ?></td>
                    <td><?php echo htmlspecialchars($row['bases']); ?></td>
                    <td title="<?php echo htmlspecialchars($row['lastPlay']); ?>">
                        <?php
                        $lp = $row['lastPlay'];
                        echo htmlspecialchars(mb_strlen($lp) > 60 ? (mb_substr($lp,0,60).'…') : $lp);
                        ?>
                    </td>
                    <td><?php echo htmlspecialchars($row['score_val']); ?></td>
                    <td><?php echo htmlspecialchars($row['h1e'] ?: '-'); ?></td>
                    <td><?php echo htmlspecialchars($row['h2e'] ?: '-'); ?></td>

                    <td>
                        <span class="rgbdot" style="background: rgb(<?php echo htmlspecialchars($row['home_rgb']); ?>)"></span>
                        <span class="muted"><?php echo htmlspecialchars($row['home_rgb']); ?></span>
                    </td>
                    <td>
                        <span class="rgbdot" style="background: rgb(<?php echo htmlspecialchars($row['away_rgb']); ?>)"></span>
                        <span class="muted"><?php echo htmlspecialchars($row['away_rgb']); ?></span>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        <div class="footer">
            Wrote <?php echo count($written_rows); ?> row(s) to <strong>sports_list</strong> (Conference: <strong>NCAABase</strong>).
        </div>
    </div>

    <div class="card">
        <div class="flex">
            <div>Downstream generator response:</div>
            <code class="muted"><?php echo htmlspecialchars(mb_substr($downstream_response,0,300)); echo (strlen($downstream_response)>300?'…':''); ?></code>
        </div>
    </div>
</div>
</body>
</html>
